#############################################################################
##
#A  vcyc.g               CHEVIE library         Meinolf Geck, Frank L\"ubeck, 
#A                                               Jean Michel, G\"otz Pfeiffer
##
#A  $Id: vcyc.g,v 1.1 1997/01/21 13:46:36 gap Exp $
##
#Y  Copyright (C) 1992 - 1996  Lehrstuhl D f\"ur Mathematik, RWTH Aachen, IWR
#Y  der Universit\"at Heidelberg, University of St. Andrews, and   University 
#Y  Paris VII.
##
##  This file contains functions dealing with polynomials represented  as 
##  'vcyc', which I use in various files of the chevie package when possible.
##  A polynomial can be represented as a vcyc if it is of the form:
##
##              c \times q^v \times \prod_{n>0} Phi_n(q)^{a_n}
##
##  where c is some constant and Phi_n is the nth cyclotomic polynomial. It
##  is then represented as a record (vcyc:=vec,coeff:=c,operations=vcycOps)
##  where vec is a list where vec[1]=v and vec[n+1]=a_n for n>0.
##
##
##  The advantage of representing polynomials which can be as vcyc should be
##  obvious: less storage, faster multiplication and division, faster
##  evaluation, and not least trivial to print as product of cyclotomic
##  polynomials. The technique can also represent rational fractions whose
##  numerators and denominators are products of cyclotomic polynomials.
##
##  Of course these functions might be re-used/modified by Frank to serve
##  in other parts of the Chevie Package.
##       Jean 13-9-96
##  
#############################################################################

vcycOps:= OperationsRecord("vcyc");

vcycOps.\*:=function(a,b)local res;
  if b=1 then return a;fi;
  if Length(a.vcyc)>Length(b.vcyc) then 
    res:=Copy(a);
    res.vcyc{[1..Length(b.vcyc)]}:=res.vcyc{[1..Length(b.vcyc)]}+b.vcyc;
  else 
    res:=Copy(b);
    res.vcyc{[1..Length(a.vcyc)]}:=res.vcyc{[1..Length(a.vcyc)]}+a.vcyc;
  fi;
  res.coeff:=a.coeff*b.coeff;
  return res;
end;

vcycOps.\/:=function(a,b)local res;
  if b=1 then return a;fi;
  if Length(a.vcyc)>Length(b.vcyc) then 
    res:=Copy(a);
    res.vcyc{[1..Length(b.vcyc)]}:=res.vcyc{[1..Length(b.vcyc)]}-b.vcyc;
  else 
    res:=Copy(b);
    res.vcyc:=-res.vcyc;
    res.vcyc{[1..Length(a.vcyc)]}:=res.vcyc{[1..Length(a.vcyc)]}+a.vcyc;
  fi;
  res.coeff:=a.coeff/b.coeff;
  return res;
end;

vcycOps.Value:=function(v,q)local res,i;
  res:=q^v.vcyc[1]*v.coeff;
  for i in [2..Length(v.vcyc)] do
    if v.vcyc[i]<>0 then 
      res:=res*FastValue([CyclotomicPol(i-1),0],q)^v.vcyc[i];
    fi;
  od;
  return res;
end;

# The following accepts two forms:
#  arg=[v,c] returns a vcyc with vcyc:=v and coeff:=c
#  arg=[a,b,eps] returns the vcyc representing q^a+eps*q^b where eps in [-1,1]
##
vcyc:=function(arg)local res,d,a,b,eps,l;
  if IsList(arg[1]) then return
    rec(vcyc:=arg[1],coeff:=arg[2],operations:=vcycOps);
  fi;
  a:=arg[1];b:=arg[2];eps:=arg[3];
  res:=rec(vcyc:=[Minimum(a,b)],operations:=vcycOps);
  d:=AbsInt(a-b);
  if d=0 then res.coeff:=1+eps;return res;
  elif eps=1 then res.coeff:=1;
    l:=Difference(DivisorsInt(2*d),DivisorsInt(d));
  else res.coeff:=SignInt(a-b);
    l:=DivisorsInt(d);
  fi;
  Append(res.vcyc,[1..Maximum(l)]*0);
  res.vcyc{l+1}:=l*0+1;
  return res;
end;
