#############################################################################
##
#A  weyld.g                 CHEVIE library                   G\"otz Pfeiffer.
##
#A  $Id: weyld.g,v 1.1 1997/01/21 13:46:49 gap Exp $
##
#Y  Copyright (C) 1992 - 1996  Lehrstuhl D f\"ur Mathematik, RWTH Aachen, IWR
#Y  der Universit\"at Heidelberg, University of St. Andrews, and   University 
#Y  Paris VII.
##
##  This file contains special functions for Coxeter groups and Iwahori-Hecke 
##  algebras  of type  D.
##

CHEVIE.D:=rec();


#############################################################################
##
#F  CHEVIE.D.ClassInfo( <n> ) . . . . . . .  conjugacy classes for type D.
##
##  'CHEVIE.D.ClassInfo' returns a record with three components:
##    classtext:   representatives of minimal length in  the  conjugacy  
##                 classes, as words in generators in standard order
##    classparams:  double partitions or [partition,sign],
##                 parameterizing the classes
##    classnames:  strings for double partitions or [partition,sign].
##  
##  The ordering corresponds to the order of the columns of  the  ordinary
##  character table of the Coxeter group of type $D_n$, as returned by the GAP
##  function 'CharTable("WeylD", <n>)'.
##
CHEVIE.D.ClassInfo:= function(n)

   local w, i, l, pi, res, tmp;

   res:= rec(classtext:=[],classnames:=[],classparams:=[]);

   #  loop over the labels for type B.
   for pi in PartitionTuples(n, 2) do

      #  take those with an even number of signed parts.
      if Length(pi[2]) mod 2 = 0 then
         w:= []; i:= 1;

         #  handle signed parts in reversed order.
         for l in Reversed(pi[2]) do

            #  the first sign is empty.
            if i = 1 then
               Append(w, [2 .. i+l-1]);
            else
               Append(w, [i, i-1 .. 3]);
               Append(w, [1 .. i+l-1]);
            fi;
            i:= i+l;
         od;
   
         #  the unsigned cycles.
         for l in pi[1] do
            Append(w, [i+1 .. i+l-1]);
            i:= i+l;
         od;

         #  cosmetics for lexicographics.
         if w <> [] and w[1] = 2 then
            w[1]:= 1;
         fi;
   
         # classes are labelled with '+', if they have representatives
	 # in parabolic subgroup of type A_{l-1}, given by {1,3,4,..}
	 if pi[2] = [  ] and ForAll( pi[1], function ( x )
		     return x mod 2 = 0;
		 end )  then
	     tmp := Copy(pi);
	     tmp[2] := '+';
	     Add( res.classtext, w);
	     Add( res.classparams, Copy(tmp));
	     Add( res.classnames, String(Concatenation( 
                                    IntListToString(tmp[1]),['.',tmp[2]])));
	     w := Copy( w );
	     w[1] := 2;
	     tmp[2] := '-';
	     Add( res.classtext, w);
             Add( res.classparams, Copy(tmp));
             Add( res.classnames, String(Concatenation(
                     IntListToString(tmp[1]),['.',tmp[2]])));
	 else
	     Add( res.classtext, w);
	     Add( res.classparams, pi);
	     Add( res.classnames, DoublePartitionToString(pi));
	 fi;

      fi;

   od;

   #  return the list.
   return res;

end;

# how to make a .charname from a .charparam
CHEVIE.D.CharName:=function(s)
  if s[2] in "+-" then
     return String(Concatenation(IntListToString(s[1]), ".", [s[2]]));
  else
     return DoublePartitionToString(s);
  fi;
end;

#############################################################################
##
#F  CHEVIE.D.CharParams( <n> )  . . . . . . . . . . . characters for type D
##  
CHEVIE.D.CharParams:=n->CharTableWeylD.charparam[1](n);

CHEVIE.D.PositionId:=function(n) 
  return Length(CHEVIE.D.CharParams(n));
end;

CHEVIE.D.PositionSgn:=function(n) 
  return Position(CHEVIE.D.CharParams(n),[[],0*[1..n]+1]);
end;

CHEVIE.D.PositionRefl:=function(n) 
  return Length(CHEVIE.D.CharParams(n))-2;
end;

#############################################################################
##
#F  CHEVIE.D.ClassParam( <n>, <w> )  . . . . . . . . . class parameter of w
##  
##  given an element w  of a Coxeter group W of type A  as word in  standard
##  generators, 'CHEVIE.D.ClassParam' returns the classparam of its conjugacy
##  class.
##  

# Used by 'ClassParamD' for distinguishing classes with '+' or '-' in label:
# (precomputed for D_n with n=4,6,8)
gensMODAforD:=
    [,,,
    [[(1,2)(7,8),(3,4)(5,6),(2,3)(6,7),(3,5)(4,6)],[[4],[,,2]],
    [[2],[1,,1]]],,
    [[(1,2)(8,11)(12,14)(15,17)(16,18)(19,21)(22,25)(31,32),
    (3,4)(5,6)(7,9)(10,13)(20,23)(24,26)(27,28)(29,30),
    (2,3)(6,8)(9,12)(13,16)(17,20)(21,24)(25,27)(30,31),
    (3,5)(4,6)(12,15)(14,17)(16,19)(18,21)(27,29)(28,30),
    (5,7)(6,9)(8,12)(11,14)(19,22)(21,25)(24,27)(26,28),
    (7,10)(9,13)(12,16)(14,18)(15,19)(17,21)(20,24)(23,26)],
    [[16],[4,,6],[1,,,,5]],[[12],[2,,6],[,2,,,4]]
    ],,
    [[(1,2)(8,11)(12,15)(16,20)(17,21)(22,26)(23,27)(28,33)
    (29,34)(30,35)(36,41)(37,42)(43,50)(44,51)(52,59)
    (60,68)(61,69)(70,77)(78,85)(79,86)(87,92)(88,93)
    (94,99)(95,100)(96,101)(102,106)(103,107)(108,112)(109,113)
    (114,117)(118,121)(127,128),(3,4)(5,6)(7,9)(10,13)
    (14,18)(19,24)(25,31)(32,38)(39,45)(40,46)(47,53)
    (48,54)(49,55)(56,62)(57,63)(58,64)(65,71)(66,72)
    (67,73)(74,80)(75,81)(76,82)(83,89)(84,90)(91,97)
    (98,104)(105,110)(111,115)(116,119)(120,122)(123,124)(125,126),
    (2,3)(6,8)(9,12)(13,17)(18,23)(20,25)(24,30)
    (26,32)(33,39)(34,40)(41,48)(42,49)(50,57)(51,58)
    (53,61)(59,67)(62,70)(68,76)(71,78)(72,79)(80,87)
    (81,88)(89,95)(90,96)(97,103)(99,105)(104,109)(106,111)
    (112,116)(117,120)(121,123)(126,127),
    (3,5)(4,6)(12,16)(15,20)(17,22)(21,26)(23,29)
    (27,34)(30,37)(35,42)(39,47)(45,53)(48,56)(54,62)
    (57,65)(58,66)(63,71)(64,72)(67,75)(73,81)(76,84)
    (82,90)(87,94)(92,99)(95,102)(100,106)(103,108)(107,112)
    (109,114)(113,117)(123,125)(124,126),
    (5,7)(6,9)(8,12)(11,15)(22,28)(26,33)(29,36)
    (32,39)(34,41)(37,44)(38,45)(40,48)(42,51)(46,54)
    (49,58)(55,64)(65,74)(71,80)(75,83)(78,87)(81,89)
    (84,91)(85,92)(88,95)(90,97)(93,100)(96,103)(101,107)
    (114,118)(117,121)(120,123)(122,124),
    (7,10)(9,13)(12,17)(15,21)(16,22)(20,26)(25,32)
    (31,38)(36,43)(41,50)(44,52)(48,57)(51,59)(54,63)
    (56,65)(58,67)(62,71)(64,73)(66,75)(70,78)(72,81)
    (77,85)(79,88)(86,93)(91,98)(97,104)(103,109)(107,113)
    (108,114)(112,117)(116,120)(119,122),
    (10,14)(13,18)(17,23)(21,27)(22,29)(26,34)(28,36)
    (32,40)(33,41)(38,46)(39,48)(45,54)(47,56)(52,60)
    (53,62)(59,68)(61,70)(67,76)(69,77)(73,82)(75,84)
    (81,90)(83,91)(88,96)(89,97)(93,101)(95,103)(100,107)
    (102,108)(106,112)(111,116)(115,119),
    (14,19)(18,24)(23,30)(27,35)(29,37)(34,42)(36,44)
    (40,49)(41,51)(43,52)(46,55)(48,58)(50,59)(54,64)
    (56,66)(57,67)(62,72)(63,73)(65,75)(70,79)(71,81)
    (74,83)(77,86)(78,88)(80,89)(85,93)(87,95)(92,100)
    (94,102)(99,106)(105,111)(110,115)],
    [[64],[16,,24],[,,32],[4,,,,20],[,,,,,,16]],
    [[56],[12,,24],[6,,28],[2,4,,,18],[1,,3,,,,14]]]];


CHEVIE.D.ClassParam:=function(n,w)
  local x, i, res, mark, cyc, j, tmp, gens;
  
  x:=();
  for i in w do
    if i=1 then x:=x*(1,n+2)(2,n+1); else x:=x*(i-1,i)(i-1+n,i+n);fi;
  od;
  
  res:=[[],[]];
  mark:=[1..n];
  for i in [1..n] do
    if mark[i]<>0 then
      cyc:=CyclePermInt(x,i);
      if i+n in cyc then
        Add(res[2],Length(cyc)/2);
      else
        Add(res[1],Length(cyc));
      fi;
      for j in cyc do
        if j>n then
          mark[j-n]:=0;
        else
          mark[j]:=0;
        fi;
      od;
    fi;
  od;
  
  if res[2]=[] and ForAll(res[1],i->i mod 2 = 0 ) then
    # for classes with '+' or '-' we use the cycle type for the
    # permutation representation on the cosets of the parabolic
    # subgroup [2..n]: (the generators for this representation are 
    # stored in the global variable 'gensMODAforD')
    
    if not IsBound(gensMODAforD[n]) then
      tmp:=CoxeterGroup("D",n);
      gens:=PermCosetsSubgroup (tmp,
                               ReflectionSubgroup(tmp,[2..n]));
      tmp:=CHEVIE.D.ClassInfo(n);
      tmp:=tmp.classtext{Filtered([1..Length(tmp.classnames)],i->
                   '+' in tmp.classnames[i] or '-' in tmp.classnames[i])};
      tmp:=List(tmp,a->CycleStructurePerm(Product(gens{a})));
      gensMODAforD[n]:=[gens,tmp{2*[1..Length(tmp)/2]-1},
                             tmp{2*[1..Length(tmp)/2]}];
    fi;
    
    tmp:=CycleStructurePerm(Product(gensMODAforD[n][1]{w}));
    if tmp in gensMODAforD[n][2] and not tmp in gensMODAforD[n][3] then
      res[2]:='+';
    elif not tmp in gensMODAforD[n][2] and tmp in gensMODAforD[n][3] then
      res[2]:='-';
#   else
#     # this is hopefully never used (surely not for n<14):
#     tmp:=CoxeterGroup("D",n);
#     if RepresentativeOperation(tmp,PermCoxeterWord (tmp,w),
#     PermCoxeterWord (tmp,CoxeterGroupOps.ChevieClassInfo(tmp).classtext[
#                 Position(CoxeterGroupOps.ChevieClassInfo(tmp).classparams,
#                                [[res[1],'+']])]))<>false then
#       res[2]:='+';
#     else
#       res[2]:='-';
#     fi;
    fi;
  fi;  
  
  Sort(res[1]);
  if IsList(res[2]) then
    Sort(res[2]);
    return [Reversed(res[1]),Reversed(res[2])];
  else
    return [Reversed(res[1]),res[2]];
  fi;
end;



#############################################################################
##
#F  CHEVIE.D.CharTable( <l> ) . . .  character table of CoxeterGroup("D",l)
##  
##  This function returns  the part of the character table the Coxeter group
##  of  type B_l on classes  inside  a reflection subgroup  of  type D_l.
##  
##  If l is even then some of the classes  and restrictions split into two
##  classes or  characters, respectively. Their  values  are given by  the
##  character  values    for W(B_l) and   those  for  the  symmetric group
##  S_(l/2). This is described in [Pfeiffer, G., Character Tables of  Weyl
##  Groups in GAP]. 
##  
CHEVIE.D.CharTable:=function(l)
  local l, hi, erg, i, j, lst, lst2, p, sy, syc, tmp, cli, pow,  S, Sval;
  
  hi:=CharTable("WeylB",l);
  if not IsBound(hi.size) then
    hi.size:=2^l*Factorial(l);
  fi;
  cli:=CHEVIE.D.ClassInfo(l);
  if l mod 2 = 0 then
    S:=CharTable("Symmetric",l/2);
    Sval:=function(aa,pp) 
      return S.irreducibles[Position(S.classparam,[1,aa[1]])]
                           [Position(S.classparam,[1,pp[1]/2])];
    end;  
  fi;

  # classes in subgroup:
  lst:=[];
  for i in [1..Length(hi.classparam)] do
    if Length(hi.classparam[i][2][2]) mod 2=0 then
      # degenerate classes:
      if hi.classparam[i][2][2] = [] and ForAll(hi.classparam[i][2][1]/2,IsInt)
         then
        Append(lst,[i,i]);
      else
        Add(lst,i);
      fi;
    fi;
  od;
  
  # similarly for characters:
  lst2:=[];
  for i in [1..Length(hi.irredinfo)] do
    sy:=hi.irredinfo[i].charparam[2];
    if sy[1]<=sy[2] then
      # degenerate characters:
      if sy[1]=sy[2] then
        Append(lst2,[i,i]);
      else
        Add(lst2,i);
      fi;
    fi;
  od;
  
  # for power maps:
  pow:=function(i,p,lst) 
    local r;
    r:=Position(lst,p[lst[i]]);
    if '-' in cli.classparams[i] and '+' in cli.classparams[r] then
        r:=r+1;
    fi;
    return r;
  end;
  
  erg:=rec(identifier:=String(Concatenation("W(D",String(l),")")));
  erg.size:=hi.size/2;
  erg.cartan:=CartanMat("D",l);
  erg.centralizers:=hi.centralizers{lst}/2;
  erg.orders:=hi.orders{lst};
  erg.classparam:=cli.classparams;
  erg.classtext:=cli.classtext;
  erg.classnames:=cli.classnames;
  erg.classes:=hi.classes{lst};
  erg.powermap:=[];
  for i in [1..Length(hi.powermap)] do
    if IsBound(hi.powermap[i]) then
      erg.powermap[i]:=List([1..Length(lst)],j->pow(j,hi.powermap[i],lst));
    fi;
  od;
  erg.text:="extracted from generic character table of type WeylB";
  erg.operations:=CharTableOps;
  erg.irredinfo:=List(List(hi.irredinfo{lst2},Copy),
                         a->rec(charparam:=a.charparam[2],
                         charname:=DoublePartitionToString(a.charparam[2])));
  erg.irreducibles:=hi.irreducibles{lst2}{lst};
  
  # corrections for degenerate classes and characters:
  for i in [1..Length(lst)] do
    if '+' in erg.classparam[i] or '-' in erg.classparam[i] then
      erg.centralizers[i]:=2*erg.centralizers[i];
      erg.classes[i]:=erg.classes[i]/2;
    fi;
  od;
    
  i:=1;
  while i<=Length(lst2) do
    sy:=erg.irredinfo[i].charparam;
    if sy[1]=sy[2] then
      erg.irreducibles[i]:=erg.irreducibles[i]/2;
      erg.irreducibles[i+1]:=erg.irreducibles[i+1]/2;
      erg.irredinfo[i].charparam[2]:='+';
      erg.irredinfo[i+1].charparam[2]:='-';
      erg.irredinfo[i].charname:=CHEVIE.D.CharName(erg.irredinfo[i].charparam);
      erg.irredinfo[i+1].charname:=CHEVIE.D.CharName(erg.irredinfo[i+1].charparam);
      for j in [1..Length(lst)] do
        syc:=erg.classparam[j];
        if '+' in syc then
          tmp:=2^(Length(syc[1])-1)*Sval(sy,syc);
          erg.irreducibles[i][j]:=erg.irreducibles[i][j]+tmp;
          erg.irreducibles[i+1][j+1]:=erg.irreducibles[i+1][j+1]+tmp;
          erg.irreducibles[i][j+1]:=erg.irreducibles[i][j+1]-tmp;
          erg.irreducibles[i+1][j]:=erg.irreducibles[i+1][j]-tmp;
        fi;
      od;
      i:=i+2;
    else
      i:=i+1;
    fi;
  od;
  
  return erg;
end;

###########################################################################
##
#F  CHEVIE.D.HeckeCharTable( <n>, <u> ) . . . .  character table of $H(D_n)$.
##
##  'CHEVIE.D.HeckeCharTable' returns the character table of the Hecke algebra 
##  associated with the finite  Coxeter group of  type  $D_n$   with parameter 
##  <u>.
##
CHEVIE.D.Hk:= Copy(CharTableWeylD);

CHEVIE.D.Hk.identifier:= "HeckeD"; 

CHEVIE.D.Hk.specializedname:= 
   (nq -> Concatenation("H(D", String(nq[1]), ")"));

CHEVIE.D.Hk.size:= (nq-> 2^(nq[1]-1) * Factorial(nq[1]));  
CHEVIE.D.Hk.order:=CHEVIE.D.Hk.size;

CHEVIE.D.Hk.domain:= function(nq)
   return IsList(nq) and Length(nq) = 2 and IsInt(nq[1]) and nq[1] > 1;
end;

CHEVIE.D.Hk.text:= "generic character table of Hecke algebras of type D";

CHEVIE.D.Hk.classparam:= [nq-> CharTableWeylD.classparam[1](nq[1])];
CHEVIE.D.Hk.charparam:= [nq-> CharTableWeylD.charparam[1](nq[1])];

CHEVIE.D.Hk.irreducibles:= [[]];
CHEVIE.D.Hk.irreducibles[1][1]:= function(nq, alpha, pi)

   local delta, val, n, q;

   n:= nq[1];  q:= nq[2];
   
   if q = 1 then
      return CharTableWeylD.irreducibles[1][1](n, alpha, pi);
   fi;
   
   # this function uses weyla and weylb
   if not IsBound(CHEVIE.A) then ReadChv("tbl/weyla");fi;
   if not IsBound(CHEVIE.B) then ReadChv("tbl/weylb");fi;

   if not IsList(alpha[2]) then
      delta:= [alpha[1], alpha[1]];
      if not IsList(pi[2]) then
         val:= CHEVIE.B.Hk.irreducibles[1][1]([n, 1, q], delta, [pi[1], []])/2;
         if alpha[2] = pi[2] then
            val:= val + (q+1)^Length(pi[1]) / 2 *
              CHEVIE.A.Hk.irreducibles[1][1]([n/2, q^2], alpha[1], pi[1]/2);
         else
            val:= val - (q+1)^Length(pi[1]) / 2 *
              CHEVIE.A.Hk.irreducibles[1][1]([n/2, q^2], alpha[1], pi[1]/2);
         fi;
      else
         val:= CHEVIE.B.Hk.irreducibles[1][1]([n, 1, q], delta, pi)/2;
      fi;
   else
      if not IsList(pi[2]) then
         val:= CHEVIE.B.Hk.irreducibles[1][1]([n, 1, q], alpha, [pi[1], []]);
      else
         val:= CHEVIE.B.Hk.irreducibles[1][1]([n, 1, q], alpha, pi);
      fi;
   fi;

   return val;

end;

CHEVIE.D.HeckeCharTable:= function(n, u)

   local r, tbl;

   tbl:= CharTableSpecialized(CHEVIE.D.Hk, [n, u]);
   tbl.cartan:= CartanMat("D", n);
   tbl.parameter:= List([1..n], x-> u);
   tbl.classtext:= CHEVIE.D.ClassInfo(n).classtext;

   for r in tbl.irredinfo do
      r.charparam:= r.charparam[2];
      r.charname:= CHEVIE.D.CharName(r.charparam);
   od;
   tbl.classparam:= List(tbl.classparam, x-> x[2]);
   tbl.classnames:= List(tbl.classparam, CHEVIE.D.CharName);

   return tbl;

end;


#############################################################################
##
#F  PoincarePolynomialD . . . . . . . . . . . Poincare polynomial for type D.
##
CHEVIE.D.PoincarePolynomial:= function(n, q)
  return Sum([0..n-1], k-> q^k) 
         * Product([1..n-1], i-> (q^i + 1) * Sum([0..i-1], k-> q^k));
end;


CHEVIE.D.symbolcharparam:= function(c)
  if c[2] in "+-" then return SymbolDpart([c[1],c[1]],0);
  else return SymbolDpart(c,0);
  fi;
end;

#############################################################################
##
#F  CHEVIE.F.vcycGenericDegree( <para> ) .  . . . Generic Degree  for type D.
##
##  'CHEVIE.F.vcycGenericDegree'  returns the  generic degree of the character
##  with parameter <para> as a vcyc (see vcyc.g).
##
##  [Reference: Lusztig, 'Irreducible Repr. of classical groups']
##
CHEVIE.D.vcycGenericDegree:=c->vcycGenericDegreeSymbol(CHEVIE.D.symbolcharparam(c));

#############################################################################
##
#F  CHEVIE.D.vcycFakeDegree( <c> ) Fake Degree of char. with charparam <c>
##
##  This returns the polynomial describing the multiplicity of the character
##  with charaparam <c> in the graded version of the regular representation 
##  given by the quotient S/I where S is the symmetric algebra of the reflection
##  representation and I is the ideal generated by the homogenous invariants
##  of positive degree in S.
##
CHEVIE.D.vcycFakeDegree:=c->vcycFakeDegreeSymbol(CHEVIE.D.symbolcharparam(c));

CHEVIE.D.Chara:=function(p) local pp, m, i, j, res;
  if p[2] in ['+','-'] then p:=[p[1],p[1]]; fi;
  pp:=SymbolDpart(p,0);if pp[1]>pp[2] then pp:=[pp[2],pp[1]]; fi;
  m:=List(pp,Length);
  res:=pp[1]*[m[1]-1,m[1]-2..0];
  if pp[2]<>[] then
    res:=res+pp[2]*[m[2]-1,m[2]-2..0];
  fi;
  for i in pp[1] do
    for j in pp[2] do
      res:=res+Minimum(i,j);
    od;
  od;
  m:=Sum(m)-2;
  while m>1 do
    res:=res-m*(m-1)/2;
    m:=m-2;
  od;
  return res;
end;

CHEVIE.D.CharA:=function(p) local pp, res, l, m, i, j;
  if p[2] in ['+','-'] then p:=[p[1],p[1]]; fi;
  pp:=SymbolDpart(p,0);if pp[1]>pp[2] then pp:=[pp[2],pp[1]]; fi;
  m:=List(pp,Length);
  l:=Sum(pp[1])+Sum(pp[2])-QuoInt((Sum(m)-1)^2,4);
  pp:=List(pp,Reversed);
  res:=l*(l+1)+pp[1]*[m[1]-1,m[1]-2..0];
  if pp[2]<>[] then
    res:=res+pp[2]*[m[2]-1,m[2]-2..0];
  fi;
  for i in pp[1] do
    for j in pp[2] do
      res:=res+Maximum(i,j);
    od;
  od;
  m:=Sum(m)-2;
  while m>1 do
    res:=res-m*(m-1)/2;
    m:=m-2;
  od;
  for i in Concatenation(pp) do
    res:=res-i*(i+1);
  od;
  return res-(Sum(pp[1])+Sum(pp[2])-QuoInt((Length(pp[1])
                 +Length(pp[2])-1)^2,4));
end;

CHEVIE.D.Charb:=function(p) local pp, m, i, j, res;
  if p[2] in ['+','-'] then p:=[p[1],p[1]]; fi;
  pp:=SymbolDpart(p,0);if pp[1]>pp[2] then pp:=[pp[2],pp[1]]; fi;
  m:=Length(pp[2]);
  if Length(pp[1])<>m then
    return -1;
  fi;
  res:=2*(pp[1]+pp[2])*[m-1,m-2..0];
  res:=res+Minimum(Sum(pp[1]),Sum(pp[2]));
  return res-m*(m-1)*(4*m-5)/6;
end;

CHEVIE.D.CharB:=p->CHEVIE.D.Chara(p)+CHEVIE.D.CharA(p)-CHEVIE.D.Charb(p);

#############################################################################
##
#E  Emacs . . . . . . . . . . . . . . . . . . . . . .  local emacs variables.
##
##  Local Variables:
##  fill-column: 77
##  fill-prefix: "##  "
##  End:
##
