#############################################################################
##
#A  coset.g                  CHEVIE library        Frank Luebeck, Jean Michel
##
#A  $Id: coset.g,v 1.1 1997/01/21 13:46:23 gap Exp $
##
#Y  Copyright (C) 1992 - 1996  Lehrstuhl D f\"ur Mathematik, RWTH Aachen, IWR
#Y  der Universit\"at Heidelberg, University of St. Andrews, and   University 
#Y  Paris VII.
##
##  This file contains general functions that deal with cosets of Coxeter 
##  groups.
##

#############################################################################
##
#F  CoxeterCosetOps . . . . . . .  operations record for Coxeter group cosets
##  
##  We  first copy  the  basic  functions   included in 'DomainOps'    and
##  overwrite some of them by more efficient ones.
##  
CoxeterCosetOps:=OperationsRecord("CoxeterCosetOps",DomainOps);

CoxeterCosetOps.IsFinite:=function(WF) return true; end;

CoxeterCosetOps.Random:=function(WF)
  return Random(WF.coxeter)*WF.phi;
end;

CoxeterCosetOps.Representative:=function(WF) return WF.phi; end;

CoxeterCosetOps.Size:= function(WF)
  return Size(WF.coxeter);
end;

CoxeterCosetOps.Elements:=function(WF)
  if WF.phi in WF.coxeter then
    return Elements(WF.coxeter);
  else
    return Set(Elements(WF.coxeter)*WF.phi);
  fi;
end;

CoxeterCosetOps.Parent:=PermGroupOps.Parent;

CoxeterCosetOps.\in:=function(x,WF)
  return x/WF.phi in WF.coxeter;
end;

CoxeterCosetOps.\=:=function(WF1,WF2)
  return IsIdentical(WF1,WF2) or 
         (WF1.F0Mat=WF2.F0Mat and WF1.coxeter=WF2.coxeter);
end;

#############################################################################
##
#F  CoxeterCosetOps.Print( <WF> ) .   . . . . . . . . . 'Print' function for
#F  . . . . . . . . . . . . . . . . . . . . . . . Coxeter group coset records
##
##  If <WF>.name is bound then this component is printed.
##
CoxeterCosetOps.Print:=function(WF)
  if IsBound(WF.name) then
    Print(WF.name);
  elif IsBound(WF.parent) then
    Print("CoxeterSubCoset(",WF.parent,", ",
          WF.coxeter.rootInclusion{[1..WF.coxeter.semisimpleRank]});
    if IsBound(WF.coxeter.omega) and Size(WF.coxeter.omega)>1 then
      Print(", ",WF.coxeter.omega);
    fi;
    if WF.F0Perm/WF.parent.F0Perm<>() then
      Print(", ",WF.F0Perm/WF.parent.F0Perm);
    fi;
    Print(")");
  else
    Print("CoxeterCoset(",WF.coxeter);
    if WF.callArgs<>[] then
      Print(", ",WF.callArgs);
    fi;
    Print(")");
  fi;
end;

#############################################################################
##
#F  CoxeterCosetOps.CartanType( <WF> ) . . . . . . . . the Coxeter group type 
##  . . . . . . . . . . . . . . . . . . . . . . . . . of a CoxeterGroup coset
##  
##  Gives  a list which describes  WF.  The list  has one element for each
##  orbit of <WF>.phi on the irreducible components of  W. This element is
##  a record with two components:
##  
##    .orbit  The CartanType of the Cartan matrix of the orbit (see  
##            CoxeterGroupOps.CartanType)
##    .phi    A permutation of .orbit[1][2] which describes the effect of
##            <WF>.phi^Length(orbit) on it.
##  
##  The indices in .orbit are given such that
##      .orbit([2][2])=OnTuples(.orbit[1][2],<WF>.phi), etc...
##  
##  If the components in .orbit are of type D_4 and <WF>.phi^Length(orbit)
##  has order  2  on .orbit[1][2] then .orbit[1][2]  is   sorted such that
##  <WF>.phi^Length(orbit) permutes the first two entries.
##  
##  If the components in .orbit are of type D_4 and <WF>.phi^Length(orbit)
##  has order  3  on .orbit[1][2] then .orbit[1][2]  is   sorted such that
##  <WF>.phi^Length(orbit) permutes the  entries 1 -> 2 -> 4 -> 1.
##  
CoxeterCosetOps.CartanType:=function(WF)
  local W, c, i, type, rls, hi, phifus;

  W:=WF.coxeter;

  type:=CartanType(W);

  c:=List(type,a->Set(W.rootInclusion{a[2]}));
  c:=Cycles(Permutation(WF.phi,c,OnSets),[1..Length(type)]);
  type:=List(c,x->type{x});
  phifus:=WF.phi^MappingPermListList(W.rootInclusion,[1..2*W.N]);
  for c in type do
    for i in [2..Length(c)] do
      c[i][2]:=OnTuples(c[i-1][2],phifus);
    od;
  od;
  type:=List(type,c->rec(orbit:=c,phi:=
                   RestrictedPerm(phifus^Length(c),c[1][2])));

  # some adjustment such that in type ^2D_4 the first two simple
  # roots are permuted by res.phi, and in type ^3D_4 the permutation 
  # of the simple roots is 1 -> 2 -> 4 -> 1:

  hi:=function(a)
    local b, rf, j;    

    if a.phi<>() and a.orbit[1][1]="D" and Length(a.orbit[1][2])=4 then
      # ^2D_4
      if a.phi^2=() then
        rf:=Filtered([1..4],i->a.orbit[1][2][i]^a.phi<>a.orbit[1][2][i]);
        for i in [1..4] do if not i in rf then Add(rf,i); fi; od;
        b:=rec(orbit:=[],phi:=Copy(a.phi));
        for j in a.orbit do Add(b.orbit,[j[1],j[2]{rf}]); od;
        return b;
      else 
        # triality group ^3D_4
        if a.orbit[1][2][1]^a.phi=a.orbit[1][2][4] then
          b:=Copy(a);
          for j in [1..Length(b.orbit)] do
            b.orbit[j][2]:=b.orbit[j][2]{[1,4,3,2]};
          od;
          return b;
        else
          return a;
        fi;
      fi;
    else
      return a;
    fi;
  end;
  type:=List(type,hi);

##    # some sorting: (now already done in CartanType(WF.coxeter))
##    hi:=List(type,a->[Length(a.orbit[1][2]),a.orbit[1][1]]);
##    SortParallel(hi,type,function(a,b) 
##                      return a[1]>b[1] or (a[1]=b[1] and a[2]<b[2]); end);  
  return type;
end;

#############################################################################
##
#F  CoxeterCosetOps.PrintDynkinDiagram( <WF> ) . . . . prints twisted Dynkin
#F  diagram of <WF>
##  
##  In  addition to 'PrintDynkinDiagram(  CoxeterGroup(<WF>) )'  information 
##  about the action of <WF>.phi on the Dynkin diagram is printed.
##  
CoxeterCosetOps.PrintDynkinDiagram:=function(WF)
  local tmp, i, o;
  tmp:=Copy(CartanType(WF));
  for i in [1..Length(tmp)] do
    tmp[i].orbit:=List(tmp[i].orbit,function(a) 
      if Length(a)=2 then return [a[1],WF.coxeter.rootInclusion{a[2]}];
      else return [a[1],WF.coxeter.rootInclusion{a[2]},a[3]];
      fi; end);
    if tmp[i].phi<>() then
      tmp[i].phi:=RestrictedPerm(WF.phi^Length(tmp[i].orbit),
                          tmp[i].orbit[1][2]);
    fi;
  od;
  for o in tmp do
    if Length(o.orbit)>1 then
      Print("phi permutes the next ",Length(o.orbit)," components\n");
    fi;
    if o.phi<>() then
      Print("phi");
      if Length(o.orbit)>1 then Print("^",Length(o.orbit));fi;
      Print(" acts as ",o.phi," on the component below\n");
    fi;
    PrintDynkinDiagram(o.orbit);
  od;
end;

#############################################################################
##
#F  CoxeterCosetOps.CartanName( <W> ) . . . . . name of a CoxeterGroup coset
##  
##  Gives a string which describes the isomorphism type of W.  An orbit of
##  phi on the  components is put in brackets  if of length x greater than
##  1, and  is  preceded by  the  order of  phi^x on it,  e.g. 2(A2xA2xA2)
##  denotes 3 components A2  permuted by phi,  and such that phi^3 induces
##  the non-trivial diagram automorphism on any of them, while 3D4 denotes
##  an orbit of  length 1 on  which phi is of order  3.  
##  
CoxeterCosetOps.CartanName:=function(WF)
  local t,name;
  name:=[];
  for t in CartanType(WF) do
    if OrderPerm(t.phi)<>1 then Append(name,String(OrderPerm(t.phi)));fi;
    if Length(t.orbit)>1 then Append(name,"(");fi;
    Append(name,CartanName(t.orbit));
    if Length(t.orbit)>1 then Append(name,")");fi;
    Append(name,"x");
  od;
  return name{[1..Length(name)-1]};
end;


#############################################################################
##
#F  CoxeterCosetOps.ConjugacyClasses( <WF> ) . . . . . conjugacy classes for 
#F  CoxeterGroup coset
##  
##  'ConjugacyClasses' returns the CoxeterGroup(<WF>) -  conjugacy classes 
##  of  the coset <WF>.
##  
##  Here     the   classification     is     used,   see   comments    for
##  'CoxeterConjugacyClasses' and 'CoxeterClassInfoIrred'.
##  
CoxeterCosetOps.ConjugacyClasses:=WF->List(CoxeterConjugacyClasses(WF),
     x->ConjugacyClass(WF.coxeter,PermCoxeterWord (WF.coxeter,
     WF.coxeter.rootInclusion{x})*WF.phi));

#############################################################################
##
#F  CoxeterCosetClassInfoIrred( <type>, <l> [,<bond>], <o> )  . . . information
#F  about F-conjugacy classes of simple Coxeter cosets
##  
##  <l> must be a list of positive integers.  <type>, Length(<l>) (and <bond>
##  if  <type>="I") determine a simple Coxeter  group W.
##  <o> is the order of a graph automorphism F on the  root system of this
##  group.     'CoxeterCosetClassInfoIrred' returns a record
##  containing  information about the F-conjugacy  classes of W, which has
##  three lists as components:
##  
##    .classtext   representatives of classes as words in elements of <l>,
##                 taken as standard generators of W
##    .classnames  corresponding names for the classes
##    .classparams  corresponding parameters for the classes
##  
##  The function  calls functions  ChevieClassInfoX,  where X  describes the
##  twisted  type. See  these  functions for  more information  about the
##  single cases.
##  
CoxeterCosetClassInfoIrred:=function(arg) local rank, r,type,cl,o;
  o:=arg[Length(arg)];
  if o=1 then return ApplyFunc(CoxeterClassInfoIrred,
                                          arg{[1..Length(arg)-1]});fi;
  type:=arg[1];
  rank:=Length(arg[2]);
  r:=CHEVIE.LoadCoset(type,rank,o);
  if type in ["A","~A","D"]  and o=2 then cl:=r.ClassInfo(rank);
  elif (type="D" and rank=4 and o=3) or (type="E" and rank=6 and o=2) then
    cl:=rec(classtext:=r.classtext,
	    classnames:=r.classnames,classparams:=r.classnames);
  elif type="I" and rank=2 and o=2 then cl:=r.ClassInfo(arg[3]);
  fi;

  r:=MappingPermListList([1..Length(arg[2])],arg[2]);
  cl.classtext:=List(cl.classtext,x->OnTuples(x,r));
  return cl;
end;

#############################################################################
##
#F  CoxeterCosetOps.ChevieClassInfo( <WF> ) . . . 'ChevieClassInfo' for cosets
##  
##  Here     the   classification     is     used,   see   comments    for
##  'CoxeterClassInfoIrred'.
##   
CoxeterCosetOps.ChevieClassInfo:=function(WF)
  local tmp, f;

  f:=function(l)
    local s;
    s:=String(Concatenation(List(l,x->Concatenation(x,","))));
    return s{[1..Length(s)-1]};
  end;

  tmp:=List(CartanType(WF),function(a)local b;
      b:=ShallowCopy(a.orbit[1]);Add(b,OrderPerm(a.phi));
      return ApplyFunc(CoxeterCosetClassInfoIrred,b);end);  
  WF.classInfo:=rec(
    classtext:=List(Cartesian(List(tmp,x->x.classtext)),Concatenation),
    classparams:=Cartesian(List(tmp,x->x.classparams)),
    classnames:=List(Cartesian(List(tmp,x->x.classnames)),f));
  return WF.classInfo;
end;

CoxeterCosetOps.Hecke:=function(arg)
  ReadChv("prg/hekcoset");return ApplyFunc(CoxeterCosetOps.Hecke,arg);
end;

#############################################################################
## 
#F PhiFactors( <WF> ) . . . . . . . Eigenvalues of phi on the invariants
##
## The eigenvalues by which .phi acts on the basic invariants of W
## in the symmetric algebra of the reflection representation
## The implementation below can be improved --- a more refined one would
## avoid constructing any subgroups.
##
PhiFactors:=function(WF) local degrees,factors,a,W,o,d,f,i,type,r,rank,n,t;
  if not IsBound(WF.factors) then
    degrees:=[];factors:=[];
    for a in CartanType(WF) do
      t:=ShallowCopy(a.orbit[1]);
      t[2]:=Length(t[2]);
      d:=ReflectionDegrees(ApplyFunc(CoxeterGroup,t));
      f:=List(d,x->1);
      o:=OrderPerm(a.phi);
      if o>1 then
	type:=a.orbit[1][1];
	rank:=Length(a.orbit[1][2]);
        r:=CHEVIE.LoadCoset(type,rank,o);
	if type in ["A","~A","D"] and o=2 then f:=r.PhiFactors(rank);
	else f:=r.phifactors;
	fi;
      fi;
      for i in [0..Length(a.orbit)-1] do
	Append(degrees,d);
	Append(factors,f*E(Length(a.orbit))^i);
      od;
    od;
    SortParallel(degrees,factors);
    W:=CoxeterGroup(WF);
    if W.rank>W.semisimpleRank then
      n:=WF.F0Mat*MatXPerm(W,WF.F0Perm)^-1;
      n:=Filtered(EigenvaluesMat(n),x->x<>1);
      factors:=Concatenation(n,List(
                       [1..W.rank-W.semisimpleRank-Length(n)],x->1),factors);
    fi;
    WF.factors:=factors;
  fi;
  return WF.factors;
end;

###########################################################################
##
#F  CharName(W,p) . . . . . . . . name of character with .charparam p
##
CoxeterCosetOps.CharName:=function(W,p)local t,i,s;
  s:=[]; t:=CartanType(W);
  for i in [1..Length(t)] do
    Append(s,CHEVIE.LoadCoset(t[i].orbit[1][1],Length(t[i].orbit[1][2]),
            OrderPerm(t[i].phi)).CharName(p[i]));
    if i<>Length(t) then Append(s,",");fi;
  od;
  return String(s);
end;

###########################################################################
##
#F  GetChevieInfo(W) . . . . . . . . . . get information 
##
CoxeterCosetOps.GetChevieInfo:=function(W,fname)local t,type,rank,res,p,o;
  res:=[];
  for t in CartanType(W) do
    rank:=Length(t.orbit[1][2]);
    type:=t.orbit[1][1];
    o:=OrderPerm(t.phi);
    p:=CHEVIE.LoadCoset(type,rank,o);
    if   type in ["A","~A","B","C","D"] and o<>3 then p:=p.(fname)(rank);
    elif type in ["D","E","H","F","G"] then p:=p.(fname)();
    elif type="I" then p:=p.(fname)(t.orbit[1][3]);
    fi;
    Add(res,p);
  od;
  return res;
end;

CoxeterCosetOps.CharParams:=CoxeterGroupOps.CharParams;

CoxeterCosetOps.PositionId:=CoxeterGroupOps.PositionId;

CoxeterCosetOps.PositionSgn:=CoxeterGroupOps.PositionSgn;

#############################################################################
##
#F  CoxeterCosetOps.ChevieCharInfo( <W> ). . . . see corresponding dispatcher 
#F  function
##
CoxeterCosetOps.ChevieCharInfo:=function(W) local t,c,type,rank,p,o;
  if not IsBound(W.charInfo) then
    t:=CharParams(W);
    W.charInfo:=rec(charparams:=t, charnames:=List(t,i->CharName(W,i)),
     positionId:=CoxeterGroupOps.PositionId(W),
     positionSgn:=CoxeterGroupOps.PositionSgn(W));
  fi;
  c:=CartanType(W);
  if Length(c)=1 then
    W.charInfo.positionRefl:=GetChevieInfo(W,"PositionRefl")[1];
    rank:=Length(c[1].orbit[1][2]);
    type:=c[1].orbit[1][1];
    o:=OrderPerm(c[1].phi);
    p:=CHEVIE.LoadCoset(type,rank,o);
    if type="F" then W.charInfo.kondo:=p.kondo;
    elif type="E" then W.charInfo.frame:=p.frame;
    fi;
  fi;
  return W.charInfo;
end;

##############################################################################
##
#F  CoxeterCosetCharTableIrred( <typ>, <inds>[, <bond>], <ord> ) . . character 
#F  for coset of irreducible Coxeter group
##  
##  Accepts an argument which is the type of  an irreducible Coxeter group
##  (like  a   component  of   the   result of  CartanType).    Arguments:
##  type,indices[,bond (type I)],orderphi
##  
CoxeterCosetCharTableIrred:=function(arg)
  local p, r, rank, type, tbl, orderphi;

  orderphi:=arg[Length(arg)];
  if orderphi=1 then 
    return ApplyFunc(CoxeterCharTableIrred,arg{[1..Length(arg)-1]});
  fi;

  type:=arg[1];
  rank:=Length(arg[2]);
  r:=CHEVIE.LoadCoset(type,rank,orderphi);
  if type in ["A","~A","D"] and orderphi=2 then tbl:=r.CharTable(rank);
  elif type="D" and orderphi=3 and rank=4 then tbl:=r.HeckeCharTable(1);
  elif type="E" and rank=6  and orderphi=2 then tbl:=r.CharTable();
  elif type="I" and rank=2  and orderphi=2 then tbl:=r.HeckeCharTable(arg[3],1);
  fi;
  tbl.classtext:=List(tbl.classtext,x->
                            OnTuples(x,MappingPermListList([1..rank],arg[2])));
  
  # change '.name' if tbl is specialized from table for Hecke algebra:
  if not IsBound(tbl.name) then
    tbl.name:= tbl.identifier;
  fi;
  p:=Position(tbl.name,'H');
  if p<>false and tbl.name[p+1]='(' then
    tbl.name[p]:='W';
    tbl.identifier:=tbl.name;
  fi;
  if not IsBound(tbl.order) then
    tbl.order:=tbl.size;
  fi;
  return tbl;
end;

#############################################################################
##
#F  CoxeterCosetOps.CharTable( <WF> ) . .  character table for Coxeter cosets
##  
##  The character  table   of a coset   W*phi is  defined as follows:  The
##  characters  of <W, phi>  with non  zero values on  the classes  in the
##  coset W*phi   are   precisely  those   whose restriction   to   W   is
##  irreducible. To each  character of W  which is extendable to  <W, phi>
##  there are |phi| extensions.
##  
##  Following Lusztig  we  choose a unique  of  these extensions  in  each
##  single  case, called the  *preferred*  extension.  This is defined  in
##  [Lusztig, Character sheaves IV, Chapter 17.2].
##  
##  The character table is  the (square) table which  gives the  values of
##  the *preferred* extensions on the  conjugacy classes contained in  the
##  coset W*F.
##  
##  It is  a GAP  character  table record, but if   <WF>.FOMat is  not the
##  identity matrix then there are no  components .powermap and .orders.
##  
CoxeterCosetOps.CharTable:=function(WF)
  local t, l, a, ntrivF0, tmp, res,cl,f,args;
  
  l:=[];
  ntrivF0:=WF.F0Mat<>IdentityMat(WF.coxeter.rank);
  for t in CartanType(WF) do
    args:=[t.orbit[1][1],WF.coxeter.rootInclusion{t.orbit[1][2]}];
    if args[1]="I" then Add(args,t.orbit[1][3]);fi;
    Add(args,OrderPerm(t.phi));
    tmp:=ApplyFunc(CoxeterCosetCharTableIrred,args);
    # for nontrival F0 the orders are  not easy to determine (and cannot 
    # be determined from the CartanType), and powermaps do not make sense:
    if ntrivF0 then
      Unbind(tmp.orders);
      Unbind(tmp.powermap);
    fi;
    
    if Length(t.orbit)>1 then
      tmp.size:=tmp.size^Length(t.orbit);
      tmp.order:=tmp.size;
      tmp.classes:=List(tmp.centralizers,a->tmp.size/a);
    fi;
    Add(l,tmp);
  od;
  if Length(l)>0 then 
    res:=l[1];
  else 
    res:=GroupOps.CharTable(WF.coxeter);
    res.classtext:=[[]];
  fi;
  for t in l{[2..Length(l)]} do
    res:=CharTableDirectProduct(res,t);
    Unbind(res.fusionsource); Unbind(res.fusions);
  od;
  # We have to do some corrections due to 'InitClassesCharTable',
  # called in 'CharTableDirectProduct':
  res.classes:=List(res.centralizers,x->res.size/x);

  res.irredinfo:=List(CharParams(WF),
		                x->rec(charparam:=x,charname:=CharName(WF,x)));
  cl:=ChevieClassInfo(WF);
  for f in RecFields(cl) do res.(f):=cl.(f);od;
  res.name:=String(Concatenation("W( ",CartanName(WF)," )"));
  res.identifier:=res.name;

  return res;
end;

#############################################################################
##
#F  CoxeterCosetOps.ClassInvariants( <W> ) . .  'ClassInvariants' for Coxeter 
##  . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . .  cosets
#F  CoxeterCosetOps.PositionClass( <W>, <w> ) . . 'PositionClass' for Coxeter
##  . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . .  cosets
##
CoxeterCosetOps.ClassInvariants:=CoxeterCosetOpsClassInvariants;
CoxeterCosetOps.PositionClass:=PermGroupOps.PositionClass;

#############################################################################
##
#F  CoxeterGroupOps.FusionConjugacyClasses( <W1>, <W2> ) . . . . . . . . . . 
#F  'FusionConjugacyClasses' for Coxeter cosets
##
CoxeterCosetOps.FusionConjugacyClasses:=function(u,g)
  if Parent(u)=Parent(g) then
    return List(List(ConjugacyClasses(u),Representative),
                x->PositionClass(g,x));
  else
    Error("Don't know how to compute fusion.\n");
  fi;
end;

#############################################################################
##
#F  CoxeterCoset( <W>[, <F0Mat> or  <perm>] ) . . . .  create a CoxeterGroup
#F  coset record
#F  CoxeterCoset( <rec> ) . . . . . . . . . return component <rec>.coxeterCoset
##  
##  In the first form <W> must be a CoxeterGroup record.
##  <F0Mat> must be a square  matrix of rank <W>.rank such that:
##    - <F0Mat> is invertible
##    - <F0Mat> has finite order
##    - <F0Mat>: X->X lets invariant the set of roots of <W> and of Parent(<W>)
##    - TransposedMat(<F0Mat>): Y->Y lets invariant the set of coroots
##        of <W> and of Parent(<W>)
##  
##  If <W>.semisimpleRank  = <W>.rank it   is allowed to give an  argument
##  <perm> (which  is a permutation) instead  of <F0Mat>. Then  <F0Mat> is
##  computed as the  unique matrix which  maps the simple  roots of <W> on
##  the roots given by i^<perm>, i in [1..<W>.rank]. 
##  
##  If only  <W> is  given then the  default for  <F0Mat> is  the identity
##  matrix.
##  
##  'CoxeterCoset' returns a record with the following components:
##    .isDomain     set to true
##    .isFinite     set to true
##    .isCoxeterCoset  set to true
##    .isGroup      set to true, not quite correct, but necessary for using
##                  some dispatcher functions in the GAP library
##    .coxeter      <W> from the argument
##    .F0Mat        as described above
##    .F0Perm       permutation on roots of Parent(<W>) induced by F0Mat: X->X
##    .w1           (unique) element in <W> such that w1*F0Perm fixes 
##                  the set of simple roots of <W>
##    .phi          w1*F0Perm
##    .operations   CoxeterCosetOps
##    .callArgs     information used by 'Print'
##  
##  In the second form 'CoxeterCoset' returns the component <rec>.coxeterCoset,
##  if present.
##   
CoxeterCoset:=function(arg)
  local W, PW, F0, F0Mat, F0perm, tmp, res;

  # just return  component .coxeterCoset:
  if Length(arg)=1 and IsRec(arg[1]) and IsBound(arg[1].coxeterCoset) then
    return arg[1].coxeterCoset;
  fi;

  W:=arg[1];
  PW:=Parent(W);
  res:=rec(isDomain:=true,
           isFinite:=true,
           isCoxeterCoset:=true,
           # unfortunately we have to set this to get the 'CharTable'
           # command working:
           isGroup:=true,
           coxeter:=W);

  if Length(arg)>1 then
    F0:=arg[2];
    if IsPerm(F0) then
      if W.rank<>W.semisimpleRank then
        Error("#I Coset representative can be given as permutation",
	      " only for semisimple datum.\n");
      else
	F0Mat:=MatXPerm(W,F0);
      fi;
    else
      F0Mat:=F0;
    fi;
  else
    F0Mat:=IdentityMat(W.rank);
  fi;
  

  if F0Mat=IdentityMat(W.rank) then
    res.F0Mat:=F0Mat;
    res.F0Perm:=();
  else
    # checking finite order of F0Mat:
    if W.rank>W.semisimpleRank then
      OrderMat(F0Mat);
    fi;
    
    tmp:=PW.roots*PW.simpleRoots;
    tmp:=PermListList(tmp,tmp*F0Mat);
    if tmp=false then
      Error("#I matrix for F0 must normalize set of roots of parent.\n");
    fi;
    
    # checking if subsystem is normalized:
    if Set(W.rootInclusion)<>Set(OnTuples(W.rootInclusion,tmp)) then
      Error("#I matrix for F0 must normalize set of roots.\n");
    fi;
    
    # checking if coroots of parent are normalized:
    if not IsNormalizing(PW.coroots*PW.simpleCoroots,TransposedMat(F0Mat)) then
      Error("#I transposed of matrix for F0 must normalize set of coroots",
            " of parent.\n");
    fi;
    res.F0Perm:=PermMatX(W,F0Mat);
    res.F0Mat:=F0Mat;
  fi;

  res.phi:=ReducedInCoxeterCoset(W,res.F0Perm);

  res.w1:=res.phi/res.F0Perm;

  res.operations:=CoxeterCosetOps;
  
  # for printing:
  if Length(arg)=1 or res.F0Mat=IdentityMat(arg[1].rank) then
    res.callArgs:=[];
  else
    # check for simplification:
    tmp:=arg[1].rootInclusion{[1..arg[1].semisimpleRank]};
    if arg[1].rank=arg[1].semisimpleRank and
       Set(OnTuples(tmp,res.F0Perm))=Set(tmp) then
      res.callArgs:=MappingPermListList(tmp, OnTuples(tmp,res.F0Perm));
    else
      res.callArgs:=arg[2];
    fi;
  fi;

  return res;
end;


#############################################################################
##
#F  CoxeterSubCoset(  <WF>, <gens>[, <Omega> ][, <w> ] ) . . . . . . subcoset 
#F  of a Coxeter coset
##   
##  The arguments <gens> and (  optional) <Omega> must be  as in the  call
##  'ReflectionSubgroup(CoxeterGroup(<WF>),   <gens>[, <Omega> ]  )'.    
##  <w> must be  an element of  CoxeterGroup(<WF>) such that the root system
##  of 'ReflectionSubgroup(CoxeterGroup(<WF>), <gens>[, <Omega>  ]  )' is 
##  invariant   under <w>*<WF>.F0Perm. The default for <w> is ().
##  
CoxeterSubCoset:=function(arg)
  local WF, W, w, tmp, Om, gens, res;
  
  if Length(arg)=1 then
    arg:=arg[1];
  fi;
  
  if IsBound(arg[1].parent) then
    WF:=arg[1].parent;
  else
    WF:=arg[1];
  fi;
  W:=WF.coxeter;
  
## gens:=W.rootInclusion{arg[2]};
  gens:=arg[2];
  
  if Length(arg)>2 then
    if IsGroup(arg[3]) then
      Om:=arg[3];
    elif arg[3] in W then
      w:=arg[3];
    else
      Error("must give w in WF.coxeter.\n");
    fi;
  fi;
  
  if Length(arg)>3 and arg[4] in W then
    w:=arg[4];
  fi;
  
  if not IsBound(w) then
    w:=();
  fi;
  
  res:=rec(isDomain:=true,
           isFinite:=true,
           # unfortunately we have to set this to get through the dispatcher
           # function for 'CharTable':
           isGroup:=true,
           parent:=WF);
  
  if IsBound(Om) then
    res.coxeter:=ReflectionSubgroup(W,gens,Om);
  else
    res.coxeter:=ReflectionSubgroup(W,gens);
  fi;

  # checking, if w*WF.F0Perm normalizes subroot system:
  tmp:=Set(res.coxeter.rootInclusion);
  if not OnSets(tmp,w*WF.F0Perm)=tmp then
    Error("must give w, such that w * WF.F0Perm normalizes ",
          "subroot system.\n");
  fi;
  
  res.F0Mat:=MatXPerm(W,w)*WF.F0Mat;
  res.F0Perm:=w*WF.F0Perm;
  
  res.phi:=ReducedInCoxeterCoset(res.coxeter,res.F0Perm);

  res.w1:=res.phi/res.F0Perm;

  res.operations:=CoxeterCosetOps;

  return res;
end;

#############################################################################
##
#F  Frobenius( <W> )  . . . . . . . .  returns function which gives the action
#F  of W.F0Perm on a permutation or on the set of roots.
##  
##  
##  The function f returned by 'Frobenius( <W> )' can be applied to:
##    a permutation x:           then f(x) = x^(<W>.F0Perm^-1)
##    a list l of integers:      then f(l) = List(l, i->i^(<W>.F0Perm^-1))
##    a record r with component   
##        .operations.Frobenius: then f(r) = r.operations.Frobenius(<W>,r)
##  
##  If the simple roots are   invariant under <W>.F0Perm then the   second
##  form gives  the action of the Frobenius  on elements of the Coxeter group
##  written as words in the generators.
##  
Frobenius:=function(W)
  local finv;
  finv:=W.F0Perm^-1;
  return function(x)
    if IsPerm(x) then 
      return x^finv;
    elif IsList(x) then 
      return OnTuples(x,finv);
    elif IsRec(x) then
      if IsBound(x.operations) and  IsBound(x.operations.Frobenius) then 
	return x.operations.Frobenius(W,x);
      else
        Error("no method Frobenius available.\n");
      fi;
   fi;
  end;
end;
