#############################################################################
##
#A  hecke.g                CHEVIE library           Meinolf Geck, Jean Michel 
##
#A  $Id: hecke.g,v 1.3 1997/03/21 15:24:50 werner Exp $
##
#Y  Copyright (C) 1992 - 1996  Lehrstuhl D f\"ur Mathematik, RWTH Aachen, and
#Y  Universite Paris VII.
##
##  This file contains  GAP functions for working with Hecke algebras and
##  their character tables.
##

##############################################################################
##
#F HeckeCharTableIrred(type,inds {,Ibond}{,param{,sqrtparam}})
## accepts an argument which is the type of an irreducible Coxeter group
## (like a component of the result of CartanType).
## If inds is a single number n it is interpreted as [1..n]
##
HeckeCharTableIrred:=function(arg)
  local inds,bond,param,sqrtparam,rank,r,tbl,type,tmp;
  type:=arg[1];
  if IsList(arg[2]) then inds:=arg[2];rank:=Length(inds);
  else rank:=arg[2];inds:=[1..rank];
  fi;
  if type="I" then bond:=arg[3];arg:=arg{[4..Length(arg)]};
  else arg:=arg{[3..Length(arg)]};
  fi;
  if IsBound(arg[1]) then param:=ShallowCopy(arg[1]);else param:=1;fi;
  if not IsList(param) then param:=List([1..rank],x->param);fi;
  if IsBound(arg[2]) then sqrtparam:=arg[2];else sqrtparam:=[];fi;
  if not IsList(sqrtparam) then sqrtparam:=List([1..rank],x->sqrtparam);fi;
  r:=CHEVIE.Load(type,rank);
  if   type in ["A","~A","D"] then tbl:=r.HeckeCharTable(rank,param[1]);
  elif type in ["B","C"]   then tbl:=r.HeckeCharTable(rank,param{[1,2]});
  elif type="E" and rank=6 then tbl:=r.HeckeCharTable(param[1]);
  elif (type="E" and rank in [7,8]) or type="H" then
    if IsBound(sqrtparam[1]) then tbl:=r.HeckeCharTable(sqrtparam[1]);
    else Error("need sqrt parameters for CharTable(H(E",rank,"))");
    fi;
  elif type="F" then tbl:=r.HeckeCharTable(param[2],param[3]);
  elif type="G" then
    if IsBound(sqrtparam[1]) then Add(param,sqrtparam[1]*sqrtparam[2]);fi;
    tbl:=ApplyFunc(r.HeckeCharTable,param);
  elif type="I" then
    param:=Concatenation([bond],param);
    if IsBound(sqrtparam[1]) then Add(param,sqrtparam[1]*sqrtparam[2]);fi;
    tbl:=ApplyFunc(r.HeckeCharTable,param);
  fi;
  tbl.classtext:=List(tbl.classtext,x->
                          OnTuples(x,MappingPermListList([1..rank],inds)));
  if not IsBound(tbl.name) then
    tbl.name:=tbl.identifier;
  fi;
  if not IsBound(tbl.order) then
    tbl.order:=tbl.size;
  fi;
  return tbl;
end;

##############################################################################
##
## HeckeAlgebraOps: operations for Hecke algebras.
##

HeckeAlgebraOps:=OperationsRecord("HeckeAlgebraOps");

HeckeAlgebraOps.Print:=function(H)
  Print("Hecke(",CoxeterGroup(H),",",H.parameter,",",H.sqrtParameter,")");
end;

HeckeAlgebraOps.CharTable:=function(H)local W,t,tmp,res,cl,f;
  W:=CoxeterGroup(H);
  tmp:=List(CartanType(W),function(t)
     return ApplyFunc(HeckeCharTableIrred,
     Concatenation(t,[H.parameter{t[2]},SublistUnbnd(H.sqrtParameter,t[2])]));
     end);
  res:=tmp[1];
  for t in tmp{[2..Length(tmp)]} do
    Unbind(res.powermap);Unbind(t.powermap);
# the above line because of a bug trying to get table of H(FFE)
    res:=CharTableDirectProduct(res,t); 
    Unbind(res.fusionsource); Unbind(res.fusions);
  od;
  res.parameter:=H.parameter; res.sqrtParameter:=H.sqrtParameter;
  res.cartan:=W.cartan;
  res.irredinfo:=List(CharParams(W),x->rec(charparam:=x,
                                           charname:=CharName(W,x)));
  cl:=ChevieClassInfo(W);
  for f in RecFields(cl) do res.(f):=cl.(f);od;
  return res;
end;

HeckeAlgebraOps.Basis:=function(H,basis)
  if not IsBound(HeckeAlgebraOps.(basis)) then
     if basis in ["C","C'","D","D'"] then ReadChv("prg/kl"); # hack
     else Error("basis ",basis," unknown");fi;
  fi;
  return function(arg) return HeckeAlgebraOps.MakeBasisElt(H,basis,arg);end;
end;

HeckeAlgebraOps.CharParams:=H->CharParams(CoxeterGroup(H));

HeckeAlgebraOps.SchurElements:=H->List(CharParams(H),p->SchurElement(H,p));

HeckeAlgebraOps.Group:=H->H.coxeter;

##############################################################################
##
## HeckeEltOps: operations for Hecke elements.
##

HeckeEltOps:=OperationsRecord("HeckeEltOps");

HeckeEltOps.Normalize:=function(t)local i,elm,coeff;
  SortParallel(t.elm,t.coeff);
  elm:=[];coeff:=[];
  for i in [1..Length(t.elm)] do
    if i=1 or t.elm[i]<>t.elm[i-1] then
	 Add(elm,t.elm[i]);Add(coeff,t.coeff[i]);
    else coeff[Length(coeff)]:=coeff[Length(coeff)]+t.coeff[i];
    fi;
  od;
  i:=List(coeff,x->x<>x*0);
  t.coeff:=ListBlist(coeff,i); t.elm:=ListBlist(elm,i);
  return t;
end;

HeckeEltOps.\+:=function(x,y)local H;
  H:=Hecke(x);
  if not IsIdentical(H,Hecke(y)) then Error("not elements of the same algebra");fi;
  if x.basis <> y.basis then return Basis(H,"T")(x)+Basis(H,"T")(y);
  else
    x:=ShallowCopy(x);
    x.elm:=Concatenation(x.elm,y.elm);
    x.coeff:=Concatenation(x.coeff,y.coeff);
    return HeckeEltOps.Normalize(x);
  fi;
end;

HeckeEltOps.\-:=function(h1,h2) return h1+(-1)*h2; end;

## The q-argument allows calculations to done in the generic Hecke algebra
## even though H.parameter may be specialized (allows "generic" Cp()) -apm

HeckeEltOps.genprod:=function(x,y,q)local H,W,res,temp,temp1,s,i,j,e,qs;
  H:=Hecke(x);W:=CoxeterGroup(H);
  res:=ShallowCopy(x);res.elm:=[];res.coeff:=[];
  for i in [1..Length(x.elm)] do
    temp:=x.coeff[i]*y;
    for s in Reversed(CoxeterWord(W,x.elm[i])) do
      temp1:=ShallowCopy(x);temp1.elm:=[];temp1.coeff:=[];
      qs:=q[W.rootRestriction[s]];
      for j in [1..Length(temp.elm)] do
        e:=temp.elm[j];
        if s^e>W.parentN then
          Add(temp1.elm,e);Add(temp1.coeff,(qs-1)*temp.coeff[j]);
          Add(temp1.elm,W.generators[W.rootRestriction[s]]*e);
          Add(temp1.coeff,qs*temp.coeff[j]);
        else
          Add(temp1.elm,W.generators[W.rootRestriction[s]]*e);
          Add(temp1.coeff,temp.coeff[j]);
        fi;
      od;
      temp:=HeckeEltOps.Normalize(temp1);
    od;
    res:=res+temp;
  od;
  return res;
end;

HeckeEltOps.\*:=function(x,y)local H,res;
  H:=Hecke(y);
  if not IsRec(x) then x:=x*H.unit;fi;
  if IsPolynomial(x) or IsInt(x) then 
     res:=ShallowCopy(y);res.coeff:=res.coeff*x;
     return res;
  fi;
  if not IsIdentical(H,Hecke(x)) then Error("not elements of the same algebra");fi;
  if IsBound(HeckeAlgebraOps.(x.basis).prod) then
    return HeckeAlgebraOps.(x.basis).prod(H,x,y);
  elif IsBound(HeckeAlgebraOps.(y.basis).prod) then
    return HeckeAlgebraOps.(y.basis).prod(H,x,y);
  elif x.basis<>y.basis then return Basis(H,"T")(x)*Basis(H,"T")(y);
  elif x.basis="T" then return HeckeEltOps.genprod(x,y,H.parameter);
  else return Basis(H,x.basis)(Basis(H,"T")(x)*Basis(H,"T")(y));
  fi;
end;

#############################################################################
##
#F AlphaInvolution(h)    
## The involution on Hecke Elements defined by T_w->T_{w^{-1}}
## (and same in other bases)
HeckeEltOps.AlphaInvolution := function(h)
    h:=ShallowCopy(h);
    h.elm:=List(h.elm,x->x^-1);
    return h;
end;

#HeckeEltOps.\=:=function(x,y)local H;
#  H:=Hecke(x);
#  if H<>Hecke(y) then return false;fi;
#  if x.basis=y.basis then return x.elm=y.elm and x.coeff=y.coeff;
#  else return Basis(H,"T")(x)=Basis(H,"T")(y);
#  fi;
#end;

HeckeEltOps.Print:=function(h)Print(String(h));end;

HeckeEltOps.String:=function(h) 
  local i,j,coeff,hcoeff,elm,helm,s,needsbrackets,order,res;
  res:="";
  if h.elm=[] then Append(res,"0");fi;
  order:=function(x,y) return Length(x)<Length(y) or
			      (Length(x)=Length(y) and x<y);end;
  hcoeff:=ShallowCopy(h.coeff);
  helm:=List(h.elm,x->CoxeterWord(CoxeterGroup(Hecke(h)),x));
  SortParallel(helm,hcoeff,order);
  for i in [1..Length(helm)] do
    coeff:=String(hcoeff[i]);
    needsbrackets:=false;
    for j in [2..Length(coeff)] do
      if (coeff[j]='+' or coeff[j]='-') and coeff[j-1]<>'^' then
	needsbrackets:=true;fi;
    od;
    if needsbrackets then coeff:=Concatenation("(",coeff,")");
    elif coeff="1" then coeff:="";
    elif coeff="-1" then coeff:="-";
    fi;
    if Position(coeff,'-')<>1 and i<>1 then Append(res,"+");fi;
    Append(res,String(coeff));
    if CHEVIE.PrintHecke="GAP" and coeff<>"" and coeff<>"-" then 
      Append(res,"*");
    fi;
    elm:="";
    for s in helm[i] do Append(elm,String(s));Add(elm,',');od;
    Append(res,h.basis);
    Append(res,Concatenation("(",String(elm{[1..Length(elm)-1]}),")"));
  od;
  return String(res);
end;

HeckeEltOps.\^:=function(h,n) local i,p,H,W;
  H:=Hecke(h);W:=CoxeterGroup(H);
  if IsRec(n) then return n^-1*h*n;
  elif n<0 then
    if Length(h.elm)=1 and h.basis="T" then
      if n=-1 then
	p:=Basis(H,"T")([()],[h.coeff[1]^-1]);
	for i in Reversed(CoxeterWord(W,h.elm[1])) do
	  p:=p*Basis(H,"T")([W.generators[i],()],
	           [H.parameter[i]^-1,H.parameter[i]^-1-1]);
	od;
	return p;
      else
	return (h^-1)^(-n);
      fi;
    else
      Error("negative exponent implemented only for single T_w");
    fi;
  else
   p:=Basis(H,h.basis)(Basis(H,"T")());
   while n>0 do
    if n mod 2 <> 0 then p:=p*h;fi;
    h:=h*h;
    n:=QuoInt(n,2);
   od;
   return p;
   fi;
end;

HeckeEltOps.\/:=function(a,b) return a*b^-1;end;

HeckeEltOps.Coefficient:= function(T,elm)local p;
  if IsList(elm) then elm:=PermCoxeterWord(CoxeterGroup(Hecke(T)),elm);fi;
  p:=Position(T.elm,elm);
  if p=false then return 0*Hecke(T).unit;else return T.coeff[p];fi;
end;

HeckeEltOps.Frobenius:=function(W,x)
  x:=ShallowCopy(x);x.elm:=List(x.elm,y->Frobenius(W)(y));
  return x;
  end;

HeckeAlgebraOps.MakeBasisEltRec:=function(H,basis,elm,coeff)
  return rec(coeff:=coeff,elm:=elm,basis:=basis,
               operations:=HeckeAlgebraOps.(basis),hecke:=H);
end;

# The function below is made member of HeckeAlgebraOps just to hide it
HeckeAlgebraOps.MakeBasisElt:=function(H,basis,V)local w,res,s,h;
#F The Hecke-element making functions accept the forms:
##  Basis(H,<basis>)(perm)
##  Basis(H,<basis>)([perms],[coeffs])
##  Basis(H,<basis>)([s_1,...,s_n])
##  Basis(H,<basis>)(s_1,...,s_n) 
##  Basis(H,<basis>)(h) 
## where [s_1,...,s_n] is a (non-necessarily reduced if <basis>="T") CoxeterWord.
## In the last form h is a hecke element and the function tries
## to convert h to <basis>. It first looks if h has a method .(<basis>),
## and if not converts h to T and then to <basis>.
  if IsBound(HeckeAlgebraOps.(basis).init) then
     HeckeAlgebraOps.(basis).init(H);
  fi;
  if Length(V)=1 and IsRec(V[1]) then
    h:=V[1];
    if h.basis=basis then return h;
    elif IsBound(h.operations.(basis)) then return h.operations.(basis)(h);
    else return HeckeAlgebraOps.(basis).(basis)(h.operations.T(h));
    fi;
  fi;
  if IsBound(HeckeAlgebraOps.(basis).MakeBasisElt) then
    return HeckeAlgebraOps.(basis).MakeBasisElt(H,V);
  fi;
  if V=[] then
    return HeckeAlgebraOps.MakeBasisEltRec(H,basis,[()],[H.unit]);
  fi;
  h:=V[1];
  if IsInt(h) then V:=[V];fi;
  if IsBound(V[2]) then 
    return HeckeAlgebraOps.MakeBasisEltRec(H,basis,h,V[2]);
  elif IsPerm(h) then
   return HeckeAlgebraOps.MakeBasisEltRec(H,basis,[h],[H.unit]);
  fi;
  # now check if the CoxeterWord is reduced -- else assume basis is "T"
  h:=V[1];
  w:=PermCoxeterWord(CoxeterGroup(H),h);
  if CoxeterLength(CoxeterGroup(H),w)=Length(h) then 
    return HeckeAlgebraOps.MakeBasisEltRec(H,basis,[w],[H.unit]);
  fi;
  if basis<>"T" then
   Error("Construction with non-reduced words implemented only for basis T");
  fi;
  res:=Basis(H,"T")();
  for s in h do res:=res*Basis(H,"T")(CoxeterGroup(H).generators[s]);od;
  return res;
end;

#############################################################################
##
#F CreateHeckeBasis(basis,ops) Create a new basis of Hecke algebras
##
## basis, a string, is the name of the new basis and ops=rec() overrides
## some operations in HeckeEltOps. ops should contain at least methods
## .T and .(basis) to convert to and from the T-basis. If it contains
## a method init(H) this is called when using Basis(H,basis)

CreateHeckeBasis:=function(basis,ops)local o, b;
  if not IsRec(ops) or not IsBound(ops.T) or not IsBound(ops.(basis))
  then Error("The operations record must contain methods .T and .",basis);
  fi;
  HeckeAlgebraOps.(basis):=OperationsRecord(
    Concatenation("Hecke",basis,"Ops"),HeckeEltOps);
  for o in RecFields(ops) do
    HeckeAlgebraOps.(basis).(o) := ops.(o);
  od;
end;

CreateHeckeBasis("T",rec(
   T:=x->x,
   BetaInvolution:=function(h)local H;
     if not IsBound(HeckeAlgebraOps.InitKL) then ReadChv("prg/kl");fi;
     H:=Hecke(h);
     HeckeAlgebraOps.InitKL(H,"beta involution");
     h:=HeckeAlgebraOps.Beta(h,"T");
     h.coeff:=H.genQ^-CoxeterGroup(H).N*h.coeff;
     return h;
   end,
   AltInvolution:=function(h)local H,i;
     if not IsBound(HeckeAlgebraOps.InitKL) then ReadChv("prg/kl");fi;
     H:=Hecke(h);
     HeckeAlgebraOps.InitKL(H,"alt involution");
     h:=HeckeAlgebraOps.Alt(h,"T");
     for i in [1..Length(h.coeff)] do
       h.coeff[i]:=(-H.genQ^-2)^CoxeterLength(CoxeterGroup(H),h.elm[i])*
                                                                h.coeff[i];
     od;
     return h;
   end
));

#############################################################################
##
#F  Hecke(<W>[,<parameters>, [<sqrtParameters>]] )  Make W into a Hecke algebra
##
##  'Hecke'  adds  to the record of the Coxeter group  <W> functions 
##  T (resp. Cp, C, D, Dp)  which produce,  for each element  w in W, the
##  corresponding standard basis element T(w) in the Hecke algebra H of W
##  (resp C'(w), C(w), D(w), D'(w) which are the 4 Kazhdan-Lusztig bases).
##  The T basis elements multiply as, if we set q:=<parameters>:
##     T(s)^2 = q[s]T(1)+(q[s]-1)T(s), for standard generators s
##  and  T(vw) = T(v) T(w), if l(vw) = l(v) + l(w). 
##  <parameters> can be one Ring element to which all parameters are set, or a
##  list of length W.semisimpleRank where parameters corresponding to conjugate
##  reflections must be equal.
##   For certain operations (character tables in types G2, E7, E8 or conversion
##  of KL bases to standard bases) the square root of the parameters need
##  to be specified.
##
##  Example: Hecke(CoxeterGroup("B",2),[q,q^2]);
##
CoxeterGroupOps.Hecke:=function(arg)local W,H,i;
  W:=arg[1]; 
  H:=rec(coxeter:=W,isGroup:=true,isDomain:=true,operations:=HeckeAlgebraOps);
# isGroup set to true for CharTable to accept H as argument
  if IsBound(arg[2]) then H.parameter:=arg[2];else H.parameter:=1;fi;
  if not IsList(H.parameter) 
  then H.parameter:=List([1..W.semisimpleRank],i->H.parameter);
  elif not ForAll([1..W.semisimpleRank], i->H.parameter[i]=
    H.parameter[W.rootRestriction[W.orbitRepresentative[i]]])
  then 
    Error("Hecke algebra parameters should be equal for conjugate reflections");
  fi;
  if IsBound(arg[3]) then H.sqrtParameter:=arg[3];else H.sqrtParameter:=[];fi;
  if not IsList(H.sqrtParameter) then
       H.sqrtParameter:=List([1..W.semisimpleRank],i->H.sqrtParameter);
  fi;
  if W.semisimpleRank=0 then H.unit:=1;else H.unit:=Product(H.parameter)^0;fi;
 # check for cached Hecke algebras in W
  if not IsBound(W.heckeAlgebras) then W.heckeAlgebras:=[];fi;
  i:=PositionProperty(W.heckeAlgebras,x->x.parameter=H.parameter
		       and x.sqrtParameter=H.sqrtParameter);
  if i=false then Add(W.heckeAlgebras,H);
  else H:=W.heckeAlgebras[i];
  fi;
  return H;
end; 

#############################################################################
##
#F  HeckeSubAlgebra( <H>, <roots> ) . . . . . . . . . . . . . . . . . . . or
#F  HeckeSubAlgebra( <H>, <subgroup> ) . . . . . . . .   Hecke Sub-Algebra
## 
##  Given a Hecke Algebra H and either a set of roots of CoxeterGroup(H) 
##  given as their index in the roots of W, or a reflection subgroup of W, 
##  return the Hecke sub-algebra generated by the T_s corresponding to 
##  these roots. The roots must be simple roots if any parameter is not 1.
##
##  As for Subgroup, a subalgebra of a subalgebra is given as a subalgebra
##  of the largest algebra.
##
HeckeSubAlgebra:=function(H,subW)local W,subroots,res;
  W:=CoxeterGroup(H);
  if IsList(subW) then subW:=ReflectionSubgroup(W,subW);fi;
  subroots:=W.rootRestriction{subW.rootInclusion{[1..subW.semisimpleRank]}};
  if ForAll(H.parameter,x->x=1) then return Hecke(subW,1,1);
  elif ForAll(subroots,x->x in [1..W.semisimpleRank]) then
  res:=Hecke(subW,H.parameter{subroots},SublistUnbnd(H.sqrtParameter,subroots));
   res.parent:=H;
   return res;
  else Error("Generators of a sub-Hecke algebra should be simple reflections");
  fi;
end;

#############################################################################
##
#F  HeckeClassPolynomials( <h> [, <reps>] )  . . . . . . the class polynomials
##
##  'HeckeClassPolynomials' returns  the  class polynomials of the element <h>
##  of the Hecke algebra <H> with  respect  to  representatives  <reps>  of 
##  minimal lengths in  the conjugacy  classes of the Coxeter group W of <H>.
##  <h>  is  an element of <H> given in any of the standard bases.
##  If absent, <reps> is taken to be 'CoxeterConjugacyClasses(W)'

HeckeClassPolynomials:=function(arg)
  local H,reps,p,h,T,q,res,min,l,max,inds,i,o,orb,W,H;
  H:=Hecke(arg[1]);T:=Basis(H,"T");h:=T(arg[1]);W:=CoxeterGroup(H);
  if IsBound(arg[2]) then 
    reps:=arg[2];
    p:=MappingPermListList(List(reps,x->PositionClass(W,PermCoxeterWord (W,x)),
        [1..Length(reps)]));
  else reps:=CoxeterConjugacyClasses(W);p:=();
  fi;

# if there exists w conjugate by cyclic shift to elm and an elementary
# reflection s such that l(sws)=l(elm)-2 then orb returns
# H.parameter*T_sws+(H.parameter-1)T_sw  else an error is signaled.

  orb:=function(elm)local orbit,w,y,s,yy,ind;
    orbit:=[elm];
    for w in orbit do
      for s in [1..W.semisimpleRank] do
	y:=W.generators[s]*w;yy:=y*W.generators[s];
	if W.rootInclusion[s]^w>W.parentN then
	  if W.rootInclusion[s]^(y^-1)>W.parentN then 
	    return T([yy,y],[H.parameter[s],H.parameter[s]-1]);
	  else if not yy in orbit then Add(orbit,yy);fi;
	  fi;
	else
	  if W.rootInclusion[s]^(y^-1)>W.parentN then 
	    if not yy in orbit then Add(orbit,yy);fi;
	  fi;
	fi; 
      od;
    od;
    Error("theory");
  end;

  min:=[1..Length(reps)]*0*H.parameter[1]^0;
  while Length(h.elm)>0 do
   l:=List(h.elm,x->CoxeterLength(W,x));
   max:=Maximum(l);
   inds:=Filtered([1..Length(h.elm)],x->max=l[x]);
   max:=T(h.elm{inds},h.coeff{inds});
   h:=h-max;
   for i in [1..Length(max.elm)] do
     o:=PositionClass(W,max.elm[i])^p;
     if Length(reps[o])=CoxeterLength(W,max.elm[i]) then 
       min[o]:=min[o]+max.coeff[i];
     else
       o:=max.coeff[i]*orb(max.elm[i]);
       Append(h.elm,o.elm);Append(h.coeff,o.coeff);
     fi;
   od;
   HeckeEltOps.Normalize(h);
  od;
  return min;
end;

#############################################################################
##
#F  HeckeCharValues( <T> [,<irreds>] )  . . character values of <irreds> on <T>
##
##  T is an element of an Iwahori-Hecke algebra H (expressed in any of the 5
##  bases) and <irreds> is a set of irreducible characters of H.
##  HeckeCharValues returns the values of irreds on T, using
##  HeckeClassPolynomials.
##   If absent, <irreds> is taken to be the set of irreducible characters
##  of <H>.

HeckeCharValues:=function(arg)local irrs;
  if IsBound(arg[2]) then irrs:=arg[2]*Hecke(arg[1]).parameter[1]^0;
  else irrs:=CharTable(Hecke(arg[1])).irreducibles;fi;
  return irrs*HeckeClassPolynomials(arg[1]);
  end;

#############################################################################
##
#F  HeckeReflectionRepresentation( <H> )  . . . . . . . . . . . . . . . . . . 
#F  the reflection representation of Hecke algebra <H>, returned as a set
#F  of matrices for the generators.
##
HeckeReflectionRepresentation:=function(H)
  local t,i,j,n,m,a,C,v,W;
  if Length(Set(H.parameter))>1 then
    Error("Refl. repr. of Hecke algebra with unequal parameters not implemented");
  fi;
  v:=H.parameter[1];
  C:=[];
  W:=CoxeterGroup(H);
  for i in [1..W.semisimpleRank] do
    C[i]:=[];
    for j in [1..i-1] do
      m:=OrderPerm(W.generators[i]*W.generators[j]);
      C[i][j]:=(2+E(m)+E(m)^(-1))*v^0;
      if m=2 then C[j][i]:=0; else C[j][i]:=v;fi;
    od;
    C[i][i]:=v+v^0;
  od;
  t:=[];
  for i in [1..W.semisimpleRank] do
    a:=[];
    for j in [1..W.semisimpleRank] do
      a[j]:=List([1..W.semisimpleRank],function(k) return 0*v; end);
      a[j][j]:=v;
      a[j][i]:=a[j][i]-C[i][j];
    od;
    Add(t,a);
  od;
  return t;
end;
  
#############################################################################
##
#F  CheckHeckeDefiningRelations( <H> , <t> )  . . . . . . . . . . . . . check
#F  the  defining  relations  of  a  Hecke algebra for a given representation
##  
##  'CheckHeckeDefiningRelations' returns true or false, according to whether
##  a given  set  <t>  of  matrices  corresponding to the standard generators
##  of the Hecke algebra <H>  defines a representation or not. 
## 
CheckHeckeDefiningRelations:=function(H,t)local i,j,n,l;
  for i in [1..CoxeterGroup(H).semisimpleRank] do
    if not t[i]^2=H.parameter[i]*t[i]^0+(H.parameter[i]-1)*t[i] then
      InfoChevie("#I  Error in ",Ordinal(i)," quadratic relation\n");
      return false;
    fi;
    for j in [i+1..CoxeterGroup(H).semisimpleRank] do
      n:=OrderPerm(CoxeterGroup(H).generators[i]*
                                   CoxeterGroup(H).generators[j]);
      l:=t{List([1..n+1],k->i+(j-i)*(k mod 2))};
      if Product(l{[1..n]})<>Product(l{[2..n+1]}) then
        InfoChevie("#I  Error in (",i,",",j,") braid relation\n");
        return false;
      fi;
    od;
  od;
  return true;
end;
  
############################################################################
##
#F  SchurElement( <H>, <p> ) .  Schur element of character with .charparam p
##
##  'SchurElement' returns $P/D_\phi$ where P is the
##  Poincare polynomial of <H>, and $\phi$ is the irreducible character
##  of $H$ with .charparam <p> and $D_\phi$ is the generic degree of $\phi$.
##
##  For types I_2,G_2 with unequal parameters, sqrtParameter needs to be bound.
#
SchurElement:=function(H,phi)local rank,param,res,sqrtparam,t,type,r,v,i;
  res:=[];
  if not IsRec(H) or not IsBound(H.parameter) then
    Error("<H> should be a Hecke algebra");
  fi;
  t:=CartanType(CoxeterGroup(H));
  for i in [1..Length(t)] do
    type:=t[i][1];
    rank:=Length(t[i][2]);
    r:=CHEVIE.Load(type,rank);
    param:=H.parameter{t[i][2]};
    sqrtparam:=SublistUnbnd(H.sqrtParameter,t[i][2]);
    if type in ["A","~A"] then Add(res,r.SchurElement(phi[i],param[1]));
    elif type="B" or type="C" then Add(res,r.SchurElement(phi[i],param));
    elif type="D" then Add(res,r.PoincarePolynomial(rank,param[1])/
                                Value(r.vcycGenericDegree(phi[i]),param[1]));
    elif type="E" then 
      Add(res,Value(ApplyFunc(vcyc,r.vcycpolschurelms[Position(r.CharParams(),
                   phi[i])]),param[1]));
    elif type="H" then
     Add(res,FastValue(r.vpolschurelms[Position(r.CharParams(),phi[i])],
                                                                 param[1]));
    elif type="F" then Add(res,r.SchurElement(phi[i],param[2],param[3]));
    elif type="G" then
      param:=Concatenation([phi[i]],param);
      if IsBound(sqrtparam[1]) then Add(param,sqrtparam[1]*sqrtparam[2]);fi;
      Add(res,ApplyFunc(r.SchurElement,param));
    elif type="I" then 
      param:=Concatenation([t[i][3],phi[i]],param);
      if IsBound(sqrtparam[1]) then Add(param,sqrtparam[1]*sqrtparam[2]);fi;
      Add(res,ApplyFunc(r.SchurElement,param));
    fi;
  od;
  return Product(res);
end;

############################################################################
##
#F  PoincarePolynomial( <H> ) . . . Poincare Polynomial of Hecke Algebra <H>
##
##  returns the Poincare Polynomial of <H> (equal to the Schur element
##  associated to the 'identity' character).
#
PoincarePolynomial:=function(H)local rank,param,res,p,t,type,r,v;
  res:=[];
  if not IsRec(H) or not IsBound(H.parameter) then
    Error("<H> should be a Hecke algebra");
  fi;
  for t in CartanType(CoxeterGroup(H)) do
    type:=t[1];
    rank:=Length(t[2]);
    r:=CHEVIE.Load(type,rank);
    p:=MappingPermListList([1..rank],t[2])^-1;
    param:=List([1..rank],i->H.parameter[i^p]);
    if type in ["A","~A","D"] then Add(res,r.PoincarePolynomial(rank,param[1]));
    elif type="B" or type="C" then Add(res,r.PoincarePolynomial(rank,param));
    elif type="E" then 
      Add(res,Value(ApplyFunc(vcyc,r.vcycpolschurelms[r.PositionId()]),
                                                                  param[1]));
    elif type="H" then
      Add(res,FastValue(r.vpolschurelms[r.PositionId()],param[1]));
    elif type="F" then 
      Add(res,r.SchurElements[r.PositionId()](param[2],param[3]));
    elif type="G" then Add(res,ApplyFunc(r.PoincarePolynomial,param));
    elif type="I" then Add(res,r.PoincarePolynomial(t[3],param[1],param[2]));
    fi;
  od;
  return Product(res);
end;

###########################################################################
##
#F  LowestPowerGenericDegrees(W)  . . . . The list of a for characters of W
##
##  returns for each character chi, the valuation of the generic degree of chi.
##
LowestPowerGenericDegrees:=function(W)local t, rank, type, res,r;
  if not IsBound(W.a) then
    res:=[];
    for t in CartanType(W) do
      type:=t[1];
      rank:=Length(t[2]);
      r:=CHEVIE.Load(type,rank);
      if type in ["F","G"] then Add(res,r.chara);
      elif type="E" then Add(res,List(r.vcycpolschurelms,x->-x[1][1]));
      elif type="H" then Add(res,List(r.vpolschurelms,x->-x[2]));
      elif type in ["A","~A","B","C","D"] then 
        Add(res,List(r.CharParams(rank),x->r.Chara(x)));
      elif type="I" then 
        Add(res,List(r.CharParams(t[3]),x->r.Chara(t[3],x)));
      fi;
    od;
    W.a:=List(Cartesian(res),Sum);
  fi;
  return W.a;
end;

###########################################################################
##
#F  HighestPowerGenericDegrees(W) . . . . The list of A for characters of W
##
##  returns for each character chi, the degree of the generic degree of chi.
##
HighestPowerGenericDegrees:=function(W)local t, rank, type, res,r;
  if not IsBound(W.A) then
    res:=[];
    for t in CartanType(W) do
      type:=t[1];
      rank:=Length(t[2]);
      r:=CHEVIE.Load(type,rank);
      if type in ["F","G"] then Add(res,r.charA);
      elif type="E" then 
	Add(res,List(r.vcycpolschurelms,x->W.N-x[1][1]-
	             Sum([2..Length(x[1])],i->x[1][i]*Phi(i-1))));
      elif type="H" then 
	Add(res,List(r.vpolschurelms,x->W.N-x[2]-Length(x[1])+1));
      elif type in ["A","~A","B","C","D"] then 
        Add(res,List(r.CharParams(rank),x->r.CharA(x)));
      elif type="I" then 
        Add(res,List(r.CharParams(t[3]),x->r.CharA(t[3],x)));
      fi;
    od;
    W.A:=List(Cartesian(res),Sum);
  fi;
  return W.A;
end;

############################################################################
##
#F  JInductionTable( <subgroup>, <group> ) . . . J-induction of characters
##  
##  This    function  works   like  'InductionTable'   but   computes  the
##  J-induction, as defined in [Lusztig-book, p.78].
##  
JInductionTable:=function(u,g)
  local it,i,j,au,ag;
  if not (IsBound(u.isCoxeterGroup) and IsBound(g.isCoxeterGroup)) then
    Error("groups must be CoxeterGroup groups.\n");
  fi;
  it:=InductionTable(u,g);
  it.headString:=String(Concatenation("J-",it.headString));
  au:=LowestPowerGenericDegrees(u);ag:=LowestPowerGenericDegrees(g);
  for i in [1..Length(au)] do
    for j in [1..Length(ag)] do
      if ag[j]<>au[i] then it.scalar[j][i]:=0;fi;
    od;
  od;
  it.operations.Print:=function(t)
       Print("JInductionTable( ",t.u,", ",t.g,")");
     end;
  return it;
end;

############################################################################
##
#F  HeckeScalarProducts( <ti>, <char1>, <char2> )  . . . . . . . . . . . . .
#F   . . . . . . . . . . . . . . . . . .  scalar products between characters
##
##  'HeckeScalarProducts' specializes the parameters to  1  and  returns the
##  matrix of ordinary scalar products between the  specialized  characters.
##  This only works if the parameters are all equal to 1 or are monomials.
##
HeckeScalarProducts:=function(ti,char1,char2)local specialize;
  if ForAny(ti.parameter,i->i<>1 and (not IsPolynomial(i) or Value(i,1)<>1))
  then Error("parameters must be equal to 1 or have value 1 at 1\n");
  fi;
  specialize:=function(p)
    if IsPolynomial(p) then
      return Value(p,1);
    else
      return p;
    fi;
  end;
  char1:=List(char1,x->List(x,specialize));
  char2:=List(char2,x->List(x,specialize));
  return List(char1,i->List(char2,j->ScalarProduct(ti,i,j)));
end;

###########################################################################
##
#F HeckeCentralMonomials( <H> ) . . . . . . . . scalars of central element 
## . . . . . . . . . . . . . . . . . . . . .  on irreduciblerepresentations
##
## 'HeckeCentralMonomials'  computes  the  scalars  by  which  the  square
## of the longest element  acts on  the irreducible representations of the 
## Iwahori-Hecke algebra.

HeckeCentralMonomials:=function(H)local ti,refl,v,W;
  W:=CoxeterGroup(H);ti:=CharTable(W);
  refl:=Filtered([1..Length(ti.classtext)],r->Length(ti.classtext[r])=1);
  v:=List(refl,i->H.parameter[W.rootRestriction[ti.classtext[i][1]]]);
  return List(ti.irreducibles,irr->Product(List([1..Length(refl)],r->
              v[r]^(ti.classes[refl[r]]*(irr[refl[r]]+irr[1])/irr[1]))));
end;

###########################################################################
##
#F HeckeCharValuesGood( <H>, <w> ) . . . character values of T_w^d for good w
##
## 'HeckeCharValuesGood'  computes the values of the irreducible characters
## of the  Iwahori-Hecke algebra <H> on T_w^d, the $d$-th power of the basis
## element corresponding to the good element <w> of the Coxeter group, where 
## $d$ is the order of <w>. The point is that the character table of the Hecke
## algebra is not used, and that all the eigenvalues of T_w^d are monomials
## in H.parameters, so this can be used to find the absolute value of the
## eigenvalues of T_w, a step towards computing the character table of <H>.
#
HeckeCharValuesGood:=function(H,w)local HI,HJ,J,eig,wd,o,W;
  W:=CoxeterGroup(H);
  wd:=GoodCoxeterWord(W,w);if wd=false then Error("");fi;
  if Length(wd)=0 or wd[1][1]<>[1..W.semisimpleRank] then
         wd:=Concatenation([[[1..W.semisimpleRank],0]],wd);fi;
  HJ:=HeckeSubAlgebra(H,[]);eig:=[1];
  for J in wd{[Length(wd),Length(wd)-1..1]} do
    HI:=HeckeSubAlgebra(H,J[1]);
    eig:=InductionTable(CoxeterGroup(HJ),CoxeterGroup(HI)).scalar*eig;
    o:=List(HeckeCentralMonomials(HI),x->x^(J[2]/2));
    eig:=List([1..Length(eig)],i->eig[i]*o[i]);
    HJ:=HI;
  od;
  return eig;
end;
