#############################################################################
##
#A  weyla.g                     CHEVIE library               G\"otz Pfeiffer.
##
#A  $Id: weyla.g,v 1.1 1997/01/21 13:46:47 gap Exp $
##
#Y  Copyright (C) 1992 - 1996  Lehrstuhl D f\"ur Mathematik, RWTH Aachen, IWR
#Y  der Universit\"at Heidelberg, University of St. Andrews, and   University 
#Y  Paris VII.
##
##  This file contains special functions for  Coxeter groups and Iwahori-Hecke 
##  algebras  of type  A.
##

CHEVIE.A:=rec();

CHEVIE.A.PositionId:=function(n) return Length(Partitions(n+1)); end;

CHEVIE.A.PositionSgn:=function(n) return 1; end;

CHEVIE.A.PositionRefl:=function(n) return Length(Partitions(n+1))-1; end;

#############################################################################
##
#F  CHEVIE.A.ClassInfo( <n> )  . . . . . . . . . . . conjugacy classes for type A
##  
##  'CHEVIE.A.ClassInfo' returns a record with three components:
##    classtext:   representatives of minimal length in  the  conjugacy  
##                 classes, as words in generators in standard order
##    classparams:  partitions, parameterizing the classes
##    classnames:  strings for partitions
##  
##  The  ordering  corresponds  to the  order of  the columns of the ordinary
##  character table of the symmetric  group $S_{n+1}$, as returned by the GAP
##  function 'CharTable("Symmetric", <n+1>)'.
##  
CHEVIE.A.ClassInfo:=function(n)
   local w, i, l, pi, res;

   res:= rec(classtext:=[],classnames:=[],classparams:=[]);

   #  loop over the labels.
   for pi in Partitions(n+1) do
      w:= [];  i:= 0;

      #  build cycle for each part.
      for l in pi do
         Append(w, [i+1 .. i+l-1]);
         i:= i+l;
      od;

      Add(res.classtext, w);
      Add(res.classparams,pi);
      Add(res.classnames,IntListToString(pi));
   od;

   #  return the list.
   return res;

end;

#how to make .charname from .charparam
CHEVIE.A.CharName:=IntListToString;

#############################################################################
##
#F  CHEVIE.A.CharParams( <n> ) . . . . . . . . . . . . . . . characters for type A
##  
CHEVIE.A.CharParams:=n->Partitions(n+1);

#############################################################################
##
#V  CHEVIE.A.ClassParam( <W>, <w> )  . . . . . .  class parameter of w
##
##  given an element w of a Coxeter group  W of type A, returns the classparam
##  of its conjugacy class.
##  
ClassParamAold:= function(W, w)
   local i, v;
   v:= ();
   for i in w do
      v:= v * (i,i+1);
   od;
   v:= CycleLengths(v, [1..W.semisimpleRank+1]);
   Sort(v);
   return Reversed(v);
end;

# faster version of 'ClassParamA':
# (now parameter is n=semisimple rank and the word w)
CHEVIE.A.ClassParam:= function(n, w)
  local i, x, res, mark, cyc;
  
  x:=();
  for i in w do
    x:=x*(i,i+1);
  od;
  
  res:=[];
  mark:=[1..n+1];
  for i in [1..n+1] do
    if mark[i]<>0 then
      cyc:=CyclePermInt(x,i);
      Add(res,Length(cyc));
      mark{cyc}:=cyc*0;
    fi;
  od;
  
  Sort(res);
  return Reversed(res);
end;

CHEVIE.A.CharTable:=function(rank)local tbl,cl,f;
  tbl:=CharTable("Symmetric",rank+1);
  if not IsBound(tbl.size) then
    tbl.size:=Factorial(rank+1);
  fi;
  tbl.identifier:=String(Concatenation("W(A",String(rank),")"));
  tbl.cartan:=CartanMat("A",rank);
  cl:=CHEVIE.A.ClassInfo(rank);
  for f in RecFields(cl) do tbl.(f):=cl.(f);od;
  tbl.irredinfo:=List(CHEVIE.A.CharParams(rank),x->rec(charparam:=x,
      charname:=CHEVIE.A.CharName(x)));
  return tbl;
end;

#############################################################################
##
#F  HeckeCharTableA( <n>, <q> )  . . . . . . . . character table of $H(A_n)$.
##
##  'HeckeCharTableA'  returns  the character  table  of the Hecke algebra of
##  type $A_n$ with parameter <q>.
##
CHEVIE.A.Hk:= Copy(CharTableSymmetric);

CHEVIE.A.Hk.identifier:= "HeckeA";  

CHEVIE.A.Hk.specializedname:= 
   (nq -> Concatenation("H(A", String(nq[1]-1), ")"));

CHEVIE.A.Hk.order:=(nq-> Factorial(nq[1]));  
CHEVIE.A.Hk.size:=CHEVIE.A.Hk.order;

CHEVIE.A.Hk.domain:= function(nq)
   return IsList(nq) and Length(nq) = 2 and IsInt(nq[1]) and nq[1] > 0;
end;

CHEVIE.A.Hk.text:= "generic character table of Hecke algebras of type A";

CHEVIE.A.Hk.classparam:= [nq -> Partitions(nq[1])];
CHEVIE.A.Hk.charparam:= CHEVIE.A.Hk.classparam;

CHEVIE.A.Hk.irreducibles[1][1]:= function(nq, gamma, pi)

   local n, q, k, val, alpha, dif;

   #  for the sake of clearness.
   n:= nq[1]; q:= nq[2];

   #  termination condition.
   if n = 0 then 
      return q^0;
   fi;

   #  get length of longest cycle.
   k:= pi[1];
   
   val:= 0 * q;

   #  loop over the partitions of n-k.
   for alpha in Partitions(n-k) do
      dif:= DifferencePartitions(gamma, alpha);
      if dif <> false then
         val:= val + (q-1)^(dif.cc-1) * (-1)^dif.ll * q^(k-dif.ll-dif.cc)
           * CHEVIE.A.Hk.irreducibles[1][1]([n-k, q], alpha, pi{[2..Length(pi)]});
      fi;
   od;

   # return the result.
   return val;

end;

CHEVIE.A.Hk.matrix:= function(nq)

   local scheme, beta, pm, i, m, n, q, k, t, res, charCol, hooks;

   n:= nq[1]; q:= nq[2];
   pm:= [];
   scheme:= [];

   #  how to encode all hooks.
   hooks:= function(beta, m)
      local i, j, hk, pr, cbs, prs, leg, hks, ll, lg, lh, gamma, new;

      hks:= List([1..m], x->[]);
      prs:= [];

      #  find all hooks.
      for i in beta do
         leg:= 0;
         for j in Reversed([0..i-1]) do
            if  j in beta then
                leg:= leg + 1;
            else
                Add(prs, [rec(from:= i, to:= j, leg:= leg)]);
            fi;
         od;
      od;

      #  construct combinations.
      cbs:= Copy(prs);
      for hk in cbs do

         #  extend.
         for pr in prs do
            if pr[1].to > hk[Length(hk)].from then
               Add(cbs, Concatenation(hk, pr));
            fi;
         od;

         #  encode.
         ll:= Sum(hk, x-> x.from - x.to);
         lg:= Sum(hk, x-> x.leg);
         lh:= Length(hk);
         new:= rec(wgt:= (-1)^lg * q^(ll-lg-lh) * (q-1)^(lh-1),
                   adr:= 1);

         #  recalculate address.
         if ll < m-1 then 
            gamma:= Difference(beta, List(hk, x-> x.from));
            UniteSet(gamma, List(hk, x-> x.to));
            if 0 in gamma then
               j:= 0;
               while gamma[j+1] = j do
                  j:= j+1;
               od;
               gamma:= Sublist(gamma, [j+1..Length(gamma)]) - j;
            fi;
            new.adr:= Position(pm[m-ll], gamma);
         fi;

         #  insert.
         Add(hks[ll], new);
      od;

      return hks;
   end;

   #  collect hook encodings.
   InfoCharTable2("#I  Scheme: \c");
   for i in [1..n] do
      InfoCharTable2(i, " \c");
      pm[i]:= List(Partitions(i), BetaSet);
      scheme[i]:= [];
      for beta in pm[i] do
         Add(scheme[i], hooks(beta, i));
      od;
   od;
   InfoCharTable2("done.\n");

   #  how to construct a new column.
   charCol:= function(n, t, k)
      local col, pi, hk, val;

      col:= [];
      for pi in scheme[n] do
         val:= 0*q;
         for hk in pi[k] do
            val:= val + hk.wgt * t[hk.adr];
         od;
         Add(col, val);
      od;
      return col;
   end;

   pm:= List([1..n-1], x-> []);

   #  construct the columns.
   InfoCharTable2("#I  Cycles: \c");
   for m in [1..QuoInt(n,2)] do
      InfoCharTable2(m, " \c");
      Add(pm[m], charCol(m, [1], m));

      for k in [m+1..n-m] do
         for t in pm[k-m] do
            Add(pm[k], charCol(k, t, m));
         od;
      od;
   od;
   InfoCharTable2("done.\n");

   InfoCharTable2("#I  Tables: \c");
   res:= [];
   for k in [1..n-1] do
      InfoCharTable2(k, " \c");
      for t in pm[n-k] do
         Add(res, charCol(n, t, k));
      od;
   od;
   InfoCharTable2("done.\n");

   Add(res, charCol(n, [1], n));

   return TransposedMat(res);

end;

CHEVIE.A.HeckeCharTable:= function(n, q)local tbl;

   tbl:= CharTableSpecialized(CHEVIE.A.Hk, [n+1, q]);
   tbl.cartan:= CartanMat("A", n);
   tbl.parameter:= List([1..n], x-> q);
   tbl.classtext:= CHEVIE.A.ClassInfo(n).classtext;
   tbl.classparam:= List(tbl.classparam, x->x[2]);
   tbl.classnames:= List(tbl.classparam, IntListToString);
   tbl.irredinfo:=List(CHEVIE.A.CharParams(n),x->
                       rec(charparam:=x,charname:=CHEVIE.A.CharName(x)));
   return tbl;
end;

#############################################################################
##
#F  CHEVIE.A.PoincarePolynomial( <n>, <q> ) . . Poincare polynomial for type A.
##
##  'CHEVIE.A.PoincarePolynomial'  returns the Poincare  polynomial of the 
##  Coxeter group $W$ of type $A_n$, ie.  the sum of $q^l(w)$ over all 
##  elements $w$ of $W$.
##
CHEVIE.A.PoincarePolynomial:= function(n, q)
   return Product([1..n], i-> Sum([0..i], k-> q^k));
end;


#############################################################################
##
#F  CHEVIE.A.SchurElement( <n>, <q> ) . . . . . . . Schur elements for type A.
##
##  'CHEVIE.A.SchurElement' returns the list of Schur elements for the character
##  table of the Hecke algebra of type $A_n$ with parameter $q$.
##
##  [Reference: Carter II, 13.5, p. 446.]
##
CHEVIE.A.SchurElement:= function(alpha, u)

   local i, j, lambda, res;

   lambda:= BetaSet(alpha);
   res:= u^Binomial(Length(lambda), 3);
   for i in lambda do
      for j in [0..i-1] do
         if j in lambda then
            res:= res/u^j;
         else
            res:= res * Sum([0..i-j-1], e-> u^e);
         fi;
      od;
   od;

   return res;

end;

CHEVIE.A.FakeDegree:=function(p,q)return
CHEVIE.A.PoincarePolynomial(Sum(p)-1,q)/CHEVIE.A.SchurElement(p,q);end;

CHEVIE.A.Charb:=p->p*[0..Length(p)-1];
CHEVIE.A.Chara:=CHEVIE.A.Charb;

CHEVIE.A.CharB:=p->Sum(p)*(Sum(p)-1)/2-CHEVIE.A.Chara(AssociatedPartition(p));
CHEVIE.A.CharA:=CHEVIE.A.CharB;
   
#############################################################################
##
#E  Emacs . . . . . . . . . . . . . . . . . . . . . .  local emacs variables.
##
##  Local Variables:
##  fill-column: 77
##  fill-prefix: "##  "
##  End:
##
