(***********************************************************************

                    Mathematica-Compatible Notebook

This notebook can be used on any computer system with Mathematica 4.0,
MathReader 4.0, or any compatible application. The data for the notebook 
starts with the line containing stars above.

To get the notebook into a Mathematica-compatible application, do one of 
the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing the 
word CacheID, otherwise Mathematica-compatible applications may try to 
use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
***********************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[     91922,       2669]*)
(*NotebookOutlinePosition[    109432,       3124]*)
(*  CellTagsIndexPosition[    109172,       3111]*)
(*WindowFrame->Normal*)



Notebook[{

Cell[CellGroupData[{
Cell["TSi ProPac: Backlash & Friction", "Title"],

Cell["Harry G. Kwatny", "Subtitle"],

Cell["\<\
Department of Mechanical Engineering & Mechanics
Drexel University
Philadelphia, PA 19104\
\>", "Subsubtitle"],

Cell[CellGroupData[{

Cell["Introduction", "Section",
  CellTags->"Introduction"],

Cell[TextData[{
  "ProPac contains version 3.0 of Dynamics. It includes functions for dealing \
with nondifferentiable nonlinearities including backlash and coulomb, and \
static friction with Stribeck effect. The primary functions are:\n",
  StyleBox["BacklashPotential\n",
    FontFamily->"Courier New"],
  StyleBox["JointFrictionPotential",
    FontFamily->"Courier New"],
  StyleBox["  ",
    FontFamily->"Courier New"],
  "\nIn addition, simplification rules/functions have been implemented \
including\n",
  StyleBox["SimplifyDiracDelta1\nUnitStep2Sign",
    FontFamily->"Courier New"],
  "\nAlso modifications have been made to several TSiDynamics functions \
including\n",
  StyleBox["CreateModel\nCreateModelSpecial",
    FontFamily->"Courier New"],
  "\nto facilitate working with nondifferentiable nonlinearities. This \
notebook illustrates the use of these functions.\nFirst we load the package."
}], "Text",
  CellTags->{"Introduction", "i:1"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(<< "\<ProPac`\>"\)], "Input",
  CellLabel->"In[1]:=",
  PageWidth->Infinity,
  AspectRatioFixed->True,
  CellTags->"Introduction"],

Cell[BoxData[
    \("\n ProPac has been successfully loaded. It contains packages\n for \
multibody dynamics, linear control and nonlinear control.\n Appropriate \
packages will be loaded automatically as required.\n Packages can be manually \
loaded by simply entering GeometricTools,\n Dynamics, LinearControl, \
NonlinearControl or MEXTools as desired.\n Once a package is loaded, enter \
?GeomtricTools, ?Dynamics,\n ?LinearControl, ?NonlinearControl, or ?MEXTools, \
respectively,\n to obtain a list of available functions. Help is also \
available\n in the Help Browser under Add-ons."\)], "Print",
  CellTags->"Introduction"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(Dynamics\)], "Input",
  CellLabel->"In[2]:=",
  CellTags->"Introduction"],

Cell[BoxData[
    \("  *** Dynamics successfully loaded ***"\)], "Print",
  CellTags->"Introduction"],

Cell[BoxData[
    \(Dynamics\)], "Output",
  CellLabel->"Out[2]=",
  CellTags->"Introduction"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(?BacklashPotential\)\)], "Input",
  CellLabel->"In[3]:=",
  CellTags->{"Introduction", "i:2"}],

Cell[BoxData[
    \("BacklashPotential[P,e,x] returns the backlash potential function\n \
associated with a Hertz impact potential function P (must be defined \n as a \
pure function), a backlash parameter e, and a coordinate x. Note\n that it \
may be appropriate to use BacklashPotential in conjunction with\n \
SpringPotential."\)], "Print",
  CellTags->{"Introduction", "i:2"}]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(?JointFrictionPotential\)\)], "Input",
  CellLabel->"In[4]:=",
  CellTags->{"Introduction", "i:4"}],

Cell[BoxData[
    \("JointFrictionPotential[v_,Fv_,Fc_,Fs_,vs_] assembles a dissipation\n \
function of Lur'e type for a joint that involves viscous, Coulomb and\n \
Stribeck effects. The potential function has meaning if the arguments\n are \
functions of the joint variables as described below. The function is \
listable.\n \n The function calls:\n \n  \
JointFrictionPotential[v_,Fv_,Fc_,Fs_,vs_] or \
JointFrictionPotential[v_,Fv_,Fc_]\n \n return the dissipation function for a \
joint friction component. The\n 'velocity' v must be an expression involving \
joint quasi-velocities.\n It may be a function of the coordinates as well.\n \
\n The friction parameters are:\n \n           Fv-viscous friction \
coefficient\n           Fc-Coulomb friction coefficient,\n           \
Fs-Static friction,\n           vs-Stribeck velocity.\n \n These coefficients \
may be functions of the joint coordinates."\)], "Print",
  CellTags->{"Introduction", "i:4"}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Backlash", "Section",
  CellTags->{"Backlash", "i:3"}],

Cell[CellGroupData[{

Cell["Two Masses & Spring With Backlash", "Subsection",
  CellTags->{"Backlash", "i:3"}],

Cell[CellGroupData[{

Cell[BoxData[{
    \(r1 = {1}; H1 = Transpose[{{0, 0, 0, 1, 0, 0}}];\), "\n", 
    \(q1 = {x1}; p1 = {v1};\), "\n", 
    \(r2 = {1}; H2 = Transpose[{{0, 0, 0, 1, 0, 0}}];\), "\n", 
    \(q2 = {x2}; p2 = {v2};\), "\n", 
    \(\(JointLst = {{r1, H1, q1, p1}, {r2, H2, q2, p2}};\)\), "\n", 
    \(\({JV, JX, JH} = Joints[JointLst];\)\)}], "Input",
  CellLabel->"In[5]:=",
  PageWidth->Infinity,
  AspectRatioFixed->True,
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[
    \("Computing joint 1 kinematics"\)], "Print",
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[
    \("Computing joint 2 kinematics"\)], "Print",
  CellTags->{"Backlash", "i:3"}]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[{
    \(com1 = {0, 0, 0}; mass1 = m1; out1 = {2, {0, 0, 0}};\), "\n", 
    \(\(Inertia1 = {{0, 0, 0}, {0, 0, 0}, {0, 0, 0}};\)\), "\n", 
    \(com2 = {0, 0, 0}; mass2 = m2; out2 = {3, {0, 0, 0}};\), "\n", 
    \(\(Inertia2 = {{0, 0, 0}, {0, 0, 0}, {0, 0, 0}};\)\), "\n", 
    \(BodyLst = {{com1, {out1}, mass1, Inertia1}, {com2, {out2}, mass2, 
          Inertia2}}\), "\n", 
    \(\(TreeLst = {{{1, 1}, {2, 2}}};\)\)}], "Input",
  CellLabel->"In[6]:=",
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[
    \({{{0, 0, 0}, {{2, {0, 0, 0}}}, 
        m1, {{0, 0, 0}, {0, 0, 0}, {0, 0, 0}}}, {{0, 0, 0}, {{3, {0, 0, 0}}}, 
        m2, {{0, 0, 0}, {0, 0, 0}, {0, 0, 0}}}}\)], "Output",
  CellLabel->"Out[6]=",
  CellTags->{"Backlash", "i:3"}]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(PEBack[x_] := \((k1*x^2)\)/2;\)\), "\n", 
    \(PE1 = BacklashPotential[PEBack, e, \((x1 - x2)\)]\), "\n", 
    \(\(PE2 = \((1/2)\)*k2*x2^2;\)\), "\n", 
    \(\(PE = PE1 + PE2;\)\), "\n", 
    \(\(Q = {F1, F2};\)\), "\n", 
    \(CreateModel[JointLst, BodyLst, TreeLst, \(-g\), PE, Q, JV, JX, 
      JH]\)}], "Input",
  CellLabel->"In[8]:=",
  PageWidth->Infinity,
  AspectRatioFixed->True,
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[
    \(1\/2\ k1\ \((\(-e\) + Abs[x1 - x2])\)\^2\ UnitStep[\(-e\) + 
          Abs[x1 - x2]]\)], "Output",
  CellLabel->"Out[8]=",
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[
    \("Computing Potential Functions"\)], "Print",
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[
    \("Computing Inertia Matrix"\)], "Print",
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[
    \("Computing Poincare Function"\)], "Print",
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[
    \({{{{1}}, {{1}}}, {{{1, 0, 0, x1}, {0, 1, 0, 0}, {0, 0, 1, 0}, {0, 0, 0, 
            1}}, {{1, 0, 0, x2}, {0, 1, 0, 0}, {0, 0, 1, 0}, {0, 0, 0, 
            1}}}, {{{0}, {0}, {0}, {1}, {0}, {0}}, {{0}, {0}, {0}, {1}, {0}, \
{0}}}, {{m1 + m2, m2}, {m2, m2}}, {{0, 0}, {0, 0}}, {\(-F1\) + 
          k1\ \((\(-e\) + Abs[x1 - x2])\)\ \((UnitStep[x1 - x2] - 
                UnitStep[\(-x1\) + x2])\)\ UnitStep[\(-e\) + 
                Abs[x1 - x2]], \(-F2\) + k2\ x2 + 
          k1\ \((\(-e\) + Abs[x1 - x2])\)\ \((\(-UnitStep[x1 - x2]\) + 
                UnitStep[\(-x1\) + x2])\)\ UnitStep[\(-e\) + 
                Abs[x1 - x2]]}, {v1, v2}, {x1, x2}}\)], "Output",
  CellLabel->"Out[9]=",
  CellTags->{"Backlash", "i:3"}]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(CreateModel[JointLst, BodyLst, TreeLst, \(-g\), PE, Q]\)], "Input",
  CellLabel->"In[10]:=",
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[
    \("Computing Joint Kinematics"\)], "Print",
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[
    \("Computing joint 1 kinematics"\)], "Print",
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[
    \("Computing joint 2 kinematics"\)], "Print",
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[
    \("Computing Potential Functions"\)], "Print",
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[
    \("Computing Inertia Matrix"\)], "Print",
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[
    \("Computing Poincare Function"\)], "Print",
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[
    \({{{{1}}, {{1}}}, {{{1, 0, 0, x1}, {0, 1, 0, 0}, {0, 0, 1, 0}, {0, 0, 0, 
            1}}, {{1, 0, 0, x2}, {0, 1, 0, 0}, {0, 0, 1, 0}, {0, 0, 0, 
            1}}}, {{{0}, {0}, {0}, {1}, {0}, {0}}, {{0}, {0}, {0}, {1}, {0}, \
{0}}}, {{m1 + m2, m2}, {m2, m2}}, {{0, 0}, {0, 0}}, {\(-F1\) + 
          k1\ \((\(-e\) + Abs[x1 - x2])\)\ \((UnitStep[x1 - x2] - 
                UnitStep[\(-x1\) + x2])\)\ UnitStep[\(-e\) + 
                Abs[x1 - x2]], \(-F2\) + k2\ x2 + 
          k1\ \((\(-e\) + Abs[x1 - x2])\)\ \((\(-UnitStep[x1 - x2]\) + 
                UnitStep[\(-x1\) + x2])\)\ UnitStep[\(-e\) + 
                Abs[x1 - x2]]}, {v1, v2}, {x1, x2}}\)], "Output",
  CellLabel->"Out[10]=",
  CellTags->{"Backlash", "i:3"}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Double Pendulum With Backlash Constrained Upper Joint", "Subsection",
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[{
    \(r1 = {1}; H1 = {{1}, {0}, {0}, {0}, {0}, {0}};\), "\n", 
    \(q1 = {a1x}; p1 = {w1x};\), "\n", 
    \(r2 = {1}; H2 = {{1}, {0}, {0}, {0}, {0}, {0}};\), "\n", 
    \(q2 = {a2x}; p2 = {w2x};\), "\n", 
    \(\(JointLst = {{r1, H1, q1, p1}, {r2, H2, q2, p2}};\)\)}], "Input",
  CellLabel->"In[11]:=",
  PageWidth->Infinity,
  AspectRatioFixed->True,
  CellTags->{"Backlash", "i:3"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(\({JV, JX, JH} = Joints[JointLst];\)\)], "Input",
  CellLabel->"In[16]:=",
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[
    \("Computing joint 1 kinematics"\)], "Print",
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[
    \("Computing joint 2 kinematics"\)], "Print",
  CellTags->{"Backlash", "i:3"}]
}, Open  ]],

Cell["Define Body Data", "Text",
  Evaluatable->False,
  CellHorizontalScrolling->False,
  AspectRatioFixed->True,
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[{
    \(com1 = {0, 0, \(-l1\)}; mass1 = m1; 
    out1 = {2, {0, 0, \(-l1\)}};\), "\n", 
    \(\(Inertia1 = DiagonalMatrix[{0, 0, 0}];\)\), "\n", 
    \(com2 = {0, 0, \(-l2\)}; mass2 = m2; 
    out2 = {3, {0, 0, \(-l2\)}};\), "\n", 
    \(\(Inertia2 = DiagonalMatrix[{0, 0, 0}];\)\), "\n", 
    \(\(BodyLst = {{com1, {out1}, mass1, Inertia1}, {com2, {out2}, mass2, 
            Inertia2}};\)\)}], "Input",
  CellLabel->"In[17]:=",
  PageWidth->Infinity,
  AspectRatioFixed->True,
  CellTags->{"Backlash", "i:3"}],

Cell["Define Interconnection Structure", "Text",
  Evaluatable->False,
  CellHorizontalScrolling->False,
  AspectRatioFixed->True,
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[
    \(\(TreeLst = {{{1, 1}, {2, 2}}};\)\)], "Input",
  CellLabel->"In[22]:=",
  PageWidth->Infinity,
  AspectRatioFixed->True,
  CellTags->{"Backlash", "i:3"}],

Cell["\<\
Define Potential Energy
In this case only gravity contributes to the potential energy. The only \
generalized forces are external torques acting at the two joints.\
\>", "Text",
  Evaluatable->False,
  CellHorizontalScrolling->False,
  AspectRatioFixed->True,
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[
    \(g = gc; PE = 0; Q = {T1, T2};\)], "Input",
  CellLabel->"In[23]:=",
  PageWidth->Infinity,
  AspectRatioFixed->True,
  CellTags->{"Backlash", "i:3"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(\({V, X, H, M, Cp, Fp, p, q} = 
        CreateModel[JointLst, BodyLst, TreeLst, g, PE, Q];\)\)], "Input",
  CellLabel->"In[24]:=",
  PageWidth->Infinity,
  AspectRatioFixed->True,
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[
    \("Computing Joint Kinematics"\)], "Print",
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[
    \("Computing joint 1 kinematics"\)], "Print",
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[
    \("Computing joint 2 kinematics"\)], "Print",
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[
    \("Computing Potential Functions"\)], "Print",
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[
    \("Computing Inertia Matrix"\)], "Print",
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[
    \("Computing Poincare Function"\)], "Print",
  CellTags->{"Backlash", "i:3"}]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(PEBack[x_] := \((k1*x^2)\)/2;\)\), "\n", 
    \(PE1 = BacklashPotential[PEBack, e, \((a1x)\)]\), "\n", 
    \(\(PE2 = \((1/2)\)*k2*a2x^2;\)\), "\n", 
    \(\(PE = PE1 + P2;\)\)}], "Input",
  CellLabel->"In[25]:=",
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[
    \(1\/2\ k1\ \((\(-e\) + Abs[a1x])\)\^2\ UnitStep[\(-e\) + 
          Abs[a1x]]\)], "Output",
  CellLabel->"Out[25]=",
  CellTags->{"Backlash", "i:3"}]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(\({JV, JX, JH, MM, Cp, Fp, pp, qq} = 
        CreateModel[JointLst, BodyLst, TreeLst, g, PE, Q, JV, JX, 
          JH];\)\)], "Input",
  CellLabel->"In[27]:=",
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[
    \("Computing Potential Functions"\)], "Print",
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[
    \("Computing Inertia Matrix"\)], "Print",
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[
    \("Computing Poincare Function"\)], "Print",
  CellTags->{"Backlash", "i:3"}]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(Fp\)], "Input",
  CellLabel->"In[28]:=",
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[
    \({\(-T1\) + 
        gc\ \((l1\ \((m1 + m2)\)\ Sin[a1x] + l2\ m2\ Sin[a1x + a2x])\) + 
        k1\ \((\(-e\) + Abs[a1x])\)\ \((\(-UnitStep[\(-a1x\)]\) + 
              UnitStep[a1x])\)\ UnitStep[\(-e\) + Abs[a1x]], \(-T2\) + 
        gc\ l2\ m2\ Sin[a1x + a2x]}\)], "Output",
  CellLabel->"Out[28]=",
  CellTags->{"Backlash", "i:3"}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Simulation: Numerical Solution", "Subsection",
  Evaluatable->False,
  CellHorizontalScrolling->False,
  AspectRatioFixed->True,
  CellTags->{"Backlash", "i:3"}],

Cell["\<\
We can easily construct a simulation within Mathematica. First assemble the \
system parameter matrices computed above into a system of ordinary \
differential equations using the TSi Dynamics function MakeODEs.\
\>", "Text",
  Evaluatable->False,
  CellHorizontalScrolling->False,
  AspectRatioFixed->True,
  CellTags->{"Backlash", "i:3"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(Equations = MakeODEs[p, q, V, M, Cp, Fp, t]\)], "Input",
  CellLabel->"In[29]:=",
  PageWidth->Infinity,
  AspectRatioFixed->True,
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{
            SuperscriptBox["a1x", "\[Prime]",
              MultilineFunction->None], "[", "t", "]"}], "==", \(w1x[t]\)}], 
        ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox["a2x", "\[Prime]",
              MultilineFunction->None], "[", "t", "]"}], "==", \(w2x[t]\)}], 
        ",", 
        RowBox[{
          
          RowBox[{\(-T1\), 
            "+", \(gc\ \((l1\ \((m1 + m2)\)\ Sin[a1x[t]] + 
                  l2\ m2\ Sin[a1x[t] + a2x[t]])\)\), 
            "+", \(k1\ \((\(-e\) + 
                  Abs[a1x[t]])\)\ \((\(-UnitStep[\(-a1x[t]\)]\) + 
                  UnitStep[a1x[t]])\)\ UnitStep[\(-e\) + Abs[a1x[t]]]\), 
            "+", \(w2x[
                t]\ \((\((\(-l1\)\ l2\ m2\ Sin[a2x[t]] + 
                        l1\^2\ m2\ Cos[a2x[t]]\ Sin[a2x[t]] - 
                        l1\ \((l2\ m2 + l1\ m2\ Cos[a2x[t]])\)\ Sin[
                            a2x[t]])\)\ w1x[t] - 
                  l1\ l2\ m2\ Sin[a2x[t]]\ w2x[t])\)\), "+", 
            
            RowBox[{\((l1\^2\ m1 + l2\^2\ m2 + l1\ l2\ m2\ Cos[a2x[t]] + 
                  l1\ Cos[a2x[t]]\ \((l2\ m2 + l1\ m2\ Cos[a2x[t]])\) + 
                  l1\^2\ m2\ Sin[a2x[t]]\^2)\), " ", 
              RowBox[{
                SuperscriptBox["w1x", "\[Prime]",
                  MultilineFunction->None], "[", "t", "]"}]}], "+", 
            RowBox[{\((l2\^2\ m2 + l1\ l2\ m2\ Cos[a2x[t]])\), " ", 
              RowBox[{
                SuperscriptBox["w2x", "\[Prime]",
                  MultilineFunction->None], "[", "t", "]"}]}]}], "==", "0"}], 
        ",", 
        RowBox[{
          
          RowBox[{\(-T2\), "+", \(gc\ l2\ m2\ Sin[a1x[t] + a2x[t]]\), 
            "-", \(1\/2\ l1\ l2\ m2\ Sin[a2x[t]]\ w1x[t]\ w2x[t]\), 
            "+", \(1\/2\ w1x[
                t]\ \((\(-\((\(-l1\)\ l2\ m2\ Sin[a2x[t]] + 
                          l1\^2\ m2\ Cos[a2x[t]]\ Sin[a2x[t]] - 
                          l1\ \((l2\ m2 + l1\ m2\ Cos[a2x[t]])\)\ Sin[
                              a2x[t]])\)\)\ w1x[t] + 
                  l1\ l2\ m2\ Sin[a2x[t]]\ w2x[t])\)\), "+", 
            RowBox[{\((l2\^2\ m2 + l1\ l2\ m2\ Cos[a2x[t]])\), " ", 
              RowBox[{
                SuperscriptBox["w1x", "\[Prime]",
                  MultilineFunction->None], "[", "t", "]"}]}], "+", 
            RowBox[{\(l2\^2\), " ", "m2", " ", 
              RowBox[{
                SuperscriptBox["w2x", "\[Prime]",
                  MultilineFunction->None], "[", "t", "]"}]}]}], "==", 
          "0"}]}], "}"}]], "Output",
  CellLabel->"Out[29]=",
  CellTags->{"Backlash", "i:3"}]
}, Open  ]],

Cell["\<\
Now, replace parameter symbols by numbers and set up initial conditions.\
\>", "Text",
  Evaluatable->False,
  CellHorizontalScrolling->False,
  AspectRatioFixed->True,
  CellTags->{"Backlash", "i:3"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(DataReplacements = {m1 \[Rule] 1, l1 \[Rule] 2, m2 \[Rule] 1, 
        l2 \[Rule] 1, gc \[Rule] 1, T1 \[Rule] 0, T2 \[Rule] 0, k1 -> 1000, 
        e ->  .1, k2 -> 0}\)], "Input",
  CellLabel->"In[30]:=",
  PageWidth->Infinity,
  AspectRatioFixed->True,
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{\(m1 \[Rule] 1\), ",", \(l1 \[Rule] 2\), ",", \(m2 \[Rule] 1\), 
        ",", \(l2 \[Rule] 1\), ",", \(gc \[Rule] 1\), ",", \(T1 \[Rule] 0\), 
        ",", \(T2 \[Rule] 0\), ",", \(k1 \[Rule] 1000\), ",", 
        RowBox[{"e", "\[Rule]", 
          StyleBox["0.1`",
            StyleBoxAutoDelete->True,
            PrintPrecision->1]}], ",", \(k2 \[Rule] 0\)}], "}"}]], "Output",
  CellLabel->"Out[30]=",
  CellTags->{"Backlash", "i:3"}]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(Equations1 = Equations /. DataReplacements\)], "Input",
  CellLabel->"In[31]:=",
  PageWidth->Infinity,
  AspectRatioFixed->True,
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{
            SuperscriptBox["a1x", "\[Prime]",
              MultilineFunction->None], "[", "t", "]"}], "==", \(w1x[t]\)}], 
        ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox["a2x", "\[Prime]",
              MultilineFunction->None], "[", "t", "]"}], "==", \(w2x[t]\)}], 
        ",", 
        RowBox[{
          
          RowBox[{\(4\ Sin[a1x[t]]\), "+", \(Sin[a1x[t] + a2x[t]]\), 
            "+", \(1000\ \((\(-0.1`\) + 
                  Abs[a1x[t]])\)\ \((\(-UnitStep[\(-a1x[t]\)]\) + 
                  UnitStep[a1x[t]])\)\ UnitStep[\(-0.1`\) + Abs[a1x[t]]]\), 
            "+", \(w2x[
                t]\ \((\((\(-2\)\ Sin[a2x[t]] + 4\ Cos[a2x[t]]\ Sin[a2x[t]] - 
                        2\ \((1 + 2\ Cos[a2x[t]])\)\ Sin[a2x[t]])\)\ w1x[t] - 
                  2\ Sin[a2x[t]]\ w2x[t])\)\), "+", 
            
            RowBox[{\((5 + 2\ Cos[a2x[t]] + 
                  2\ Cos[a2x[t]]\ \((1 + 2\ Cos[a2x[t]])\) + 
                  4\ Sin[a2x[t]]\^2)\), " ", 
              RowBox[{
                SuperscriptBox["w1x", "\[Prime]",
                  MultilineFunction->None], "[", "t", "]"}]}], "+", 
            RowBox[{\((1 + 2\ Cos[a2x[t]])\), " ", 
              RowBox[{
                SuperscriptBox["w2x", "\[Prime]",
                  MultilineFunction->None], "[", "t", "]"}]}]}], "==", "0"}], 
        ",", 
        RowBox[{
          
          RowBox[{\(Sin[a1x[t] + a2x[t]]\), 
            "-", \(Sin[a2x[t]]\ w1x[t]\ w2x[t]\), 
            "+", \(1\/2\ w1x[
                t]\ \((\(-\((\(-2\)\ Sin[a2x[t]] + 
                          4\ Cos[a2x[t]]\ Sin[a2x[t]] - 
                          2\ \((1 + 2\ Cos[a2x[t]])\)\ Sin[a2x[t]])\)\)\ w1x[
                      t] + 2\ Sin[a2x[t]]\ w2x[t])\)\), "+", 
            RowBox[{\((1 + 2\ Cos[a2x[t]])\), " ", 
              RowBox[{
                SuperscriptBox["w1x", "\[Prime]",
                  MultilineFunction->None], "[", "t", "]"}]}], "+", 
            RowBox[{
              SuperscriptBox["w2x", "\[Prime]",
                MultilineFunction->None], "[", "t", "]"}]}], "==", "0"}]}], 
      "}"}]], "Output",
  CellLabel->"Out[31]=",
  CellTags->{"Backlash", "i:3"}]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(vars = Join[p, q]\)], "Input",
  CellLabel->"In[32]:=",
  PageWidth->Infinity,
  AspectRatioFixed->True,
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[
    \({w1x, w2x, a1x, a2x}\)], "Output",
  CellLabel->"Out[32]=",
  CellTags->{"Backlash", "i:3"}]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(InitialConditions = 
      Table[ToExpression[
          ToString[
              vars\[LeftDoubleBracket]
                i\[RightDoubleBracket]] <> "\<[0]==0\>"], {i, 
          Length[vars]}]\)], "Input",
  CellLabel->"In[33]:=",
  PageWidth->Infinity,
  AspectRatioFixed->True,
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[
    \({w1x[0] == 0, w2x[0] == 0, a1x[0] == 0, a2x[0] == 0}\)], "Output",
  CellLabel->"Out[33]=",
  CellTags->{"Backlash", "i:3"}]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(InitialConditions = {w1x[0] == 0, w2x[0] == 0, a1x[0] ==  .1, 
        a2x[0] ==  .2}\)], "Input",
  CellLabel->"In[34]:=",
  PageWidth->Infinity,
  AspectRatioFixed->True,
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{\(w1x[0] == 0\), ",", \(w2x[0] == 0\), ",", 
        RowBox[{\(a1x[0]\), "==", 
          StyleBox["0.1`",
            StyleBoxAutoDelete->True,
            PrintPrecision->1]}], ",", 
        RowBox[{\(a2x[0]\), "==", 
          StyleBox["0.2`",
            StyleBoxAutoDelete->True,
            PrintPrecision->1]}]}], "}"}]], "Output",
  CellLabel->"Out[34]=",
  CellTags->{"Backlash", "i:3"}]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(Equations2 = Join[Equations1, InitialConditions]\)], "Input",
  CellLabel->"In[35]:=",
  PageWidth->Infinity,
  AspectRatioFixed->True,
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{
            SuperscriptBox["a1x", "\[Prime]",
              MultilineFunction->None], "[", "t", "]"}], "==", \(w1x[t]\)}], 
        ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox["a2x", "\[Prime]",
              MultilineFunction->None], "[", "t", "]"}], "==", \(w2x[t]\)}], 
        ",", 
        RowBox[{
          
          RowBox[{\(4\ Sin[a1x[t]]\), "+", \(Sin[a1x[t] + a2x[t]]\), 
            "+", \(1000\ \((\(-0.1`\) + 
                  Abs[a1x[t]])\)\ \((\(-UnitStep[\(-a1x[t]\)]\) + 
                  UnitStep[a1x[t]])\)\ UnitStep[\(-0.1`\) + Abs[a1x[t]]]\), 
            "+", \(w2x[
                t]\ \((\((\(-2\)\ Sin[a2x[t]] + 4\ Cos[a2x[t]]\ Sin[a2x[t]] - 
                        2\ \((1 + 2\ Cos[a2x[t]])\)\ Sin[a2x[t]])\)\ w1x[t] - 
                  2\ Sin[a2x[t]]\ w2x[t])\)\), "+", 
            
            RowBox[{\((5 + 2\ Cos[a2x[t]] + 
                  2\ Cos[a2x[t]]\ \((1 + 2\ Cos[a2x[t]])\) + 
                  4\ Sin[a2x[t]]\^2)\), " ", 
              RowBox[{
                SuperscriptBox["w1x", "\[Prime]",
                  MultilineFunction->None], "[", "t", "]"}]}], "+", 
            RowBox[{\((1 + 2\ Cos[a2x[t]])\), " ", 
              RowBox[{
                SuperscriptBox["w2x", "\[Prime]",
                  MultilineFunction->None], "[", "t", "]"}]}]}], "==", "0"}], 
        ",", 
        RowBox[{
          
          RowBox[{\(Sin[a1x[t] + a2x[t]]\), 
            "-", \(Sin[a2x[t]]\ w1x[t]\ w2x[t]\), 
            "+", \(1\/2\ w1x[
                t]\ \((\(-\((\(-2\)\ Sin[a2x[t]] + 
                          4\ Cos[a2x[t]]\ Sin[a2x[t]] - 
                          2\ \((1 + 2\ Cos[a2x[t]])\)\ Sin[a2x[t]])\)\)\ w1x[
                      t] + 2\ Sin[a2x[t]]\ w2x[t])\)\), "+", 
            RowBox[{\((1 + 2\ Cos[a2x[t]])\), " ", 
              RowBox[{
                SuperscriptBox["w1x", "\[Prime]",
                  MultilineFunction->None], "[", "t", "]"}]}], "+", 
            RowBox[{
              SuperscriptBox["w2x", "\[Prime]",
                MultilineFunction->None], "[", "t", "]"}]}], "==", "0"}], 
        ",", \(w1x[0] == 0\), ",", \(w2x[0] == 0\), ",", 
        RowBox[{\(a1x[0]\), "==", 
          StyleBox["0.1`",
            StyleBoxAutoDelete->True,
            PrintPrecision->1]}], ",", 
        RowBox[{\(a2x[0]\), "==", 
          StyleBox["0.2`",
            StyleBoxAutoDelete->True,
            PrintPrecision->1]}]}], "}"}]], "Output",
  CellLabel->"Out[35]=",
  CellTags->{"Backlash", "i:3"}]
}, Open  ]],

Cell["\<\
Finally, integrate the equations using the Mathematica function NDSolve and \
plot the results.\
\>", "Text",
  Evaluatable->False,
  CellHorizontalScrolling->False,
  AspectRatioFixed->True,
  CellTags->{"Backlash", "i:3"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(sols = 
      NDSolve[Equations2, vars, {t, 0, 19}, PrecisionGoal -> 10]\)], "Input",
  CellLabel->"In[36]:=",
  PageWidth->Infinity,
  AspectRatioFixed->True,
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{"{", 
        RowBox[{
          RowBox[{"w1x", "\[Rule]", 
            
            TagBox[\(InterpolatingFunction[{{0.`, 19.0000000000000017`}}, 
                "<>"]\),
              False,
              Editable->False]}], ",", 
          RowBox[{"w2x", "\[Rule]", 
            
            TagBox[\(InterpolatingFunction[{{0.`, 19.0000000000000017`}}, 
                "<>"]\),
              False,
              Editable->False]}], ",", 
          RowBox[{"a1x", "\[Rule]", 
            
            TagBox[\(InterpolatingFunction[{{0.`, 19.0000000000000017`}}, 
                "<>"]\),
              False,
              Editable->False]}], ",", 
          RowBox[{"a2x", "\[Rule]", 
            
            TagBox[\(InterpolatingFunction[{{0.`, 19.0000000000000017`}}, 
                "<>"]\),
              False,
              Editable->False]}]}], "}"}], "}"}]], "Output",
  CellLabel->"Out[36]=",
  CellTags->{"Backlash", "i:3"}]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(Plot[Evaluate[a1x[t] /. sols], {t, 0, 19}]\)], "Input",
  CellLabel->"In[37]:=",
  PageWidth->Infinity,
  AspectRatioFixed->True,
  CellTags->{"Backlash", "i:3"}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.0501253 0.307091 2.72469 [
[.14912 .29459 -9 -9 ]
[.14912 .29459 9 0 ]
[.27444 .29459 -3 -9 ]
[.27444 .29459 3 0 ]
[.39975 .29459 -9 -9 ]
[.39975 .29459 9 0 ]
[.52506 .29459 -6 -9 ]
[.52506 .29459 6 0 ]
[.65038 .29459 -12 -9 ]
[.65038 .29459 12 0 ]
[.77569 .29459 -6 -9 ]
[.77569 .29459 6 0 ]
[.901 .29459 -12 -9 ]
[.901 .29459 12 0 ]
[.01131 .03462 -24 -4.5 ]
[.01131 .03462 0 4.5 ]
[.01131 .17086 -30 -4.5 ]
[.01131 .17086 0 4.5 ]
[.01131 .44332 -24 -4.5 ]
[.01131 .44332 0 4.5 ]
[.01131 .57956 -18 -4.5 ]
[.01131 .57956 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
.14912 .30709 m
.14912 .31334 L
s
[(2.5)] .14912 .29459 0 1 Mshowa
.27444 .30709 m
.27444 .31334 L
s
[(5)] .27444 .29459 0 1 Mshowa
.39975 .30709 m
.39975 .31334 L
s
[(7.5)] .39975 .29459 0 1 Mshowa
.52506 .30709 m
.52506 .31334 L
s
[(10)] .52506 .29459 0 1 Mshowa
.65038 .30709 m
.65038 .31334 L
s
[(12.5)] .65038 .29459 0 1 Mshowa
.77569 .30709 m
.77569 .31334 L
s
[(15)] .77569 .29459 0 1 Mshowa
.901 .30709 m
.901 .31334 L
s
[(17.5)] .901 .29459 0 1 Mshowa
.125 Mabswid
.04887 .30709 m
.04887 .31084 L
s
.07393 .30709 m
.07393 .31084 L
s
.099 .30709 m
.099 .31084 L
s
.12406 .30709 m
.12406 .31084 L
s
.17419 .30709 m
.17419 .31084 L
s
.19925 .30709 m
.19925 .31084 L
s
.22431 .30709 m
.22431 .31084 L
s
.24937 .30709 m
.24937 .31084 L
s
.2995 .30709 m
.2995 .31084 L
s
.32456 .30709 m
.32456 .31084 L
s
.34962 .30709 m
.34962 .31084 L
s
.37469 .30709 m
.37469 .31084 L
s
.42481 .30709 m
.42481 .31084 L
s
.44987 .30709 m
.44987 .31084 L
s
.47494 .30709 m
.47494 .31084 L
s
.5 .30709 m
.5 .31084 L
s
.55013 .30709 m
.55013 .31084 L
s
.57519 .30709 m
.57519 .31084 L
s
.60025 .30709 m
.60025 .31084 L
s
.62531 .30709 m
.62531 .31084 L
s
.67544 .30709 m
.67544 .31084 L
s
.7005 .30709 m
.7005 .31084 L
s
.72556 .30709 m
.72556 .31084 L
s
.75063 .30709 m
.75063 .31084 L
s
.80075 .30709 m
.80075 .31084 L
s
.82581 .30709 m
.82581 .31084 L
s
.85088 .30709 m
.85088 .31084 L
s
.87594 .30709 m
.87594 .31084 L
s
.92607 .30709 m
.92607 .31084 L
s
.95113 .30709 m
.95113 .31084 L
s
.97619 .30709 m
.97619 .31084 L
s
.25 Mabswid
0 .30709 m
1 .30709 L
s
.02381 .03462 m
.03006 .03462 L
s
[(-0.1)] .01131 .03462 1 0 Mshowa
.02381 .17086 m
.03006 .17086 L
s
[(-0.05)] .01131 .17086 1 0 Mshowa
.02381 .44332 m
.03006 .44332 L
s
[(0.05)] .01131 .44332 1 0 Mshowa
.02381 .57956 m
.03006 .57956 L
s
[(0.1)] .01131 .57956 1 0 Mshowa
.125 Mabswid
.02381 .06187 m
.02756 .06187 L
s
.02381 .08912 m
.02756 .08912 L
s
.02381 .11636 m
.02756 .11636 L
s
.02381 .14361 m
.02756 .14361 L
s
.02381 .1981 m
.02756 .1981 L
s
.02381 .22535 m
.02756 .22535 L
s
.02381 .2526 m
.02756 .2526 L
s
.02381 .27984 m
.02756 .27984 L
s
.02381 .33434 m
.02756 .33434 L
s
.02381 .36158 m
.02756 .36158 L
s
.02381 .38883 m
.02756 .38883 L
s
.02381 .41608 m
.02756 .41608 L
s
.02381 .47057 m
.02756 .47057 L
s
.02381 .49782 m
.02756 .49782 L
s
.02381 .52507 m
.02756 .52507 L
s
.02381 .55231 m
.02756 .55231 L
s
.02381 .00737 m
.02756 .00737 L
s
.02381 .60681 m
.02756 .60681 L
s
.25 Mabswid
.02381 0 m
.02381 .61803 L
s
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
.5 Mabswid
.02381 .57956 m
.02846 .57998 L
.02952 .58014 L
.03068 .5803 L
.03178 .58042 L
.03279 .58049 L
.03399 .58052 L
.03527 .58047 L
.03635 .58037 L
.03754 .58022 L
.04019 .5798 L
.04145 .57963 L
.04262 .57952 L
.04383 .57945 L
.04515 .57945 L
.04631 .5795 L
.04755 .5796 L
.04824 .57968 L
.04898 .57977 L
.05033 .57996 L
.05153 .58011 L
.05285 .58024 L
.05411 .58029 L
.05528 .58028 L
.05652 .5802 L
.05722 .58012 L
.05785 .58004 L
.06244 .57929 L
.06747 .57864 L
.07281 .57792 L
.07559 .57742 L
.07866 .57667 L
.08146 .57576 L
.08408 .57464 L
.08659 .57328 L
.08923 .57151 L
.09395 .5673 L
.09902 .561 L
.10458 .55166 L
.10986 .54011 L
.11485 .52661 L
.12419 .49419 L
.13424 .44867 L
.14514 .38761 L
.16463 .25502 L
.17496 .17833 L
.18606 .09683 L
.19152 .05868 L
.1942 .04061 L
.19551 .03203 L
Mistroke
.1967 .02479 L
.19786 .01912 L
.19895 .01576 L
.2001 .01472 L
.20135 .01663 L
.20263 .02161 L
.20328 .02503 L
.20399 .02927 L
.20642 .0451 L
.22546 .17193 L
.23551 .23004 L
.24501 .27391 L
.25026 .29273 L
.25303 .301 L
.25596 .3085 L
.25843 .31377 L
.26115 .31848 L
.26362 .32176 L
.26594 .32397 L
.26712 .32476 L
.26838 .32537 L
.26957 .32572 L
.27065 .32584 L
.27192 .32575 L
.27264 .32559 L
.2733 .32537 L
.27451 .32479 L
.2758 .32394 L
.27703 .32288 L
.27835 .32149 L
.28074 .31833 L
.28616 .30814 L
.29113 .29543 L
.29584 .28086 L
.30468 .24884 L
.31343 .21503 L
.32287 .18212 L
.32828 .1672 L
.33065 .16185 L
.33324 .15694 L
.33569 .15325 L
.33798 .15068 L
.33924 .14964 L
.34039 .14893 L
.34103 .14863 L
.34172 .14838 L
.34298 .14816 L
.3442 .1482 L
.34535 .14848 L
.34661 .14906 L
Mistroke
.34732 .14951 L
.34796 .15001 L
.34926 .15122 L
.35067 .15288 L
.35325 .15685 L
.35563 .16158 L
.35816 .16775 L
.36273 .18174 L
.3684 .20425 L
.37374 .23044 L
.38373 .29146 L
.40212 .43551 L
.41256 .5248 L
.41765 .56702 L
.42007 .58618 L
.42113 .59364 L
.4223 .5999 L
.42345 .60326 L
.42467 .60332 L
.42531 .60189 L
.426 .59929 L
.42724 .59223 L
.43193 .55536 L
.44225 .46696 L
.46332 .28752 L
.47371 .21252 L
.48338 .15596 L
.48866 .13116 L
.49346 .11261 L
.49879 .09653 L
.50172 .08975 L
.50448 .08467 L
.50687 .08129 L
.50911 .07899 L
.5104 .07803 L
.51157 .07739 L
.51222 .07713 L
.51293 .07691 L
.51421 .07672 L
.51491 .07672 L
.51557 .07679 L
.5168 .07709 L
.51749 .07735 L
.51812 .07765 L
.51956 .07853 L
.52197 .08063 L
.52451 .08364 L
.52907 .09084 L
.53332 .0993 L
.54297 .12244 L
Mistroke
.55345 .14871 L
.55861 .16019 L
.56337 .16922 L
.56769 .17579 L
.5699 .17848 L
.57228 .18084 L
.57352 .18184 L
.57465 .18262 L
.57581 .18326 L
.57688 .18374 L
.57812 .18413 L
.57878 .18427 L
.57948 .18437 L
.58077 .18441 L
.58195 .18428 L
.58315 .18399 L
.58427 .18357 L
.5855 .18295 L
.58682 .18209 L
.58951 .17973 L
.59196 .17686 L
.5967 .16944 L
.60118 .16013 L
.61115 .13204 L
.622 .09163 L
.6269 .07095 L
.63219 .04775 L
.63463 .03697 L
.63595 .03114 L
.63719 .02626 L
.63824 .02308 L
.63937 .02112 L
.64063 .02098 L
.64177 .02273 L
.64305 .02656 L
.64423 .0313 L
.6469 .04329 L
.65175 .06569 L
.66277 .11868 L
.70216 .31169 L
.74004 .47558 L
.75947 .54856 L
.76507 .56816 L
.76783 .57757 L
.76917 .58206 L
.77039 .58572 L
.77158 .58842 L
.7727 .58981 L
.77388 .58988 L
.77516 .58834 L
Mistroke
.77648 .58527 L
.77787 .58098 L
.78036 .57276 L
.78986 .54267 L
.80003 .51537 L
.80496 .50478 L
.80963 .49661 L
.81382 .49092 L
.8162 .4884 L
.81838 .48655 L
.82072 .48505 L
.82194 .48447 L
.82324 .48401 L
.82436 .48373 L
.82541 .48359 L
.82656 .48354 L
.82779 .48363 L
.82849 .48374 L
.82914 .48388 L
.82985 .48408 L
.8306 .48433 L
.83185 .48488 L
.83324 .48565 L
.83579 .48749 L
.83818 .48971 L
.84348 .49623 L
.84832 .50388 L
.8592 .52525 L
.86843 .54483 L
.87349 .55465 L
.87825 .56254 L
.88092 .56618 L
.88337 .56893 L
.88609 .57124 L
.88751 .57212 L
.8883 .5725 L
.88902 .57277 L
.89021 .57309 L
.89149 .57322 L
.89267 .57315 L
.89378 .5729 L
.89508 .5724 L
.89628 .57172 L
.8976 .57072 L
.899 .56939 L
.90152 .56623 L
.90391 .56231 L
.90841 .55249 L
.91325 .53828 L
.91854 .51846 L
Mistroke
.92803 .47183 L
.93684 .41677 L
.97598 .09448 L
.97619 .09272 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[37]:=",
  ImageSize->{288, 177.938},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/A000`40O003h00Oogoo8Goo003oOolQ
Ool00?moob5oo`00ogoo8Goo003oOolQOol00?moob5oo`00ogoo8Goo003oOolQOol0025oo`03001o
ogoo0?aoo`008Goo00<007ooOol0o7oo000QOol2003mOol0025oo`03001oogoo0?aoo`008Goo00<0
07ooOol0:Woo0P00d7oo000QOol00`00Oomoo`0ZOol2001aOol00`00Oomoo`1LOol0025oo`03001o
ogoo02Yoo`80071oo`8005ioo`004Goo00<007ooOol00Woo00<007ooOol01@000goo00<007ooOol0
:Goo00@007ooOol006moo`03001oo`0005eoo`0047oo00<007oo000027oo00<007ooOol00goo00<0
07ooOol0:Goo00@007ooOol006moo`03001oo`0005eoo`002goo0`000Woo00<007oo000027oo00<0
07ooOol00goo0P00:Woo00@007ooOol006moo`03001oo`0005eoo`0047oo00<007oo000027oo00<0
07ooOol00goo00<007ooOol0:Goo00@007ooOol006moo`04001oogoo001LOol0011oo`03001oo`00
00Qoo`03001oogoo00=oo`03001oogoo02Uoo`04001oogoo001^Ool01@00Oomoogoo0000G7oo000A
Ool00`00Oomoo`06Ool20005Ool00`00Oomoo`0YOol01000Oomoo`00KWoo00D007ooOomoo`0005ao
o`008Goo00<007ooOol0:7oo00D007ooOomoo`0006ioo`05001oogooOol0001LOol0025oo`03001o
ogoo02Qoo`05001oogooOol0001^Ool00`00Oomoo`02Ool00`00Oomoo`1IOol0025oo`03001oogoo
02Qoo`03001oogoo009oo`03001oogoo06Yoo`03001oogoo00=oo`03001oogoo05Uoo`008Goo0P00
:Goo00<007ooOol00Woo00<007ooOol0JWoo00<007ooOol00goo00<007ooOol0FGoo000QOol00`00
Oomoo`0XOol00`00Oomoo`02Ool00`00Oomoo`1ZOol00`00Oomoo`03Ool00`00Oomoo`1IOol0025o
o`03001oogoo02Qoo`03001oogoo009oo`03001oogoo06Yoo`03001oogoo00=oo`03001oogoo05Uo
o`008Goo00<007ooOol0:7oo00<007ooOol00Woo00<007ooOol0JWoo00<007ooOol00goo00<007oo
Ool0FGoo000QOol00`00Oomoo`0WOol00`00Oomoo`03Ool00`00Oomoo`1;Ool4000JOol00`00Oomo
o`05Ool00`00Oomoo`1HOol0025oo`03001oogoo02Moo`03001oogoo00Aoo`03001oogoo04Uoo`04
001oogooOol2000IOol00`00Oomoo`05Ool00`00Oomoo`1HOol0025oo`03001oogoo02Moo`03001o
ogoo00Aoo`03001oogoo04Uoo`03001oogoo00=oo`03001oogoo01Ioo`03001oogoo00Eoo`03001o
ogoo05Qoo`008Goo0P00:7oo00<007ooOol017oo00<007ooOol0B7oo00<007ooOol017oo00<007oo
Ool05Goo00<007ooOol01Woo00<007ooOol0F7oo000QOol00`00Oomoo`0WOol00`00Oomoo`04Ool0
0`00Oomoo`17Ool00`00Oomoo`05Ool00`00Oomoo`0EOol00`00Oomoo`07Ool00`00Oomoo`1?Ool0
0`00Oomoo`05Ool0025oo`03001oogoo02Moo`03001oogoo00Aoo`03001oogoo04Moo`03001oogoo
00Ioo`03001oogoo01Aoo`03001oogoo00Moo`03001oogoo04moo`03001oogoo00Eoo`008Goo00<0
07ooOol09goo00<007ooOol017oo00<007ooOol0Agoo00<007ooOol01Woo00<007ooOol04goo00<0
07ooOol027oo00<007ooOol0Cgoo00<007ooOol01Goo000QOol00`00Oomoo`0WOol00`00Oomoo`04
Ool00`00Oomoo`16Ool00`00Oomoo`08Ool00`00Oomoo`0BOol00`00Oomoo`08Ool00`00Oomoo`1?
Ool00`00Oomoo`05Ool0025oo`03001oogoo02Moo`03001oogoo00Eoo`03001oogoo04Eoo`03001o
ogoo00Qoo`03001oogoo019oo`03001oogoo00Qoo`03001oogoo04moo`03001oogoo00Eoo`008Goo
00<007ooOol09Woo00<007ooOol01Woo00<007ooOol0AGoo00<007ooOol02Goo00<007ooOol047oo
00<007ooOol02Woo00<007ooOol0CGoo00<007ooOol01Woo000QOol2000WOol00`00Oomoo`06Ool0
0`00Oomoo`15Ool00`00Oomoo`09Ool00`00Oomoo`0@Ool00`00Oomoo`0:Ool00`00Oomoo`1=Ool0
0`00Oomoo`06Ool0025oo`03001oogoo02Ioo`03001oogoo00Ioo`03001oogoo04Eoo`03001oogoo
00Yoo`03001oogoo00moo`03001oogoo00Yoo`03001oogoo04eoo`03001oogoo00Ioo`008Goo00<0
07ooOol09Woo00<007ooOol01Woo00<007ooOol0A7oo00<007ooOol02goo00<007ooOol03goo00<0
07ooOol02Woo00<007ooOol0CGoo00<007ooOol01Woo000QOol00`00Oomoo`0VOol00`00Oomoo`06
Ool00`00Oomoo`14Ool00`00Oomoo`0;Ool00`00Oomoo`0>Ool00`00Oomoo`0;Ool00`00Oomoo`1=
Ool00`00Oomoo`06Ool0025oo`03001oogoo02Ioo`03001oogoo00Ioo`03001oogoo04Aoo`03001o
ogoo00aoo`03001oogoo00eoo`03001oogoo00aoo`03001oogoo04aoo`03001oogoo00Ioo`008Goo
00<007ooOol09Woo00<007ooOol01goo00<007ooOol0@goo00<007ooOol037oo00<007ooOol03Goo
00<007ooOol037oo00<007ooOol0C7oo00<007ooOol01Woo000QOol00`00Oomoo`0UOol00`00Oomo
o`08Ool00`00Oomoo`12Ool00`00Oomoo`0=Ool00`00Oomoo`0<Ool00`00Oomoo`0=Ool00`00Oomo
o`1<Ool00`00Oomoo`06Ool0025oo`8002Ioo`03001oogoo00Qoo`03001oogoo049oo`03001oogoo
00ioo`03001oogoo00]oo`03001oogoo00eoo`03001oogoo04]oo`03001oogoo00Moo`008Goo00<0
07ooOol09Goo00<007ooOol027oo00<007ooOol077oo0`008goo00<007ooOol03Woo00<007ooOol0
2goo00<007ooOol03Goo00<007ooOol0Bgoo00<007ooOol01goo000QOol00`00Oomoo`0UOol00`00
Oomoo`08Ool00`00Oomoo`0KOol200000goo0000000QOol00`00Oomoo`0@Ool00`00Oomoo`0:Ool0
0`00Oomoo`0>Ool00`00Oomoo`1:Ool00`00Oomoo`07Ool0025oo`03001oogoo02Eoo`03001oogoo
00Qoo`03001oogoo01Yoo`8000=oo`03001oogoo01moo`03001oogoo011oo`03001oogoo00Uoo`03
001oogoo00moo`03001oogoo04Yoo`03001oogoo00Moo`008Goo00<007ooOol09Goo00<007ooOol0
2Goo00<007ooOol06Goo00<007ooOol00goo00<007ooOol07Woo00<007ooOol04Goo00<007ooOol0
27oo00<007ooOol03goo00<007ooOol0BWoo00<007ooOol01goo000<Ool00`00Oomoo`02Ool00`00
Oomoo`02Ool00`00Oomoo`02Ool00`00Oomoo`03Ool00`00Oomoo`0UOol00`00Oomoo`09Ool00`00
Oomoo`0IOol00`00Oomoo`03Ool00`00Oomoo`0NOol00`00Oomoo`0AOol00`00Oomoo`08Ool00`00
Oomoo`0?Ool00`00Oomoo`1:Ool00`00Oomoo`07Ool000]oo`03001oo`0000Moo`03001oo`00009o
o`03001oo`0000Aoo`03001oogoo02Aoo`03001oogoo00Yoo`03001oogoo01Qoo`03001oogoo00Eo
o`03001oogoo01eoo`03001oogoo019oo`03001oogoo00Ioo`03001oogoo011oo`03001oogoo04Yo
o`03001oogoo00Moo`001Woo0`000Woo00<007oo00001goo00<007oo000017oo00<007ooOol00Woo
0P009Goo00<007ooOol02Woo00<007ooOol05goo00<007ooOol01Woo00<007ooOol077oo00<007oo
Ool04goo00<007ooOol01Woo00<007ooOol04Goo00<007ooOol0BGoo00<007ooOol01goo000;Ool0
0`00Ool00007Ool00`00Ool00002Ool20005Ool00`00Oomoo`0TOol00`00Oomoo`0:Ool00`00Oomo
o`0GOol00`00Oomoo`07Ool00`00Oomoo`0KOol00`00Oomoo`0DOol00`00Oomoo`04Ool00`00Oomo
o`0BOol00`00Oomoo`18Ool00`00Oomoo`08Ool000]oo`03001oo`0000Moo`03001oo`00009oo`03
001oogoo00Aoo`03001oogoo02Aoo`03001oogoo00Yoo`03001oogoo01Ioo`03001oogoo00Qoo`03
001oogoo01]oo`03001oogoo01Eoo`03001oogoo009oo`03001oogoo01=oo`03001oogoo04Qoo`03
001oogoo00Qoo`0037oo00<007ooOol01goo00@007ooOomoo`<000Aoo`03001oogoo02Aoo`03001o
ogoo00]oo`03001oogoo01Eoo`03001oogoo00Qoo`03001oogoo01]oo`03001oogoo01Eoo`D001Io
o`03001oogoo04Qoo`03001oogoo00Qoo`008Goo00<007ooOol097oo00<007ooOol02goo00<007oo
Ool05Goo00<007ooOol027oo00<007ooOol06goo00<007ooOol05goo0P005goo00<007ooOol0B7oo
00<007ooOol027oo000QOol00`00Oomoo`0TOol00`00Oomoo`0;Ool00`00Oomoo`0EOol00`00Oomo
o`08Ool00`00Oomoo`0KOol00`00Oomoo`0aOol00`00Oomoo`17Ool00`00Oomoo`08Ool0025oo`03
001oogoo02=oo`03001oogoo00aoo`03001oogoo01Aoo`03001oogoo00Yoo`03001oogoo01Yoo`03
001oogoo035oo`03001oogoo04Moo`03001oogoo00Qoo`008Goo0P0097oo00<007ooOol037oo00<0
07ooOol057oo00<007ooOol02Woo00<007ooOol06Woo00<007ooOol0<Goo00<007ooOol0Agoo00<0
07ooOol027oo000QOol00`00Oomoo`0SOol00`00Oomoo`0=Ool00`00Oomoo`0COol00`00Oomoo`0:
Ool00`00Oomoo`0IOol00`00Oomoo`0bOol00`00Oomoo`17Ool00`00Oomoo`08Ool0025oo`03001o
ogoo02=oo`03001oogoo00eoo`03001oogoo01=oo`03001oogoo00Yoo`03001oogoo01Uoo`03001o
ogoo039oo`03001oogoo04Moo`03001oogoo00Qoo`008Goo00<007ooOol08goo00<007ooOol03Goo
00<007ooOol04Woo00<007ooOol02goo00<007ooOol06Goo00<007ooOol0<goo00<007ooOol0AGoo
00<007ooOol02Goo000QOol00`00Oomoo`0SOol00`00Oomoo`0=Ool00`00Oomoo`0BOol00`00Oomo
o`0<Ool00`00Oomoo`0HOol00`00Oomoo`0cOol00`00Oomoo`15Ool00`00Oomoo`09Ool0025oo`03
001oogoo029oo`03001oogoo00ioo`03001oogoo019oo`03001oogoo00aoo`03001oogoo01Qoo`03
001oogoo03=oo`03001oogoo04Eoo`03001oogoo00Uoo`008Goo00<007ooOol08Woo00<007ooOol0
3Woo00<007ooOol04Woo00<007ooOol037oo00<007ooOol067oo00<007ooOol0<goo00<007ooOol0
AGoo00<007ooOol02Goo000QOol2000SOol00`00Oomoo`0?Ool00`00Oomoo`0@Ool00`00Oomoo`0=
Ool00`00Oomoo`0HOol00`00Oomoo`0cOol00`00Oomoo`15Ool00`00Oomoo`09Ool0025oo`03001o
ogoo029oo`03001oogoo00moo`03001oogoo011oo`03001oogoo00ioo`03001oogoo01Ioo`03001o
ogoo03Eoo`03001oogoo04Aoo`03001oogoo00Uoo`008Goo00<007ooOol08Woo00<007ooOol03goo
00<007ooOol047oo00<007ooOol03Woo00<007ooOol05Woo00<007ooOol0=Goo00<007ooOol0A7oo
00<007ooOol02Goo000QOol00`00Oomoo`0ROol00`00Oomoo`0?Ool00`00Oomoo`0@Ool00`00Oomo
o`0>Ool00`00Oomoo`0FOol00`00Oomoo`0eOol00`00Oomoo`14Ool00`00Oomoo`09Ool0025oo`03
001oogoo029oo`03001oogoo00moo`03001oogoo00moo`03001oogoo00moo`03001oogoo01Ioo`03
001oogoo03Eoo`03001oogoo04=oo`03001oogoo00Yoo`008Goo00<007ooOol08Goo00<007ooOol0
4Goo00<007ooOol03Woo00<007ooOol03goo00<007ooOol05Woo00<007ooOol0=Goo00<007ooOol0
@goo00<007ooOol02Woo000QOol00`00Oomoo`0QOol00`00Oomoo`0AOol00`00Oomoo`0>Ool00`00
Oomoo`0@Ool00`00Oomoo`0EOol00`00Oomoo`0fOol00`00Oomoo`12Ool00`00Oomoo`0:Ool0025o
o`80029oo`03001oogoo015oo`03001oogoo00ioo`03001oogoo011oo`03001oogoo01Aoo`03001o
ogoo03Moo`03001oogoo049oo`03001oogoo00Yoo`008Goo00<007ooOol08Goo00<007ooOol04Goo
00<007ooOol03Goo00<007ooOol04Goo00<007ooOol057oo00<007ooOol0=goo00<007ooOol0@Woo
00<007ooOol02Woo000QOol00`00Oomoo`0QOol00`00Oomoo`0AOol00`00Oomoo`0=Ool00`00Oomo
o`0AOol00`00Oomoo`0DOol00`00Oomoo`0gOol00`00Oomoo`12Ool00`00Oomoo`0:Ool0025oo`03
001oogoo01Moo`<000=oo`04001oogooOol2000BOol00`00Oomoo`05Ool00`00Oomoo`05Ool00`00
Oomoo`0AOol00`00Oomoo`02Ool00`00Oomoo`02Ool00`00Oomoo`0:Ool00`00Oomoo`07Ool50002
Ool00`00Oomoo`0BOol5000017oo000000000goo00<007ooOol00Woo00@007ooOol0011oo`D0009o
o`03001oogoo019oo`D0009oo`03001oogoo009oo`03001oogoo009oo`03001oogoo00Aoo`03001o
ogoo00Yoo`008Goo00<007ooOol05goo00<007ooOol01goo00<007oo00004Woo00<007ooOol00goo
00<007oo00001Goo00<007ooOol04Woo00<007ooOol01Woo00<007oo00002goo00<007ooOol02Goo
00L007ooOomoo`00Ool0000EOol01@00Oomoogoo00002Goo00<007oo00000Woo00<007ooOol03goo
00L007ooOomoo`00Ool0000EOol00`00Oomoo`02Ool00`00Oomoo`06Ool00`00Ool00005Ool00`00
Oomoo`0:Ool0025oo`03001oogoo01Qoo`03001oogoo00Eoo`04001oogoo000BOol00`00Oomoo`05
Ool00`00Oomoo`03Ool00`00Oomoo`0BOol00`00Oomoo`08Ool00`00Oomoo`09Ool00`00Oomoo`09
Ool01`00Oomoogoo001oo`0001Eoo`03001oogoo009oo`03001oogoo00Qoo`04001oogoo000AOol0
0`00Oomoo`03Ool00`00Oomoo`0COol00`00Oomoo`02Ool00`00Oomoo`08Ool00`00Oomoo`03Ool0
0`00Oomoo`0:Ool0025oo`03001oogoo01Qoo`03001oogoo00Eoo`<001=oo`03001oogoo00=oo`80
00Ioo`03001oogoo019oo`8000Moo`8000]oo`03001oogoo00Yoo`07001oogooOol007oo00005Goo
00<007ooOol00Woo00<007ooOol01Woo0P000goo00<007ooOol03goo00@007ooOomoo`8001Ioo`03
001oogoo00=oo`03001oogoo00Eoo`8000Eoo`03001oogoo00]oo`008Goo0P0067oo00<007oo0000
1Woo0P0057oo00<007ooOol00goo00<007ooOol017oo00<007ooOol04Woo00<007oo00001goo00<0
07ooOol02Woo00<007ooOol02Woo00L007ooOomoo`00Ool0000EOol01`00Oomoogoo001oo`0000Mo
o`03001oogoo009oo`03001oogoo00moo`05001oogooOol0000GOol01`00Oomoogoo001oo`0000Mo
o`03001oogoo00Aoo`03001oogoo00]oo`008Goo00<007ooOol067oo00<007ooOol01Goo10004goo
00<007ooOol00Woo0`0017oo00<007ooOol04Woo0`001goo0`002Woo00<007ooOol02Goo0P0017oo
00<007ooOol04goo0P0017oo00<007ooOol01Woo0`000Woo00<007ooOol03Woo0P000goo0`0057oo
0P000goo0`001goo0`0017oo00<007ooOol02goo000QOol00`00Oomoo`0POol00`00Oomoo`0DOol0
0`00Oomoo`09Ool00`00Oomoo`0DOol00`00Oomoo`0BOol00`00Oomoo`0jOol00`00Oomoo`0oOol0
0`00Oomoo`0;Ool0025oo`03001oogoo021oo`03001oogoo01Eoo`03001oogoo00Moo`03001oogoo
01Eoo`03001oogoo019oo`03001oogoo03Yoo`03001oogoo03moo`03001oogoo00]oo`008Goo00<0
07ooOol087oo00<007ooOol05Goo00<007ooOol01goo00<007ooOol05Goo00<007ooOol04Woo00<0
07ooOol0>Woo00<007ooOol0?goo00<007ooOol02goo000QOol00`00Oomoo`0OOol00`00Oomoo`0F
Ool00`00Oomoo`07Ool00`00Oomoo`0EOol00`00Oomoo`0BOol00`00Oomoo`0jOol00`00Oomoo`0o
Ool00`00Oomoo`0;Ool0025oo`03001oogoo01moo`03001oogoo01Ioo`03001oogoo00Ioo`03001o
ogoo01Ioo`03001oogoo019oo`03001oogoo03Yoo`03001oogoo03moo`03001oogoo00]oo`008Goo
00<007ooOol07goo00<007ooOol05goo00<007ooOol01Goo00<007ooOol05goo00<007ooOol047oo
00<007ooOol0?7oo00<007ooOol0?Goo00<007ooOol037oo000KOooo00060000025oo`03001oogoo
00=oo`03001oogoo00Aoo`03001oogoo00=oo`03001oogoo00Aoo`03001oogoo00=oo`03001oo`00
00Aoo`03001oogoo00=oo`03001oogoo00Aoo`03001oogoo00=oo`04001oogoo0003Ool00`00Ool0
0003Ool00`00Oomoo`04Ool00`00Oomoo`03Ool00`00Oomoo`04Ool01000Oomoo`000Woo00<007oo
Ool017oo00<007ooOol017oo00<007oo00000goo00<007ooOol017oo00<007ooOol00goo00<007oo
Ool017oo00<007ooOol00goo00<007ooOol017oo00<007ooOol00goo00<007ooOol017oo00<007oo
Ool00goo00<007ooOol017oo00<007ooOol00goo00<007ooOol017oo00<007ooOol00goo00<007oo
Ool017oo00<007ooOol00goo00<007ooOol017oo00<007ooOol00goo00<007ooOol017oo00<007oo
Ool00goo00<007ooOol00goo0P001Goo00<007ooOol01Goo000QOol00`00Oomoo`0OOol00`00Oomo
o`0HOol00`00Oomoo`03Ool00`00Oomoo`0HOol00`00Oomoo`0@Ool00`00Oomoo`0lOol00`00Oomo
o`0mOol00`00Oomoo`0<Ool0025oo`03001oogoo01ioo`03001oogoo01Yoo`05001oogooOol0000K
Ool00`00Oomoo`0@Ool00`00Oomoo`0lOol00`00Oomoo`0mOol00`00Oomoo`0<Ool0025oo`03001o
ogoo01ioo`03001oogoo01]oo`@001]oo`03001oogoo011oo`03001oogoo03eoo`03001oogoo03ao
o`03001oogoo00aoo`008Goo00<007ooOol07Woo00<007ooOol0>Woo00<007ooOol047oo00<007oo
Ool0?Goo00<007ooOol0?7oo00<007ooOol037oo000QOol00`00Oomoo`0NOol00`00Oomoo`0jOol0
0`00Oomoo`0@Ool00`00Oomoo`0mOol00`00Oomoo`0lOol00`00Oomoo`0<Ool0025oo`8001moo`03
001oogoo03]oo`03001oogoo00moo`03001oogoo03eoo`03001oogoo03]oo`03001oogoo00eoo`00
8Goo00<007ooOol07Woo00<007ooOol0>goo00<007ooOol03goo00<007ooOol0?Woo00<007ooOol0
>Woo00<007ooOol03Goo000QOol00`00Oomoo`0NOol00`00Oomoo`0kOol00`00Oomoo`0>Ool00`00
Oomoo`0oOol00`00Oomoo`0jOol00`00Oomoo`0=Ool0025oo`03001oogoo01eoo`03001oogoo03ao
o`03001oogoo00ioo`03001oogoo03moo`03001oogoo03Yoo`03001oogoo00eoo`008Goo00<007oo
Ool07Goo00<007ooOol0?7oo00<007ooOol03Woo00<007ooOol0?goo00<007ooOol0>Woo00<007oo
Ool03Goo000QOol00`00Oomoo`0MOol00`00Oomoo`0lOol00`00Oomoo`0>Ool00`00Oomoo`10Ool0
0`00Oomoo`0iOol00`00Oomoo`0=Ool0025oo`03001oogoo01eoo`03001oogoo03aoo`03001oogoo
00ioo`03001oogoo041oo`03001oogoo03Uoo`03001oogoo00eoo`008Goo0P007Woo00<007ooOol0
?Goo00<007ooOol03Goo00<007ooOol0@7oo00<007ooOol0>Goo00<007ooOol03Goo000QOol00`00
Oomoo`0MOol00`00Oomoo`0mOol00`00Oomoo`0=Ool00`00Oomoo`10Ool00`00Oomoo`0iOol00`00
Oomoo`0=Ool0025oo`03001oogoo01eoo`03001oogoo03eoo`03001oogoo00eoo`03001oogoo045o
o`03001oogoo03Moo`03001oogoo00ioo`008Goo00<007ooOol077oo00<007ooOol0?Woo00<007oo
Ool03Goo00<007ooOol0@Goo00<007ooOol0=goo00<007ooOol03Woo000QOol00`00Oomoo`0LOol0
0`00Oomoo`0nOol00`00Oomoo`0<Ool00`00Oomoo`12Ool00`00Oomoo`0gOol00`00Oomoo`0>Ool0
025oo`03001oogoo01aoo`03001oogoo03ioo`03001oogoo00aoo`03001oogoo049oo`03001oogoo
03Moo`03001oogoo00ioo`008Goo00<007ooOol077oo00<007ooOol0?Woo00<007ooOol037oo00<0
07ooOol0@goo00<007ooOol0=Woo00<007ooOol03Woo000QOol2000MOol00`00Oomoo`0nOol00`00
Oomoo`0<Ool00`00Oomoo`13Ool00`00Oomoo`0fOol00`00Oomoo`0>Ool0025oo`03001oogoo01ao
o`03001oogoo03moo`03001oogoo00]oo`03001oogoo04=oo`03001oogoo03Ioo`03001oogoo00io
o`008Goo00<007ooOol06goo00<007ooOol0@7oo00<007ooOol02goo00<007ooOol0@goo00<007oo
Ool0=Woo00<007ooOol03Woo000QOol00`00Oomoo`0KOol00`00Oomoo`10Ool00`00Oomoo`0;Ool0
0`00Oomoo`13Ool00`00Oomoo`0eOol00`00Oomoo`0?Ool0025oo`03001oogoo01]oo`03001oogoo
041oo`03001oogoo00]oo`03001oogoo04Aoo`03001oogoo03Aoo`03001oogoo00moo`008Goo00<0
07ooOol06goo00<007ooOol0@7oo00<007ooOol02goo00<007ooOol0A7oo00<007ooOol0=7oo00<0
07ooOol03goo000QOol00`00Oomoo`0KOol00`00Oomoo`10Ool00`00Oomoo`0:Ool00`00Oomoo`15
Ool00`00Oomoo`0dOol00`00Oomoo`0?Ool0025oo`8001]oo`03001oogoo045oo`03001oogoo00Yo
o`03001oogoo04Eoo`03001oogoo03Aoo`03001oogoo00moo`008Goo00<007ooOol06Woo00<007oo
Ool0@Goo00<007ooOol02Woo00<007ooOol0AWoo00<007ooOol0<goo00<007ooOol03goo000QOol0
0`00Oomoo`0JOol00`00Oomoo`12Ool00`00Oomoo`09Ool00`00Oomoo`16Ool00`00Oomoo`0cOol0
0`00Oomoo`0?Ool0025oo`03001oogoo01Yoo`03001oogoo049oo`03001oogoo00Uoo`03001oogoo
04Ioo`03001oogoo03=oo`03001oogoo00moo`008Goo00<007ooOol06Woo00<007ooOol0@Woo00<0
07ooOol02Goo00<007ooOol0AWoo00<007ooOol0<Woo00<007ooOol047oo000<Ool00`00Oomoo`02
Ool00`00Oomoo`02Ool00`00Oomoo`02Ool00`00Oomoo`03Ool00`00Oomoo`0JOol00`00Oomoo`12
Ool00`00Oomoo`09Ool00`00Oomoo`17Ool00`00Oomoo`0aOol00`00Oomoo`0@Ool000]oo`03001o
o`0000Moo`03001oo`00009oo`03001oo`0000Aoo`03001oogoo01Uoo`03001oogoo04=oo`03001o
ogoo00Uoo`03001oogoo04Moo`03001oogoo035oo`03001oogoo011oo`002goo00<007oo00001goo
00<007oo000017oo00<007ooOol00Woo0P006Woo00<007ooOol0@goo00<007ooOol02Goo00<007oo
Ool0Agoo00<007ooOol0<Goo00<007ooOol047oo000;Ool00`00Ool00007Ool00`00Ool00002Ool2
0005Ool00`00Oomoo`0IOol00`00Oomoo`13Ool00`00Oomoo`09Ool00`00Oomoo`17Ool00`00Oomo
o`0aOol00`00Oomoo`0@Ool000]oo`03001oo`0000Moo`03001oo`00009oo`03001oogoo00Aoo`03
001oogoo01Uoo`03001oogoo04Aoo`03001oogoo00Moo`03001oogoo04Uoo`03001oogoo031oo`03
001oogoo011oo`0037oo00<007ooOol01goo00@007ooOomoo`<000Aoo`03001oogoo01Uoo`03001o
ogoo04Aoo`03001oogoo00Moo`03001oogoo04Uoo`03001oogoo031oo`03001oogoo011oo`008Goo
00<007ooOol067oo00<007ooOol0AGoo00<007ooOol01goo00<007ooOol0BGoo00<007ooOol0;goo
00<007ooOol04Goo000QOol00`00Oomoo`0HOol00`00Oomoo`15Ool00`00Oomoo`07Ool00`00Oomo
o`19Ool00`00Oomoo`0_Ool00`00Oomoo`0AOol0025oo`03001oogoo01Qoo`03001oogoo04Eoo`03
001oogoo00Moo`03001oogoo04Yoo`03001oogoo02ioo`03001oogoo015oo`008Goo0P006Goo00<0
07ooOol0AGoo00<007ooOol01goo00<007ooOol0BWoo00<007ooOol0;Woo00<007ooOol04Goo000Q
Ool00`00Oomoo`0HOol00`00Oomoo`15Ool00`00Oomoo`07Ool00`00Oomoo`1:Ool00`00Oomoo`0^
Ool00`00Oomoo`0AOol0025oo`03001oogoo01Qoo`03001oogoo04Eoo`03001oogoo00Moo`03001o
ogoo04Yoo`03001oogoo02eoo`03001oogoo019oo`008Goo00<007ooOol05goo00<007ooOol0Agoo
00<007ooOol01Goo00<007ooOol0C7oo00<007ooOol04Woo0P0067oo00<007ooOol04Woo000QOol0
0`00Oomoo`0GOol00`00Oomoo`17Ool00`00Oomoo`05Ool00`00Oomoo`1<Ool00`00Oomoo`0@Ool6
000FOol00`00Oomoo`0BOol0025oo`03001oogoo01Moo`03001oogoo04Moo`03001oogoo00Eoo`03
001oogoo04aoo`03001oogoo00moo`8000Aoo`03001oogoo01Aoo`03001oogoo019oo`008Goo00<0
07ooOol05goo00<007ooOol0Agoo00<007ooOol01Goo00<007ooOol0C7oo00<007ooOol03Woo00<0
07ooOol01Goo00<007ooOol04Woo00<007ooOol04goo000QOol2000HOol00`00Oomoo`17Ool00`00
Oomoo`05Ool00`00Oomoo`1=Ool00`00Oomoo`0=Ool00`00Oomoo`06Ool00`00Oomoo`0AOol00`00
Oomoo`0COol0025oo`03001oogoo01Ioo`03001oogoo04Qoo`03001oogoo00Eoo`03001oogoo04eo
o`03001oogoo00aoo`03001oogoo00Qoo`03001oogoo011oo`03001oogoo01=oo`008Goo00<007oo
Ool05Woo00<007ooOol0B7oo00<007ooOol01Goo00<007ooOol0CGoo00<007ooOol037oo00<007oo
Ool027oo00<007ooOol047oo00<007ooOol04goo000QOol00`00Oomoo`0FOol00`00Oomoo`18Ool0
0`00Oomoo`05Ool00`00Oomoo`1=Ool00`00Oomoo`0<Ool00`00Oomoo`09Ool00`00Oomoo`0>Ool0
0`00Oomoo`0DOol0025oo`03001oogoo01Eoo`03001oogoo04Yoo`03001oogoo00=oo`03001oogoo
04moo`03001oogoo00Yoo`03001oogoo00Yoo`03001oogoo00ioo`03001oogoo01Aoo`008Goo00<0
07ooOol05Goo00<007ooOol0BWoo00<007ooOol00goo00<007ooOol0Cgoo00<007ooOol02Woo00<0
07ooOol02goo00<007ooOol03Goo00<007ooOol057oo000QOol00`00Oomoo`0EOol00`00Oomoo`1:
Ool00`00Oomoo`03Ool00`00Oomoo`1?Ool00`00Oomoo`0:Ool00`00Oomoo`0;Ool00`00Oomoo`0=
Ool00`00Oomoo`0DOol0025oo`8001Eoo`03001oogoo04]oo`03001oogoo00=oo`03001oogoo04mo
o`03001oogoo00Uoo`03001oogoo00eoo`03001oogoo00aoo`03001oogoo01Aoo`008Goo00<007oo
Ool057oo00<007ooOol0Bgoo00<007ooOol00goo00<007ooOol0D7oo00<007ooOol027oo00<007oo
Ool03Goo00<007ooOol02goo00<007ooOol05Goo000QOol00`00Oomoo`0DOol00`00Oomoo`1;Ool0
0`00Oomoo`03Ool00`00Oomoo`1@Ool00`00Oomoo`07Ool00`00Oomoo`0?Ool00`00Oomoo`0:Ool0
0`00Oomoo`0EOol0025oo`03001oogoo01=oo`03001oogoo04aoo`03001oogoo00=oo`03001oogoo
051oo`03001oogoo00Moo`03001oogoo00moo`03001oogoo00Yoo`03001oogoo01Eoo`008Goo00<0
07ooOol04goo00<007ooOol0C7oo00<007ooOol00goo00<007ooOol0D7oo00<007ooOol01Woo00<0
07ooOol04Goo00<007ooOol02Goo00<007ooOol05Goo000QOol00`00Oomoo`0COol00`00Oomoo`1<
Ool00`00Oomoo`02Ool00`00Oomoo`1BOol00`00Oomoo`05Ool00`00Oomoo`0AOol00`00Oomoo`09
Ool00`00Oomoo`0EOol0025oo`03001oogoo019oo`03001oogoo04ioo`05001oogooOol0001DOol0
0`00Oomoo`05Ool00`00Oomoo`0BOol00`00Oomoo`07Ool00`00Oomoo`0FOol0025oo`8001=oo`03
001oogoo04ioo`05001oogooOol0001DOol00`00Oomoo`04Ool00`00Oomoo`0COol00`00Oomoo`07
Ool00`00Oomoo`0FOol0025oo`03001oogoo015oo`03001oogoo04moo`05001oogooOol0001DOol0
0`00Oomoo`04Ool00`00Oomoo`0DOol00`00Oomoo`06Ool00`00Oomoo`0FOol0025oo`03001oogoo
015oo`03001oogoo04moo`05001oogooOol0001EOol00`00Oomoo`02Ool00`00Oomoo`0FOol00`00
Oomoo`04Ool00`00Oomoo`0GOol0025oo`03001oogoo011oo`03001oogoo051oo`05001oogooOol0
001EOol00`00Oomoo`02Ool00`00Oomoo`0FOol00`00Oomoo`04Ool00`00Oomoo`0GOol0025oo`03
001oogoo00moo`03001oogoo055oo`05001oogooOol0001EOol00`00Oomoo`02Ool00`00Oomoo`0G
Ool00`00Oomoo`02Ool00`00Oomoo`0HOol0015oo`03001oogoo009oo`03001oogoo00D000=oo`03
001oogoo00ioo`03001oogoo059oo`05001oogooOol0001EOol01@00Oomoogoo00006Woo0P0000=o
o`0000006goo000@Ool00`00Ool00008Ool00`00Oomoo`03Ool00`00Oomoo`0<Ool2001EOol01000
Oomoo`00Egoo00@007ooOol001]oo`<001aoo`0047oo00<007oo000027oo00<007ooOol00goo0P00
0Woo2`00F7oo00<007oo0000Egoo00@007ooOol003Yoo`0047oo00<007oo000027oo00<007ooOol0
0goo10000goo0P00GWoo00<007oo0000Egoo00@007ooOol003Yoo`0047oo00<007oo000027oo00<0
07ooOol00goo00<007ooOol0I7oo00<007oo0000F7oo0P00>goo000AOol00`00Oomoo`06Ool20005
Ool00`00Oomoo`1TOol00`00Ool0001HOol2000kOol0025oo`03001oogoo06Aoo`03001oo`0009Eo
o`008Goo00<007ooOol0I7oo0P00UWoo000QOol00`00Oomoo`1TOol2002FOol0025oo`8006Ioo`03
001oogoo09Aoo`008Goo00<007ooOol0o7oo000QOol00`00Oomoo`3lOol0025oo`03001oogoo0?ao
o`00ogoo8Goo003oOolQOol00?moob5oo`00ogoo8Goo003oOolQOol00?moob5oo`00ogoo8Goo003o
OolQOol00001\
\>"],
  ImageRangeCache->{{{0, 287}, {176.938, 0}} -> {-2.55194, -0.124206, \
0.0767493, 0.00141193}},
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[
    TagBox[\(\[SkeletonIndicator]  Graphics  \[SkeletonIndicator]\),
      False,
      Editable->False]], "Output",
  CellLabel->"Out[37]=",
  CellTags->{"Backlash", "i:3"}]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(Plot[Evaluate[a2x[t] /. sols], {t, 0, 19}]\)], "Input",
  CellLabel->"In[38]:=",
  PageWidth->Infinity,
  AspectRatioFixed->True,
  CellTags->{"Backlash", "i:3"}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.0501253 0.314177 0.876487 [
[.14912 .30168 -9 -9 ]
[.14912 .30168 9 0 ]
[.27444 .30168 -3 -9 ]
[.27444 .30168 3 0 ]
[.39975 .30168 -9 -9 ]
[.39975 .30168 9 0 ]
[.52506 .30168 -6 -9 ]
[.52506 .30168 6 0 ]
[.65038 .30168 -12 -9 ]
[.65038 .30168 12 0 ]
[.77569 .30168 -6 -9 ]
[.77569 .30168 6 0 ]
[.901 .30168 -12 -9 ]
[.901 .30168 12 0 ]
[.01131 .05123 -24 -4.5 ]
[.01131 .05123 0 4.5 ]
[.01131 .13888 -24 -4.5 ]
[.01131 .13888 0 4.5 ]
[.01131 .22653 -24 -4.5 ]
[.01131 .22653 0 4.5 ]
[.01131 .40183 -18 -4.5 ]
[.01131 .40183 0 4.5 ]
[.01131 .48947 -18 -4.5 ]
[.01131 .48947 0 4.5 ]
[.01131 .57712 -18 -4.5 ]
[.01131 .57712 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
.14912 .31418 m
.14912 .32043 L
s
[(2.5)] .14912 .30168 0 1 Mshowa
.27444 .31418 m
.27444 .32043 L
s
[(5)] .27444 .30168 0 1 Mshowa
.39975 .31418 m
.39975 .32043 L
s
[(7.5)] .39975 .30168 0 1 Mshowa
.52506 .31418 m
.52506 .32043 L
s
[(10)] .52506 .30168 0 1 Mshowa
.65038 .31418 m
.65038 .32043 L
s
[(12.5)] .65038 .30168 0 1 Mshowa
.77569 .31418 m
.77569 .32043 L
s
[(15)] .77569 .30168 0 1 Mshowa
.901 .31418 m
.901 .32043 L
s
[(17.5)] .901 .30168 0 1 Mshowa
.125 Mabswid
.04887 .31418 m
.04887 .31793 L
s
.07393 .31418 m
.07393 .31793 L
s
.099 .31418 m
.099 .31793 L
s
.12406 .31418 m
.12406 .31793 L
s
.17419 .31418 m
.17419 .31793 L
s
.19925 .31418 m
.19925 .31793 L
s
.22431 .31418 m
.22431 .31793 L
s
.24937 .31418 m
.24937 .31793 L
s
.2995 .31418 m
.2995 .31793 L
s
.32456 .31418 m
.32456 .31793 L
s
.34962 .31418 m
.34962 .31793 L
s
.37469 .31418 m
.37469 .31793 L
s
.42481 .31418 m
.42481 .31793 L
s
.44987 .31418 m
.44987 .31793 L
s
.47494 .31418 m
.47494 .31793 L
s
.5 .31418 m
.5 .31793 L
s
.55013 .31418 m
.55013 .31793 L
s
.57519 .31418 m
.57519 .31793 L
s
.60025 .31418 m
.60025 .31793 L
s
.62531 .31418 m
.62531 .31793 L
s
.67544 .31418 m
.67544 .31793 L
s
.7005 .31418 m
.7005 .31793 L
s
.72556 .31418 m
.72556 .31793 L
s
.75063 .31418 m
.75063 .31793 L
s
.80075 .31418 m
.80075 .31793 L
s
.82581 .31418 m
.82581 .31793 L
s
.85088 .31418 m
.85088 .31793 L
s
.87594 .31418 m
.87594 .31793 L
s
.92607 .31418 m
.92607 .31793 L
s
.95113 .31418 m
.95113 .31793 L
s
.97619 .31418 m
.97619 .31793 L
s
.25 Mabswid
0 .31418 m
1 .31418 L
s
.02381 .05123 m
.03006 .05123 L
s
[(-0.3)] .01131 .05123 1 0 Mshowa
.02381 .13888 m
.03006 .13888 L
s
[(-0.2)] .01131 .13888 1 0 Mshowa
.02381 .22653 m
.03006 .22653 L
s
[(-0.1)] .01131 .22653 1 0 Mshowa
.02381 .40183 m
.03006 .40183 L
s
[(0.1)] .01131 .40183 1 0 Mshowa
.02381 .48947 m
.03006 .48947 L
s
[(0.2)] .01131 .48947 1 0 Mshowa
.02381 .57712 m
.03006 .57712 L
s
[(0.3)] .01131 .57712 1 0 Mshowa
.125 Mabswid
.02381 .06876 m
.02756 .06876 L
s
.02381 .08629 m
.02756 .08629 L
s
.02381 .10382 m
.02756 .10382 L
s
.02381 .12135 m
.02756 .12135 L
s
.02381 .15641 m
.02756 .15641 L
s
.02381 .17394 m
.02756 .17394 L
s
.02381 .19147 m
.02756 .19147 L
s
.02381 .209 m
.02756 .209 L
s
.02381 .24406 m
.02756 .24406 L
s
.02381 .26159 m
.02756 .26159 L
s
.02381 .27912 m
.02756 .27912 L
s
.02381 .29665 m
.02756 .29665 L
s
.02381 .33171 m
.02756 .33171 L
s
.02381 .34924 m
.02756 .34924 L
s
.02381 .36677 m
.02756 .36677 L
s
.02381 .3843 m
.02756 .3843 L
s
.02381 .41936 m
.02756 .41936 L
s
.02381 .43688 m
.02756 .43688 L
s
.02381 .45441 m
.02756 .45441 L
s
.02381 .47194 m
.02756 .47194 L
s
.02381 .507 m
.02756 .507 L
s
.02381 .52453 m
.02756 .52453 L
s
.02381 .54206 m
.02756 .54206 L
s
.02381 .55959 m
.02756 .55959 L
s
.02381 .0337 m
.02756 .0337 L
s
.02381 .01617 m
.02756 .01617 L
s
.02381 .59465 m
.02756 .59465 L
s
.02381 .61218 m
.02756 .61218 L
s
.25 Mabswid
.02381 0 m
.02381 .61803 L
s
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
.5 Mabswid
.02381 .48947 m
.02499 .48937 L
.02605 .48911 L
.02729 .48861 L
.02846 .48796 L
.03053 .48646 L
.03279 .48444 L
.0379 .47874 L
.04262 .47151 L
.0472 .46179 L
.05224 .44831 L
.06244 .41645 L
.08274 .33371 L
.09394 .28527 L
.10458 .24446 L
.11464 .21473 L
.11739 .20849 L
.12033 .20284 L
.12287 .19881 L
.12562 .19537 L
.12694 .19407 L
.12833 .19294 L
.12951 .19218 L
.13082 .19156 L
.13204 .19117 L
.13335 .19098 L
.13459 .19101 L
.13571 .19121 L
.13688 .19159 L
.13797 .1921 L
.13915 .19283 L
.14041 .19381 L
.14253 .1959 L
.14487 .19885 L
.14978 .20718 L
.15496 .21883 L
.16421 .24588 L
.1743 .2817 L
.18507 .32271 L
.19039 .34237 L
.193 .35161 L
.19541 .3598 L
.19655 .36322 L
.1976 .36538 L
.19871 .3659 L
.19992 .36394 L
.20115 .35901 L
.20228 .3521 L
.20481 .33093 L
.22318 .16906 L
Mistroke
.23239 .10233 L
.23745 .07271 L
.2422 .05006 L
.24492 .03953 L
.24785 .0302 L
.25051 .02359 L
.25178 .02108 L
.25296 .01912 L
.25418 .01747 L
.25528 .01631 L
.25651 .01538 L
.25782 .01483 L
.25908 .01472 L
.25979 .01484 L
.26045 .01507 L
.26166 .01578 L
.26294 .01696 L
.26408 .01836 L
.26533 .02029 L
.26784 .02539 L
.27017 .03157 L
.27234 .03854 L
.27723 .0586 L
.28245 .0863 L
.29204 .15236 L
.30074 .22569 L
.32159 .4173 L
.33073 .49044 L
.33558 .52311 L
.34078 .5525 L
.34584 .57498 L
.35046 .58992 L
.35268 .59516 L
.35385 .59741 L
.3551 .5994 L
.35618 .6008 L
.35738 .602 L
.35847 .60275 L
.3595 .60317 L
.36065 .60332 L
.36192 .60308 L
.36324 .60237 L
.36447 .60131 L
.36562 .59995 L
.36667 .59842 L
.36902 .59394 L
.37156 .58752 L
.3743 .57873 L
.3793 .55805 L
.38874 .50343 L
Mistroke
.39915 .42415 L
.41013 .3277 L
.41494 .28461 L
.41748 .26222 L
.42018 .23901 L
.42085 .23376 L
.42158 .22873 L
.42289 .222 L
.42417 .21911 L
.42535 .21987 L
.42646 .22338 L
.42768 .22968 L
.43014 .24582 L
.43961 .31282 L
.45053 .39117 L
.46049 .455 L
.46529 .48108 L
.47052 .505 L
.47545 .52282 L
.48001 .53486 L
.4825 .53954 L
.48364 .54123 L
.48484 .5427 L
.48548 .54335 L
.48617 .54396 L
.4874 .54477 L
.48809 .54507 L
.48883 .54528 L
.48953 .54537 L
.49016 .54535 L
.49083 .54523 L
.49154 .54499 L
.49282 .54428 L
.49353 .54372 L
.49419 .5431 L
.4957 .54132 L
.49823 .5372 L
.50093 .5312 L
.5062 .51501 L
.51107 .49488 L
.52212 .43317 L
.54223 .28502 L
.56108 .15163 L
.56624 .12288 L
.57169 .09772 L
.57634 .08084 L
.57896 .0733 L
.58143 .06752 L
.58278 .06493 L
.58424 .06257 L
Mistroke
.58546 .06097 L
.5868 .05957 L
.58755 .05897 L
.58825 .05852 L
.58899 .05816 L
.58979 .05791 L
.59052 .0578 L
.59123 .05781 L
.59254 .05812 L
.59379 .05877 L
.59448 .05928 L
.59511 .05984 L
.59749 .06273 L
.59883 .0649 L
.60028 .06768 L
.6029 .07386 L
.60801 .09004 L
.61341 .11277 L
.62309 .16629 L
.63256 .23089 L
.63506 .24925 L
.63637 .25894 L
.63777 .26836 L
.63899 .27509 L
.63964 .27795 L
.64034 .28043 L
.64156 .28325 L
.64271 .28432 L
.64388 .28424 L
.64515 .2835 L
.64772 .28186 L
.6523 .27921 L
.6565 .27715 L
.66105 .27538 L
.66379 .27457 L
.665 .27429 L
.66633 .27402 L
.66749 .27383 L
.66875 .27367 L
.66944 .2736 L
.67007 .27355 L
.67131 .27349 L
.6725 .27348 L
.6736 .27351 L
.67481 .27359 L
.6761 .27372 L
.67733 .2739 L
.67849 .27412 L
.68065 .27465 L
.6832 .27547 L
Mistroke
.68589 .27658 L
.69071 .27917 L
.6959 .28279 L
.70159 .2877 L
.72113 .31034 L
.74206 .33833 L
.75225 .3505 L
.76333 .36102 L
.7684 .36464 L
.76964 .36549 L
.77033 .36616 L
.77097 .367 L
.77168 .36827 L
.77242 .37006 L
.77379 .37467 L
.77495 .37994 L
.77619 .38676 L
.77841 .40081 L
.78346 .43235 L
.79372 .48755 L
.79951 .51181 L
.80484 .52901 L
.80709 .53469 L
.8095 .53971 L
.81089 .54208 L
.81216 .54391 L
.81336 .54535 L
.81463 .54657 L
.81596 .54748 L
.81718 .54802 L
.8185 .54826 L
.81925 .54824 L
.81995 .54811 L
.82122 .54764 L
.82238 .54692 L
.82365 .54582 L
.825 .54431 L
.82756 .54043 L
.82998 .53557 L
.83452 .52343 L
.8394 .5061 L
.8447 .48266 L
.86306 .37289 L
.88251 .24091 L
.89283 .1818 L
.89853 .1557 L
.90394 .13608 L
.90639 .12897 L
.90901 .12267 L
.91052 .11967 L
Mistroke
.91188 .11734 L
.91327 .11535 L
.91455 .11388 L
.91572 .11282 L
.91695 .112 L
.91819 .11149 L
.91889 .11134 L
.91955 .1113 L
.92074 .11144 L
.92182 .11182 L
.92307 .11256 L
.92426 .11355 L
.92656 .11627 L
.92899 .12032 L
.93338 .13049 L
.93809 .14542 L
.94329 .16633 L
.95913 .25229 L
.97619 .36073 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[38]:=",
  ImageSize->{288, 177.938},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/A000`40O003h00Oogoo8Goo003oOolQ
Ool00?moob5oo`00ogoo8Goo003oOolQOol00?moob5oo`006goo00<007ooOol0ogoo0goo000KOol0
0`00Oomoo`3oOol3Ool001]oo`03001oogoo0?moo`=oo`006goo00<007ooOol0ogoo0goo000KOol2
000kOol30035Ool001]oo`03001oogoo03Uoo`80009oo`03001oogoo0<9oo`006goo00<007ooOol0
>Goo00<007ooOol00Woo00<007ooOol0`Goo000KOol00`00Oomoo`0hOol00`00Oomoo`03Ool00`00
Oomoo`31Ool001]oo`03001oogoo03Qoo`03001oogoo00=oo`03001oogoo0<5oo`006goo0P00>Goo
00<007ooOol017oo00<007ooOol0`7oo000KOol00`00Oomoo`0hOol00`00Oomoo`04Ool00`00Oomo
o`30Ool000]oo`03001oogoo009oo`04001oogooOol20005Ool00`00Oomoo`0hOol00`00Oomoo`04
Ool00`00Oomoo`30Ool000Yoo`03001oo`0000Uoo`03001oogoo009oo`03001oogoo03Moo`03001o
ogoo00Eoo`03001oogoo0<1oo`001Goo0`000Woo00<007oo000027oo00<007ooOol00goo0P00>7oo
00<007ooOol01Woo00<007ooOol0_goo000:Ool00`00Ool00009Ool00`00Oomoo`02Ool00`00Oomo
o`0gOol00`00Oomoo`06Ool00`00Oomoo`2oOol000Yoo`03001oo`0000Moo`03001oo`0000Aoo`03
001oogoo03Moo`03001oogoo00Ioo`03001oogoo051oo`8006eoo`002goo00<007ooOol01goo00<0
07ooOol00goo00<007ooOol0=Woo00<007ooOol01goo00<007ooOol0Cgoo0P0000=oo`00Ool0Jgoo
000KOol00`00Oomoo`0fOol00`00Oomoo`08Ool00`00Oomoo`1>Ool00`00Oomoo`02001[Ool001]o
o`8003Moo`03001oogoo00Qoo`03001oogoo04eoo`03001oogoo009oo`03001oogoo06Uoo`006goo
00<007ooOol0=Woo00<007ooOol027oo00<007ooOol0CGoo00<007ooOol00goo00<007ooOol0J7oo
000KOol00`00Oomoo`0fOol00`00Oomoo`08Ool00`00Oomoo`1<Ool00`00Oomoo`04Ool00`00Oomo
o`1XOol001]oo`03001oogoo03Eoo`03001oogoo00Yoo`03001oogoo04]oo`03001oogoo00Aoo`03
001oogoo06Qoo`006goo00<007ooOol0=Goo00<007ooOol02Woo00<007ooOol0Bgoo00<007ooOol0
1Goo00<007ooOol0Igoo000KOol2000fOol00`00Oomoo`0:Ool00`00Oomoo`1;Ool00`00Oomoo`05
Ool00`00Oomoo`1WOol001]oo`03001oogoo03Eoo`03001oogoo00Yoo`03001oogoo04Yoo`03001o
ogoo00Ioo`03001oogoo06Moo`006goo00<007ooOol0=7oo00<007ooOol02goo00<007ooOol0BWoo
00<007ooOol01Woo00<007ooOol0Igoo000KOol00`00Oomoo`0dOol00`00Oomoo`0;Ool00`00Oomo
o`1:Ool00`00Oomoo`07Ool00`00Oomoo`1VOol001]oo`8003Eoo`03001oogoo00]oo`03001oogoo
04Yoo`03001oogoo00Moo`03001oogoo06Ioo`006goo00<007ooOol0=7oo00<007ooOol037oo00<0
07ooOol0B7oo00<007ooOol027oo00<007ooOol0IWoo000KOol00`00Oomoo`0dOol00`00Oomoo`0<
Ool00`00Oomoo`18Ool00`00Oomoo`09Ool00`00Oomoo`1>Ool00`00Oomoo`0DOol001]oo`03001o
ogoo03Aoo`03001oogoo00aoo`03001oogoo04Qoo`03001oogoo00Uoo`03001oogoo04eoo`03001o
o`0001Eoo`006goo00<007ooOol0=7oo00<007ooOol037oo00<007ooOol0B7oo00<007ooOol02Goo
00<007ooOol0C7oo00D007ooOomoo`0001Aoo`006goo0P00=7oo00<007ooOol03Goo00<007ooOol0
Agoo00<007ooOol02Woo00<007ooOol0Bgoo0P0017oo00<007ooOol04Goo000KOol00`00Oomoo`0c
Ool00`00Oomoo`0=Ool00`00Oomoo`17Ool00`00Oomoo`0:Ool00`00Oomoo`1;Ool00`00Oomoo`03
Ool00`00Oomoo`0AOol001]oo`03001oogoo03=oo`03001oogoo00eoo`03001oogoo04Moo`03001o
ogoo00]oo`03001oogoo04Yoo`03001oogoo00Aoo`03001oogoo011oo`002goo00<007ooOol00Woo
00@007ooOomoo`<000Aoo`03001oogoo03=oo`03001oogoo00eoo`03001oogoo04Moo`03001oogoo
00]oo`03001oogoo04Uoo`03001oogoo00Eoo`03001oogoo011oo`002Woo00<007oo00001goo00<0
07ooOol017oo00<007ooOol0<goo00<007ooOol03Woo00<007ooOol0AWoo00<007ooOol02goo00<0
07ooOol0BGoo00<007ooOol01Goo00<007ooOol047oo0005Ool30002Ool00`00Ool00008Ool00`00
Oomoo`03Ool2000dOol00`00Oomoo`0>Ool00`00Oomoo`15Ool00`00Oomoo`0<Ool00`00Oomoo`19
Ool00`00Oomoo`06Ool00`00Oomoo`0?Ool000Yoo`03001oo`0000Qoo`03001oogoo00=oo`03001o
ogoo03=oo`03001oogoo00ioo`03001oogoo04Eoo`03001oogoo00aoo`03001oogoo04Uoo`03001o
ogoo00Ioo`03001oogoo00moo`002Woo00<007oo00001goo00<007oo000017oo00<007ooOol0<goo
00<007ooOol03Woo00<007ooOol0AGoo00<007ooOol037oo00<007ooOol0B7oo00<007ooOol01goo
00<007ooOol03goo000;Ool00`00Oomoo`07Ool00`00Oomoo`03Ool00`00Oomoo`0cOol00`00Oomo
o`0>Ool00`00Oomoo`15Ool00`00Oomoo`0<Ool00`00Oomoo`18Ool00`00Oomoo`07Ool00`00Oomo
o`0?Ool001]oo`8003=oo`03001oogoo00moo`03001oogoo04Eoo`03001oogoo00eoo`03001oogoo
04Moo`03001oogoo00Moo`03001oogoo00moo`006goo00<007ooOol0<Woo00<007ooOol03goo00<0
07ooOol0AGoo00<007ooOol03Goo00<007ooOol0Agoo00<007ooOol027oo00<007ooOol03Woo000K
Ool00`00Oomoo`0bOol00`00Oomoo`0?Ool00`00Oomoo`15Ool00`00Oomoo`0=Ool00`00Oomoo`16
Ool00`00Oomoo`09Ool00`00Oomoo`0>Ool001]oo`03001oogoo039oo`03001oogoo00moo`03001o
ogoo04Aoo`03001oogoo00ioo`03001oogoo04Ioo`03001oogoo00Uoo`03001oogoo00ioo`006goo
00<007ooOol0<Woo00<007ooOol047oo00<007ooOol0@goo00<007ooOol03Woo00<007ooOol0AWoo
00<007ooOol02Goo00<007ooOol03Woo000KOol2000cOol00`00Oomoo`0@Ool00`00Oomoo`13Ool0
0`00Oomoo`0>Ool00`00Oomoo`16Ool00`00Oomoo`09Ool00`00Oomoo`0>Ool001]oo`03001oogoo
039oo`03001oogoo011oo`03001oogoo04=oo`03001oogoo00moo`03001oogoo04Aoo`03001oogoo
00]oo`03001oogoo00eoo`006goo00<007ooOol0<Woo00<007ooOol047oo00<007ooOol0@goo00<0
07ooOol03goo00<007ooOol0A7oo00<007ooOol02goo00<007ooOol03Goo000KOol00`00Oomoo`0b
Ool00`00Oomoo`0@Ool00`00Oomoo`13Ool00`00Oomoo`0?Ool00`00Oomoo`14Ool00`00Oomoo`0;
Ool00`00Oomoo`0=Ool001]oo`03001oogoo035oo`03001oogoo015oo`03001oogoo04=oo`03001o
ogoo00moo`03001oogoo04Aoo`03001oogoo00]oo`03001oogoo00eoo`006goo0P006Goo1@0057oo
00<007ooOol04Goo00<007ooOol0@Woo00<007ooOol047oo00<007ooOol0@goo00<007ooOol03Goo
00<007ooOol037oo000KOol00`00Oomoo`0HOol01@00Oomoogoo000057oo00<007ooOol04Goo00<0
07ooOol0@Woo00<007ooOol047oo00<007ooOol0@goo00<007ooOol03Goo00<007ooOol037oo000K
Ool00`00Oomoo`0GOol00`00Oomoo`03Ool00`00Oomoo`0AOol00`00Oomoo`0AOol00`00Oomoo`12
Ool00`00Oomoo`0AOol00`00Oomoo`12Ool00`00Oomoo`0=Ool00`00Oomoo`0<Ool001]oo`03001o
ogoo01Moo`03001oogoo00Aoo`03001oogoo011oo`03001oogoo015oo`03001oogoo049oo`03001o
ogoo015oo`03001oogoo049oo`03001oogoo00eoo`03001oogoo00aoo`006goo0P005goo00<007oo
Ool01Goo00<007ooOol047oo00<007ooOol04Woo00<007ooOol0@Goo00<007ooOol04Goo00<007oo
Ool0@Woo00<007ooOol03Goo00<007ooOol037oo000KOol00`00Oomoo`0EOol00`00Oomoo`07Ool0
0`00Oomoo`0?Ool00`00Oomoo`0BOol00`00Oomoo`11Ool00`00Oomoo`0AOol00`00Oomoo`11Ool0
0`00Oomoo`0?Ool00`00Oomoo`0;Ool001]oo`03001oogoo01Eoo`03001oogoo00Moo`03001oogoo
00moo`03001oogoo019oo`03001oogoo045oo`03001oogoo015oo`03001oogoo045oo`03001oogoo
00moo`03001oogoo00]oo`002goo00<007ooOol00Woo00<007ooOol01@000goo00<007ooOol05Goo
00<007ooOol027oo00<007ooOol03Goo00<007ooOol04goo00<007ooOol07Woo0P0087oo00<007oo
Ool04Woo00<007ooOol0@Goo00<007ooOol03goo00<007ooOol02goo000:Ool00`00Ool00008Ool0
0`00Oomoo`03Ool00`00Oomoo`0DOol00`00Oomoo`09Ool00`00Oomoo`0=Ool00`00Oomoo`0COol0
0`00Oomoo`0NOol2000POol00`00Oomoo`0COol00`00Oomoo`10Ool00`00Oomoo`0?Ool00`00Oomo
o`0;Ool000Eoo`<0009oo`03001oo`0000Qoo`03001oogoo00=oo`8001Eoo`03001oogoo00Yoo`03
001oogoo00aoo`03001oogoo01=oo`03001oogoo01ioo`80021oo`03001oogoo01=oo`03001oogoo
041oo`03001oogoo00moo`03001oogoo00]oo`002Woo00<007oo000027oo00<007ooOol00goo00<0
07ooOol04goo00<007ooOol02goo00<007ooOol037oo00<007ooOol04goo00<007ooOol07Woo0P00
87oo00<007ooOol04goo00<007ooOol0@7oo00<007ooOol047oo00<007ooOol02Woo000:Ool00`00
Ool00008Ool00`00Oomoo`03Ool00`00Oomoo`0COol00`00Oomoo`0;Ool00`00Oomoo`0<Ool00`00
Oomoo`0COol00`00Oomoo`0MOol00`00Ool0000POol00`00Oomoo`0COol00`00Oomoo`0oOol00`00
Oomoo`0AOol00`00Oomoo`0:Ool000]oo`03001oogoo00Ioo`8000Eoo`03001oogoo01=oo`03001o
ogoo00]oo`03001oogoo00aoo`03001oogoo01=oo`03001oogoo01eoo`04001oogoo000OOol00`00
Oomoo`0COol00`00Oomoo`0oOol00`00Oomoo`0AOol00`00Oomoo`0:Ool001]oo`03001oogoo019o
o`03001oogoo00eoo`03001oogoo00]oo`03001oogoo01=oo`03001oogoo01eoo`04001oogoo000O
Ool00`00Oomoo`0COol00`00Oomoo`0oOol00`00Oomoo`0AOol00`00Oomoo`0:Ool001]oo`8001=o
o`03001oogoo00eoo`03001oogoo00]oo`03001oogoo01Aoo`03001oogoo01aoo`04001oogoo000O
Ool00`00Oomoo`0COol00`00Oomoo`0oOol00`00Oomoo`0BOol00`00Oomoo`09Ool001]oo`03001o
ogoo019oo`03001oogoo00eoo`03001oogoo00]oo`03001oogoo01Aoo`03001oogoo01aoo`04001o
ogoo000NOol00`00Oomoo`0DOol00`00Oomoo`0oOol00`00Oomoo`0BOol00`00Oomoo`09Ool001]o
o`03001oogoo015oo`03001oogoo00moo`03001oogoo00Uoo`03001oogoo01Eoo`03001oogoo01ao
o`04001oogoo000NOol00`00Oomoo`0DOol00`00Oomoo`0oOol00`00Oomoo`0BOol00`00Oomoo`09
Ool001]oo`03001oogoo015oo`03001oogoo00moo`03001oogoo00Uoo`03001oogoo01Eoo`03001o
ogoo01aoo`04001oogoo000NOol00`00Oomoo`0EOol00`00Oomoo`0mOol00`00Oomoo`0COol00`00
Oomoo`09Ool001]oo`80019oo`03001oogoo00moo`03001oogoo00Uoo`03001oogoo01Eoo`03001o
ogoo01aoo`04001oogoo000NOol00`00Oomoo`0EOol00`00Oomoo`0mOol00`00Oomoo`0COol00`00
Oomoo`09Ool001]oo`03001oogoo015oo`03001oogoo011oo`03001oogoo00Qoo`03001oogoo01Eo
o`03001oogoo01aoo`05001oogooOol0000MOol00`00Oomoo`0EOol00`00Oomoo`0mOol00`00Oomo
o`0COol00`00Oomoo`09Ool001]oo`03001oogoo011oo`03001oogoo015oo`03001oogoo00Qoo`03
001oogoo01Eoo`03001oogoo01aoo`05001oogooOol0000MOol00`00Oomoo`0EOol00`00Oomoo`0m
Ool00`00Oomoo`0DOol00`00Oomoo`08Ool001]oo`03001oogoo011oo`03001oogoo00Eoo`<000=o
o`03001oogoo009oo`8000Yoo`03001oogoo00eoo`03001oogoo00Eoo`03001oogoo015oo`03001o
ogoo009oo`03001oogoo009oo`8000=oo`03001oogoo011oo`D0009oo`05001oogooOol0000@Ool5
000017oo000000000goo00<007ooOol00Woo00<007ooOol04goo1@000Woo00<007ooOol04Woo1@00
0Woo00<007ooOol00Woo00<007ooOol00Woo00<007ooOol01goo00<007ooOol027oo000KOol00`00
Oomoo`0@Ool00`00Oomoo`05Ool00`00Oomoo`07Ool00`00Ool0000:Ool00`00Oomoo`0<Ool00`00
Ool00007Ool00`00Oomoo`0@Ool00`00Oomoo`06Ool30003Ool00`00Oomoo`0BOol02@00Oomoogoo
001oo`00Ool0000COol01000Oomoogoo0P001Goo200057oo00L007ooOomoo`00Ool0000EOol00`00
Ool00002Ool00`00Oomoo`06Ool00`00Ool00008Ool00`00Oomoo`08Ool001]oo`80015oo`03001o
ogoo00Ioo`03001oogoo00Qoo`8000Uoo`03001oogoo00ioo`03001oogoo00Eoo`03001oogoo011o
o`03001oogoo00Moo`8000=oo`03001oogoo019oo`09001oogooOol007oo001oo`0001=oo`03001o
ogoo009oo`80009oo`8000Eoo`04001oogoo000COol00`00Oomoo`03Ool00`00Oomoo`0COol00`00
Ool00002Ool00`00Oomoo`08Ool00`00Oomoo`06Ool00`00Oomoo`08Ool001]oo`03001oogoo00mo
o`03001oogoo00Moo`03001oogoo00Ioo`800003Ool007oo00Qoo`03001oogoo00aoo`8000Qoo`03
001oogoo015oo`03001oogoo00Eoo`8000Aoo`03001oogoo019oo`09001oogooOol007oo001oo`00
01=oo`03001oogoo009oo`@000Eoo`8000Aoo`80015oo`04001oogooOol2000FOol20004Ool00`00
Oomoo`05Ool20009Ool00`00Oomoo`08Ool001]oo`03001oogoo00moo`03001oogoo00Ioo`03001o
o`0000Moo`04001oogoo0008Ool00`00Oomoo`0=Ool00`00Oomoo`07Ool00`00Oomoo`0?Ool00`00
Ool00007Ool20004Ool00`00Oomoo`0BOol02@00Oomoogoo001oo`00Ool0000COol01`00Oomoogoo
001oo`0000Moo`03001oogoo00Eoo`03001oogoo00ioo`05001oogooOol0000GOol20002Ool00`00
Ool00007Ool00`00Oomoo`08Ool00`00Oomoo`08Ool001]oo`03001oogoo00moo`03001oogoo00Mo
o`03001oogoo00Ioo`<00003Ool007oo00Ioo`03001oogoo00eoo`<000Moo`03001oogoo00moo`<0
00Moo`<000=oo`03001oogoo015oo`8000Aoo`04001oogoo000BOol20004Ool00`00Oomoo`06Ool3
0006Ool00`00Oomoo`0<Ool20003Ool3000DOol30002Ool30007Ool30008Ool00`00Oomoo`08Ool0
01]oo`03001oogoo00moo`03001oogoo01Aoo`03001oogoo00Eoo`03001oogoo01Moo`03001oogoo
01Yoo`03001oogoo009oo`03001oogoo01Yoo`03001oogoo02Uoo`03001oogoo02Uoo`03001oogoo
01Ioo`03001oogoo00Moo`006goo0P0047oo00<007ooOol057oo00<007ooOol01Goo00<007ooOol0
5goo00<007ooOol06Woo00<007ooOol00goo00<007ooOol06Goo00<007ooOol0:Woo00<007ooOol0
:7oo00<007ooOol05Woo00<007ooOol01goo000KOol00`00Oomoo`0>Ool00`00Oomoo`0EOol00`00
Oomoo`05Ool00`00Oomoo`0GOol00`00Oomoo`0JOol00`00Oomoo`03Ool00`00Oomoo`0HOol00`00
Oomoo`0[Ool00`00Oomoo`0XOol00`00Oomoo`0FOol00`00Oomoo`07Ool001]oo`03001oogoo00io
o`03001oogoo01Ioo`03001oogoo00Aoo`03001oogoo01Moo`03001oogoo01Yoo`03001oogoo00=o
o`03001oogoo01Qoo`03001oogoo02aoo`03001oogoo02Moo`03001oogoo01Ioo`03001oogoo00Mo
o`006goo00<007ooOol03Woo00<007ooOol05Woo00<007ooOol017oo00<007ooOol067oo00<007oo
Ool067oo00<007ooOol017oo00<007ooOol067oo00<007ooOol0;Goo00<007ooOol09Goo00<007oo
Ool05goo00<007ooOol01goo000DOooo000=000001]oo`03001oogoo00=oo`03001oogoo00Aoo`05
001oogooOol00002Ool00`00Oomoo`03Ool00`00Oomoo`04Ool00`00Oomoo`04Ool00`00Ool00003
Ool00`00Ool00004Ool00`00Oomoo`04Ool00`00Oomoo`03Ool00`00Oomoo`04Ool01000Oomoo`00
0goo00<007ooOol00goo00<007ooOol017oo00<007ooOol017oo00@007ooOol0009oo`05001oogoo
Ool00002Ool00`00Oomoo`04Ool00`00Oomoo`03Ool00`00Oomoo`04Ool01@00Oomoogoo00000Woo
00<007ooOol00goo00<007ooOol017oo00<007ooOol017oo00<007ooOol00goo00<007ooOol017oo
00<007ooOol017oo00<007ooOol00goo0P001Goo00<007ooOol017oo00<007ooOol00goo00<007oo
Ool017oo00<007ooOol017oo00<007ooOol00Woo0P001Woo00<007ooOol017oo00<007ooOol017oo
00L007ooOomoo`00Ool00007Ool001]oo`03001oogoo00ioo`03001oogoo00eoo`03001oogoo00Mo
o`03001oogoo009oo`03001oogoo00moo`03001oogoo00Moo`03001oogoo01Eoo`04001oogoo0006
Ool00`00Oomoo`0DOol01@00Oomoogoo000077oo00<007ooOol04Woo00<007ooOol02Woo00<007oo
Ool05Woo00<007ooOol01Goo00<007ooOol03goo00<007ooOol01goo000KOol00`00Oomoo`0=Ool0
0`00Oomoo`0HOol00`00Oomoo`02Ool00`00Oomoo`0IOol00`00Oomoo`0HOol00`00Oomoo`04Ool0
0`00Oomoo`0GOol00`00Oomoo`0aOol00`00Oomoo`0ROol00`00Oomoo`0HOol00`00Oomoo`06Ool0
01]oo`03001oogoo00eoo`03001oogoo01Qoo`03001oogoo009oo`03001oogoo01Uoo`03001oogoo
01Qoo`03001oogoo00Eoo`03001oogoo01Ioo`03001oogoo039oo`03001oogoo025oo`03001oogoo
01Qoo`03001oogoo00Ioo`006goo0P003Woo00<007ooOol067oo00<007ooOol00Woo00<007ooOol0
6Goo00<007ooOol067oo00<007ooOol01Goo00<007ooOol05Woo00<007ooOol0<goo00<007ooOol0
87oo00<007ooOol067oo00<007ooOol01Woo000KOol00`00Oomoo`0=Ool00`00Oomoo`0IOol01@00
Oomoogoo00006goo00<007ooOol067oo00<007ooOol01Goo00<007ooOol05Woo00<007ooOol0<goo
00<007ooOol07goo00<007ooOol06Goo00<007ooOol01Woo000KOol00`00Oomoo`0=Ool00`00Oomo
o`0IOol01@00Oomoogoo000077oo00<007ooOol05goo00<007ooOol01Goo00<007ooOol05Woo00<0
07ooOol0=7oo00<007ooOol07Woo00<007ooOol06Goo00<007ooOol01Woo000KOol00`00Oomoo`0<
Ool00`00Oomoo`0JOol01000Oomoo`007Goo00<007ooOol05goo00<007ooOol01Goo00<007ooOol0
5Woo00<007ooOol0=Goo00<007ooOol07Goo00<007ooOol06Goo00<007ooOol01Woo000KOol00`00
Oomoo`0<Ool00`00Oomoo`0KOol00`00Ool0000MOol00`00Oomoo`0FOol00`00Oomoo`06Ool00`00
Oomoo`0FOol00`00Oomoo`0fOol00`00Oomoo`0LOol00`00Oomoo`0IOol00`00Oomoo`06Ool001]o
o`8000eoo`03001oogoo01]oo`03001oo`0001eoo`03001oogoo01Ioo`03001oogoo00Ioo`03001o
ogoo01Eoo`03001oogoo03Qoo`03001oogoo01]oo`03001oogoo01Yoo`03001oogoo00Eoo`006goo
00<007ooOol037oo00<007ooOol06goo00<007oo00007Goo00<007ooOol05Woo00<007ooOol01goo
00<007ooOol057oo00<007ooOol0>Goo00<007ooOol06Woo00<007ooOol06Woo00<007ooOol01Goo
000KOol00`00Oomoo`0;Ool00`00Oomoo`0LOol00`00Ool0000MOol00`00Oomoo`0FOol00`00Oomo
o`07Ool00`00Oomoo`0DOol00`00Oomoo`0jOol00`00Oomoo`0IOol00`00Oomoo`0JOol00`00Oomo
o`05Ool001]oo`03001oogoo00]oo`03001oogoo01aoo`<001eoo`03001oogoo01Ioo`03001oogoo
00Moo`03001oogoo01Aoo`03001oogoo03]oo`03001oogoo01Moo`03001oogoo01]oo`03001oogoo
00Eoo`006goo0P0037oo00<007ooOol07Goo0P007Goo00<007ooOol05Woo00<007ooOol01goo00<0
07ooOol057oo00<007ooOol0?7oo0P005goo00<007ooOol08goo000KOol00`00Oomoo`0;Ool00`00
Oomoo`0lOol00`00Oomoo`0FOol00`00Oomoo`07Ool00`00Oomoo`0DOol00`00Oomoo`0mOol00`00
Oomoo`0EOol00`00Oomoo`0SOol001]oo`03001oogoo00Yoo`03001oogoo03ioo`03001oogoo01Eo
o`03001oogoo00Moo`03001oogoo01=oo`03001oogoo03ioo`03001oogoo01Eoo`03001oogoo02=o
o`006goo00<007ooOol02Woo00<007ooOol0?Woo00<007ooOol057oo00<007ooOol027oo00<007oo
Ool04goo00<007ooOol0?goo00<007ooOol057oo00<007ooOol08goo000KOol00`00Oomoo`0:Ool0
0`00Oomoo`0nOol00`00Oomoo`0DOol00`00Oomoo`09Ool00`00Oomoo`0BOol00`00Oomoo`0oOol0
0`00Oomoo`0DOol00`00Oomoo`0SOol001]oo`8000Yoo`03001oogoo03moo`03001oogoo01Aoo`03
001oogoo00Uoo`03001oogoo019oo`03001oogoo03moo`03001oogoo01=oo`03001oogoo02Aoo`00
6goo00<007ooOol02Goo00<007ooOol0?goo00<007ooOol057oo00<007ooOol02Goo00<007ooOol0
4Woo00<007ooOol0?goo00<007ooOol04goo00<007ooOol097oo000KOol00`00Oomoo`09Ool00`00
Oomoo`0oOol00`00Oomoo`0DOol00`00Oomoo`09Ool00`00Oomoo`0BOol00`00Oomoo`0oOol00`00
Oomoo`0COol00`00Oomoo`0TOol000]oo`03001oogoo009oo`03001oogoo00D000=oo`03001oogoo
00Uoo`03001oogoo03moo`03001oogoo01Aoo`03001oogoo00Uoo`03001oogoo019oo`03001oogoo
041oo`03001oogoo019oo`03001oogoo02Aoo`002Woo00<007oo000027oo00<007ooOol00goo00<0
07ooOol027oo00<007ooOol0@7oo00<007ooOol057oo00<007ooOol02Goo00<007ooOol04Goo00<0
07ooOol0@Goo00<007ooOol04Woo00<007ooOol097oo000:Ool00`00Ool00008Ool00`00Oomoo`03
Ool20009Ool00`00Oomoo`11Ool00`00Oomoo`0COol00`00Oomoo`0:Ool00`00Oomoo`0@Ool00`00
Oomoo`11Ool00`00Oomoo`0BOol00`00Oomoo`0TOol000Yoo`03001oo`0000Qoo`03001oogoo00=o
o`03001oogoo00Qoo`03001oogoo045oo`03001oogoo01=oo`03001oogoo00Yoo`03001oogoo011o
o`03001oogoo045oo`03001oogoo015oo`03001oogoo02Eoo`002Woo00<007oo000027oo00<007oo
Ool00goo00<007ooOol027oo00<007ooOol0@Goo00<007ooOol04Woo00<007ooOol02goo00<007oo
Ool047oo00<007ooOol0@Goo00<007ooOol04Goo00<007ooOol09Goo000;Ool00`00Oomoo`06Ool2
0005Ool00`00Oomoo`07Ool00`00Oomoo`12Ool00`00Oomoo`0BOol00`00Oomoo`0;Ool00`00Oomo
o`0@Ool00`00Oomoo`11Ool00`00Oomoo`0AOol00`00Oomoo`0UOol001]oo`8000Qoo`03001oogoo
049oo`03001oogoo019oo`03001oogoo00]oo`03001oogoo011oo`03001oogoo049oo`03001oogoo
011oo`03001oogoo02Eoo`006goo00<007ooOol01goo00<007ooOol0@Woo00<007ooOol04Woo00<0
07ooOol02goo00<007ooOol03goo00<007ooOol0@goo00<007ooOol047oo00<007ooOol09Goo000K
Ool00`00Oomoo`06Ool00`00Oomoo`13Ool00`00Oomoo`0BOol00`00Oomoo`0<Ool00`00Oomoo`0>
Ool00`00Oomoo`13Ool00`00Oomoo`0@Ool00`00Oomoo`0UOol001]oo`03001oogoo00Ioo`03001o
ogoo04=oo`03001oogoo019oo`03001oogoo00aoo`03001oogoo00ioo`03001oogoo04=oo`03001o
ogoo00moo`03001oogoo02Ioo`006goo00<007ooOol01Goo00<007ooOol0A7oo00<007ooOol04Woo
00<007ooOol037oo00<007ooOol03Woo00<007ooOol0@goo00<007ooOol03goo00<007ooOol09Woo
000KOol20006Ool00`00Oomoo`15Ool00`00Oomoo`0AOol00`00Oomoo`0<Ool00`00Oomoo`0>Ool0
0`00Oomoo`13Ool00`00Oomoo`0?Ool00`00Oomoo`0VOol001]oo`03001oogoo00Eoo`03001oogoo
04Eoo`03001oogoo011oo`03001oogoo00eoo`03001oogoo00ioo`03001oogoo04=oo`03001oogoo
00moo`03001oogoo02Ioo`006goo00<007ooOol017oo00<007ooOol0AWoo00<007ooOol047oo00<0
07ooOol03Goo00<007ooOol03Goo00<007ooOol0AGoo00<007ooOol03Woo00<007ooOol09Woo000K
Ool00`00Oomoo`04Ool00`00Oomoo`16Ool00`00Oomoo`0@Ool00`00Oomoo`0>Ool00`00Oomoo`0<
Ool00`00Oomoo`15Ool00`00Oomoo`0>Ool00`00Oomoo`0VOol001]oo`03001oogoo00Aoo`03001o
ogoo04Ioo`03001oogoo011oo`03001oogoo00ioo`03001oogoo00aoo`03001oogoo04Eoo`03001o
ogoo00eoo`03001oogoo02Moo`006goo0P0017oo00<007ooOol0Agoo00<007ooOol047oo00<007oo
Ool03Woo00<007ooOol037oo00<007ooOol0AGoo00<007ooOol03Goo00<007ooOol09goo000KOol0
0`00Oomoo`03Ool00`00Oomoo`17Ool00`00Oomoo`0@Ool00`00Oomoo`0>Ool00`00Oomoo`0<Ool0
0`00Oomoo`15Ool00`00Oomoo`0=Ool00`00Oomoo`0WOol001]oo`03001oogoo00=oo`03001oogoo
04Moo`03001oogoo011oo`03001oogoo00ioo`03001oogoo00]oo`03001oogoo04Moo`03001oogoo
00aoo`03001oogoo02Moo`006goo00<007ooOol00Woo00<007ooOol0B7oo00<007ooOol03goo00<0
07ooOol03goo00<007ooOol02goo00<007ooOol0Agoo00<007ooOol037oo00<007ooOol09goo000K
Ool20003Ool00`00Oomoo`18Ool00`00Oomoo`0?Ool00`00Oomoo`0@Ool00`00Oomoo`0:Ool00`00
Oomoo`17Ool00`00Oomoo`0<Ool00`00Oomoo`0WOol001]oo`05001oogooOol0001<Ool00`00Oomo
o`0>Ool00`00Oomoo`0@Ool00`00Oomoo`0:Ool00`00Oomoo`17Ool00`00Oomoo`0;Ool00`00Oomo
o`0XOol001]oo`05001oogooOol0001<Ool00`00Oomoo`0>Ool00`00Oomoo`0@Ool00`00Oomoo`0:
Ool00`00Oomoo`17Ool00`00Oomoo`0;Ool00`00Oomoo`0XOol000]oo`03001oogoo009oo`04001o
ogooOol30004Ool01000Oomoo`00CGoo00<007ooOol03Woo00<007ooOol047oo00<007ooOol02Woo
00<007ooOol0B7oo00<007ooOol02Woo00<007ooOol0:7oo000:Ool00`00Ool00007Ool00`00Oomo
o`04Ool00`00Ool0001>Ool00`00Oomoo`0>Ool00`00Oomoo`0@Ool00`00Oomoo`09Ool00`00Oomo
o`19Ool00`00Oomoo`0:Ool00`00Oomoo`0XOol000Yoo`03001oo`0000Qoo`03001oogoo00=oo`80
04moo`03001oogoo00ioo`03001oogoo015oo`03001oogoo00Qoo`03001oogoo04Uoo`03001oogoo
00Yoo`03001oogoo02Qoo`002Woo00<007oo000027oo00<007ooOol00goo00<007ooOol0CWoo00<0
07ooOol03Goo00<007ooOol04Woo00<007ooOol027oo00<007ooOol0BGoo00<007ooOol02Goo00<0
07ooOol0:Goo000:Ool00`00Ool00007Ool00`00Ool00004Ool00`00Oomoo`1>Ool00`00Oomoo`0=
Ool00`00Oomoo`0BOol00`00Oomoo`08Ool00`00Oomoo`19Ool00`00Oomoo`09Ool00`00Oomoo`0Y
Ool000]oo`03001oogoo00Moo`03001oogoo00=oo`03001oogoo04moo`03001oogoo00aoo`03001o
ogoo01=oo`03001oogoo00Moo`03001oogoo04Yoo`03001oogoo00Qoo`03001oogoo02Uoo`006goo
00<007ooOol0Cgoo00<007ooOol037oo00<007ooOol04goo00<007ooOol01Woo00<007ooOol0Bgoo
00<007ooOol027oo00<007ooOol0:Goo000KOol2001@Ool00`00Oomoo`0<Ool00`00Oomoo`0COol0
0`00Oomoo`06Ool00`00Oomoo`1;Ool00`00Oomoo`08Ool00`00Oomoo`0YOol001]oo`03001oogoo
04moo`03001oogoo00aoo`03001oogoo01=oo`03001oogoo00Ioo`03001oogoo04]oo`03001oogoo
00Moo`03001oogoo02Yoo`006goo00<007ooOol0D7oo00<007ooOol02Woo00<007ooOol05Goo00<0
07ooOol01Goo00<007ooOol0Bgoo00<007ooOol01goo00<007ooOol0:Woo000KOol00`00Oomoo`1@
Ool00`00Oomoo`0:Ool00`00Oomoo`0EOol00`00Oomoo`05Ool00`00Oomoo`1<Ool00`00Oomoo`05
Ool00`00Oomoo`0[Ool001]oo`80055oo`03001oogoo00Yoo`03001oogoo01Eoo`03001oogoo00Ao
o`03001oogoo04eoo`03001oogoo00Eoo`03001oogoo02]oo`006goo00<007ooOol0D7oo00<007oo
Ool02Woo00<007ooOol05Goo00<007ooOol017oo00<007ooOol0CWoo00<007ooOol017oo00<007oo
Ool0:goo000KOol00`00Oomoo`1@Ool00`00Oomoo`09Ool00`00Oomoo`0GOol00`00Oomoo`03Ool0
0`00Oomoo`1>Ool00`00Oomoo`03Ool00`00Oomoo`0/Ool001]oo`03001oogoo051oo`03001oogoo
00Uoo`03001oogoo01Moo`03001oogoo009oo`03001oogoo04moo`03001oogoo00=oo`03001oogoo
02aoo`006goo00<007ooOol0DGoo00<007ooOol027oo00<007ooOol067oo00D007ooOomoo`00059o
o`03001oogoo009oo`03001oogoo02aoo`006goo0P00DWoo00<007ooOol027oo00<007ooOol067oo
0P0000=oo`00Ool0DWoo00<007ooOol00Woo00<007ooOol0;7oo000KOol00`00Oomoo`1AOol00`00
Oomoo`08Ool00`00Oomoo`0IOol3001DOol4000_Ool001]oo`03001oogoo055oo`03001oogoo00Mo
o`03001oogoo079oo`80031oo`006goo00<007ooOol0DGoo00<007ooOol01goo00<007ooOol0Y7oo
000KOol00`00Oomoo`1AOol00`00Oomoo`07Ool00`00Oomoo`2TOol001]oo`80059oo`03001oogoo
00Moo`03001oogoo0:Aoo`006goo00<007ooOol0DWoo00<007ooOol01Woo00<007ooOol0Y7oo000;
Ool00`00Oomoo`02Ool01000Oomoogoo0P001Goo00<007ooOol0DWoo00<007ooOol01Goo00<007oo
Ool0YGoo000:Ool00`00Ool00009Ool00`00Oomoo`02Ool00`00Oomoo`1BOol00`00Oomoo`05Ool0
0`00Oomoo`2UOol000Yoo`03001oo`0000Qoo`03001oogoo00=oo`8005=oo`03001oogoo00Eoo`03
001oogoo0:Eoo`002Woo00<007oo00002Goo00<007ooOol00Woo00<007ooOol0Dgoo00<007ooOol0
17oo00<007ooOol0YGoo000:Ool00`00Ool00007Ool00`00Ool00004Ool00`00Oomoo`1COol00`00
Oomoo`03Ool00`00Oomoo`2VOol000]oo`03001oogoo00Moo`03001oogoo00=oo`03001oogoo05Ao
o`03001oogoo009oo`03001oogoo0:Ioo`006goo00<007ooOol0E7oo00<007ooOol00Woo00<007oo
Ool0YWoo000KOol2001EOol00`00Oomoo`02Ool00`00Oomoo`2VOol001]oo`03001oogoo05Eoo`04
001oogoo002YOol001]oo`03001oogoo05Eoo`<00:Yoo`006goo00<007ooOol0ogoo0goo000KOol0
0`00Oomoo`3oOol3Ool001]oo`800?moo`Aoo`006goo00<007ooOol0ogoo0goo003oOolQOol00?mo
ob5oo`00ogoo8Goo003oOolQOol00?moob5oo`00ogoo8Goo0000\
\>"],
  ImageRangeCache->{{{0, 287}, {176.938, 0}} -> {-2.04942, -0.385338, \
0.0749984, 0.00428907}},
  CellTags->{"Backlash", "i:3"}],

Cell[BoxData[
    TagBox[\(\[SkeletonIndicator]  Graphics  \[SkeletonIndicator]\),
      False,
      Editable->False]], "Output",
  CellLabel->"Out[38]=",
  CellTags->{"Backlash", "i:3"}]
}, Open  ]]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell["Friction", "Section",
  CellTags->{"Friction", "i:5"}],

Cell[CellGroupData[{

Cell["Two Masses With Backlash & Friction", "Subsection",
  CellTags->{"Friction", "i:5"}],

Cell[CellGroupData[{

Cell[BoxData[{
    \(r1 = {1}; H1 = Transpose[{{0, 0, 0, 1, 0, 0}}];\), "\n", 
    \(q1 = {x1}; p1 = {v1};\), "\n", 
    \(r2 = {1}; H2 = Transpose[{{0, 0, 0, 1, 0, 0}}];\), "\n", 
    \(q2 = {x2}; p2 = {v2};\), "\n", 
    \(\(JointLst = {{r1, H1, q1, p1}, {r2, H2, q2, p2}};\)\), "\n", 
    \(\({JV, JX, JH} = Joints[JointLst];\)\)}], "Input",
  CellLabel->"In[39]:=",
  PageWidth->Infinity,
  AspectRatioFixed->True,
  CellTags->{"Friction", "i:5"}],

Cell[BoxData[
    \("Computing joint 1 kinematics"\)], "Print",
  CellTags->{"Friction", "i:5"}],

Cell[BoxData[
    \("Computing joint 2 kinematics"\)], "Print",
  CellTags->{"Friction", "i:5"}]
}, Open  ]],

Cell[BoxData[{
    \(com1 = {0, 0, 0}; mass1 = m1; out1 = {2, {0, 0, 0}};\), "\n", 
    \(\(Inertia1 = {{0, 0, 0}, {0, 0, 0}, {0, 0, 0}};\)\), "\n", 
    \(com2 = {0, 0, 0}; mass2 = m2; out2 = {3, {0, 0, 0}};\), "\n", 
    \(\(Inertia2 = {{0, 0, 0}, {0, 0, 0}, {0, 0, 0}};\)\), "\n", 
    \(\(BodyLst = {{com1, {out1}, mass1, Inertia1}, {com2, {out2}, mass2, 
            Inertia2}};\)\), "\n", 
    \(\(TreeLst = {{{1, 1}, {2, 2}}};\)\)}], "Input",
  CellLabel->"In[40]:=",
  CellTags->{"Friction", "i:5"}],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(PEBack[x_] := \((k1*x^2)\)/2;\)\), "\n", 
    \(PE1 = BacklashPotential[PEBack, e, x2]\), "\n", 
    \(\(PE2 = \((1/2)\)*k2*\((x1 + x2)\)^2;\)\), "\n", 
    \(\(PE = PE1 + PE2;\)\), "\n", 
    \(\(Q = {FF1, FF2};\)\), "\n", 
    \(DissPot1 = JointFrictionPotential[v1, cv, cc, cs, vs]\), "\n", 
    \(\({JV, JX, JH, MM, Cp, Fp, pp, qq} = 
        CreateModel[JointLst, BodyLst, TreeLst, \(-g\), PE, DissPot1, Q, JV, 
          JX, JH];\)\n\)}], "Input",
  CellLabel->"In[41]:=",
  PageWidth->Infinity,
  AspectRatioFixed->True,
  CellTags->{"Friction", "i:5"}],

Cell[BoxData[
    \(1\/2\ k1\ \((\(-e\) + Abs[x2])\)\^2\ UnitStep[\(-e\) + 
          Abs[x2]]\)], "Output",
  CellLabel->"Out[41]=",
  CellTags->{"Friction", "i:5"}],

Cell[BoxData[
    \(\(cv\ v1\^2\)\/2 + cc\ Abs[v1] + 
      1\/2\ \((\(-cc\) + cs)\)\ \@\[Pi]\ vs\ Erf[v1\/vs]\ Sign[
          v1]\)], "Output",
  CellLabel->"Out[42]=",
  CellTags->{"Friction", "i:5"}],

Cell[BoxData[
    \("Computing Potential Functions"\)], "Print",
  CellTags->{"Friction", "i:5"}],

Cell[BoxData[
    \("Computing Inertia Matrix"\)], "Print",
  CellTags->{"Friction", "i:5"}],

Cell[BoxData[
    \("Computing Poincare Function"\)], "Print",
  CellTags->{"Friction", "i:5"}]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(Fp\)], "Input",
  CellLabel->"In[44]:=",
  CellTags->{"Friction", "i:5"}],

Cell[BoxData[
    \({\(-FF1\) + cv\ v1 + 
        k2\ \((x1 + x2)\) + \((\(-cc\) + 
              cs)\)\ E\^\(-\(v1\^2\/vs\^2\)\)\ Sign[v1] + 
        cc\ \((\(-UnitStep[\(-v1\)]\) + UnitStep[v1])\), \(-FF2\) + 
        k2\ \((x1 + x2)\) + 
        k1\ \((\(-e\) + Abs[x2])\)\ \((\(-UnitStep[\(-x2\)]\) + 
              UnitStep[x2])\)\ UnitStep[\(-e\) + Abs[x2]]}\)], "Output",
  CellLabel->"Out[44]=",
  CellTags->{"Friction", "i:5"}]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(UnitStep2Sign[Fp]\)], "Input",
  CellLabel->"In[45]:=",
  CellTags->{"Friction", "i:5"}],

Cell[BoxData[
    \({\(-FF1\) + cv\ v1 + k2\ x1 + k2\ x2 + 
        E\^\(-\(v1\^2\/vs\^2\)\)\ \((cs + 
              cc\ \((\(-1\) + E\^\(v1\^2\/vs\^2\))\))\)\ Sign[v1], \(-FF2\) + 
        k2\ x1 + \(k1\ x2\)\/2 + k2\ x2 - 1\/2\ e\ k1\ Sign[x2] + 
        1\/2\ k1\ x2\ Sign[\(-e\) + Abs[x2]] - 
        1\/2\ e\ k1\ Sign[x2]\ Sign[\(-e\) + Abs[x2]]}\)], "Output",
  CellLabel->"Out[45]=",
  CellTags->{"Friction", "i:5"}]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Open  ]]
},
FrontEndVersion->"4.0 for Microsoft Windows",
ScreenRectangle->{{0, 800}, {0, 527}},
ShowPageBreaks->False,
WindowSize->{656, 387},
WindowMargins->{{0, Automatic}, {Automatic, 5}},
Visible->True,
TaggingRules:>{"NextIndexingNumber" -> 6, "IndexEntries" -> {{"i:1", \
"BacklashPotential"}, {"i:1", "JointFrictionPotential"}, {"i:2", \
"BacklashPotential"}, {"i:3", "BacklashPotential"}, {"i:4", \
"JointFrictionPotential"}, {"i:5", "JointFrictionPotential"}}},
ShowCellTags->True,
StyleDefinitions -> "Textbook.nb"
]


(***********************************************************************
Cached data follows.  If you edit this Notebook file directly, not using
Mathematica, you must remove the line containing CacheID at the top of 
the file.  The cache data will then be recreated when you save this file 
from within Mathematica.
***********************************************************************)

(*CellTagsOutline
CellTagsIndex->{
  "Introduction"->{
    Cell[1973, 63, 59, 1, 99, "Section",
      CellTags->"Introduction"],
    Cell[2035, 66, 958, 21, 306, "Text",
      CellTags->{"Introduction", "i:1"}],
    Cell[3018, 91, 150, 5, 35, "Input",
      CellTags->"Introduction"],
    Cell[3171, 98, 632, 9, 151, "Print",
      CellTags->"Introduction"],
    Cell[3840, 112, 93, 3, 35, "Input",
      CellTags->"Introduction"],
    Cell[3936, 117, 101, 2, 29, "Print",
      CellTags->"Introduction"],
    Cell[4040, 121, 94, 3, 29, "Output",
      CellTags->"Introduction"],
    Cell[4171, 129, 116, 3, 35, "Input",
      CellTags->{"Introduction", "i:2"}],
    Cell[4290, 134, 381, 6, 83, "Print",
      CellTags->{"Introduction", "i:2"}],
    Cell[4708, 145, 121, 3, 35, "Input",
      CellTags->{"Introduction", "i:4"}],
    Cell[4832, 150, 957, 14, 327, "Print",
      CellTags->{"Introduction", "i:4"}]},
  "i:1"->{
    Cell[2035, 66, 958, 21, 306, "Text",
      CellTags->{"Introduction", "i:1"}]},
  "i:2"->{
    Cell[4171, 129, 116, 3, 35, "Input",
      CellTags->{"Introduction", "i:2"}],
    Cell[4290, 134, 381, 6, 83, "Print",
      CellTags->{"Introduction", "i:2"}]},
  "i:4"->{
    Cell[4708, 145, 121, 3, 35, "Input",
      CellTags->{"Introduction", "i:4"}],
    Cell[4832, 150, 957, 14, 327, "Print",
      CellTags->{"Introduction", "i:4"}]},
  "Backlash"->{
    Cell[5838, 170, 60, 1, 119, "Section",
      CellTags->{"Backlash", "i:3"}],
    Cell[5923, 175, 88, 1, 54, "Subsection",
      CellTags->{"Backlash", "i:3"}],
    Cell[6036, 180, 450, 10, 144, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[6489, 192, 96, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[6588, 196, 96, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[6721, 203, 499, 9, 144, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[7223, 214, 249, 5, 57, "Output",
      CellTags->{"Backlash", "i:3"}],
    Cell[7509, 224, 445, 11, 144, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[7957, 237, 175, 4, 51, "Output",
      CellTags->{"Backlash", "i:3"}],
    Cell[8135, 243, 97, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[8235, 247, 92, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[8330, 251, 95, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[8428, 255, 744, 12, 152, "Output",
      CellTags->{"Backlash", "i:3"}],
    Cell[9209, 272, 145, 3, 44, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[9357, 277, 94, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[9454, 281, 96, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[9553, 285, 96, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[9652, 289, 97, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[9752, 293, 92, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[9847, 297, 95, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[9945, 301, 745, 12, 152, "Output",
      CellTags->{"Backlash", "i:3"}],
    Cell[10739, 319, 108, 1, 54, "Subsection",
      CellTags->{"Backlash", "i:3"}],
    Cell[10850, 322, 400, 9, 124, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[11275, 335, 127, 3, 44, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[11405, 340, 96, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[11504, 344, 96, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[11615, 349, 147, 4, 39, "Text",
      Evaluatable->False,
      CellTags->{"Backlash", "i:3"}],
    Cell[11765, 355, 524, 12, 124, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[12292, 369, 163, 4, 39, "Text",
      Evaluatable->False,
      CellTags->{"Backlash", "i:3"}],
    Cell[12458, 375, 173, 5, 44, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[12634, 382, 302, 8, 81, "Text",
      Evaluatable->False,
      CellTags->{"Backlash", "i:3"}],
    Cell[12939, 392, 169, 5, 44, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[13133, 401, 232, 6, 44, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[13368, 409, 94, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[13465, 413, 96, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[13564, 417, 96, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[13663, 421, 97, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[13763, 425, 92, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[13858, 429, 95, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[13990, 436, 269, 6, 104, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[14262, 444, 168, 4, 51, "Output",
      CellTags->{"Backlash", "i:3"}],
    Cell[14467, 453, 212, 5, 64, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[14682, 460, 97, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[14782, 464, 92, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[14877, 468, 95, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[15009, 475, 93, 3, 44, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[15105, 480, 352, 7, 76, "Output",
      CellTags->{"Backlash", "i:3"}],
    Cell[15506, 493, 167, 4, 54, "Subsection",
      Evaluatable->False,
      CellTags->{"Backlash", "i:3"}],
    Cell[15676, 499, 350, 8, 56, "Text",
      Evaluatable->False,
      CellTags->{"Backlash", "i:3"}],
    Cell[16051, 511, 183, 5, 44, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[16237, 518, 2668, 59, 306, "Output",
      CellTags->{"Backlash", "i:3"}],
    Cell[18920, 580, 211, 6, 39, "Text",
      Evaluatable->False,
      CellTags->{"Backlash", "i:3"}],
    Cell[19156, 590, 306, 7, 44, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[19465, 599, 484, 10, 38, "Output",
      CellTags->{"Backlash", "i:3"}],
    Cell[19986, 614, 182, 5, 44, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[20171, 621, 2263, 53, 253, "Output",
      CellTags->{"Backlash", "i:3"}],
    Cell[22471, 679, 157, 5, 44, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[22631, 686, 112, 3, 38, "Output",
      CellTags->{"Backlash", "i:3"}],
    Cell[22780, 694, 333, 10, 44, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[23116, 706, 144, 3, 38, "Output",
      CellTags->{"Backlash", "i:3"}],
    Cell[23297, 714, 225, 6, 44, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[23525, 722, 442, 12, 38, "Output",
      CellTags->{"Backlash", "i:3"}],
    Cell[24004, 739, 188, 5, 44, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[24195, 746, 2589, 61, 291, "Output",
      CellTags->{"Backlash", "i:3"}],
    Cell[26799, 810, 234, 7, 39, "Text",
      Evaluatable->False,
      CellTags->{"Backlash", "i:3"}],
    Cell[27058, 821, 212, 6, 44, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[27273, 829, 991, 29, 95, "Output",
      CellTags->{"Backlash", "i:3"}],
    Cell[28301, 863, 182, 5, 44, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[28486, 870, 28328, 783, 200, 7679, 522, "GraphicsData", 
    "PostScript", "Graphics",
      CellTags->{"Backlash", "i:3"}],
    Cell[56817, 1655, 188, 5, 38, "Output",
      CellTags->{"Backlash", "i:3"}],
    Cell[57042, 1665, 182, 5, 44, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[57227, 1672, 30604, 858, 200, 8457, 579, "GraphicsData", 
    "PostScript", "Graphics",
      CellTags->{"Backlash", "i:3"}],
    Cell[87834, 2532, 188, 5, 38, "Output",
      CellTags->{"Backlash", "i:3"}]},
  "i:3"->{
    Cell[5838, 170, 60, 1, 119, "Section",
      CellTags->{"Backlash", "i:3"}],
    Cell[5923, 175, 88, 1, 54, "Subsection",
      CellTags->{"Backlash", "i:3"}],
    Cell[6036, 180, 450, 10, 144, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[6489, 192, 96, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[6588, 196, 96, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[6721, 203, 499, 9, 144, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[7223, 214, 249, 5, 57, "Output",
      CellTags->{"Backlash", "i:3"}],
    Cell[7509, 224, 445, 11, 144, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[7957, 237, 175, 4, 51, "Output",
      CellTags->{"Backlash", "i:3"}],
    Cell[8135, 243, 97, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[8235, 247, 92, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[8330, 251, 95, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[8428, 255, 744, 12, 152, "Output",
      CellTags->{"Backlash", "i:3"}],
    Cell[9209, 272, 145, 3, 44, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[9357, 277, 94, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[9454, 281, 96, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[9553, 285, 96, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[9652, 289, 97, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[9752, 293, 92, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[9847, 297, 95, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[9945, 301, 745, 12, 152, "Output",
      CellTags->{"Backlash", "i:3"}],
    Cell[10739, 319, 108, 1, 54, "Subsection",
      CellTags->{"Backlash", "i:3"}],
    Cell[10850, 322, 400, 9, 124, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[11275, 335, 127, 3, 44, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[11405, 340, 96, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[11504, 344, 96, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[11615, 349, 147, 4, 39, "Text",
      Evaluatable->False,
      CellTags->{"Backlash", "i:3"}],
    Cell[11765, 355, 524, 12, 124, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[12292, 369, 163, 4, 39, "Text",
      Evaluatable->False,
      CellTags->{"Backlash", "i:3"}],
    Cell[12458, 375, 173, 5, 44, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[12634, 382, 302, 8, 81, "Text",
      Evaluatable->False,
      CellTags->{"Backlash", "i:3"}],
    Cell[12939, 392, 169, 5, 44, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[13133, 401, 232, 6, 44, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[13368, 409, 94, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[13465, 413, 96, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[13564, 417, 96, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[13663, 421, 97, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[13763, 425, 92, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[13858, 429, 95, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[13990, 436, 269, 6, 104, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[14262, 444, 168, 4, 51, "Output",
      CellTags->{"Backlash", "i:3"}],
    Cell[14467, 453, 212, 5, 64, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[14682, 460, 97, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[14782, 464, 92, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[14877, 468, 95, 2, 42, "Print",
      CellTags->{"Backlash", "i:3"}],
    Cell[15009, 475, 93, 3, 44, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[15105, 480, 352, 7, 76, "Output",
      CellTags->{"Backlash", "i:3"}],
    Cell[15506, 493, 167, 4, 54, "Subsection",
      Evaluatable->False,
      CellTags->{"Backlash", "i:3"}],
    Cell[15676, 499, 350, 8, 56, "Text",
      Evaluatable->False,
      CellTags->{"Backlash", "i:3"}],
    Cell[16051, 511, 183, 5, 44, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[16237, 518, 2668, 59, 306, "Output",
      CellTags->{"Backlash", "i:3"}],
    Cell[18920, 580, 211, 6, 39, "Text",
      Evaluatable->False,
      CellTags->{"Backlash", "i:3"}],
    Cell[19156, 590, 306, 7, 44, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[19465, 599, 484, 10, 38, "Output",
      CellTags->{"Backlash", "i:3"}],
    Cell[19986, 614, 182, 5, 44, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[20171, 621, 2263, 53, 253, "Output",
      CellTags->{"Backlash", "i:3"}],
    Cell[22471, 679, 157, 5, 44, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[22631, 686, 112, 3, 38, "Output",
      CellTags->{"Backlash", "i:3"}],
    Cell[22780, 694, 333, 10, 44, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[23116, 706, 144, 3, 38, "Output",
      CellTags->{"Backlash", "i:3"}],
    Cell[23297, 714, 225, 6, 44, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[23525, 722, 442, 12, 38, "Output",
      CellTags->{"Backlash", "i:3"}],
    Cell[24004, 739, 188, 5, 44, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[24195, 746, 2589, 61, 291, "Output",
      CellTags->{"Backlash", "i:3"}],
    Cell[26799, 810, 234, 7, 39, "Text",
      Evaluatable->False,
      CellTags->{"Backlash", "i:3"}],
    Cell[27058, 821, 212, 6, 44, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[27273, 829, 991, 29, 95, "Output",
      CellTags->{"Backlash", "i:3"}],
    Cell[28301, 863, 182, 5, 44, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[28486, 870, 28328, 783, 200, 7679, 522, "GraphicsData", 
    "PostScript", "Graphics",
      CellTags->{"Backlash", "i:3"}],
    Cell[56817, 1655, 188, 5, 38, "Output",
      CellTags->{"Backlash", "i:3"}],
    Cell[57042, 1665, 182, 5, 44, "Input",
      CellTags->{"Backlash", "i:3"}],
    Cell[57227, 1672, 30604, 858, 200, 8457, 579, "GraphicsData", 
    "PostScript", "Graphics",
      CellTags->{"Backlash", "i:3"}],
    Cell[87834, 2532, 188, 5, 38, "Output",
      CellTags->{"Backlash", "i:3"}]},
  "Friction"->{
    Cell[88083, 2544, 60, 1, 57, "Section",
      CellTags->{"Friction", "i:5"}],
    Cell[88168, 2549, 90, 1, 45, "Subsection",
      CellTags->{"Friction", "i:5"}],
    Cell[88283, 2554, 451, 10, 108, "Input",
      CellTags->{"Friction", "i:5"}],
    Cell[88737, 2566, 96, 2, 29, "Print",
      CellTags->{"Friction", "i:5"}],
    Cell[88836, 2570, 96, 2, 29, "Print",
      CellTags->{"Friction", "i:5"}],
    Cell[88947, 2575, 507, 9, 108, "Input",
      CellTags->{"Friction", "i:5"}],
    Cell[89479, 2588, 583, 13, 138, "Input",
      CellTags->{"Friction", "i:5"}],
    Cell[90065, 2603, 166, 4, 37, "Output",
      CellTags->{"Friction", "i:5"}],
    Cell[90234, 2609, 203, 5, 39, "Output",
      CellTags->{"Friction", "i:5"}],
    Cell[90440, 2616, 97, 2, 29, "Print",
      CellTags->{"Friction", "i:5"}],
    Cell[90540, 2620, 92, 2, 29, "Print",
      CellTags->{"Friction", "i:5"}],
    Cell[90635, 2624, 95, 2, 29, "Print",
      CellTags->{"Friction", "i:5"}],
    Cell[90767, 2631, 93, 3, 35, "Input",
      CellTags->{"Friction", "i:5"}],
    Cell[90863, 2636, 436, 9, 63, "Output",
      CellTags->{"Friction", "i:5"}],
    Cell[91336, 2650, 108, 3, 35, "Input",
      CellTags->{"Friction", "i:5"}],
    Cell[91447, 2655, 423, 8, 80, "Output",
      CellTags->{"Friction", "i:5"}]},
  "i:5"->{
    Cell[88083, 2544, 60, 1, 57, "Section",
      CellTags->{"Friction", "i:5"}],
    Cell[88168, 2549, 90, 1, 45, "Subsection",
      CellTags->{"Friction", "i:5"}],
    Cell[88283, 2554, 451, 10, 108, "Input",
      CellTags->{"Friction", "i:5"}],
    Cell[88737, 2566, 96, 2, 29, "Print",
      CellTags->{"Friction", "i:5"}],
    Cell[88836, 2570, 96, 2, 29, "Print",
      CellTags->{"Friction", "i:5"}],
    Cell[88947, 2575, 507, 9, 108, "Input",
      CellTags->{"Friction", "i:5"}],
    Cell[89479, 2588, 583, 13, 138, "Input",
      CellTags->{"Friction", "i:5"}],
    Cell[90065, 2603, 166, 4, 37, "Output",
      CellTags->{"Friction", "i:5"}],
    Cell[90234, 2609, 203, 5, 39, "Output",
      CellTags->{"Friction", "i:5"}],
    Cell[90440, 2616, 97, 2, 29, "Print",
      CellTags->{"Friction", "i:5"}],
    Cell[90540, 2620, 92, 2, 29, "Print",
      CellTags->{"Friction", "i:5"}],
    Cell[90635, 2624, 95, 2, 29, "Print",
      CellTags->{"Friction", "i:5"}],
    Cell[90767, 2631, 93, 3, 35, "Input",
      CellTags->{"Friction", "i:5"}],
    Cell[90863, 2636, 436, 9, 63, "Output",
      CellTags->{"Friction", "i:5"}],
    Cell[91336, 2650, 108, 3, 35, "Input",
      CellTags->{"Friction", "i:5"}],
    Cell[91447, 2655, 423, 8, 80, "Output",
      CellTags->{"Friction", "i:5"}]}
  }
*)

(*CellTagsIndex
CellTagsIndex->{
  {"Introduction", 92884, 2693},
  {"i:1", 93755, 2716},
  {"i:2", 93850, 2719},
  {"i:4", 94028, 2724},
  {"Backlash", 94213, 2729},
  {"i:3", 100367, 2885},
  {"Friction", 106526, 3041},
  {"i:5", 107842, 3074}
  }
*)

(*NotebookFileOutline
Notebook[{

Cell[CellGroupData[{
Cell[1739, 51, 48, 0, 59, "Title"],
Cell[1790, 53, 35, 0, 33, "Subtitle"],
Cell[1828, 55, 120, 4, 68, "Subsubtitle"],

Cell[CellGroupData[{
Cell[1973, 63, 59, 1, 99, "Section",
  CellTags->"Introduction"],
Cell[2035, 66, 958, 21, 306, "Text",
  CellTags->{"Introduction", "i:1"}],

Cell[CellGroupData[{
Cell[3018, 91, 150, 5, 35, "Input",
  CellTags->"Introduction"],
Cell[3171, 98, 632, 9, 151, "Print",
  CellTags->"Introduction"]
}, Open  ]],

Cell[CellGroupData[{
Cell[3840, 112, 93, 3, 35, "Input",
  CellTags->"Introduction"],
Cell[3936, 117, 101, 2, 29, "Print",
  CellTags->"Introduction"],
Cell[4040, 121, 94, 3, 29, "Output",
  CellTags->"Introduction"]
}, Open  ]],

Cell[CellGroupData[{
Cell[4171, 129, 116, 3, 35, "Input",
  CellTags->{"Introduction", "i:2"}],
Cell[4290, 134, 381, 6, 83, "Print",
  CellTags->{"Introduction", "i:2"}]
}, Open  ]],

Cell[CellGroupData[{
Cell[4708, 145, 121, 3, 35, "Input",
  CellTags->{"Introduction", "i:4"}],
Cell[4832, 150, 957, 14, 327, "Print",
  CellTags->{"Introduction", "i:4"}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[5838, 170, 60, 1, 119, "Section",
  CellTags->{"Backlash", "i:3"}],

Cell[CellGroupData[{
Cell[5923, 175, 88, 1, 54, "Subsection",
  CellTags->{"Backlash", "i:3"}],

Cell[CellGroupData[{
Cell[6036, 180, 450, 10, 144, "Input",
  CellTags->{"Backlash", "i:3"}],
Cell[6489, 192, 96, 2, 42, "Print",
  CellTags->{"Backlash", "i:3"}],
Cell[6588, 196, 96, 2, 42, "Print",
  CellTags->{"Backlash", "i:3"}]
}, Open  ]],

Cell[CellGroupData[{
Cell[6721, 203, 499, 9, 144, "Input",
  CellTags->{"Backlash", "i:3"}],
Cell[7223, 214, 249, 5, 57, "Output",
  CellTags->{"Backlash", "i:3"}]
}, Open  ]],

Cell[CellGroupData[{
Cell[7509, 224, 445, 11, 144, "Input",
  CellTags->{"Backlash", "i:3"}],
Cell[7957, 237, 175, 4, 51, "Output",
  CellTags->{"Backlash", "i:3"}],
Cell[8135, 243, 97, 2, 42, "Print",
  CellTags->{"Backlash", "i:3"}],
Cell[8235, 247, 92, 2, 42, "Print",
  CellTags->{"Backlash", "i:3"}],
Cell[8330, 251, 95, 2, 42, "Print",
  CellTags->{"Backlash", "i:3"}],
Cell[8428, 255, 744, 12, 152, "Output",
  CellTags->{"Backlash", "i:3"}]
}, Open  ]],

Cell[CellGroupData[{
Cell[9209, 272, 145, 3, 44, "Input",
  CellTags->{"Backlash", "i:3"}],
Cell[9357, 277, 94, 2, 42, "Print",
  CellTags->{"Backlash", "i:3"}],
Cell[9454, 281, 96, 2, 42, "Print",
  CellTags->{"Backlash", "i:3"}],
Cell[9553, 285, 96, 2, 42, "Print",
  CellTags->{"Backlash", "i:3"}],
Cell[9652, 289, 97, 2, 42, "Print",
  CellTags->{"Backlash", "i:3"}],
Cell[9752, 293, 92, 2, 42, "Print",
  CellTags->{"Backlash", "i:3"}],
Cell[9847, 297, 95, 2, 42, "Print",
  CellTags->{"Backlash", "i:3"}],
Cell[9945, 301, 745, 12, 152, "Output",
  CellTags->{"Backlash", "i:3"}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[10739, 319, 108, 1, 54, "Subsection",
  CellTags->{"Backlash", "i:3"}],
Cell[10850, 322, 400, 9, 124, "Input",
  CellTags->{"Backlash", "i:3"}],

Cell[CellGroupData[{
Cell[11275, 335, 127, 3, 44, "Input",
  CellTags->{"Backlash", "i:3"}],
Cell[11405, 340, 96, 2, 42, "Print",
  CellTags->{"Backlash", "i:3"}],
Cell[11504, 344, 96, 2, 42, "Print",
  CellTags->{"Backlash", "i:3"}]
}, Open  ]],
Cell[11615, 349, 147, 4, 39, "Text",
  Evaluatable->False,
  CellTags->{"Backlash", "i:3"}],
Cell[11765, 355, 524, 12, 124, "Input",
  CellTags->{"Backlash", "i:3"}],
Cell[12292, 369, 163, 4, 39, "Text",
  Evaluatable->False,
  CellTags->{"Backlash", "i:3"}],
Cell[12458, 375, 173, 5, 44, "Input",
  CellTags->{"Backlash", "i:3"}],
Cell[12634, 382, 302, 8, 81, "Text",
  Evaluatable->False,
  CellTags->{"Backlash", "i:3"}],
Cell[12939, 392, 169, 5, 44, "Input",
  CellTags->{"Backlash", "i:3"}],

Cell[CellGroupData[{
Cell[13133, 401, 232, 6, 44, "Input",
  CellTags->{"Backlash", "i:3"}],
Cell[13368, 409, 94, 2, 42, "Print",
  CellTags->{"Backlash", "i:3"}],
Cell[13465, 413, 96, 2, 42, "Print",
  CellTags->{"Backlash", "i:3"}],
Cell[13564, 417, 96, 2, 42, "Print",
  CellTags->{"Backlash", "i:3"}],
Cell[13663, 421, 97, 2, 42, "Print",
  CellTags->{"Backlash", "i:3"}],
Cell[13763, 425, 92, 2, 42, "Print",
  CellTags->{"Backlash", "i:3"}],
Cell[13858, 429, 95, 2, 42, "Print",
  CellTags->{"Backlash", "i:3"}]
}, Open  ]],

Cell[CellGroupData[{
Cell[13990, 436, 269, 6, 104, "Input",
  CellTags->{"Backlash", "i:3"}],
Cell[14262, 444, 168, 4, 51, "Output",
  CellTags->{"Backlash", "i:3"}]
}, Open  ]],

Cell[CellGroupData[{
Cell[14467, 453, 212, 5, 64, "Input",
  CellTags->{"Backlash", "i:3"}],
Cell[14682, 460, 97, 2, 42, "Print",
  CellTags->{"Backlash", "i:3"}],
Cell[14782, 464, 92, 2, 42, "Print",
  CellTags->{"Backlash", "i:3"}],
Cell[14877, 468, 95, 2, 42, "Print",
  CellTags->{"Backlash", "i:3"}]
}, Open  ]],

Cell[CellGroupData[{
Cell[15009, 475, 93, 3, 44, "Input",
  CellTags->{"Backlash", "i:3"}],
Cell[15105, 480, 352, 7, 76, "Output",
  CellTags->{"Backlash", "i:3"}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[15506, 493, 167, 4, 54, "Subsection",
  Evaluatable->False,
  CellTags->{"Backlash", "i:3"}],
Cell[15676, 499, 350, 8, 56, "Text",
  Evaluatable->False,
  CellTags->{"Backlash", "i:3"}],

Cell[CellGroupData[{
Cell[16051, 511, 183, 5, 44, "Input",
  CellTags->{"Backlash", "i:3"}],
Cell[16237, 518, 2668, 59, 306, "Output",
  CellTags->{"Backlash", "i:3"}]
}, Open  ]],
Cell[18920, 580, 211, 6, 39, "Text",
  Evaluatable->False,
  CellTags->{"Backlash", "i:3"}],

Cell[CellGroupData[{
Cell[19156, 590, 306, 7, 44, "Input",
  CellTags->{"Backlash", "i:3"}],
Cell[19465, 599, 484, 10, 38, "Output",
  CellTags->{"Backlash", "i:3"}]
}, Open  ]],

Cell[CellGroupData[{
Cell[19986, 614, 182, 5, 44, "Input",
  CellTags->{"Backlash", "i:3"}],
Cell[20171, 621, 2263, 53, 253, "Output",
  CellTags->{"Backlash", "i:3"}]
}, Open  ]],

Cell[CellGroupData[{
Cell[22471, 679, 157, 5, 44, "Input",
  CellTags->{"Backlash", "i:3"}],
Cell[22631, 686, 112, 3, 38, "Output",
  CellTags->{"Backlash", "i:3"}]
}, Open  ]],

Cell[CellGroupData[{
Cell[22780, 694, 333, 10, 44, "Input",
  CellTags->{"Backlash", "i:3"}],
Cell[23116, 706, 144, 3, 38, "Output",
  CellTags->{"Backlash", "i:3"}]
}, Open  ]],

Cell[CellGroupData[{
Cell[23297, 714, 225, 6, 44, "Input",
  CellTags->{"Backlash", "i:3"}],
Cell[23525, 722, 442, 12, 38, "Output",
  CellTags->{"Backlash", "i:3"}]
}, Open  ]],

Cell[CellGroupData[{
Cell[24004, 739, 188, 5, 44, "Input",
  CellTags->{"Backlash", "i:3"}],
Cell[24195, 746, 2589, 61, 291, "Output",
  CellTags->{"Backlash", "i:3"}]
}, Open  ]],
Cell[26799, 810, 234, 7, 39, "Text",
  Evaluatable->False,
  CellTags->{"Backlash", "i:3"}],

Cell[CellGroupData[{
Cell[27058, 821, 212, 6, 44, "Input",
  CellTags->{"Backlash", "i:3"}],
Cell[27273, 829, 991, 29, 95, "Output",
  CellTags->{"Backlash", "i:3"}]
}, Open  ]],

Cell[CellGroupData[{
Cell[28301, 863, 182, 5, 44, "Input",
  CellTags->{"Backlash", "i:3"}],
Cell[28486, 870, 28328, 783, 200, 7679, 522, "GraphicsData", "PostScript", \
"Graphics",
  CellTags->{"Backlash", "i:3"}],
Cell[56817, 1655, 188, 5, 38, "Output",
  CellTags->{"Backlash", "i:3"}]
}, Open  ]],

Cell[CellGroupData[{
Cell[57042, 1665, 182, 5, 44, "Input",
  CellTags->{"Backlash", "i:3"}],
Cell[57227, 1672, 30604, 858, 200, 8457, 579, "GraphicsData", "PostScript", \
"Graphics",
  CellTags->{"Backlash", "i:3"}],
Cell[87834, 2532, 188, 5, 38, "Output",
  CellTags->{"Backlash", "i:3"}]
}, Open  ]]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{
Cell[88083, 2544, 60, 1, 57, "Section",
  CellTags->{"Friction", "i:5"}],

Cell[CellGroupData[{
Cell[88168, 2549, 90, 1, 45, "Subsection",
  CellTags->{"Friction", "i:5"}],

Cell[CellGroupData[{
Cell[88283, 2554, 451, 10, 108, "Input",
  CellTags->{"Friction", "i:5"}],
Cell[88737, 2566, 96, 2, 29, "Print",
  CellTags->{"Friction", "i:5"}],
Cell[88836, 2570, 96, 2, 29, "Print",
  CellTags->{"Friction", "i:5"}]
}, Open  ]],
Cell[88947, 2575, 507, 9, 108, "Input",
  CellTags->{"Friction", "i:5"}],

Cell[CellGroupData[{
Cell[89479, 2588, 583, 13, 138, "Input",
  CellTags->{"Friction", "i:5"}],
Cell[90065, 2603, 166, 4, 37, "Output",
  CellTags->{"Friction", "i:5"}],
Cell[90234, 2609, 203, 5, 39, "Output",
  CellTags->{"Friction", "i:5"}],
Cell[90440, 2616, 97, 2, 29, "Print",
  CellTags->{"Friction", "i:5"}],
Cell[90540, 2620, 92, 2, 29, "Print",
  CellTags->{"Friction", "i:5"}],
Cell[90635, 2624, 95, 2, 29, "Print",
  CellTags->{"Friction", "i:5"}]
}, Open  ]],

Cell[CellGroupData[{
Cell[90767, 2631, 93, 3, 35, "Input",
  CellTags->{"Friction", "i:5"}],
Cell[90863, 2636, 436, 9, 63, "Output",
  CellTags->{"Friction", "i:5"}]
}, Open  ]],

Cell[CellGroupData[{
Cell[91336, 2650, 108, 3, 35, "Input",
  CellTags->{"Friction", "i:5"}],
Cell[91447, 2655, 423, 8, 80, "Output",
  CellTags->{"Friction", "i:5"}]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}
]
*)




(***********************************************************************
End of Mathematica Notebook file.
***********************************************************************)

