(***********************************************************************

                    Mathematica-Compatible Notebook

This notebook can be used on any computer system with Mathematica 4.0,
MathReader 4.0, or any compatible application. The data for the notebook 
starts with the line containing stars above.

To get the notebook into a Mathematica-compatible application, do one of 
the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing the 
word CacheID, otherwise Mathematica-compatible applications may try to 
use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
***********************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[     45970,        956]*)
(*NotebookOutlinePosition[     70952,       1828]*)
(*  CellTagsIndexPosition[     70626,       1813]*)
(*WindowFrame->Normal*)



Notebook[{

Cell[CellGroupData[{
Cell["\<\
Variable Structure Control of Partially Linearizable Dynamics\
\>", "Title"],

Cell["Harry G. Kwatny", "Subtitle"],

Cell["\<\
Department of Mechanical Engineering & Mechanics
Drexel University\
\>", "Subsubtitle"],

Cell[CellGroupData[{

Cell["Introduction", "Section",
  CellTags->"Introduction"],

Cell["\<\
ProPac contains functions that assist in the design of variable structure \
control systems. The constructions that are included apply to partially \
feedback linearizable nonlinear systems. In this notebook we will describe \
tools for design of the switching surfaces and assembly of the switching \
controller. Smoothing and moderating functions can be included for the \
purpose of eliminating or reducing chattering as appropriate. These options \
are also illustrated. With the inclusion of an observer and exogenous signal \
generator, these tools provide a nonlinear generalization of the variable \
structure servomechanism.
\tFirst we load the package:\
\>", "Text",
  TextJustification->1,
  CellTags->"Introduction"],

Cell[CellGroupData[{

Cell[BoxData[
    \(<< ProPac`\)], "Input",
  CellLabel->"In[1]:=",
  CellTags->"Introduction"],

Cell[BoxData[
    \("\n    This is a development version of ProPac - Version 1.xx\n \n \
ProPac has been successfully loaded. It contains packages\n for multibody \
dynamics, linear control and nonlinear control.\n Appropriate packages will \
be loaded automatically as required.\n Packages can be manually loaded by \
simply entering GeometricTools,\n Dynamics, LinearControl, NonlinearControl \
or MEXTools as desired.\n Once a package is loaded, enter ?GeomtricTools, \
?Dynamics,\n ?LinearControl, ?NonlinearControl, or ?MEXTools, respectively,\n \
to obtain a list of available functions. Help is also available\n in the Help \
Browser under Add-ons."\)], "Print"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["The Sliding Surface", "Section",
  CellTags->{"Sliding", "i:1"}],

Cell[TextData[{
  "There are several methods for determining the sliding surface once the \
system has been reduced to normal form. ProPac includes a function ",
  StyleBox["SlidingSurface",
    FontFamily->"Courier"],
  StyleBox[" that implements two alternatives depending on the arguments \
provided. The function may be called via",
    FontFamily->"Times New Roman"],
  "\n\t\t\t",
  StyleBox["{rho,s}=SlidingSurface[f,g,h,x,lam]",
    FontFamily->"Courier"],
  "\nor\n\t\t\t",
  StyleBox["s=SlidingSurface[rho,vro,z,lam]",
    FontFamily->"Courier"],
  "\nIn the first case the data provided is the nonlinear system definition \
f,g,h,x and an m-vector lam which contains a list of desired exponential \
decay rates, one for each channel. The function returns the decoupling matrix \
rho and the switching surfaces s as functions of the state x.\nThe second use \
of the function assumes that the input-output linearization has already been \
performed so that the decoupling matrix rho, the vector relative degree and \
the normal coordinate (partial) transformation z are known. In this case the \
dimension of each of the m switching surfaces is known so that it is possible \
to specify a complete set of eigenvalues for each surface. Thus, lam is a \
list of m-sublists containing the specified eigenvalues. Only the switching \
surfaces are returned."
}], "Text",
  CellTags->{"Sliding", "i:1"}],

Cell[CellGroupData[{

Cell["Example", "Subsection",
  CellTags->{"Sliding", "i:1"}],

Cell["\<\
As a first example, we take a simple linear problem. This will show in a \
transparent way how the basic functions are used. \
\>", "Text",
  CellTags->{"Sliding", "i:1"}],

Cell[BoxData[{
    \(\(f = {x2, \(-x1\)};\)\), "\n", 
    \(\(g = {0, 1};\)\), "\n", 
    \(\(h = {x1};\)\)}], "Input",
  CellLabel->"In[2]:=",
  CellTags->{"Sliding", "i:1"}],

Cell[CellGroupData[{

Cell[BoxData[
    \({rho, alpha, vro, control} = IOLinearize[f, g, h, {x1, x2}]\)], "Input",\

  CellLabel->"In[3]:=",
  CellTags->{"Sliding", "i:1"}],

Cell[BoxData[
    \("  *** LinearControl successfully loaded ***"\)], "Print",
  CellTags->"i:1"],

Cell[BoxData[
    \("  *** NonlinearControl successfully loaded ***"\)], "Print",
  CellTags->"i:1"],

Cell[BoxData[
    \("Computing Decoupling Matrix"\)], "Print",
  CellTags->"i:1"],

Cell[BoxData[
    \("Computing linearizing/decoupling control"\)], "Print",
  CellTags->"i:1"],

Cell[BoxData[
    \({{{1}}, {\(-x1\)}, {2}, {v1 + x1}}\)], "Output",
  CellLabel->"Out[3]=",
  CellTags->"i:1"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(?NormalCoordinates\)\)], "Input",
  CellLabel->"In[4]:=",
  CellTags->{"Sliding", "i:1"}],

Cell[BoxData[
    \("NormalCoordinates[f,g,h,x,Vro] returns the functions z(x) which\n \
define the linearizable states."\)], "Print",
  CellTags->"i:1"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(z = NormalCoordinates[f, g, h, {x1, x2}, vro]\)], "Input",
  CellLabel->"In[5]:=",
  CellTags->{"Sliding", "i:1"}],

Cell[BoxData[
    \({x1, x2}\)], "Output",
  CellLabel->"Out[5]=",
  CellTags->"i:1"]
}, Open  ]],

Cell["\<\
Now, this data can be used to design a switching surface that locates the \
sliding poles at {-2,-3}.\
\>", "Text",
  CellTags->{"Sliding", "i:1"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(s = SlidingSurface[rho, vro, z, {{\(-2\)}}]\)], "Input",
  CellLabel->"In[6]:=",
  CellTags->{"Sliding", "i:1"}],

Cell[BoxData[
    \({2\ x1 + x2}\)], "Output",
  CellLabel->"Out[6]=",
  CellTags->"i:1"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(SwitchingControl[rho, s, {{\(-1\), 1}}, {{1}}]\)], "Input",
  CellLabel->"In[7]:=",
  CellTags->"i:1"],

Cell[BoxData[
    \({Sign[\(-2\)\ x1 - x2]}\)], "Output",
  CellLabel->"Out[7]=",
  CellTags->"i:1"]
}, Open  ]],

Cell["\<\
Or, we can determine a switching surface from scratch with a specified decay \
rate of 2.\
\>", "Text",
  CellTags->{"Sliding", "i:1"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(SlidingSurface[f, g, h, {x1, x2}, {2}]\)], "Input",
  CellLabel->"In[8]:=",
  CellTags->{"Sliding", "i:1"}],

Cell[BoxData[
    \("Computing Decoupling Matrix"\)], "Print",
  CellTags->"i:1"],

Cell[BoxData[
    \("Computing linearizing/decoupling control"\)], "Print",
  CellTags->"i:1"],

Cell[BoxData[
    \({{{1}}, {x1\/\(\(-2\) + \@5\) + x2}}\)], "Output",
  CellLabel->"Out[8]=",
  CellTags->"i:1"]
}, Open  ]]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell["Reaching", "Section",
  CellTags->{"Reaching", "i:2"}],

Cell[TextData[{
  "Once the sliding surface is defined it is possible to construct the \
switching controller. This is accompished with the function ",
  StyleBox["SwitchingControl",
    FontFamily->"Courier"],
  ". "
}], "Text",
  CellTags->{"Reaching", "i:2"}],

Cell[CellGroupData[{

Cell["Example", "Subsection",
  CellTags->{"Reaching", "i:2"}],

Cell[BoxData[{
    \(\(f = {x2, \(-x1\)};\)\), "\n", 
    \(\(g = {0, 1};\)\), "\n", 
    \(\(h = {x1};\)\)}], "Input",
  CellLabel->"In[9]:=",
  CellTags->{"Reaching", "i:2"}],

Cell[CellGroupData[{

Cell[BoxData[
    \({rho, s} = SlidingSurface[f, g, h, {x1, x2}, {2}]\)], "Input",
  CellLabel->"In[10]:=",
  CellTags->{"Reaching", "i:2"}],

Cell[BoxData[
    \("Computing Decoupling Matrix"\)], "Print",
  CellTags->"i:2"],

Cell[BoxData[
    \("Computing linearizing/decoupling control"\)], "Print",
  CellTags->"i:2"],

Cell[BoxData[
    \({{{1}}, {x1\/\(\(-2\) + \@5\) + x2}}\)], "Output",
  CellLabel->"Out[10]=",
  CellTags->"i:2"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(vsc = SwitchingControl[rho, s, {{\(-1\), 1}}, {{1}}]\)], "Input",
  CellLabel->"In[11]:=",
  CellTags->{"Reaching", "i:2"}],

Cell[BoxData[
    \({Sign[x1\/\(2 - \@5\) - x2]}\)], "Output",
  CellLabel->"Out[11]=",
  CellTags->"i:2"]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Smooth Approximations", "Section",
  CellTags->{"Smoothing", "i:3"}],

Cell[TextData[{
  "Smoothing of the switching mechanism is a commonly used method for \
reducing chattering and improving the efficency of numerical simulation. \
However, some caution is required because there are exceptional cases where \
smoothing can cause instability. The function ",
  StyleBox["SwitchingControl",
    FontFamily->"Courier"],
  " has an option which allows the designer to introduce user defined \
smoothing functions."
}], "Text",
  CellTags->{"Smoothing", "i:3"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(?SmoothingFunctions\)\)], "Input",
  CellLabel->"In[12]:=",
  CellTags->{"Smoothing", "i:3"}],

Cell[BoxData[
    \("SmoothingFunctions is an option of SwitchingControl[].\n UnitStep \
functions are multiplied by 'smoothing' functions defined in the statement\n \
SmoothingFunctions[x_]->{function1[x],...,functionm[x]}, where m is the \
number\n of control variables. Typical smoothing functions are:\n \n \
Sat[x/eps], (1-Exp[-x/eps]), x/(eps+Abs[x]), Tanh[x/eps] \n \n In each case \
eps is a 'small' positive constant. "\)], "Print",
  CellTags->"i:3"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(vsc = 
      SwitchingControl[rho, s, {{\(-1\), 1}}, {{1}}, 
        SmoothingFunctions[x_] -> {\((1 - Exp[\(-Abs[x/10]\)])\)}]\)], "Input",\

  CellLabel->"In[13]:=",
  CellTags->{"Smoothing", "i:3"}],

Cell[BoxData[
    \({Sign[\(-\(x1\/\(\(-2\) + \@5\)\)\) - x2] - 
        E\^\(\(-\(1\/10\)\)\ Abs[\(-\(x1\/\(\(-2\) + \@5\)\)\) - x2]\)\ Sign[\
\(-\(x1\/\(\(-2\) + \@5\)\)\) - x2]}\)], "Output",
  CellLabel->"Out[13]=",
  CellTags->"i:3"]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell["Moderating Functions", "Section",
  CellTags->{"Moderating", "i:4"}],

Cell["\<\
Moderating functions present another means for reducing chettering. The main \
idea is to reduce the control control bounds as the regulated variables \
approach their desired values. Roughly stated this reduces the effective loop \
gain so that a limit cycle cannot be sustained, thereby quenching any  \
chattering behavior as the equilibrium is approached.\
\>", "Text",
  CellTags->{"Moderating", "i:4"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(?ModeratingFunctions\)\)], "Input",
  CellLabel->"In[14]:=",
  CellTags->{"Moderating", "i:4"}],

Cell[BoxData[
    \("ModeratingFunctions is an option of SwitchingControl[].\n UnitStep \
functions are multiplied by 'moderating' functions defined in the statement\n \
ModeratingFunctions->{function1[z],...,functionm[z]}, where m is the number\n \
of control variables. Typical moderating functions are:\n \n \
(1-Exp[-norm[z]/cons]), norm[z]/(cons+norm[z]), Tanh[norm[z]/cons] \n \n \
where 'norm' is some appropriate norm in the normal coordinates z.\n In each \
case 'cons' is a positive constant. "\)], "Print",
  CellTags->"i:4"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(vsc = 
      SwitchingControl[rho, s, {{\(-1\), 1}}, {{1}}, 
        ModeratingFunctions -> {Abs[
                z[\([1]\)]]/\(( .1 + Abs[z[\([1]\)]])\)}]\)], "Input",
  CellLabel->"In[15]:=",
  CellTags->{"Moderating", "i:4"}],

Cell[BoxData[
    RowBox[{"{", 
      FractionBox[\(Abs[x1]\ Sign[\(-\(x1\/\(\(-2\) + \@5\)\)\) - x2]\), 
        RowBox[{
          RowBox[{
            StyleBox["0.1`",
              StyleBoxAutoDelete->True,
              PrintPrecision->1], "\[InvisibleSpace]"}], "+", \(Abs[x1]\)}]], 
      "}"}]], "Output",
  CellLabel->"Out[15]=",
  CellTags->"i:4"]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell["Bibliography", "Section",
  CellTags->"Bibliography"],

Cell[OLEData["\<\
d<lAh:6a6^4000000000000000000000?P030?ko2@0600000000000000010000
0@000000000040004P000040003noooo00000000003ooooooooooooooooooooo
oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
ooooooooooooooooooooooooooooooooooooooooooomoooo500000<000040000
1@0000H000070000200000T0003noooo2`0000`0000=00003P0000l0000@0000
4@000?koooonooooo_ooookooolF00005`0001P0000I00006P0001/0000L0000
7@000?kooolO000080000240000R00008`0002@0000U0000o_ooobL0000X0000
:@0002X0000[0000;00002d0003noooooooooooooooooooooooooooooooooooo
oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
oooooooooooooooooooooe80K`1_07@0801506h0M01b07T00000000000000000
0000000000000000000000000000000000000000000F00D0ooooooooool30000
1PT20000003000000000AP000000000000000602Jll9U[`14`00080000000000
A01Q07@0H@000000000000000000000000000000000000000000000000000000
0000000000000000000000X00P7oooooooooooooool000000000000000000000
0000000000000000000000000002000001000000000a05@0H@1R06`0I@000000
0000000000000000000000000000000000000000000000000000000000000000
3P020040003oooooooooo`000000000000000000000000000000000000000000
000000X00000400000000040@`1_06d0L01?0680JP0000000000000000000000
0000000000000000000000000000000000000000000B00810P0000@0003ooooo
000000000000000000000000000000000000000000000000000006X000000000
V00004@000000000000000000000000000000000000000000000000000000000
000000000000000000000000003U01TPZ03^00008`2S06P0EP0@0000200b0<l0
I@1^0680HP0003<0d`2a058100040:<0n002000000020=`0W@110400<0000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000001803`0:0040F`0?008000000000:000@?7o0P0X0000
1P1>06l0LP1]0640K0000080000806e82@A^B0T4000000000000000000000000
?011@?;oX@0l00005P1406D0IP1Q07D0K01d0200D01Q0780H@1W0780H@1`06P0
801606l0KP1d00000000000000000000003^0P001`0040000@3ooooo00@00>h6
0005000000@00>h60006000000@00>h600070000000000l0003/0P00kP8001=A
U?nES0X0O?oooe2FI9[o3ol?o`oo3ol?o`oo3ol?o`l107gooomleHPYo`oo3ol?
o`oo3ol?o`oo3ol?0@1noooo@<iPk?l?o`oo3ol?o`oo3ol?o`oo3`40Oooooc01
@3[o3ol?o`oo3ol?o`oo3ol?o`l1083ooomBh0iWo`oo3ol?o`oo3ol?o`oo3ol?
0@21ooooW=c^/_l?o`oo3ol?o`oo3ol?o`oo3`40P_ooo`j`TT_o3ol?o`oo3ol?
o`oo3ol?o`l108?ooolfKK93o`oo3ol?o`oo3ol?o`oo3ol?0@28ooooeV]<Eol?
o`oo3ol?o`oo3ol?o`oo3`40ROooodPn3<?o3ol?o`oo3ol?o`oo3ol?o`l10040
0000004000000000000000000000000@000?Q0P74HBHoQG61@0120L60P0002h0
0@0000000@0000000000000000000000010000n4X0DAQ9Sn5LH5006P1@H20000
;P010000000100000000000000000000000040003h@h1164V?hEaPD00CP41P80
000^00400000004000000000000000000000000@000?Q=024HBHoQG61@01d086
0P0002h00@0001L00000000000000000000000002a0000n420LAQ9Sn5LH50048
1`I?BP40DDX106lX0040]o0100005`0000000000000000000000000;40003hBP
1A64V?hEaPD00J051Tm:0@1ABP40KbP00@2gl040000G00000000000000000000
000000/@000?Q3P44HBHoQG61@01>0@6CdX1055:0@1_:0010;O`0@0001L00000
000000000000000000002a0000n4d08AQ9Sn5LH5007@0PI?BP40DDX106lX0040
]o010000000100000000000000000000000040003hAX0A64V?hEaPD00FP11P80
000^0040000G00000000000000000000000000/@000?Q6P14HBHoQG61@01J046
CdX1055:0@1_:0010;O`2P0008Woool00000000000000023oooo000000000000
0000P_ooo`00000000000000087oool00000000000000020oooo000000000000
0000R?ooo`0000000000000007ooool0000000000000001noooo000000000000
0000OOooo`0000000000000007coool0000000000000003ooooooooooooooooo
ooooooooooooooooooooooooooooooooooooool:000000000000000000000000
0000003oo`<0100E04D0CP1O04@0K`1S05l0AP1_06h0M01O04`0J@1c07@0G`1>
0640K@1U06@CKP4E04D0CP1O04`0J@1R05l0CP1Q06d0I@1O04`0J@1c07@0G`1>
0640K@1U03@7KP4G04D0CP1O04d0H@1Y06h0G`1206l0I01i05l0D`1d07T0K01U
05l0CP1Q06d0I@2P4fh16`1L02L0<00a05`09`0`0340G00W0300IP1D06T0K@1U
07<0801>06D0M`0P0580K`1]0640KP0>0340<P1306l0KP1d05l0K01Y0680;P1U
06h0K00?0440M@1d06P0K`1b02d0A01Q07@0I@0^06D0KP1c0?m00Q0000000000
kP80070000P0@0000`0004LFT04000821P<510D20`@300000000000000000000
0@000000001D06T0K@1U07<0801>06D0M`0P0580K`1]0640KP0003DFT04200D5
0@811`H21@L0000000004000000000000000P000001C07T0K@1R06l0K00003<V
T040008;1P@20P820P@3000000000000000000000@00000000110780J@1Q06`0
000R00@000286000d08006P100000<J[5bK6ZaLV00000080000006`0001Z0P00
0@010000100340D000000000000000400@00004000000000]080000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000018`0000000000000000003f0P00
0000000000000000000000000000000000000000000000000000000000000000
00000000000000000000008000000?oo4P000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000010000o_oooooooooooooo
oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
ooooooooooooooooooooo`40o_l32P00ooooo`H90P000000`000000004HH0000
CFUSLVmcKfId85M_LV@PA6mSMFeUKW@02P0004eCEfmbI4A_H`0@0000EfmbI2i4
Kf=eKFE^M2hh0?@i/W4000000000000000000000000000000000000000000000
0@000P0000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
E`1_0780I01406l0H`1e06d0I@1^07@000000000000000000000000000000000
0000000000000000000001X00P3ooooo1@000?ooool000000000000000000000
000000000000000000000000000E00008Q000000000505<0M@1]06d0H@1b07T0
B@1^06H0K`1b06d0H@1d06T0K`1^000000000000000000000000000000000000
:0020Oooool60000ooooo`000000000000000000000000000000000000000000
000001h000004000000000D0A01_06<0M@1]06D0KP1d05<0M@1]06d0H@1b07T0
B@1^06H0K`1b06d0H@1d06T0K`1^000000000000000h0080oooooooooooooooo
0000000000000000000000000000000000000000000000009P00000@00000000
0000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000003oooooooooooooool000000000000000000000
0000000000000000000000000000000000000000003/YL40A`09100021:o0000
0000400000001000kPH000h0HVYRJXkIS]T000000000000000000000000911H0
8Q000>bc0@3//`40kP80000000000000000000000000000000000000003oo`l0
00000000003oo`l000000000003oo`l00000000000000000000005d0000009H0
00000000UP0009H000000000UP000000002F0000000009H000000000UP0001@0
0000000000000:X000000000ZP000000002Z000000000:X000000000ZP0000`0
002f000030000:X000000000;0H00;H0003>00005P000>@000000000i0000000
003T000000000>@000000000i0000000003T000000000>@0000000005@H00080
000G1P00000001L6000000005`H00000000G1P00000001L6000000005`H00000
003R1P00m0400=H8000Z00005`H001D0000000000000000000000000UP000000
003T0000000000000000000000000000003T000000000>@000000000i0000000
003T0000000001L600000000i0000000002F0000000009H000000000i0000000
0000000000000>@000000000cP000000003T000000000>@000000000i0000000
003T0000000009H000000000i0000000002F000000000>@0000000001@D00101
000000000000000000000000ZP000000002Z0000000009H000000000UP000000
002F0000000009H000000000i0000000000E1P0000000>@000000000i0000000
003T00006P4001l4003<0000UP000000002F0000000000000000000000000000
000000000000000000000000000000000000000000000000000000001@D00000
003T000000000<80000<0000h4Q0c`VF_06Z000000000:X000000000i0000000
003[10006P000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
4b11A4A9CR15CT9YHR0154]gHGA^NB`PB2hPAbhPHFiT84X^849ULVLP:34i>CDY
;R1FHG9YHF9/IB1CM79eHgAeLVDPDVEWMFaQM6U_KR1_IR1@KgMULR1@K65^M214
LWE]84aUMVE/;R1CNG=dIFec865^I213KfidLVm/85AXIFmbNB1VKg8PD6mgIG8P
DgUcM6E]LbhPBRhP@fQ_Mb`PDRhPBRhPE6Q_KF5c865^I21@;R1F;R1;Kf]_M6mf
JF<^84iUMb1IKg9[;21CL79YKVMULYIFIG9/HFLj838`=Bdb<c@^3D]gHGA^NB`P
B2hPAbhPHFiT84P^84]YKB0X<CTi<2T^89=FHG9YHF9/IB1CM79eHgAeLVDPDVEW
MFaQM6U_KR1_IR1@HG9dJF5/K7TPC6U^IF5bJGYQHVaU84AiKV5]JF=c;Y@PDgUc
M6E]Lb0V84=_KWAbKf`PC6EdM6EbLb0a=CXP=SNF>30^3D]gHGA^NB`PB2hPAbhP
HFiT84/^84@^85U_MFiW82Pa>CPa:BhPTeAXIB1FHG9YHF9/IB1CM79eHgAeLVDP
LfEbMVm]IF=XHFiYLfd^U21CNG=dIFec865^I21SKfidLVm/84aUM7AULW<P<BPc
:CXP<CPd;C4i<Bh=EGA[JFh/85H^84T^82Pa>CLd820XJFhPDWEcLfUQKRTP<CTg
>20XJFhPAFiWK6UcJ2TY;R1CK6UTJFiW84e_I6Ec865^I21DJ6EYLR11L71/JF=Q
M6U_KRhPCFmcHfmg;21=BE8^3EU_MFiW;21;;R14;R1QKV@PB2hPAbhPBgMQM6ii
82Pa>CPb:BhPTeIQLVUQHVaU85=dLWESM7EbIB1CIG9fKfeUHfQQKVUcKB1QKV@P
JGAc845`L6aYHf5dJFm^87A_84mfIG9cL6EUI21@LVmdIF=dJFm^84=_KWAbKf`^
U211MGA_KF5dJF=Q834h:3@Y>R0c>3D]=30`;QD=000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
00@00044000>10003`@00104001[1000U`@00=d4003O1000C@D006H5001W1@00
J@D00<45003L1@00g@D00=h5000Q1P00A0H00=86003L1P00g@H00=l6003/1P00
k@H00>h6003j0?;j0?00kP3`0>h0l03^0?00l03^0?X000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
000000000000000000000000000000000000000000000000000000000003=@R1
0chZ0@l22843JP000006205E20490fX00000E@P101T01000j0@00785003/1@00
D`H00>h6003h0000000000000000n0000000000000000?P0000000000000003h
0000000000000000n00000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
00H0019TJ04101>TN0001@04003^1P00oP000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000020@41
800LD0407k3@;b2`h3dQ/0P78[081b>@X0DTT:059K0000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000003no`001002000000000000
0000000000010000h8FOl_U?J12[T@P0:bNcfC00001<0@004000004000280000
0P00090000030000W00000@0002X00001@000;@000070000`00000P0003D0000
2@000>00000B0000k00000X000080@00300001@1000=0000804000h0000/0@00
3`0003@1000@0000?04001<000140@000P000>@4000N00000@000000L`0N0000
0@000000L`0N00000@000000L`0N00000@000000L`0N00002`0004i_LVeQK2iT
Kg@001h00001000006mbKAh000020000<P1bKAh0000C0000CFUSLVmcKfId85M_
LV@P>2h`00100000000000000010000001a880VF_050000001a880VF_0430000
0@0000<0001/00000`0006X20003000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
o_l000@00P00000000000000000000000P0000;EcMFL;Q/@TiL802//nJi40000
1MG=eI`^6a2CU`P0:bci[R`1003X000030000040001X00003`00070000050000
O00000H0002400004@0008`0000G0000U00000/0002L000040000:@0000C0000
[00001H0002d00003@000;`0000<0000b@000080003T10007P000080000P0000
0`0000D0000300000@0000<0003f0P000`000;<=200;0000000000/000000000
2`000000000;0000000001h@000100000@00000<40000P0001h000060000E6Ud
K6D00`0000400000V00000<00000000080000040000f00000P0003h000010000
0P0000X0001OD4U4GdMEBD@00P000>@400110000CP0007/0<`1304H0@P0g0440
A@1502d0<00a04@0>@0]0340<@140340;@0i03H0A@1402d0>@0i0380=@0c03T0
<P120340=00b04@0O@0000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000000000000000000000000000000000000000000000
0000000000000000000000\>"], "Graphics",
  ImageSize->{397, 264.5},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  CellTags->"Bibliography"]
}, Closed]]
}, Open  ]]
},
FrontEndVersion->"4.0 for Microsoft Windows",
ScreenRectangle->{{0, 800}, {0, 527}},
WindowSize->{641, 471},
WindowMargins->{{0, Automatic}, {Automatic, 10}},
TaggingRules:>{"NextIndexingNumber" -> 5, "IndexEntries" -> {{"i:1", \
"IOLinearize"}, {"i:1", "NormalCoordinates"}, {"i:1", "SlidingSurface"}, \
{"i:1", "SwitchingControl"}, {"i:2", "SwitchingControl"}, {"i:3", \
"SmoothingFunctions"}, {"i:4", "ModeratingFunctions"}}},
ShowCellTags->True,
StyleDefinitions -> Notebook[{
  
  Cell[CellGroupData[{
  Cell["Style Definitions", "Subtitle"],
  
  Cell["\<\
Modify the definitions below to change the default appearance of all cells in \
a given style.  Make modifications to any definition using commands in the \
Format menu.\
\>", "Text"],
  
  Cell[CellGroupData[{
  
  Cell["Style Environment Names", "Subsection"],
  
  Cell[StyleData[All, "Working"],
    CellBracketOptions->{"Color"->RGBColor[0, 0.2, 1]},
    ScriptMinSize->9],
  
  Cell[StyleData[All, "Printout"],
    PageWidth->PaperWidth,
    PrivateFontOptions->{"FontType"->"Outline"}]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell["Notebook Options", "Subsection"],
  
  Cell["\<\
  The options defined for the style below will be used at the Notebook level. \
 \
\>", "Text"],
  
  Cell[StyleData["Notebook"],
    PageHeaders->{{Cell[ 
            TextData[ {
              CounterBox[ "Page"]}], "PageNumber"], None, Cell[ 
            TextData[ {
              OptionValueBox[ "FileName"]}], "Header"]}, {Cell[ 
            TextData[ {
              OptionValueBox[ "FileName"]}], "Header"], None, Cell[ 
            TextData[ {
              CounterBox[ "Page"]}], "PageNumber"]}},
    PrintingOptions->{"FirstPageHeader"->False,
    "FacingPages"->True},
    CellLabelAutoDelete->False,
    CellFrameLabelMargins->6,
    StyleMenuListing->None]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell["Headings", "Subsection"],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["Title"],
    CellMargins->{{12, Inherited}, {2, 20}},
    CellGroupingRules->{"TitleGrouping", 0},
    PageBreakBelow->False,
    CounterIncrements->"Title",
    CounterAssignments->{{"Section", 0}, {"Equation", 0}, {"Figure", 0}, {
        "Subtitle", 0}, {"Subsubtitle", 0}},
    FontFamily->"Helvetica",
    FontSize->30,
    FontWeight->"Bold"],
  
  Cell[StyleData["Title", "Printout"],
    CellMargins->{{2, 10}, {2, 30}},
    FontSize->24]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["Subtitle"],
    CellMargins->{{12, Inherited}, {2, 2}},
    CellGroupingRules->{"TitleGrouping", 10},
    PageBreakBelow->False,
    CounterIncrements->"Subtitle",
    CounterAssignments->{{"Section", 0}, {"Equation", 0}, {"Figure", 0}, {
        "Subsubtitle", 0}},
    FontFamily->"Helvetica",
    FontSize->24,
    FontColor->RGBColor[0, 0, 0.6]],
  
  Cell[StyleData["Subtitle", "Printout"],
    CellMargins->{{2, 10}, {2, 4}},
    FontSize->18,
    FontColor->GrayLevel[0]]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["Subsubtitle"],
    CellMargins->{{12, Inherited}, {2, 12}},
    CellGroupingRules->{"TitleGrouping", 20},
    PageBreakBelow->False,
    CounterIncrements->"Subsubtitle",
    CounterAssignments->{{"Section", 0}, {"Equation", 0}, {"Figure", 0}},
    FontFamily->"Helvetica",
    FontSize->14,
    FontSlant->"Italic",
    FontColor->RGBColor[0, 0, 0.6]],
  
  Cell[StyleData["Subsubtitle", "Printout"],
    CellMargins->{{2, 10}, {2, 8}},
    FontSize->12,
    FontColor->GrayLevel[0]]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["Section"],
    CellFrame->{{0, 0}, {2, 0}},
    CellMargins->{{10, 4}, {2, 50}},
    CellElementSpacings->{"ClosedGroupTopMargin"->18},
    CellGroupingRules->{"SectionGrouping", 30},
    PageBreakBelow->False,
    CounterIncrements->"Section",
    CounterAssignments->{{"Subsection", 0}, {"Subsubsection", 0}},
    FontFamily->"Helvetica",
    FontSize->18,
    FontWeight->"Bold"],
  
  Cell[StyleData["Section", "Printout"],
    CellMargins->{{2, 4}, {2, 80}},
    FontSize->14]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["Subsection"],
    CellDingbat->"\[FilledSquare]",
    CellMargins->{{24, 4}, {2, 18}},
    CellElementSpacings->{"ClosedGroupTopMargin"->12},
    CellGroupingRules->{"SectionGrouping", 40},
    PageBreakBelow->False,
    CellFrameLabelMargins->6,
    CounterIncrements->"Subsection",
    CounterAssignments->{{"Subsubsection", 0}},
    FontFamily->"Helvetica",
    FontSize->15,
    FontWeight->"Bold"],
  
  Cell[StyleData["Subsection", "Printout"],
    CellMargins->{{2, 4}, {2, 18}},
    FontSize->12]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["Subsubsection"],
    CellMargins->{{10, 4}, {2, 18}},
    CellElementSpacings->{"ClosedGroupTopMargin"->12},
    CellGroupingRules->{"SectionGrouping", 50},
    PageBreakBelow->False,
    CounterIncrements->"Subsubsection",
    FontFamily->"Helvetica",
    FontWeight->"Bold"],
  
  Cell[StyleData["Subsubsection", "Printout"],
    CellMargins->{{2, 4}, {2, 18}},
    FontSize->10]
  }, Closed]]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell["Text", "Subsection"],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["Text"],
    CellMargins->{{10, 4}, {0, 8}},
    TextJustification->1,
    ParagraphSpacing->{0, 8},
    CounterIncrements->"Text"],
  
  Cell[StyleData["Text", "Printout"],
    CellMargins->{{2, 4}, {0, 8}},
    FontSize->10]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["MathCaption"],
    CellFrame->{{4, 0}, {0, 0}},
    CellMargins->{{47, 62}, {0, 14}},
    CellFrameMargins->5,
    CellFrameColor->RGBColor[0, 0.2, 1],
    LineSpacing->{1, 1},
    ParagraphSpacing->{0, 8},
    FontColor->RGBColor[0, 0, 0.6]],
  
  Cell[StyleData["MathCaption", "Printout"],
    CellMargins->{{34, 62}, {0, 14}},
    CellFrameColor->GrayLevel[0.700008],
    FontSize->10,
    FontColor->GrayLevel[0]]
  }, Closed]]
  }, Open  ]],
  
  Cell[CellGroupData[{
  
  Cell["Input/Output", "Subsection"],
  
  Cell["\<\
The cells in this section define styles used for input and output to the \
kernel.  Be careful when modifying, renaming, or removing these styles, \
because the front end associates special meanings with these style names.    \
\
\>", "Text"],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["Input"],
    CellMargins->{{56, 4}, {3, 9}},
    Evaluatable->True,
    CellGroupingRules->"InputGrouping",
    PageBreakWithin->False,
    GroupPageBreakWithin->False,
    CellLabelAutoDelete->False,
    CellLabelMargins->{{21, Inherited}, {Inherited, Inherited}},
    DefaultFormatType->DefaultInputFormatType,
    AutoItalicWords->{},
    FormatType->StandardForm,
    ShowStringCharacters->True,
    NumberMarks->True,
    CounterIncrements->"Input",
    FontWeight->"Bold"],
  
  Cell[StyleData["Input", "Printout"],
    ShowCellBracket->False,
    CellMargins->{{42, 4}, {3, 8}},
    FontSize->10]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["Output"],
    CellMargins->{{57, 4}, {5, 2}},
    CellEditDuplicate->True,
    CellGroupingRules->"OutputGrouping",
    PageBreakWithin->False,
    GroupPageBreakWithin->False,
    GeneratedCell->True,
    CellAutoOverwrite->True,
    CellLabelAutoDelete->False,
    CellLabelMargins->{{21, Inherited}, {Inherited, Inherited}},
    DefaultFormatType->DefaultOutputFormatType,
    AutoItalicWords->{},
    FormatType->StandardForm,
    CounterIncrements->"Output"],
  
  Cell[StyleData["Output", "Printout"],
    ShowCellBracket->False,
    CellMargins->{{42, 4}, {4, 2}},
    FontSize->10]
  }, Open  ]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["Message"],
    CellMargins->{{56, 4}, {3, 8}},
    CellGroupingRules->"OutputGrouping",
    PageBreakWithin->False,
    GroupPageBreakWithin->False,
    GeneratedCell->True,
    CellAutoOverwrite->True,
    ShowCellLabel->False,
    CellLabelMargins->{{21, Inherited}, {Inherited, Inherited}},
    DefaultFormatType->DefaultOutputFormatType,
    AutoItalicWords->{},
    FormatType->InputForm,
    CounterIncrements->"Message",
    StyleMenuListing->None,
    FontColor->RGBColor[0, 0.2, 1]],
  
  Cell[StyleData["Message", "Printout"],
    ShowCellBracket->False,
    CellMargins->{{42, 4}, {4, 2}},
    FontSize->10,
    FontColor->GrayLevel[0]]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["Print"],
    CellMargins->{{56, 4}, {3, 8}},
    CellGroupingRules->"OutputGrouping",
    PageBreakWithin->False,
    GroupPageBreakWithin->False,
    GeneratedCell->True,
    CellAutoOverwrite->True,
    ShowCellLabel->False,
    CellLabelMargins->{{21, Inherited}, {Inherited, Inherited}},
    DefaultFormatType->DefaultOutputFormatType,
    AutoItalicWords->{},
    FormatType->InputForm,
    CounterIncrements->"Print",
    StyleMenuListing->None],
  
  Cell[StyleData["Print", "Printout"],
    ShowCellBracket->False,
    CellMargins->{{42, 4}, {4, 2}},
    FontSize->10]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["Graphics"],
    CellMargins->{{56, Inherited}, {Inherited, Inherited}},
    CellGroupingRules->"GraphicsGrouping",
    CellHorizontalScrolling->True,
    PageBreakWithin->False,
    GeneratedCell->True,
    CellAutoOverwrite->True,
    ShowCellLabel->False,
    DefaultFormatType->DefaultOutputFormatType,
    FormatType->InputForm,
    CounterIncrements->"Graphics",
    StyleMenuListing->None],
  
  Cell[StyleData["Graphics", "Printout"],
    CellMargins->{{40, 4}, {4, 2}},
    ImageSize->{250, 250},
    FontSize->9]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["CellLabel"],
    StyleMenuListing->None,
    FontFamily->"Helvetica",
    FontSize->9,
    FontColor->RGBColor[0, 0.2, 1]],
  
  Cell[StyleData["CellLabel", "Printout"],
    FontSize->7,
    FontSlant->"Oblique",
    FontColor->GrayLevel[0]]
  }, Closed]]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell["Unique Styles", "Subsection"],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["TextTop"],
    CellFrame->{{0, 0}, {0, 2}},
    CellMargins->{{10, 4}, {2, 80}},
    CellHorizontalScrolling->True,
    CellFrameMargins->4,
    ShowSpecialCharacters->Automatic,
    ParagraphSpacing->{0, 8},
    CounterIncrements->"Text"],
  
  Cell[StyleData["TextTop", "Printout"],
    CellMargins->{{2, 4}, {2, 80}},
    FontSize->10]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["ItemizedText"],
    CellMargins->{{20, 4}, {0, 8}},
    ShowSpecialCharacters->Automatic,
    ParagraphSpacing->{0, 8},
    ParagraphIndent->-15,
    CounterIncrements->"Text"],
  
  Cell[StyleData["ItemizedText", "Printout"],
    FontSize->10]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["ItemizedTextNote"],
    CellMargins->{{35, 4}, {0, 4}},
    ShowSpecialCharacters->Automatic,
    ParagraphSpacing->{0, 4},
    CounterIncrements->"Text"],
  
  Cell[StyleData["ItemizedTextNote", "Printout"],
    FontSize->10]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["IndentedText"],
    CellMargins->{{20, 4}, {0, 6}},
    ShowSpecialCharacters->Automatic,
    ParagraphSpacing->{0, 8},
    CounterIncrements->"Text"],
  
  Cell[StyleData["IndentedText", "Printout"],
    FontSize->10]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["Note"],
    CellFrame->True,
    CellMargins->{{10, 4}, {0, 8}},
    CellHorizontalScrolling->True,
    ShowSpecialCharacters->Automatic,
    ParagraphSpacing->{0, 8},
    CounterIncrements->"Text",
    FontFamily->"Helvetica",
    FontSize->10],
  
  Cell[StyleData["Note", "Printout"],
    CellMargins->{{2, 4}, {0, 8}},
    FontSize->8]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["1ColumnBox"],
    CellFrame->True,
    CellMargins->{{10, 4}, {0, 8}},
    CellHorizontalScrolling->True,
    LineIndent->0,
    Background->GrayLevel[0.8],
    FrameBoxOptions->{BoxMargins->{{1, 1}, {1.5, 1.5}}},
    GridBoxOptions->{ColumnSpacings->1}],
  
  Cell[StyleData["1ColumnBox", "Printout"],
    CellMargins->{{2, 4}, {0, 8}},
    FontSize->10,
    Background->GrayLevel[0.900008]]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["2ColumnBox"],
    CellFrame->True,
    CellMargins->{{10, 4}, {0, 8}},
    CellHorizontalScrolling->True,
    LineIndent->0,
    Background->GrayLevel[0.8],
    FrameBoxOptions->{BoxMargins->{{1, 1}, {1.5, 1.5}}},
    GridBoxOptions->{ColumnWidths->{0.39, 0.59}}],
  
  Cell[StyleData["2ColumnBox", "Printout"],
    CellMargins->{{2, 4}, {0, 8}},
    FontSize->10,
    Background->GrayLevel[0.900008]]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["2ColumnSmallBox"],
    CellFrame->True,
    CellMargins->{{10, 4}, {0, 8}},
    CellHorizontalScrolling->True,
    LineIndent->0,
    Background->GrayLevel[0.8],
    FrameBoxOptions->{BoxMargins->{{1, 1}, {1.5, 1.5}}},
    GridBoxOptions->{ColumnSpacings->1.5,
    ColumnWidths->0.35,
    ColumnAlignments->{Right, Left}}],
  
  Cell[StyleData["2ColumnSmallBox", "Printout"],
    CellMargins->{{2, 4}, {0, 8}},
    FontSize->10,
    Background->GrayLevel[0.900008]]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["3ColumnBox"],
    CellFrame->True,
    CellMargins->{{10, 4}, {0, 8}},
    CellHorizontalScrolling->True,
    LineIndent->0,
    Background->GrayLevel[0.8],
    FrameBoxOptions->{BoxMargins->{{1, 1}, {1.5, 1.5}}},
    GridBoxOptions->{ColumnWidths->0.325}],
  
  Cell[StyleData["3ColumnBox", "Printout"],
    CellMargins->{{2, 4}, {0, 8}},
    FontSize->10,
    Background->GrayLevel[0.900008]]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["3ColumnSmallBox"],
    CellFrame->True,
    CellMargins->{{10, 4}, {0, 8}},
    CellHorizontalScrolling->True,
    LineIndent->0,
    Background->GrayLevel[0.8],
    FrameBoxOptions->{BoxMargins->{{1, 1}, {1.5, 1.5}}},
    GridBoxOptions->{ColumnSpacings->1.5,
    ColumnWidths->0.23,
    ColumnAlignments->{Right, Center, Left}}],
  
  Cell[StyleData["3ColumnSmallBox", "Printout"],
    CellMargins->{{2, 4}, {0, 8}},
    FontSize->10,
    Background->GrayLevel[0.900008]]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["4ColumnBox"],
    CellFrame->True,
    CellMargins->{{10, 4}, {0, 8}},
    CellHorizontalScrolling->True,
    LineIndent->0,
    Background->GrayLevel[0.8],
    FrameBoxOptions->{BoxMargins->{{1, 1}, {1.5, 1.5}}},
    GridBoxOptions->{ColumnWidths->{0.145, 0.345, 0.145, 0.345}}],
  
  Cell[StyleData["4ColumnBox", "Printout"],
    CellMargins->{{2, 4}, {0, 8}},
    FontSize->10,
    Background->GrayLevel[0.900008]]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["5ColumnBox"],
    CellFrame->True,
    CellMargins->{{10, 4}, {0, 8}},
    CellHorizontalScrolling->True,
    LineIndent->0,
    Background->GrayLevel[0.8],
    FrameBoxOptions->{BoxMargins->{{1, 1}, {1.5, 1.5}}},
    GridBoxOptions->{ColumnWidths->0.195}],
  
  Cell[StyleData["5ColumnBox", "Printout"],
    CellMargins->{{2, 4}, {0, 8}},
    FontSize->10,
    Background->GrayLevel[0.900008]]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["6ColumnBox"],
    CellFrame->True,
    CellMargins->{{10, 4}, {0, 8}},
    CellHorizontalScrolling->True,
    LineIndent->0,
    Background->GrayLevel[0.8],
    FrameBoxOptions->{BoxMargins->{{1, 1}, {1.5, 1.5}}},
    GridBoxOptions->{ColumnWidths->{0.13, 0.23, 0.13, 0.13, 0.23, 0.13}}],
  
  Cell[StyleData["6ColumnBox", "Printout"],
    CellMargins->{{2, 4}, {0, 8}},
    FontSize->10,
    Background->GrayLevel[0.900008]]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["Picture"],
    CellMargins->{{10, Inherited}, {0, 8}},
    CellHorizontalScrolling->True],
  
  Cell[StyleData["Picture", "Printout"],
    CellMargins->{{2, Inherited}, {0, 8}}]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["Caption"],
    CellMargins->{{10, 50}, {0, 3}},
    PageBreakAbove->False,
    FontFamily->"Helvetica",
    FontSize->9],
  
  Cell[StyleData["Caption", "Printout"],
    CellMargins->{{2, 50}, {2, 4}},
    FontSize->7]
  }, Closed]]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell["Tables", "Subsection"],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["2ColumnTable"],
    CellMargins->{{10, 4}, {0, 8}},
    CellHorizontalScrolling->True,
    GridBoxOptions->{ColumnWidths->{0.39, 0.59},
    ColumnAlignments->{Left}}],
  
  Cell[StyleData["2ColumnTable", "Printout"],
    CellMargins->{{2, 4}, {0, 8}},
    FontSize->10]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["3ColumnTable"],
    CellMargins->{{10, 4}, {0, 8}},
    CellHorizontalScrolling->True,
    StyleMenuListing->None,
    GridBoxOptions->{ColumnWidths->0.325,
    ColumnAlignments->{Left}}],
  
  Cell[StyleData["3ColumnTable", "Printout"],
    CellMargins->{{2, 4}, {0, 8}},
    FontSize->10]
  }, Closed]]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell["Formulas and Programming", "Subsection"],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["ChemicalFormula"],
    CellMargins->{{42, Inherited}, {Inherited, Inherited}},
    CellHorizontalScrolling->True,
    DefaultFormatType->DefaultInputFormatType,
    AutoSpacing->False,
    ScriptLevel->1,
    ScriptBaselineShifts->{0.6, Automatic},
    SingleLetterItalics->False,
    ZeroWidthTimes->True],
  
  Cell[StyleData["ChemicalFormula", "Printout"],
    CellMargins->{{34, Inherited}, {Inherited, Inherited}},
    FontSize->10]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["DisplayFormula"],
    CellMargins->{{42, Inherited}, {Inherited, Inherited}},
    CellHorizontalScrolling->True,
    DefaultFormatType->DefaultInputFormatType,
    ScriptLevel->0,
    SingleLetterItalics->True,
    SpanMaxSize->Infinity,
    UnderoverscriptBoxOptions->{LimitsPositioning->True},
    GridBoxOptions->{ColumnWidths->Automatic}],
  
  Cell[StyleData["DisplayFormula", "Printout"],
    CellMargins->{{34, Inherited}, {Inherited, Inherited}},
    FontSize->10]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["Program"],
    CellMargins->{{10, 4}, {0, 8}},
    CellHorizontalScrolling->True,
    FontFamily->"Courier"],
  
  Cell[StyleData["Program", "Printout"],
    CellMargins->{{2, Inherited}, {Inherited, Inherited}},
    FontSize->9.5]
  }, Closed]]
  }, Closed]]
  }, Open  ]]
  }]
]


(***********************************************************************
Cached data follows.  If you edit this Notebook file directly, not using
Mathematica, you must remove the line containing CacheID at the top of 
the file.  The cache data will then be recreated when you save this file 
from within Mathematica.
***********************************************************************)

(*CellTagsOutline
CellTagsIndex->{
  "Introduction"->{
    Cell[1988, 64, 59, 1, 99, "Section",
      CellTags->"Introduction"],
    Cell[2050, 67, 738, 13, 149, "Text",
      CellTags->"Introduction"],
    Cell[2813, 84, 95, 3, 44, "Input",
      CellTags->"Introduction"]},
  "Sliding"->{
    Cell[3629, 104, 70, 1, 99, "Section",
      CellTags->{"Sliding", "i:1"}],
    Cell[3702, 107, 1407, 26, 300, "Text",
      CellTags->{"Sliding", "i:1"}],
    Cell[5134, 137, 61, 1, 54, "Subsection",
      CellTags->{"Sliding", "i:1"}],
    Cell[5198, 140, 181, 4, 56, "Text",
      CellTags->{"Sliding", "i:1"}],
    Cell[5382, 146, 175, 5, 84, "Input",
      CellTags->{"Sliding", "i:1"}],
    Cell[5582, 155, 150, 4, 44, "Input",
      CellTags->{"Sliding", "i:1"}],
    Cell[6267, 185, 111, 3, 44, "Input",
      CellTags->{"Sliding", "i:1"}],
    Cell[6571, 198, 134, 3, 44, "Input",
      CellTags->{"Sliding", "i:1"}],
    Cell[6808, 209, 157, 4, 39, "Text",
      CellTags->{"Sliding", "i:1"}],
    Cell[6990, 217, 132, 3, 44, "Input",
      CellTags->{"Sliding", "i:1"}],
    Cell[7491, 241, 145, 4, 39, "Text",
      CellTags->{"Sliding", "i:1"}],
    Cell[7661, 249, 127, 3, 44, "Input",
      CellTags->{"Sliding", "i:1"}]},
  "i:1"->{
    Cell[3629, 104, 70, 1, 99, "Section",
      CellTags->{"Sliding", "i:1"}],
    Cell[3702, 107, 1407, 26, 300, "Text",
      CellTags->{"Sliding", "i:1"}],
    Cell[5134, 137, 61, 1, 54, "Subsection",
      CellTags->{"Sliding", "i:1"}],
    Cell[5198, 140, 181, 4, 56, "Text",
      CellTags->{"Sliding", "i:1"}],
    Cell[5382, 146, 175, 5, 84, "Input",
      CellTags->{"Sliding", "i:1"}],
    Cell[5582, 155, 150, 4, 44, "Input",
      CellTags->{"Sliding", "i:1"}],
    Cell[5735, 161, 97, 2, 42, "Print",
      CellTags->"i:1"],
    Cell[5835, 165, 100, 2, 42, "Print",
      CellTags->"i:1"],
    Cell[5938, 169, 81, 2, 42, "Print",
      CellTags->"i:1"],
    Cell[6022, 173, 94, 2, 42, "Print",
      CellTags->"i:1"],
    Cell[6119, 177, 111, 3, 38, "Output",
      CellTags->"i:1"],
    Cell[6267, 185, 111, 3, 44, "Input",
      CellTags->{"Sliding", "i:1"}],
    Cell[6381, 190, 153, 3, 61, "Print",
      CellTags->"i:1"],
    Cell[6571, 198, 134, 3, 44, "Input",
      CellTags->{"Sliding", "i:1"}],
    Cell[6708, 203, 85, 3, 38, "Output",
      CellTags->"i:1"],
    Cell[6808, 209, 157, 4, 39, "Text",
      CellTags->{"Sliding", "i:1"}],
    Cell[6990, 217, 132, 3, 44, "Input",
      CellTags->{"Sliding", "i:1"}],
    Cell[7125, 222, 89, 3, 38, "Output",
      CellTags->"i:1"],
    Cell[7251, 230, 122, 3, 44, "Input",
      CellTags->"i:1"],
    Cell[7376, 235, 100, 3, 38, "Output",
      CellTags->"i:1"],
    Cell[7491, 241, 145, 4, 39, "Text",
      CellTags->{"Sliding", "i:1"}],
    Cell[7661, 249, 127, 3, 44, "Input",
      CellTags->{"Sliding", "i:1"}],
    Cell[7791, 254, 81, 2, 42, "Print",
      CellTags->"i:1"],
    Cell[7875, 258, 94, 2, 42, "Print",
      CellTags->"i:1"],
    Cell[7972, 262, 113, 3, 55, "Output",
      CellTags->"i:1"]},
  "Reaching"->{
    Cell[8146, 272, 60, 1, 67, "Section",
      CellTags->{"Reaching", "i:2"}],
    Cell[8209, 275, 262, 7, 56, "Text",
      CellTags->{"Reaching", "i:2"}],
    Cell[8496, 286, 62, 1, 54, "Subsection",
      CellTags->{"Reaching", "i:2"}],
    Cell[8561, 289, 176, 5, 84, "Input",
      CellTags->{"Reaching", "i:2"}],
    Cell[8762, 298, 140, 3, 44, "Input",
      CellTags->{"Reaching", "i:2"}],
    Cell[9237, 319, 143, 3, 44, "Input",
      CellTags->{"Reaching", "i:2"}]},
  "i:2"->{
    Cell[8146, 272, 60, 1, 67, "Section",
      CellTags->{"Reaching", "i:2"}],
    Cell[8209, 275, 262, 7, 56, "Text",
      CellTags->{"Reaching", "i:2"}],
    Cell[8496, 286, 62, 1, 54, "Subsection",
      CellTags->{"Reaching", "i:2"}],
    Cell[8561, 289, 176, 5, 84, "Input",
      CellTags->{"Reaching", "i:2"}],
    Cell[8762, 298, 140, 3, 44, "Input",
      CellTags->{"Reaching", "i:2"}],
    Cell[8905, 303, 81, 2, 42, "Print",
      CellTags->"i:2"],
    Cell[8989, 307, 94, 2, 42, "Print",
      CellTags->"i:2"],
    Cell[9086, 311, 114, 3, 55, "Output",
      CellTags->"i:2"],
    Cell[9237, 319, 143, 3, 44, "Input",
      CellTags->{"Reaching", "i:2"}],
    Cell[9383, 324, 106, 3, 55, "Output",
      CellTags->"i:2"]},
  "Smoothing"->{
    Cell[9550, 334, 74, 1, 99, "Section",
      CellTags->{"Smoothing", "i:3"}],
    Cell[9627, 337, 488, 10, 90, "Text",
      CellTags->{"Smoothing", "i:3"}],
    Cell[10140, 351, 115, 3, 44, "Input",
      CellTags->{"Smoothing", "i:3"}],
    Cell[10755, 368, 221, 6, 64, "Input",
      CellTags->{"Smoothing", "i:3"}]},
  "i:3"->{
    Cell[9550, 334, 74, 1, 99, "Section",
      CellTags->{"Smoothing", "i:3"}],
    Cell[9627, 337, 488, 10, 90, "Text",
      CellTags->{"Smoothing", "i:3"}],
    Cell[10140, 351, 115, 3, 44, "Input",
      CellTags->{"Smoothing", "i:3"}],
    Cell[10258, 356, 460, 7, 213, "Print",
      CellTags->"i:3"],
    Cell[10755, 368, 221, 6, 64, "Input",
      CellTags->{"Smoothing", "i:3"}],
    Cell[10979, 376, 238, 5, 66, "Output",
      CellTags->"i:3"]},
  "Moderating"->{
    Cell[11266, 387, 74, 1, 67, "Section",
      CellTags->{"Moderating", "i:4"}],
    Cell[11343, 390, 418, 7, 73, "Text",
      CellTags->{"Moderating", "i:4"}],
    Cell[11786, 401, 117, 3, 44, "Input",
      CellTags->{"Moderating", "i:4"}],
    Cell[12479, 419, 248, 6, 64, "Input",
      CellTags->{"Moderating", "i:4"}]},
  "i:4"->{
    Cell[11266, 387, 74, 1, 67, "Section",
      CellTags->{"Moderating", "i:4"}],
    Cell[11343, 390, 418, 7, 73, "Text",
      CellTags->{"Moderating", "i:4"}],
    Cell[11786, 401, 117, 3, 44, "Input",
      CellTags->{"Moderating", "i:4"}],
    Cell[11906, 406, 536, 8, 232, "Print",
      CellTags->"i:4"],
    Cell[12479, 419, 248, 6, 64, "Input",
      CellTags->{"Moderating", "i:4"}],
    Cell[12730, 427, 357, 10, 70, "Output",
      CellTags->"i:4"]},
  "Bibliography"->{
    Cell[13136, 443, 59, 1, 67, "Section",
      CellTags->"Bibliography"],
    Cell[13198, 446, 32744, 506, 273, "Graphics",
      CellTags->"Bibliography"]}
  }
*)

(*CellTagsIndex
CellTagsIndex->{
  {"Introduction", 64572, 1643},
  {"Sliding", 64808, 1650},
  {"i:1", 65760, 1675},
  {"Reaching", 67560, 1726},
  {"i:2", 68050, 1739},
  {"Smoothing", 68806, 1760},
  {"i:3", 69141, 1769},
  {"Moderating", 69617, 1782},
  {"i:4", 69957, 1791},
  {"Bibliography", 70441, 1804}
  }
*)

(*NotebookFileOutline
Notebook[{

Cell[CellGroupData[{
Cell[1739, 51, 86, 2, 96, "Title"],
Cell[1828, 55, 35, 0, 33, "Subtitle"],
Cell[1866, 57, 97, 3, 50, "Subsubtitle"],

Cell[CellGroupData[{
Cell[1988, 64, 59, 1, 99, "Section",
  CellTags->"Introduction"],
Cell[2050, 67, 738, 13, 149, "Text",
  CellTags->"Introduction"],

Cell[CellGroupData[{
Cell[2813, 84, 95, 3, 44, "Input",
  CellTags->"Introduction"],
Cell[2911, 89, 669, 9, 237, "Print"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[3629, 104, 70, 1, 99, "Section",
  CellTags->{"Sliding", "i:1"}],
Cell[3702, 107, 1407, 26, 300, "Text",
  CellTags->{"Sliding", "i:1"}],

Cell[CellGroupData[{
Cell[5134, 137, 61, 1, 54, "Subsection",
  CellTags->{"Sliding", "i:1"}],
Cell[5198, 140, 181, 4, 56, "Text",
  CellTags->{"Sliding", "i:1"}],
Cell[5382, 146, 175, 5, 84, "Input",
  CellTags->{"Sliding", "i:1"}],

Cell[CellGroupData[{
Cell[5582, 155, 150, 4, 44, "Input",
  CellTags->{"Sliding", "i:1"}],
Cell[5735, 161, 97, 2, 42, "Print",
  CellTags->"i:1"],
Cell[5835, 165, 100, 2, 42, "Print",
  CellTags->"i:1"],
Cell[5938, 169, 81, 2, 42, "Print",
  CellTags->"i:1"],
Cell[6022, 173, 94, 2, 42, "Print",
  CellTags->"i:1"],
Cell[6119, 177, 111, 3, 38, "Output",
  CellTags->"i:1"]
}, Open  ]],

Cell[CellGroupData[{
Cell[6267, 185, 111, 3, 44, "Input",
  CellTags->{"Sliding", "i:1"}],
Cell[6381, 190, 153, 3, 61, "Print",
  CellTags->"i:1"]
}, Open  ]],

Cell[CellGroupData[{
Cell[6571, 198, 134, 3, 44, "Input",
  CellTags->{"Sliding", "i:1"}],
Cell[6708, 203, 85, 3, 38, "Output",
  CellTags->"i:1"]
}, Open  ]],
Cell[6808, 209, 157, 4, 39, "Text",
  CellTags->{"Sliding", "i:1"}],

Cell[CellGroupData[{
Cell[6990, 217, 132, 3, 44, "Input",
  CellTags->{"Sliding", "i:1"}],
Cell[7125, 222, 89, 3, 38, "Output",
  CellTags->"i:1"]
}, Open  ]],

Cell[CellGroupData[{
Cell[7251, 230, 122, 3, 44, "Input",
  CellTags->"i:1"],
Cell[7376, 235, 100, 3, 38, "Output",
  CellTags->"i:1"]
}, Open  ]],
Cell[7491, 241, 145, 4, 39, "Text",
  CellTags->{"Sliding", "i:1"}],

Cell[CellGroupData[{
Cell[7661, 249, 127, 3, 44, "Input",
  CellTags->{"Sliding", "i:1"}],
Cell[7791, 254, 81, 2, 42, "Print",
  CellTags->"i:1"],
Cell[7875, 258, 94, 2, 42, "Print",
  CellTags->"i:1"],
Cell[7972, 262, 113, 3, 55, "Output",
  CellTags->"i:1"]
}, Open  ]]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{
Cell[8146, 272, 60, 1, 67, "Section",
  CellTags->{"Reaching", "i:2"}],
Cell[8209, 275, 262, 7, 56, "Text",
  CellTags->{"Reaching", "i:2"}],

Cell[CellGroupData[{
Cell[8496, 286, 62, 1, 54, "Subsection",
  CellTags->{"Reaching", "i:2"}],
Cell[8561, 289, 176, 5, 84, "Input",
  CellTags->{"Reaching", "i:2"}],

Cell[CellGroupData[{
Cell[8762, 298, 140, 3, 44, "Input",
  CellTags->{"Reaching", "i:2"}],
Cell[8905, 303, 81, 2, 42, "Print",
  CellTags->"i:2"],
Cell[8989, 307, 94, 2, 42, "Print",
  CellTags->"i:2"],
Cell[9086, 311, 114, 3, 55, "Output",
  CellTags->"i:2"]
}, Open  ]],

Cell[CellGroupData[{
Cell[9237, 319, 143, 3, 44, "Input",
  CellTags->{"Reaching", "i:2"}],
Cell[9383, 324, 106, 3, 55, "Output",
  CellTags->"i:2"]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[9550, 334, 74, 1, 99, "Section",
  CellTags->{"Smoothing", "i:3"}],
Cell[9627, 337, 488, 10, 90, "Text",
  CellTags->{"Smoothing", "i:3"}],

Cell[CellGroupData[{
Cell[10140, 351, 115, 3, 44, "Input",
  CellTags->{"Smoothing", "i:3"}],
Cell[10258, 356, 460, 7, 213, "Print",
  CellTags->"i:3"]
}, Open  ]],

Cell[CellGroupData[{
Cell[10755, 368, 221, 6, 64, "Input",
  CellTags->{"Smoothing", "i:3"}],
Cell[10979, 376, 238, 5, 66, "Output",
  CellTags->"i:3"]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{
Cell[11266, 387, 74, 1, 67, "Section",
  CellTags->{"Moderating", "i:4"}],
Cell[11343, 390, 418, 7, 73, "Text",
  CellTags->{"Moderating", "i:4"}],

Cell[CellGroupData[{
Cell[11786, 401, 117, 3, 44, "Input",
  CellTags->{"Moderating", "i:4"}],
Cell[11906, 406, 536, 8, 232, "Print",
  CellTags->"i:4"]
}, Open  ]],

Cell[CellGroupData[{
Cell[12479, 419, 248, 6, 64, "Input",
  CellTags->{"Moderating", "i:4"}],
Cell[12730, 427, 357, 10, 70, "Output",
  CellTags->"i:4"]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{
Cell[13136, 443, 59, 1, 67, "Section",
  CellTags->"Bibliography"],
Cell[13198, 446, 32744, 506, 273, "Graphics",
  CellTags->"Bibliography"]
}, Closed]]
}, Open  ]]
}
]
*)




(***********************************************************************
End of Mathematica Notebook file.
***********************************************************************)

