/*
 * This source code (.java) file is Copyright  2000 Bill Venners. All rights reserved. 
 * This file accompanies the Jini Service UI Draft Specification, written by Bill
 * Venners and published on the World Wide Web at:
 * 
 *     http://www.artima.com/jini/serviceui/DraftSpec.html,
 *
 * This source file may not be copied, modified, or redistributed EXCEPT as allowed
 * by the following statements: From February 7, 2000 through March 6, 2000, you may
 * copy and/or modify these files to test and experiment with the Service UI API,
 * described in the Jini Service UI Draft Specification. Any bug fixes must be given
 * back to Bill Venners. You may not redistribute this file or any binary (such
 * as .class) files generated from this file. You may not distribute modified versions
 * this files or any binary (such as .class) files generated from modified versions of
 * this file. You may not remove this copyright notice. You may not use this file in
 * printed media without the express permission of Bill Venners. And if that weren't
 * enough, you must destroy all copies of this file, and any binary (such as
 * .class) files generated from this file, by March 7, 2000.
 *
 * BILL VENNERS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THIS
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR PARTICULAR PURPOSE, OR NON-INFRINGEMENT.
 * BILL VENNERS SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY A LICENSEE AS A RESULT
 * OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
*/
package net.jini.lookup.entry;

import net.jini.entry.AbstractEntry;
import java.util.Set;
import java.rmi.MarshalledObject;
import java.io.IOException;
import java.security.AccessController;
import java.security.PrivilegedAction;

/**
 * <CODE>Entry</CODE> that enables a UI for a service to be associated
 * with the service in the attribute sets of the service item.
 * <CODE>UIDescriptor</CODE> holds a marshalled UI factory object, as
 * well as a role string, a sub-role string, and set of attributes that describe the
 * UI generated by the factory.
*/
public class UIDescriptor extends AbstractEntry {

    /**
     * Gives the fully qualified name of the interface that represents
     * the role of the UI generated by the marshalled UI factory.
     * If the client program unmarshals the UI factory and invokes a factory method, the
     * UI returned by the factory method must implement the role the interface specified by
     * <CODE>role</CODE>.
     *
     * <P>
     * For a client program to be able to use a UI, the client has to have prior knowledge
     * of the UI semantics, which is defined by the UI's role type. Thus, for a client
     * to be able to use a UI, the client must understand the semantics
     * of the type whose fully qualified name is given in the <CODE>String</CODE>
     * referenced from the <CODE>role</CODE> field of that UI's <CODE>UIDescriptor</CODE>.
     *
     * <P>
     * For example, two role types that are defined in the <CODE>net.jini.lookup.ui</CODE>
     * package by the Jini Service UI Specification are <CODE>MainUI</CODE>, for a main UI
     * to a Jini service, and <CODE>AdminUI</CODE>, for an administration UI. Other role types
     * may be defined by the Jini Service UI Specification and by individual Jini service API
     * specifications.
     *
     * <P>
     * As the strings referenced from the <CODE>role</CODE> field are Java type names, they
     * are intended to be manipulated by client programs only. They should not be shown to a user.
     *
    */
    public String role;

    /**
     * A String to facilitate searching whose value represents the main UI toolkit (for example
     * Swing or AWT) which the produced UI makse use of. The value to which this field should
     * be set is defined in by the semantics of the factory type. (This field is intended
     * to facilitate searches. For example, a client can search for all blender services that have Swing
     * MainUI's.)
    */
    public String toolkit;

    /**
     * A set of objects that describe the UI generated by the marshalled UI factory.
    */
    public Set attributes;

    /**
     * The <CODE>get()</CODE> method of this <CODE>MarshalledObject</CODE>
     * must return an object that implements one or more UI factory interfaces. The actual
     * UI factory type or types implemented by the returned object
     * must be described by a <CODE>UIFactoryTypes</CODE> attribute placed in
     * the attributes set of this <CODE>UIDescriptor</CODE>.
    */
    public MarshalledObject factory;

    /**
     * Constructs a <CODE>UIDescriptor</CODE> with all fields set to <CODE>null</CODE>.
    */
    public UIDescriptor() {
    }

    /**
     * Constructs a <CODE>UIDescriptor</CODE> with the fields set to passed values.
    */
    public UIDescriptor(String role, Set attributes, MarshalledObject factory) {

        this.role = role;
        this.attributes = attributes;
        this.factory = factory;
    }

    /**
     * A convenience method for unmarshalling the UI factory stored
     * in the <CODE>MarshalledObject</CODE> referenced from the
     * <CODE>marshalledUIFactory</CODE> field. This method saves a reference
     * to the current context class loader, sets the context class loader
     * to the class loader passed as <CODE>parentLoader</CODE>, invokes
     * <CODE>get()</CODE> on the marshalled object, then resets the
     * context class loader to the saved reference before returning
     * the object produced by <CODE>get()</CODE>.
     * 
     * <P>The class loader
     * passed in <CODE>parentLoader</CODE> should be able to load classes
     * needed when the UI interacts with the <CODE>roleObject</CODE> passed as the first
     * parameter to the factory method. For example, if the <CODE>roleObject</CODE> is
     * the service item (as it is for the <CODE>MainUI</CODE> and <CODE>AdminUI</CODE>
     * roles), the class loader passed in <CODE>parentLoader</CODE> could be
     * the class loader with which the service item was loaded.
    */
    public final Object getUIFactory(final ClassLoader parentLoader) 
        throws IOException, ClassNotFoundException {

        Object uiFactory;
        final Thread currentThread = Thread.currentThread();

        // First, get a reference to the current context class loader,
        // so it can be restored after the unmarshalling
        final ClassLoader original = (ClassLoader) AccessController.doPrivileged(
            new PrivilegedAction() {
                public Object run() {
                    return currentThread.getContextClassLoader();
                }
            }
        );
        try {
            AccessController.doPrivileged(
                new PrivilegedAction() {
                    public Object run() {
                        currentThread.setContextClassLoader(parentLoader);
                        return null;
                    }
                }
            );
            uiFactory = factory.get();
        }
        finally {
            AccessController.doPrivileged(
                new PrivilegedAction() {
                    public Object run() {
                        currentThread.setContextClassLoader(original);
                        return null;
                    }
                }
            );
        }
        return uiFactory;
    }
}