/**
 * This provides an encapsulation of a client-side object
 * to locate RCX devices on the network and to send commands
 * to them. RCX devices are filtered by a set of Entry
 * attributes to select the ones with desired properties.
 * If more than one matches, then commands may be sent to
 * them all, and replies received from all. It is not possible
 * to further narrow the set - use a tighter set of Entry
 * attributes
 */ 
package rcx.jini;

import net.jini.discovery.LookupDiscovery;
import net.jini.discovery.DiscoveryListener;
import net.jini.discovery.DiscoveryEvent;
import net.jini.core.lookup.ServiceRegistrar;
import net.jini.core.lookup.ServiceTemplate;
import net.jini.core.event.RemoteEventListener;
import net.jini.core.event.RemoteEvent;
import net.jini.core.event.UnknownEventException;
import net.jini.core.entry.Entry;
import net.jini.core.discovery.LookupLocator;
import net.jini.core.lookup.ServiceMatches;
import net.jini.core.lookup.ServiceItem;
import net.jini.core.lookup.ServiceID;

import java.rmi.RMISecurityManager;
import java.rmi.server.UnicastRemoteObject;
import java.rmi.RemoteException;

import java.util.Vector;
import java.util.Enumeration;

import rcx.RCXListener;


/**
 * JiniRCXPort.java
 *
 *
 * Created: Wed Mar 17 14:29:15 1999
 *
 * @author Jan Newmarch
 * @version 1.0
 */

public class JiniRCXPort implements DiscoveryListener {

    protected Entry[] entries = null;
    /**
     * List of ports discovered
     */
    protected Vector ports = new Vector();
    /**
     * List of serviceIDs found from locators
     */
    protected Vector serviceIDs = new Vector();
    /**
     * List of listeners for messages from RCX's
     */    
    protected Vector listeners = new Vector();

    /**
     * Locate all RCX devices using multicast for all groups
     */
    public JiniRCXPort() {
	this(null, LookupDiscovery.ALL_GROUPS, null);
    }
    
    /**
     * Locate RCX devices using multicast for all groups
     * @param entries the set of Entry objects to select the RCX
     */
    public JiniRCXPort(Entry[] entries) {
	this(entries, LookupDiscovery.ALL_GROUPS, null);
    }
    
    /**
     * Locate RCX devices using multicast for the specified groups
     * @param entries the set of Entry objects to select the RCX
     * @param groups the group properties to select service locators
     */
    public JiniRCXPort(Entry[] entries,
		       java.lang.String[] groups) {
	this(entries, groups, null);
    }

    /**
     * Locate RCX devices. 
     * Multicast lookup is used to find local service locators
     * but a set of (probably) remote locators can be specified
     * for unicast lookup.
     * @param entries the set of Entry objects to select the RCX
     * @param groups the group properties to select service locators
     * @param locators the set of unicast locators specified
     */
    public JiniRCXPort(Entry[] entries,
		       java.lang.String[] groups, 
		       LookupLocator[] locators) {
	System.setSecurityManager(new RMISecurityManager());
	this.entries = entries;

	// Try all the unicast entries specified
	if (locators != null) {
	    ServiceRegistrar registrar = null;

	    for (int n = 0; n < locators.length; n++) {
		try {
		    registrar = locators[n].getRegistrar();
		} catch (java.io.IOException e) {
		    e.printStackTrace();
		    continue;
		} catch (java.lang.ClassNotFoundException e) {
		    e.printStackTrace();
		    continue;
		}
		lookupService(registrar);
	    }
	}

	LookupDiscovery discover = null;
	try {
	    discover = new LookupDiscovery(LookupDiscovery.ALL_GROUPS);
	} catch(Exception e) {
	    System.err.println(e.toString());
	    System.exit(1);
	}
	
        discover.addDiscoveryListener(this);

    }

    protected void lookupService(ServiceRegistrar registrar) {
	RCXPortInterface port = null;
	Class [] classes = new Class[] {RCXPortInterface.class};
	ServiceTemplate template = new ServiceTemplate(null, classes, 
						       entries);
	ServiceMatches matches = null;

	try {
	    matches = registrar.lookup(template, 10);
	} catch(java.rmi.RemoteException e) {
	    e.printStackTrace();
	    return;
	}

	ServiceItem[] items = matches.items;
	for (int m = 0; m < items.length; m++) {
	    ServiceID id = items[m].serviceID;
	    if (serviceIDs.indexOf(id) != -1) {
		// found a new serviceID - record it and use it
		port = (RCXPortProxy) items[m].service;
		if (port == null) {
		    continue;
		}

		serviceIDs.add(id);
		ports.add(port);

		// add an EventHandler as an RCX Port listener
		try {
		    port.addListener(new EventHandler(port));
		} catch(Exception e) {
		    e.printStackTrace();
		}
	    }
	}
    }

    public void discovered(DiscoveryEvent evt) {

        ServiceRegistrar[] registrars = evt.getRegistrars();
 
        for (int n = 0; n < registrars.length; n++) {
	    // System.out.println("Service found");
            ServiceRegistrar registrar = registrars[n];
	    lookupService(registrar);
	}
    }
    
    public void discarded(DiscoveryEvent evt) {
	// empty
    }
    
    public void addRCXListener(RCXListener listener) {
	listeners.add(listener);
    }
    
    public boolean write(byte[] bArray) {
	RCXPortProxy port;
	boolean status = true;

	for (Enumeration enum = ports.elements() ; enum.hasMoreElements() ;) {
	     port = (RCXPortProxy) enum.nextElement();
	     try {
		 if (! port.write(bArray)) {
		     status = false;
		 }
	     } catch(java.rmi.RemoteException e) {
		 e.printStackTrace();
	     }
	}	
	return status;
    }
    
    public byte[] parseString(String str) {
	RCXPortProxy port;
	
	// any one of the ports will do for this
	port = (RCXPortProxy) ports.elementAt(0);
	try {
	    return port.parseString(str);
	} catch(java.rmi.RemoteException e) {
	    e.printStackTrace();
	    return new byte[] {};
	}
    }

    class EventHandler extends UnicastRemoteObject 
                       implements RemoteEventListener {

	protected RCXPortInterface port = null;

        public EventHandler(RCXPortInterface port) throws RemoteException {
            super() ;
	    this.port = port;
        }

        public void notify(RemoteEvent evt) throws UnknownEventException, 
                                                 java.rmi.RemoteException {
	    // System.out.println(evt.toString());
    
	    // Time consuming work should be done elsewhere
	    // than in this method e.g. in a new Thread
	    long id = evt.getID();
	    long seqNo = evt.getSequenceNumber();
	    RCXListener listener;

	    if (id == RCXPortInterface.MESSAGE_EVENT) {
		// System.out.println("MESSAGE: " + port.getMessage(seqNo));

		for (Enumeration enum = listeners.elements() ; enum.hasMoreElements() ;) {
		    listener = (RCXListener) enum.nextElement();
		    listener.receivedMessage(port.getMessage(seqNo));
		}	

	    } else if (id == RCXPortInterface.ERROR_EVENT) {
		// System.out.println("ERROR: " + port.getError(seqNo));

		for (Enumeration enum = listeners.elements() ; enum.hasMoreElements() ;) {
		    listener = (RCXListener) enum.nextElement();
		    listener.receivedError(port.getError(seqNo));
		}	
	    } else {
		throw new UnknownEventException("Unknown message " + evt.getID());
	    }
        }
    }
} // JiniRCXPort
