/** 
 *  The intent of this class is to demonstrate how a service 
 *  functions as a Transaction participant.
 *  
 *  This class should be run in conjunction with a client class which
 *  will discover each lookup, query them for the service that was
 *  registered by this class, and then invoke the methods under
 *  a transaction.  The client class TxnClient.
 *
 *  It is also useful to employ the lookup browser utility to verify that
 *  the service, along with its associated test attributes, has indeed
 *  been registered in each lookup.
 */

package examples.txn;

import net.jini.core.discovery.LookupLocator;
import net.jini.core.entry.Entry;
import net.jini.core.lookup.ServiceRegistrar;
import net.jini.core.lookup.ServiceItem;
import net.jini.core.lookup.ServiceRegistration;
import net.jini.core.lease.Lease;
import net.jini.core.lease.LeaseMap;
import net.jini.core.lease.LeaseDeniedException;
import net.jini.core.lease.UnknownLeaseException;

import net.jini.discovery.LookupDiscovery;
import net.jini.discovery.DiscoveryListener;
import net.jini.discovery.DiscoveryEvent;
import net.jini.lookup.entry.ServiceInfo;
import net.jini.lookup.entry.Name;

import net.jini.core.transaction.*;
import net.jini.core.transaction.server.*;

import com.sun.jini.lookup.entry.BasicServiceType;

import java.rmi.RMISecurityManager;
import java.rmi.Remote;
import java.rmi.server.UnicastRemoteObject;
import java.rmi.RemoteException;

import java.util.Properties;
import java.util.StringTokenizer;
import java.util.Vector;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.net.URL;
import java.net.MalformedURLException;
import java.io.File;
import java.io.InputStream;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.FileNotFoundException;
import java.sql.Time;
import java.lang.Long;
import java.lang.Integer;

public class TxnServerImpl implements TxnServer {

    private static final boolean DEBUG = true;

    private static final String SEP = System.getProperty("file.separator");
    private static final int N_ITER = 21;
    private static final int MIN_N_ARGS = 0;
    private static final int NUMBER_OF_LEASEMAP_UPDATES = 5;
    private static final String PUBLIC_GROUP = "";

    private static final long LEASE_DUR_WITH_LOOKUP = 3 * 60 *1000; // seconds

    private static final String PRODUCT
                             = "TxnServer - An Always-Up Service";
    private static final String MANUFACTURER = "Sun Microsystems, Inc.";
    private static final String VENDOR = MANUFACTURER;
    private static final String VERSION = "Jini 1.0";

    private static TxnServer thisClass1;
    private static TxnServer thisClass2;

    private LookupDiscovery lookupDiscovery;
    private Vector srvcLeaseVec = new Vector();
    private Vector lookupSrvcVec = new Vector();
    
    private long crashCount = 1;
    private long txnId = 0;
    private HashMap txnMap = new HashMap();
    private HashMap txnIdMap = new HashMap();
    private HashMap unDoMap = new HashMap();
    private int balance = 0;
    private int unDoCount = 0;
    private static final int SEED_MONEY1 = 100;
    private static final int SEED_MONEY2 = 200;
    private static final String ACCOUNT_NAME1 = "Account1";
    private static final String ACCOUNT_NAME2 = "Account2";
    private String accountName;
   
    private final static class UnDoReg{
        public static final int UNDO_DEPOSIT = 1;
        public static final int UNDO_WITHDRAW = 2;
        public int operation;
        public int amount;
        
        public UnDoReg(int operation, int amount){
            this.operation = operation;
            this.amount = amount;
        }                    
    }

    private Entry[] serviceAttrs = new Entry[]{new ServiceInfo(PRODUCT,
                                                               MANUFACTURER,
                                                               VENDOR,
                                                               VERSION,
                                                               null, null),
                                              new BasicServiceType("Service"),
                                              new Name("")};
                                                                                           
    public TxnServerImpl(int seedmoney, String name) throws RemoteException {
        super();
        this.balance = seedmoney;
        this.accountName = name;
        UnicastRemoteObject.exportObject(this);
        
    }

    /////////////////////////////////////////////////////
    //
    // Bank Account remote methods
    //
    /////////////////////////////////////////////////////
    
    public int startAccount(int SeedMoney) throws RemoteException {
        balance = SeedMoney;
        return balance;
    }
    
    public int balance() throws RemoteException{
        return balance;
    }
    
    public int deposit(Transaction txn, int amount) throws RemoteException {
        
        System.out.println("TxnServerImpl: deposit() called on " + accountName);
   	
        // Check if this transaction is new
        CheckIfTransactionIsNew(txn);
        
        // Keep undo operation
        UnDoReg undo = new UnDoReg(UnDoReg.UNDO_DEPOSIT, amount);
        unDoMap.put(new Integer(unDoCount++), undo);
        
        balance += amount;
        return balance;
    }
    
    public int withdraw(Transaction txn, int amount) throws RemoteException {

        System.out.println("\n");
        System.out.println("TxnServerImpl: withdraw() called on " + accountName);
        
        // Check if this transaction is new
        CheckIfTransactionIsNew(txn);
        
        // Keep undo operation
        UnDoReg undo = new UnDoReg(UnDoReg.UNDO_WITHDRAW, amount);
        unDoMap.put(new Integer(unDoCount++), undo);
        
        balance -= amount;
        return balance;            
    }
        
    /////////////////////////////////////////////////////
    //
    // TransactionParticipant methods
    //
    /////////////////////////////////////////////////////

    public int prepare(TransactionManager mgr,
                   long txnId)
            throws UnknownTransactionException,
                   java.rmi.RemoteException{
 
         System.out.println("TxnServerImpl: prepare() called on " + accountName);
                  
        // Check if this is something I know
        if (txnMap.get(new Long(txnId)) == null)
            throw new UnknownTransactionException();
        
        // In production, things should be persistently
        // saved.  In this sample code, we just assume
        // we did.
        return PREPARED;
            
    }
                   
    public void abort(TransactionManager mgr, long txnId) 
            throws UnknownTransactionException,
                   java.rmi.RemoteException {
                   
        System.out.println("TxnServerImpl: abort() called on " + accountName);
 
        // Check if this is something I know
        if (txnMap.get(new Long(txnId)) == null)
            throw new UnknownTransactionException();
        
        // Undo the previous operations
	for (int i = (unDoCount - 1); i >= 0; i--) {
	
	    UnDoReg undo = (UnDoReg)unDoMap.get(new Integer(i));
	    if (undo.operation == UnDoReg.UNDO_DEPOSIT){
	        balance -= undo.amount;
	    }
	    if (undo.operation == UnDoReg.UNDO_WITHDRAW){
	        balance += undo.amount;
	    }
	    unDoMap.remove(new Integer(i));
 
	}  
	
	// Remove the transaction
	Transaction txn = (Transaction) txnMap.get(new Long(txnId));
        txnIdMap.remove(txn);
        txnMap.remove(new Long(txnId));       
	
	return;         
                               
    }
                  
    public int prepareAndCommit(TransactionManager mgr,
                            long id)
            throws UnknownTransactionException,
                   java.rmi.RemoteException{

        System.out.println("TxnServerImpl: prepareAndCommit() called on " + accountName);
                   
        // Check if this is something I know
        if (txnMap.get(new Long(txnId)) == null)
            throw new UnknownTransactionException();
        
        commit(mgr, id);
        return COMMITTED;
            
    }
                            
    public void commit(TransactionManager mgr,
                   long txnId)
            throws UnknownTransactionException,
                   java.rmi.RemoteException{
 
        System.out.println("TxnServerImpl: commit() called on " + accountName);
                   
        // Check if this is something I know
        if (txnMap.get(new Long(txnId)) == null)
            throw new UnknownTransactionException();  
            
        // Remove undo records                 
	for (int i = (unDoCount - 1); i >= 0; i--) {
	
	    UnDoReg undo = (UnDoReg)unDoMap.get(new Integer(i));
	    unDoMap.remove(new Integer(i)); 
	}     
	
	// Remove the transaction
	Transaction txn = (Transaction) txnMap.get(new Long(txnId));
        txnIdMap.remove(txn);
	txnMap.remove(new Long(txnId));                
    }
    
    private void CheckIfTransactionIsNew(Transaction txn)
        throws RemoteException{
    
        ServerTransaction serverTxn = null;
                  
        try{
        
            if (txnIdMap.get(txn) == null){
                serverTxn = (ServerTransaction) txn;
                txnId = serverTxn.id;
                txnIdMap.put(txn, new Long(txnId));
                txnMap.put(new Long(txnId), txn);
 
                serverTxn.mgr.join(serverTxn.id, 
                           this,
                           crashCount);
            }
        }
        catch (UnknownTransactionException e){
             System.out.println("1111");
            e.printStackTrace();
        }
        catch (CannotJoinException e){
            e.printStackTrace();
        }
        catch (CrashCountException e){
            e.printStackTrace();
        }
        catch (RemoteException e){
            e.printStackTrace();
            throw new RemoteException();
        }
    
    }

    ///////////////////////////////////////////////////////////////
    //  End Txn related mothods
    ///////////////////////////////////////////////////////////////
     

    public static void main(String[] args) {
        String[] groupsToJoin = {""};
        /* ----------------------------------------------------------------- */
        // parse the arguments
        if (args.length < MIN_N_ARGS) {
            System.out.println("Usage: TxnServer [group,...]");
            System.out.println("    use \"all\" to discover all groups, "+
                               "\"public\" to specify group \"\"");
            System.out.println("    groups default to \"public\"");
            return;
        }
        // parse the 'groups' arguments
        int i = MIN_N_ARGS;
        int j = MIN_N_ARGS;
        if (i < args.length &&
            !args[i].startsWith("-") &&
            args[i].indexOf(File.separatorChar) < 0)
        {
            if (args[i].equals("all")) {
                groupsToJoin = LookupDiscovery.ALL_GROUPS;
            } else {
                StringTokenizer st = new StringTokenizer(args[i]," \t\n\r\f,");
                groupsToJoin = new String[st.countTokens()];
                for (int k = 0; st.hasMoreTokens(); k++) {
                    String group = st.nextToken();
                    if (group.equals("public"))
                        group = "";
                    groupsToJoin[k] = group;
                }
            }
            i++;
        }
        /* ----------------------------------------------------------------- */
       if(DEBUG == true) {
            if(groupsToJoin != LookupDiscovery.ALL_GROUPS) {
                for(i=0;i<groupsToJoin.length;i++){
                    System.out.println("groupsToJoin["+i+
                                       "] = "+groupsToJoin[i]);
                }
            } else {
                System.out.println("groupsToJoin == all");
            }
        }
        /* ----------------------------------------------------------------- */
        try {
            thisClass1 = new TxnServerImpl(SEED_MONEY1, ACCOUNT_NAME1);
            thisClass2 = new TxnServerImpl(SEED_MONEY2, ACCOUNT_NAME2);
        } catch (RemoteException e) {
            System.out.println("RemoteException while exporting this object "
                               +e.toString());
            System.exit(-1);
        }
        ((TxnServerImpl)thisClass1).setup(groupsToJoin);
        ((TxnServerImpl)thisClass1).go();                
    }

    private void setup(String[] groupsToJoin) {
        /* ----------------------------------------------------------------- */
        /* WARNING:  An RMISecurityManager must be set!
         *
         *           If a Security Manager is not set, exceptions can occur
         *           during discovery (if needed classes are downloaded),
         *           but those exceptions are "swallowed" by LookupDiscovery;
         *           so no disovery will occur, and it will appear as if
         *           nothing is wrong.
         *
         *           Also if no SecurityManager is set, RMI will prevent
         *           the loading of classes not found in the classpath; so
         *           a lookup will return null instead of the expected
         *           references to service items or attributes.
         */
        System.setSecurityManager(new RMISecurityManager());
        /* ----------------------------------------------------------------- */

        /* ----------------------------------------------------------------- */
        /* Use LookupDiscovery to find a Lookup Service */
        try {
            lookupDiscovery = new LookupDiscovery(groupsToJoin);
        } catch (IOException e) {
            System.out.println("IOException from LookupDiscovery constructor");
        }
        lookupDiscovery.addDiscoveryListener
                                   (new LookupDiscoveryListener(groupsToJoin));
        /* ----------------------------------------------------------------- */
    }

    private void go() {
        waitForDiscoveryAndRegistration();
        System.out.println("Discovery & Service Registration Complete ...");
    }

    private void waitForDiscoveryAndRegistration() {
        for(int i=0;i<N_ITER;i++) {
            try {Thread.sleep(1000);} catch (InterruptedException e) { }
            System.out.print(".");
        }
        System.out.println("\n");
    }

    /** Class whose discovered() method is invoked by threads in the
     *  LookupDiscovery class whenever a new lookup service is discovered
     */
    private class LookupDiscoveryListener implements DiscoveryListener
    {
        private String[] groupsToJoin;
        public LookupDiscoveryListener(String[] groupsToJoin) {
            super();
            this.groupsToJoin = groupsToJoin;
        }
        public void discovered(DiscoveryEvent evnt) {
            System.out.println("LookupDiscoveryListener:  discovered...");
            ServiceRegistrar[] regs = evnt.getRegistrars();
            /* Note: the tendancy here is to immediately try to register
             *       the service after discovering a lookup. But this
             *       registration should be done in a separate thread
             *       because the registration involves a remote call
             *       which may prevent/impede the LookupDiscovery class,
             *       which invokes this method, from returning to its
             *       primary duty of discovering other lookups.
             */
            Thread registerThread = new RegisterThread(regs,groupsToJoin);
            registerThread.start();
        }
        public void discarded(DiscoveryEvent evnt) {
            /* Note: once a lookup service is discovered, there is no on-going
             *       communication between LookupDiscovery and a discovered
             *       lookup service. This means that if a lookup service goes
             *       away, there will be no automatic notification that such
             *       an event has occurred.
             *
             *       Thus, if a client or service attempts to use a lookup
             *       service but receives a RemoteException as a result of
             *       that attempt, it is the responsibility of the client or
             *       service to invoke the discard method of the
             *       LookupDiscovery object instantiated by the client or
             *       service. Doing so will flush the lookup service from the
             *       LookupDiscovery object's cache; causing this method
             *       to be invoked.
             */
            System.out.println("LookupDiscoveryListener: discarded ...");
            /* Retrieve the discarded lookup service(s) from the event */
            ServiceRegistrar[] regs = evnt.getRegistrars();
            for(int i=0; i<regs.length;i++){
                System.out.println("  Discarded Lookup: "+regs[i]);
            }
        }
    }

    /** Daemon thread in which the service is registered with lookup.
     *
     *  A new instance of this thread is created each time a new lookup
     *  belonging to a group-of-interest is discovered.
     *
     *  Rather than performing the service registration in the
     *  DiscoveryListener's discovered() method, that registration is
     *  performed in this thread. It is very important that the
     *  discovered() method do whatever it needs to do and return
     *  as quickly. This is why a potentially time-consuming task such
     *  registering a service with lookup (a remote invocation) is
     *  performed in this separate thread.
     *
     *  Upon completion of the registration process, this thread will exit.
     */
    private class RegisterThread extends Thread {
        private ServiceRegistrar[] regs;
        private String[] groupsToJoin;
        public RegisterThread(ServiceRegistrar[] regs, String[] groupsToJoin) {
            super("registerThread");
            setDaemon(true);
            this.regs = regs;
            this.groupsToJoin = groupsToJoin;
        }

        public void run() {
            for(int i=0; i<regs.length;i++){
                boolean registerSrvc = false;
                try {
                    try {
                        /* Retrieve the groups of all the discovered lookups */
                        String[] regGroups = regs[i].getGroups();
                        LookupLocator loc = regs[i].getLocator();
                        String loc_str = loc.toString();
                        /* Look for only lookups in groups that were input */
                        System.out.println("  Lookup on host "+loc_str+":");
                        for(int j=0; j<regGroups.length; j++) {
                            if(regGroups[j].compareTo(PUBLIC_GROUP) == 0) {
                                System.out.println("    regGroups["+j+
                                                 "] belongs to Group: public");
                            } else {
                                System.out.println("    regGroups["+j+
                                          "] belongs to Group: "+regGroups[j]);
                            }
                            if(groupsToJoin != null) {
                                for(int k=0; k<groupsToJoin.length; k++) {
                                    if(regGroups[j].compareTo
                                                       (groupsToJoin[k]) == 0){
                                        registerSrvc = true;
                                    }
                               }
                            } else {
                                registerSrvc = true;
                            }
                        }
                    } catch (java.security.AccessControlException e) {
                        /* Connection Disallowed */
                        System.out.println
                                     ("    Security Restriction: Policy"
                                      +" file of discovered Lookup"
                                      +"\n                         "
                                      +" does not allow service registrations,"
                                      +"\n                         "
                                      +" service lookups, or other remote"
                                      +"\n                         "
                                      +" invocations from the current host");
                    }
                } catch (RemoteException e) {
                    System.out.println
                     ("RemoteException on call to getLocator() "+e.toString());
                }
                if(registerSrvc == true) {
                    registerItem(thisClass1,regs[i], ACCOUNT_NAME1);
                    registerItem(thisClass2,regs[i], ACCOUNT_NAME2);
                }
            }
        }

        private ServiceRegistration registerItem(Object service,
                                                 ServiceRegistrar lookupSrvc,
                                                 String name)
        {
            /* Create a ServiceItem from the service instance */
            serviceAttrs[serviceAttrs.length-1] = new Name(name);
            ServiceItem srvcItem = new ServiceItem(null,service, serviceAttrs);
            /* Register the Service with the Lookup Service */
            ServiceRegistration srvcRegistration = null;
            try {
                srvcRegistration = lookupSrvc.register(srvcItem, LEASE_DUR_WITH_LOOKUP);
                lookupSrvcVec.add(lookupSrvc);
                srvcLeaseVec.add(srvcRegistration.getLease());
                System.out.println("Registered ServiceID:  "
                                +(srvcRegistration.getServiceID()).toString());
            } catch (RemoteException e) {
                System.out.println("RemoteException while registering the "
                                   +"Service: "+service+"\n"+e.toString());
                lookupDiscovery.discard(lookupSrvc);
                return null;
            }
            return srvcRegistration;
        }
    }

}
