/*************************************************************
 *
 *  GridBagParsedConstraints
 *
 *  class to simplify using the GridBagLayout manager and make the
 *  application code more readable
 *
 *  which accepts a
 *  string representation of the constraints. For example a call which looks like:
 *
 * all other parameters will take default values
 *
 * magic string tokens:
 *  w = gridwidth, takes an int or one of the tokens rem (REMAINDER) or rel (RELATIVE)
 *  h = gridheight, ditto
 *  x = gridx, takes an int or the token rel
 *  y = gridy, ditto
 *  f = fill, takes one of the tokens none hor (HORIZONTAL) ver (VERTICAL) both
 *  a = anchor, takes one of n s e w ne nw se sw c for the compass point
 *  ix= ipadx, takes an int
 *  iy= ipady, takes an int
 *  in= insets, takes four ints separated by commas
 *  wx= weightx, takes an int or a float
 *  wy= weighty, ditto
 *
 * a parse error throws a generic error. This makes error detection
 * simple during development, without cluttering up the code with
 * try/catch blocks to catch exceptions which would never occur in production
 *
 * Author: Brian Jeltema
 *
 ********************************************************************************/

import java.awt.GridBagConstraints ;
import java.util.StringTokenizer ;
import java.util.NoSuchElementException ;
import java.awt.Insets ;

class GridBagParsedConstraints extends GridBagConstraints {

  static final boolean debug = false ;

  GridBagParsedConstraints(String constraintString) {
    super() ;
    StringTokenizer stok = new StringTokenizer(constraintString) ;

    try {
      while (stok.hasMoreTokens()) {
	StringTokenizer thisTok = new StringTokenizer(stok.nextToken(),"=") ;
	String constraint = thisTok.nextToken() ;
	String constraintValue = thisTok.nextToken() ;

	if (constraint.equals("x")) {
	  if (constraintValue.equals("rel")) 
	    gridx = GridBagConstraints.RELATIVE ;
	  else
	    gridx = Integer.parseInt(constraintValue) ;
	  continue ;
	}

	if (constraint.equals("y")) {
	  if (constraintValue.equals("rel")) 
	    gridy = GridBagConstraints.RELATIVE ;
	  else
	    gridy = Integer.parseInt(constraintValue) ;
	  continue ;
	}

	if (constraint.equals("w")) {
	  if (constraintValue.equals("rel"))
	    gridwidth = GridBagConstraints.RELATIVE ;
	  else if (constraintValue.equals("rem"))
	    gridwidth = GridBagConstraints.REMAINDER ;
	  else
	    gridwidth = Integer.parseInt(constraintValue) ;
	  continue ;
	}

	if (constraint.equals("h")) {
	  if (constraintValue.equals("rel")) 
	    gridheight = GridBagConstraints.RELATIVE ;
	  else if (constraintValue.equals("rem"))
	    gridheight = GridBagConstraints.REMAINDER ;
	  else
	    gridheight = Integer.parseInt(constraintValue) ;
	  continue ;
	}

	if (constraint.equals("f")) {
	  if (constraintValue.equals("none")) 
	    fill = GridBagConstraints.NONE ;
	  else if (constraintValue.equals("ver"))
	    fill = GridBagConstraints.VERTICAL ;
	  else if (constraintValue.equals("hor"))
	    fill = GridBagConstraints.HORIZONTAL ;
	  else if (constraintValue.equals("both"))
	    fill = GridBagConstraints.BOTH ;
	  else throw new Error("Parse error: " + constraintString) ;
	  continue ;
	}

	if (constraint.equals("ix")) {
	  ipadx = Integer.parseInt(constraintValue) ;
	  continue ;
	}

	if (constraint.equals("iy")) {
	  ipady = Integer.parseInt(constraintValue) ;
	  continue ;
	}

	if (constraint.equals("in")) {
	  StringTokenizer inVals = new StringTokenizer(constraintValue,",") ;
	  insets = new Insets(
            Integer.parseInt(inVals.nextToken()),
            Integer.parseInt(inVals.nextToken()),
            Integer.parseInt(inVals.nextToken()),
            Integer.parseInt(inVals.nextToken())) ;
	  continue ;
	}

	if (constraint.equals("a")) {
	  if (constraintValue.equals("n"))
	    anchor = GridBagConstraints.NORTH ;
	  else if (constraintValue.equals("s"))
	    anchor = GridBagConstraints.SOUTH ;
	  else if (constraintValue.equals("e"))
	    anchor = GridBagConstraints.EAST ;
	  else if (constraintValue.equals("w"))
	    anchor = GridBagConstraints.WEST ;
	  else if (constraintValue.equals("ne"))
	    anchor = GridBagConstraints.NORTHEAST ;
	  else if (constraintValue.equals("nw"))
	    anchor = GridBagConstraints.NORTHWEST ;
	  else if (constraintValue.equals("se"))
	    anchor = GridBagConstraints.SOUTHEAST ;
	  else if (constraintValue.equals("sw"))
	    anchor = GridBagConstraints.SOUTHWEST ;
	  else if (constraintValue.equals("c"))
	    anchor = GridBagConstraints.CENTER ;
	  else throw new Error("Parse error: " + constraintString) ;
	  continue ;
	}

	if (constraint.equals("wx")) {
	  Float f = new Float(constraintValue) ;
	  weightx = f.floatValue() ;
          continue ;
	}

	if (constraint.equals("wy")) {
	  Float f = new Float(constraintValue) ;
	  weighty = f.floatValue() ;
          continue ;
	}
	throw new Error("Parse error: " + constraintString) ;
      }

    }

    catch (NumberFormatException e) {
      throw new Error("Parse error: " + constraintString) ;
    }

    catch (NoSuchElementException e) {
      throw new Error("Parse error: " + constraintString) ;
    }

    if (debug) {
      System.out.println("constraints:") ;
      System.out.println("x = " + gridx) ;
      System.out.println("y = " + gridy) ;
      System.out.println("w = " + gridwidth) ;
      System.out.println("h = " + gridheight) ;
      System.out.println("f = " + fill) ;
      System.out.println("ix = " + ipadx) ;
      System.out.println("iy = " + ipady) ;
      System.out.println("in = " + insets) ;
      System.out.println("a = " + anchor) ;
      System.out.println("wx = " + weightx) ;
      System.out.println("wy= " + weighty) ;
    }

  }
}
