/*
 * This source code (.java) file is Copyright  2000 Bill Venners. All rights reserved. 
 * This file accompanies the Jini Service UI Draft Specification, written by Bill
 * Venners and published on the World Wide Web at:
 * 
 *     http://www.artima.com/jini/serviceui/DraftSpec.html,
 *
 * This source file may not be copied, modified, or redistributed EXCEPT as allowed
 * by the following statements: From February 7, 2000 through March 6, 2000, you may
 * copy and/or modify these files to test and experiment with the Service UI API,
 * described in the Jini Service UI Draft Specification. Any bug fixes must be given
 * back to Bill Venners. You may not redistribute this file or any binary (such
 * as .class) files generated from this file. You may not distribute modified versions
 * this files or any binary (such as .class) files generated from modified versions of
 * this file. You may not remove this copyright notice. You may not use this file in
 * printed media without the express permission of Bill Venners. And if that weren't
 * enough, you must destroy all copies of this file, and any binary (such as
 * .class) files generated from this file, by March 7, 2000.
 *
 * BILL VENNERS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THIS
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR PARTICULAR PURPOSE, OR NON-INFRINGEMENT.
 * BILL VENNERS SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY A LICENSEE AS A RESULT
 * OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
*/
package net.jini.lookup.ui.attribute;

import java.util.Locale;
import java.util.Iterator;
import java.util.Set;
import java.util.List;
import java.util.Collections;

/**
 * UI attribute that lists the locales supported
 * by a generated UI.
*/
public class Locales implements java.io.Serializable {

    private Set supportedLocales;

    /**
     * Constructs a <CODE>Locales</CODE> using the
     * passed <CODE>Set</CODE>. The <CODE>Set</CODE> can
     * be mutable or immutable, and must contain only
     * <CODE>java.util.Locale</CODE> objects. Each <CODE>Locale</CODE> must
     * represent a locale that is supported by the UI generated
     * by the UI factory stored in marshalled form in the
     * same <CODE>UIDescriptor</CODE>.
     * 
     * @param locales A <CODE>Set</CODE> of <CODE>Locale</CODE>
     * objects. Each element must be non-null and an instance
     * of <CODE>java.util.Locale</CODE>.
     * 
     * @throws IllegalArgumentException if <CODE>locales</CODE> 
     * is <CODE>null</CODE> or any element of <CODE>locales</CODE>
     * set is either <CODE>null</CODE>
     * or not an instance of <CODE>java.util.Locale</CODE>.
    */
    public Locales(Set locales) {
        
        if (locales == null) {
            throw new IllegalArgumentException();
        }

        Iterator it = locales.iterator();
        while (it.hasNext()) {
            Object o = it.next();
            if ((o == null) || (!(o instanceof Locale))) {
                throw new IllegalArgumentException();
            }
        }

        supportedLocales = locales;
    }

    /**
     * Indicates whether or not a locale is supported
     * by the UI generated by the UI factory stored in
     * the marshalled object of the same <CODE>UIDescriptor</CODE>.
     * This method returns <CODE>true</CODE> only if the passed
     * <CODE>Locale</CODE> exactly matches a <CODE>Locale</CODE>
     * supported by the UI, as defined by the <CODE>equals()</CODE>
     * method of class <CODE>java.util.Locale</CODE>. For example, imagine the UI supports the
     * "en" (General English) locale, but not the "en_US" (US English) locale. 
     * Then, if "en_US" is passed to this method, the method will return
     * <CODE>false</CODE>.
    */
    public boolean isLocaleSupported(Locale locale) {

        if (locale == null) {
            throw new IllegalArgumentException();
        }

        return supportedLocales.contains(locale);

    }

    /**
     * Looks through the passed array of <CODE>Locale</CODE>s
     * (in the order they appear in the array)
     * and returns the first <CODE>Locale</CODE> that is
     * supported by the UI (as defined by <CODE>isLocaleSupported()</CODE>),
     * or <CODE>null</CODE>, if none of the <CODE>Locale</CODE>s in
     * the passed array are supported by the UI.
    */
    public Locale getFirstSupportedLocale(Locale[] locales) {

        if (locales == null) {
            throw new IllegalArgumentException();
        }

        for (int i = 0; i < locales.length; ++i) {
            if (supportedLocales.contains(locales[i])) {
                return locales[i];
            }
        }

        return null;
    }

    /**
     * Iterates through the passed <CODE>List</CODE> of <CODE>Locale</CODE>s
     * and returns the first <CODE>Locale</CODE> that is
     * supported by the UI (as defined by <CODE>isLocaleSupported()</CODE>),
     * or <CODE>null</CODE>, if none of the <CODE>Locale</CODE>s in
     * the passed array are supported by the UI.
    */
    public Locale getFirstSupportedLocale(List locales) {

        if (locales == null) {
            throw new IllegalArgumentException();
        }

        for (int i = 0; i < locales.size(); ++i) {
            if (supportedLocales.contains(locales.get(i))) {
                try {
                    return (Locale) locales.get(i);
                }
                catch (ClassCastException cce) {
                    // This should never happen given
                    // the checks in the constructor.
                    continue;
                }
            }
        }

        return null;
    }

    /**
     * Returns an iterator over the set of <CODE>java.util.Locale</CODE>
     * objects, one for each locale supported
     * by the UI generated by the UI factory stored in
     * the marshalled object of the same <CODE>UIDescriptor</CODE>.
     * The returned <CODE>Iterator</CODE> does not support
     * <CODE>remove()</CODE>.
    */
    public Iterator iterator() {

        // Make sure the iterator that gets passed back
        // doesn't allow client to change the Set.
        return new Iterator() {

            Iterator it = supportedLocales.iterator();

            public boolean hasNext() {
                return it.hasNext();
            }

            public Object next() {
                return it.next();
            }

            public void remove() {
                throw new UnsupportedOperationException();
            }
        };
    }

    /**
     * Returns an unmodifiable <CODE>java.util.Set</CODE> that contains
     * <CODE>java.util.Locale</CODE> objects, one for each locale supported
     * by the UI generated by the UI factory stored in
     * the marshalled object of the same <CODE>UIDescriptor</CODE>.
    */
    public Set getLocales() {

        return Collections.unmodifiableSet(supportedLocales);
    }
}


