#if !defined __FXT_H
#define      __FXT_H

#include "types.h"   // for ulong

// for pfafft() see pfafft.h !

// for modfft() (=ntt) see mod/mfft.h


// General format of arguments: (double *f, ulong ldn)
// f := pointer to data array, 
// ldn := base 2 log of array length (length n=2**n)
//
// A funktion some_func0(...) (note the '0')
// expects input data with higher half zeros.
//
// Data in the arrays is generally changed. 

//------- HARTLEY transforms --------------------------

// fht.cc:
void fht(double *f, ulong ldn);

// fht0.cc:
void fht0(double *f, ulong ldn);

// fhtdit2.cc:
//void dit2_fht(double *f, ulong ldn);

// fhtslow.cc:
void slow_ht(double *f, ulong n);


//------- FOURIER transforms --------------------------

// typical format of arguments: (double *fr, double *fi, ulong ldn, int is)
// fr := pointer to data array (real part), 
// fi := pointer to data array (imag part), 
// ldn := base 2 log of array length
// is := sign of exponent in fourier kernel

// naming (semi-) conventions:
// blabla_fft() := fft implementation with blabla-algorithm
// blabla_fft0() := fft for zero paddad data 
//             (i.e. second half of the data is zero)

// fftfht.cc:
void fht_fft(double *fr, double *fi, ulong ldn, int is);
void fht_fft0(double *fr, double *fi, ulong ldn, int is);

// fourstep.cc:
void four_step_fft(double *fr, double *fi, ulong ldn, int is);
void four_step_fft0(double *fr, double *fi, ulong ldn, int is);

// fftdif4.cc:
void dif4_fft(double *fr, double *fi, ulong ldn, int is);

// fftdit4.cc:
void dit4_fft(double *fr, double *fi, ulong ldn, int is);

// fftduh.cc:
void duhamel_fft(double *fr, double *fi, ulong ldn, int is);

//------- REAL FOURIER ---------------------

// info about the ordering of the data is in each source file

// rfftfht.cc: 
void fht_real_complex_fft(double *f, ulong ldn);
void fht_real_complex_fft0(double *f, ulong ldn);
void fht_complex_real_fft(double *f, ulong ldn);

// rfftwrap.cc:
void wrap_real_complex_fft(double *f, ulong ldn);
void wrap_real_complex_fft0(double *f, ulong ldn);
void wrap_complex_real_fft(double *f, ulong ldn);

// rfftsplr.cc:
void split_radix_real_complex_fft(double *x, ulong ldn);
void split_radix_real_complex_fft0(double *x, ulong ldn);

// rfftsplc.cc:
void split_radix_complex_real_fft(double *x, ulong ldn);


//-------- REAL CONVOLUTION ---------------------
// fhtcnvl.cc:
void fht_convolution(double *f, double *g, ulong ldn);
void fht_convolution0(double *f, double *g, ulong ldn);

// fhtcnvla.cc:
void fht_auto_convolution(double *f, ulong ldn);  
void fht_auto_convolution0(double *f, ulong ldn);  

// fhtcnvl2.cc:
void fht_auto_convolution0_2proc(double *f, ulong ldn);  

// fftcnvl.cc:
void fht_fft_convolution(double *f, double *g, ulong ldn);
void wrap_fft_convolution(double *f, double *g, ulong ldn);
void split_radix_fft_convolution(double *f, double *g, ulong ldn);
void fht_fft_convolution0(double *f, double *g, ulong ldn);
void wrap_fft_convolution0(double *f, double *g, ulong ldn);
void split_radix_fft_convolution0(double *f, double *g, ulong ldn);

// fftcnvla.cc:
void fht_fft_auto_convolution(double *f, ulong ldn);
void wrap_fft_auto_convolution(double *f, ulong ldn);
void split_radix_fft_auto_convolution(double *f, ulong ldn);
void fht_fft_auto_convolution0(double *f, ulong ldn);
void wrap_fft_auto_convolution0(double *f, ulong ldn);
void split_radix_fft_auto_convolution0(double *f, ulong ldn);

// slowcnvl.cc:
void slow_auto_convolution(double *f, ulong nu);
void slow_auto_convolution0(double *f, ulong nu);
void slow_convolution(double *w, double *f, ulong nu);
void slow_convolution0(double *w, double *f, ulong nu);


//-------- REAL CORRELATION ---------------------
// fhtcorr.cc:
void fht_correlation0(double *f, double *g, ulong ldn);
void fht_auto_correlation0(double *f, ulong ldn);

// fftcorr.cc:
void fft_correlation0(double *f, double *g, ulong ldn);
void fft_auto_correlation0(double *f, ulong ldn);

// slowcorr.cc:
void slow_correlation(double *w, double *f, ulong nu);
void slow_correlation0(double *w, double *f, ulong nu);
void slow_auto_correlation(double *f, ulong nu);
void slow_auto_correlation0(double *f, ulong nu);


//-------- COmplex COnvolution & COrrelation ---------------------
// cococo.cc:
void fft_complex_convolution(double *f, double *g, double *r, double *s, ulong ldn);
void fft_complex_auto_convolution(double *f, double *g, ulong ldn);
void fft_complex_correlation(double *f, double *g, double *r, double *s, ulong ldn);
void fft_complex_auto_correlation(double *f, double *g, ulong ldn);

// slowcoco.cc:
void slow_complex_convolution(double *wr, double *wi, double *fr, double *fi, ulong n);
void slow_complex_auto_convolution(double *fr, double *fi, ulong n);
void slow_complex_correlation(double *wr, double *wi, double *fr, double *fi, ulong n);
void slow_complex_auto_correlation(double *fr, double *fi, ulong n);


//-------- SPECTRUM ------------------------
// fhtspect.cc:
void fht_spectrum(double *f, ulong ldn);

// fftspect.cc:
void fft_spectrum(double *f, ulong ldn);


// ----- LEARNERS STRAIGHT & SIMPLE VERSIONS: ----
// even simpler (plain C) stuff is in simplfft/
//
// fftdif2l.cc: 
void dif2l_fft(double *fr, double *fi, ulong ldn, int is);
//
// fftdit2l.cc:
void dit2l_fft(double *fr, double *fi, ulong ldn, int is);
//
// fftdif4l.cc:
void dif4l_fft(double *fr, double *fi, ulong ldn, int is);
//
// fftdit4l.cc:
void dit4l_fft(double *fr, double *fi, ulong ldn, int is);



//-------- OTHER FOURIER STUFF ------------------
// fftslow.cc:
void slow_ft(double *fr, double *fi, ulong n, int is);

// fftchirp.cc: (arbitrary length fft)
void chirp_fft(double *fr, double *fi, ulong n, int is);

// fftfract.cc: (fractional fft)
void frft(double *x, double *y, ulong n, int is, double v);

// fftndim.cc: (2...5 dim fft)
void fft_ndim(double *fr, double *fi, ulong ndim, ulong *ldn, int is);


//---------- WRAPPERS FOR TYPE COMPLEX FOURIER ----------

/*
#include <complex.h>
#if defined __COMPLEXT__
#define Complex complex<double>
#endif

// cfftwrap.cc:
void complex_fft(Complex *c, int ldn, int is);
void real_imag_fft(double *fr, double *fi, int ldn, int is);
*/

//---------- WALSH TRANSFORM ----------

// walshdif2.cc:
void dif2_walsh_wak(double *f, ulong ldn);
void dif2_walsh_pal(double *f, ulong ldn);
void dif2_walsh_wal(double *f, ulong ldn);

// walshdit2.cc:
void dit2_walsh_wal(double *f, ulong ldn);
void dit2_walsh_wal0(double *f, ulong ldn);
void dit2_walsh_pal(double *f, ulong ldn);
void dit2_walsh_pal0(double *f, ulong ldn);
void dit2_walsh_wak(double *f, ulong ldn);


// walshndim.cc: (2...5 dim walsh)
void walsh_ndim(double *f, ulong ndim, ulong *ldn);

// dyadiccnvl.cc:
void dyadic_convolution(double *f, double *g, ulong ldn);
void dyadic_auto_convolution(double *f, ulong ldn);
void dyadic_convolution0(double *f, double *g, ulong ldn);
void dyadic_auto_convolution0(double *f, ulong ldn);


//-------------------------------------------------


/*
// following moved to the 'bucket' directory:

// cssnfht.cc:
void fht_cosine_transform(double *f, ulong ldn);
void fht_sine_transform(double *f, ulong ldn);

// rspltfft.cc:
void split_radix_real_complex_fft(double *x, ulong ldn);

// fhttable.cc:
void table_fht(double *f, ulong ldn);

// fftdif4x.cc: (same as fftdif4.cc without the sumdiffs)
void dif4x_fft(double *fr, double *fi, ulong ldn, int is);

// fftdit4x.cc: (same as fftdit4.cc without the sumdiffs)
void dit4x_fft(double *fr, double *fi, ulong ldn, int is);

// fftdif2.cc:
void dif2_fft(double *fr, double *fi, ulong ldn, int is);

// fftdit2.cc:
void dit2_fft(double *fr, double *fi, ulong ldn, int is);

// fftdif4c.cc:  (uses type complex, optimized, but still slow)
void dif4c_fft(double *fr, double *fi, ulong ldn, int is);

// fftdit4c.cc: (uses type complex, optimized, but still slow)
void dit4c_fft(double *fr, double *fi, ulong ldn, int is);

// fftdif20.cc:
void dif2_fft0(double *fr, double *fi, ulong ldn, int is);

*/


#endif // !defined __FXT_H
