/**********************************************************
*
* This ANSI C-Program computes the p-th hexadecimal digit
* of pi  (plus the following 7 digits).
*
* p comes from the command line.
* 0 <= p <= 2^29 (approx 500 million).
* p=1 means the first digit after the radix point.
*
* The program uses the BBP-algorithm of
* D. Bailey, P. Borwein und S. Plouffe.
* Its central function is expm() which performs the
* "binary modulo exponentation".
*
* After D. Bailey 960429
*
**********************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <float.h>

/* Limits:
   d_max = p_max - 1
   ak_max = 8 * d_max + 6
   r_max = ak_max - 1
   r_max^2 < 2^(LDBL_MANT_DIG).
   With mantissa length = 2m (usually 2m = 64):
   r_max  <= 2^m-1
   ak_max <= 2^m
   d_max  < 2^(m-3)
   p_max  < 2^(m-3)
*/
#define p_max  pow(2, LDBL_MANT_DIG*0.5 - 3)

/**********************************************************
*
*       expm(long n, double ak)
*
*  Computes 16^n mod ak using binary modulo exponentation.
*
**********************************************************/

double expm(long n, double ak)
{
   long double r = 16.0;
   long nt;

   if (ak == 1) return 0.;
   if (n == 0)  return 1;
   if (n == 1)  return fmod(16.0, ak);

   /*  nt: largest power of 2 not greater han n/2. */
   for (nt=1; nt <= n; nt <<=1)
      ;
   nt >>= 2;

   /*  Binary exponentiation modulo ak. */
   do
   {
      r = fmod/*l*/(r*r, ak);
      if ((n & nt) != 0)
      r = fmod/*l*/(16.0*r, ak);
      nt >>= 1;
   } while (nt != 0);
   return r;
}

/**********************************************************
*
*            series(double m, long d)
*
*  computes sum_k {16^(d-k)mod(8k+m)}/(8k+m),
*                          (k=0, 1, 2, ..., d-1)
*
**********************************************************/

double series(double m, long d)
{
   long   k;
   double ak = m;
   double s = 0., t = 1., x;

   /*  sum of terms 0..d-1 */
   for (k = 0; k < d; k++)
   {
      x = expm (d-k, ak);
      s += x / ak;
      s = fmod(s, 1.0);
      ak += 8.0;
   }

   /* Some additional terms for 8 hex digit accurracy */
   #define eps 0.25e-12 /* = 16^{-10} / 4 */

   while ( (x=t/ak) > eps)
   {
      s += x;
      t /= 16.0;
      ak += 8.0;
   }
   return s;
}

/**********************************************************
*
*              main()
*
**********************************************************/

#pragma argsused
int main(int argc, char *argv[])
{
   clock_t t_beg, t_end;
   double s;
   int p;

   printf("This program computes the p-th hexadecimal digit of pi.\n");
   printf("It uses the Bailey-Borwein-Plouffe (BBP) algorithm (1995).\n");
   do
   {
      /* try p on cmdline */
      if (argc > 1)
         p = atoi(argv[1]);
      else
      {
         /* Get p from user */
         printf("\nEnter n (or -1 to exit): ");
         scanf("%d", &p);
      }
      if (p < 0)
     	   break;
     	if (p < 1) p = 1;
      if (p > p_max) p = p_max;
   	printf("Hex digits %d to %d of pi: ", p, p+7);
   	t_beg=clock();

   	s =  4*series (1, p-1)     /*  4, 8i+1 */
      	- 2*series (4, p-1)     /* -2, 8i+4 */
      	-   series (5, p-1)     /* -1, 8i+5 */
      	-   series (6, p-1);    /* -1, 8i+6 */
   	s += 4;        /* ensure s >= 0 */

   	t_end=clock();
   	printf("%08lX\n", (unsigned long)(s*pow(2, 32)));
   	printf("Elapsed time was: %.1f sec.\n",
			                 1.0*(t_end-t_beg)/CLOCKS_PER_SEC);
   } while (argc <= 1);
	return 0;
}
