/**********************************************************
*
* This ANSI C-Programm computes the p-th hexadecimal digit
* of pi (plus the following 7 digits).
*
* p comes from the command line.
* 3 <= p <= 2^30 (approx 1 billion).
* p=1 means the first digit after the radix point.
*
* The program uses the algorithm of Fabrice Bellard.
* pi = (-32*S1-S2+256*S3-64*S4-4*S5-4*S6+S7) / 2^6.
* S1 = \sum_{k=0}^\infty (-1)^k / (1024^k * ( 4*k+1))
* S2 = \sum_{k=0}^\infty (-1)^k / (1024^k * ( 4*k+3))
* S3 = \sum_{k=0}^\infty (-1)^k / (1024^k * (10*k+1))
* S4 = \sum_{k=0}^\infty (-1)^k / (1024^k * (10*k+3))
* S5 = \sum_{k=0}^\infty (-1)^k / (1024^k * (10*k+5))
* S6 = \sum_{k=0}^\infty (-1)^k / (1024^k * (10*k+7))
* S7 = \sum_{k=0}^\infty (-1)^k / (1024^k * (10*k+9))
*
* It uses the "BBP"-formula of Fabrice Bellard.
*
**********************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <float.h>

typedef unsigned long ULONG;

/*  Limits:
   d_max = p_max - 1
   dq_max = (4*d_max-6) / 10
   ak_max = 10 * dq_max + 9
   r_max = ak_max - 1
   r_max^2 < 2^(Mantissenlaenge von LONGDOUBLE).
   Mit 2m = 64 ergibt sich:
   r_max <= 2^32-1
   ak_max <= 2^32
   dq_max < 2^32 / 10
   4*d_max - 6 < 2^32
   d_max < 2^30
   p_max <=  2^30
*/
#define p_max  pow(2, LDBL_MANT_DIG*0.5 - 3)

/**********************************************************
*
*       expm(ULONG n, ULONG ak, ULONG base)
*
*  Computes 16^n mod ak using binary modulo exponentation.
*
**********************************************************/

double expm(ULONG k, double ak)
{
   long double r = 1024.;
   ULONG nt;

   if (ak == 1)   return 0.;
   if (k == 0)    return 1.;
   if (k == 1)    return fmod(1024, ak);

   /*  nt: largest power of 2 not greater than n/2. */
   for (nt=1; nt <= k; nt <<=1)
      ;
   nt >>= 2;

   /*  Binary modulo exponentiation modulo ak. */
   do
   {
      r = fmod/*l*/(r*r, ak);
      if ((k & nt) != 0)
	      r = fmod/*l*/(r*1024.0, ak);
      nt >>= 1;
   } while (nt != 0);
   return (double)r;
}

/**********************************************************
*
*  series(double p, double q, double base, double eps, int sno)
*
*  Computes the (sno-th) series
*  sum_k {(-1)^k * 1024^(dq-k)mod(pk+q)}/(pk+q),
*                          (k=0, 1, 2, ..., dq-1)
*
**********************************************************/

double series(double p, double q, double eps, ULONG dq, int sno)
{
   ULONG k;
   double ak = q;
   double s = 0.0, t=1.0, x;
   int sign = 0;

   /*  Sum of the terms 0..d-1 */
   for (k = 0; k < dq; ++k ) {
      /* Progress indicator: */
      /*
      if ( (k & 4095) == 0 )
         printf("%2ld%%\b\b\b", sno*14+(k*14)/dq);
      */
      /* Modulo Exponentation */
      x = expm(dq-k, ak);
      x /= ak;
      s += sign == 0 ? +x : -x;
      s = fmod(s, 1);
      sign = 1 - sign;
      ak += p;
   }

   /* Some additional terms for 8 hex digits accuracy */
   while ( (x=t/ak) > eps)
   {
      s += sign == 0 ? +x : -x;
      sign = 1 - sign;
      t /= 1024;
      ak += p;
   }
   return s;
}

/**********************************************************
*
*              main()
*
**********************************************************/

/* Constants in Bellards formula */
double F[]    = {-32, -1, 256, -64, -4, -4,  1};
double P[]    = {  4,  4,  10,  10, 10, 10, 10};
double Q[]    = {  1,  3,   1,   3,  5,  7,  9};

#pragma argsused
int main(int argc, char *argv[])
{
        clock_t  t_beg, t_end;
        double  p;              /* default p */
        double  S=0.0, PS;
        int     i;
        double  eps, eps0;
        ULONG   d, dq, dr;

	printf("bell16: computes the n'th hexadecimal digit of pi.\n");
	printf("It uses a BBP series of Fabrice Bellard.\n");
	do
	{
                // try cmdline
	        if (argc > 1)
	                sscanf(argv[1], "%lf", &p);
	        else { // ask user
                        printf("\nEnter n (or -1 to exit): ");
                        scanf("%lf", &p);
	        }
	        if (p < 0) break;
   	        p = floor(p);
   	        if (p < 3.0)   p = 3;   /* p > 2 for 4*(p-1) >= 6 */
   	        if (p > p_max) p = p_max;
   	        d  = p-1;               /* Shift of 64*pi by 1024^d */
   	        dq = (4*d-6) / 10;
   	        dr = (4*d-6) % 10;
   	        /* Basic accuracy: 8*4+dr + 4 binary digits */
   	        eps0 = 1.0 / pow(2.0, dr + 40.0);

   	        printf("Hexadecimal digits %.0f to %.0f of pi: ", p, p+7);
   	        t_beg=clock();
   	        for (i=0; i < 7; ++i)
   	        {
      	                /* Adjust accuracy to series factor */
                        eps = eps0 / F[i];
                        PS = F[i] * series(P[i], Q[i], fabs(eps), dq, i);
      	                PS = fmod(PS, 1.0);
      	                S += PS;
   	        }
   	        S += 7.0;  /* ensure S >= 0 */
   	        S = fmod(S, 1.0);
   	        S *= pow(2, dr);   /* Shift by (4d-6) % 10 digits */
   	        t_end=clock();
   	        printf("%08lX\n", (ULONG)(S*pow(2, 32)));
   	        printf("Elapsed time was: %.1f sec.\n",
			    	1.0*(t_end-t_beg)/CLOCKS_PER_SEC);
        } while (argc <= 1);
        return 0;
}
