\section{\Digit}
%===============

@d included files of the file \texttt{"{}digit.h"{}}
@{#ifdef _Old_STD_
@<standard system include file \texttt{<stdlib.h>}@>
#include <iostream.h>
#include <ctype.h>
#include <assert.h>
@<standard system include file \texttt{<limits.h>}@>

# define bool int
# define true 1
# define false 0

#else
// New standard

# include <iostream>
# include <cstdlib>
# include <cctype>
# include <cassert>
# include <climits>

# include <algorithm>
# include <utility>

using namespace std;

#endif
@}

@d macro definitions of the file \texttt{"{}digit.h"{}}
@{#define _DigitAsm_          // use assembler if ever it is possible
//#define _Piologie_Debug_    // Debug modus for assert
@| _DigitAsm_ _Piologie_Debug_ @}


@d declarations of error-handling
@{// error-handling:
typedef void (*piologie_error_handler_t)(const int, const char*);
piologie_error_handler_t set_piologie_error_handler(piologie_error_handler_t);
@}


@D function prototypes of the file \texttt{"{}digit.h"{}}
@{#ifndef STATIC_VS_INLINE        // some compiler have problems with inline
#define STATIC_VS_INLINE inline
#endif

STATIC_VS_INLINE Digit log2(Digit);
Digit pow10(size_t);
inline void  swap(Digit&, Digit&);
inline Digit sqrt(Digit, Digit);
Digit sqrt(Digit);
void  sqrt(Digit, Digit&, Digit&);
Digit sqrt(Digit, Digit, Digit&, Digit&);
Digit gcd(Digit, Digit);
@}

@d constants of \Piologie\
@{#ifdef DEFINE_VS_CONST        // some compiler have problems with const
# define BETA (sizeof(Digit)*CHAR_BIT)
#else
@<constant $\beta$@>
#endif
@<constant $\gamma$@>
@<constant $\gamma_{low}$ and $\gamma_{high}$@>
@<constant $\delta$@>
@<constant $\alpha$@>
@}


@d new operator
@{#ifndef _Old_STD_
# define NOTHROW_NEW  new(nothrow)
#else
# define NOTHROW_NEW  new
#endif
@}

\section{\NumberBase}
%====================

@d static variable declarations of \NumberBase\
@{protected:
  static unsigned char* primes;
  static unsigned char* primPos;
  static Digit          primNumber;
  static unsigned char  idx;
@}

@d protected function prototypes
@{size_t* quad_convergence_sizes(const size_t, size_t&) const;

void digitmul(const Digit, const Digit, const Digit, Digit*) const;
void digitmul(const Digit, const Digit, const Digit, const Digit, Digit*) const;
void digitsqr(const Digit, const Digit, Digit*) const;
@}


@d function prototypes for prime numbers in \NumberBase\
@{Digit  firstPrime() const;
bool   nextPrime(Digit&) const;
Digit  lastPrime() const;
size_t NumberOfPrimes(Digit) const;@}

@d function prototypes for \Digit operations in \NumberBase\
@{void digitmul(const Digit, const Digit, Digit&, Digit&) const;
void digitdiv(const Digit, const Digit, const Digit, Digit&, Digit&) const;
void digitmod(Digit a, Digit b, const Digit c, Digit& r) const;

void gcd(Digit, Digit, Digit, Digit, Digit&, Digit&) const;@}

@D inline implementation of the file \texttt{"{}digit.h"{}}
@{/////////////////// Inline-Implementation ///////////////////////

@<default constructor of \NumberBase\@>
@<destructor of \NumberBase\@>
#ifdef _DigitAsm_
@<digitmul and digitdiv in assembler@>
#endif // _DigitAsm_

#ifndef _DigitAsm_
@<digitmul in ANSI-\cpp\@>
#endif

@<digitmod of a double \Digit\@>
@<digitmul of a \Digit\ with a double \Digit\@>
@<digitmul of a double \Digit\@>
@<digitsqr of a double \Digit\@>
@<getting first prime@>
@<binary logarithm for \Digit s@>
@<the function swap for \Digit s@>
@<square root of a double \Digit\@>

#if defined(_Old_STD_) && !defined(__MINMAX_DEFINED) || defined(_MSC_VER) 
     // Borland Compiler

#ifdef min
# undef min
#endif
#ifdef max
# undef max
#endif

template <class T>
inline const T& min(const T& a, const T& b)
{
  return (a < b)? a : b;
}

template <class T>
inline const T& max(const T& a, const T& b)
{
  return (a < b)? b : a;
}
#endif

@<binder for the arguments of an operator@>
@}

% /*
% // Bug: Visual C++ 5.0
% #if _M_IX86 >= 300 && _MSC_VER
% inline ostream& operator<<(ostream& out, const Digit a)
% {
%   const Digit ALPHA = 1000000000;
%   const size_t ALPHA_WIDTH = 9;
%   Digit b = a/ALPHA;
%   const unsigned int d = (unsigned int)(a-b*ALPHA);
%   if (b) {
%     size_t c = out.width();
%     if (b >= ALPHA) {
%       out.width(1);
%       out << int(b/ALPHA);
%       b %= ALPHA; --c;
%     }
%     out.width(c-ALPHA_WIDTH);
%     out << ((unsigned int)b);
%     out.width(ALPHA_WIDTH);
%     out.fill('0');
%   }
%   return out << d;
% }
% #endif
% */


@D digitmul and digitdiv in assembler
@{#if defined(__386__) && defined(__WATCOMC__)
@<digitmul in assembler for a i386 processor with the Watcom-Compiler@>
@<digitdiv in assembler for a i386 processor with the Watcom-Compiler@>
#elif _M_IX86 >= 300 && _MSC_VER
// no inline support!
#elif (defined (__i386__) || defined (__i486__)) && defined (__GNUC__)
@<digitmul in assembler for a i386 processor with the GNU-Compiler@>
@<digitdiv in assembler for a i386 processor with the GNU-Compiler@>
#elif defined (__sparc_v8__) && defined (__GNUC__)
@<digitmul in assembler for a SPARC v8 processor with the GNU-Compiler@>
@<digitdiv in assembler for a SPARC v8 processor with the GNU-Compiler@>
#else
# undef _DigitAsm_
#endif
@}


@O digit.cpp
@{///////////////////////////////
//
// Piologie V1.2.1
// multi-precision arithmetic
// Digit / NumberBase
//
// Sebastian Wedeniwski
// 03/18/1998
//

#include "digit.h"

@<macro definitions for internal conditions@>
@<new operator@>


///////////////////// Class NumberBase /////////////////////////////

const size_t     SieveSize = (size_t(1) << ((BETA > 32)? 16 : BETA/2)) - 1;

class NumberBase_init : private NumberBase {
private:
  static NumberBase_init PrimeNumbers;

public:
  NumberBase_init(char);                // generating prime numbers
  ~NumberBase_init();
};

@<static variable definitions of \NumberBase\@>
@<private constructor of \NumberBase\_init for generating prime numbers@>
@<getting sizes for quadratic convergence algorithm@>
@<definitions of error-handling@>
@<getting next prime@>
@<getting last prime@>
@<getting the number of primes@>

#ifdef _DigitAsm_
#if _M_IX86 >= 300 && _MSC_VER
@<digitmul in assembler for a i386 processor with the Visual-Compiler@>
@<digitdiv in assembler for a i386 processor with the Visual-Compiler@>
#endif
#endif

#ifndef _DigitAsm_
@<digitdiv in ANSI-\cpp\@>
#endif

@<calculates 10 to the power of a \Digit\@>
@<square root of a \Digit\@>
@<square root and remainder of a \Digit\@>
@<square root and remainder of a double \Digit\@>
@<greatest common divisor of two \Digit s@>
@<greatest common divisor of two double \Digit s@>
@}

@d static variable definitions of \NumberBase\
@{unsigned char*  NumberBase::primes     = 0;
unsigned char*  NumberBase::primPos    = 0;
Digit           NumberBase::primNumber = 2;
unsigned char   NumberBase::idx        = 1;
NumberBase_init NumberBase_init::PrimeNumbers(' ');
@}

@D private constructor of \NumberBase\_init for generating prime numbers
@{NumberBase_init::NumberBase_init(char)
{
  CONDITION(primes == 0);

  // Testing the bits for low ending
  @<testing the bits for low ending@>
  // Generating prime numbers
  @<generating prime numbers@>
}

NumberBase_init::~NumberBase_init()
{
  if (this == &PrimeNumbers) {
    delete[] primes;
    primPos = primes = 0;
  }
}
@| NumberBase_init::NumberBase_init NumberBase_init::~NumberBase_init @}


@d default constructor of \NumberBase\
@{inline NumberBase::NumberBase()
{
}
@| NumberBase::NumberBase @}

@D destructor of \NumberBase\
@{inline NumberBase::~NumberBase()
{
}
@| NumberBase::~NumberBase NumberBase_init::~NumberBase_init @}

@d calculates 10 to the power of a \Digit\
@{Digit pow10(size_t a)
// Algorithm:  b := pow10(a)
// Input:      a in size_t where a <= ALPHA_WIDTH.
// Output:     b in Digit such that b = 10^a ||
{
  CONDITION(a <= ALPHA_WIDTH);

  static const Digit c[10] = {1, 10, 100, Digit(1000), Digit(10000),
                              Digit(100000), Digit(1000000),
                              Digit(10000000), Digit(100000000),
                              Digit(1000000000) };

  if (ALPHA_WIDTH < 10 || a < 10) return c[a];
  Digit b = c[9];
  for (a -= 9; a >= 9; a -= 9) b *= c[9];
  return b*c[a];
}
@| pow10 @}


\section{\Natural}
%=================

@O natural.h
@{///////////////////////////////
//
// Piologie V1.2.1
// multi-precision arithmetic
// Natural
//
// Sebastian Wedeniwski
// 03/18/1998
//


#ifndef _Include_Natural_H_
#define _Include_Natural_H_

#include "digit.h"

#ifdef setbit       // some "math.h" libraries have a defined setbit
# undef setbit
#endif

struct Natural_plus_tag {};
struct Natural_minus_tag {};
struct Natural_multiplies_tag {};
struct Natural_divides_tag {};
struct Natural_modulus_tag {};
struct Natural_square_root_tag {};
struct Natural_lshift_tag {};
struct Natural_rshift_tag {};
struct Natural_and_tag {};
struct Natural_or_tag {};
struct Natural_xor_tag {};
struct Natural_not_tag {};


class Natural : public NumberBase {
public:
  @<output variables for representation of a \Natural\@>
private:
  @<variables for representation of a \Natural\@>
  @<static variables for memory management of \Natural s@>

  void          inc(Digit*);
  void          add_with_inc(const Digit*, Digit*, const Digit*);
  void          add_with_inc(const Digit*, Digit*, const Digit*, const Digit*);
  bool          add_no_inc(const Digit*, Digit*, const Digit*) const;
  bool          add_no_inc(const Digit*, Digit*, const Digit*, const Digit*) const;
  void          sub(Digit*, Digit*, const Digit*);
  void          sub(const Digit*, Digit*, const Digit*, const Digit*);
  bool          sub_no_dec(const Digit*, Digit*, const Digit*) const;
  int           abs(Digit*, const Digit*, const Digit*, size_t) const;

  void          sqr(const Digit*, Digit*, size_t) const;
  Digit         mul(const Digit*, const Digit*, Digit*, const Digit) const;
  Digit*        muladd(const Digit*, const Digit*, Digit*, const Digit) const;
  void          mul(const Digit*, const Digit*, const size_t, Digit*) const;
  void          cmul(const Digit*, size_t, const Digit*, const size_t, Digit*) const;
  void          mul(const Digit*, size_t, const Digit*, const size_t, Digit*) const;
  Digit         mod_div(const Digit);

protected:
  void          dec(Digit*);      // bad! because Integer::setbit
  void          get_memory(const size_t);

  Natural(const size_t, char);
  Natural(const Digit, size_t);
  Natural(size_t, const Natural&);
  Natural(const Natural&, size_t);

  const Digit*  first() const;
  Digit*        last() const;
  size_t        rootsize() const;
  void          normalize();
  size_t        trailing_zeros(Digit*&) const;
  Digit*        setsize(const size_t);
  Natural&      copy(const Natural&, const size_t);
  void          enlarge(const size_t);
  void          fast_rshift(const size_t);
  void          fast_append(const size_t);

  int           compare(const Natural&) const;
  void          add(const Natural&, const Natural&);
  void          sub(const Natural&, const Natural&);
  void          mul(const Natural&, const Natural&);
  void          sqr(const Natural&);
  void          div(const Natural&, Natural);
  void          sqrt(const Natural&);
  void          newton_sqrt(Natural);

  void          lshift(const Natural&, size_t);
  void          rshift(const Natural&, size_t);

  void          bitwise_and(const Natural&, const Natural&);
  void          bitwise_or(const Natural&, const Natural&);
  void          bitwise_xor(const Natural&, const Natural&);
  void          bitwise_not(const Natural&);

  void          add(const Natural&, const Digit);
  void          sub(const Natural&, const Digit);
  void          mul(const Natural&, const Digit);
  void          muladd(const Natural&, const Digit);
  void          mulsub(const Natural&, const Digit);


public:
  static size_t NumberOfDecimals(const size_t);
  static size_t NumberOfDigits(const size_t);
  static void   RestoreSize();

  @<public constructors and destructor of \Natural\@>
  inline Natural& operator=(const Natural&);
  Natural&        operator+=(const Natural&);
  Natural&        operator-=(const Natural&);
  Natural&        operator*=(const Natural&);
  Natural&        operator/=(const Natural&);
  Natural&        operator%=(const Natural&);
  Natural&        operator&=(const Natural&);
  Natural&        operator|=(const Natural&);
  Natural&        operator^=(const Natural&);

  Digit           operator=(const Digit);
  Natural&        operator+=(const Digit);
  Natural&        operator-=(const Digit);
  Natural&        operator*=(const Digit);
  Natural&        operator/=(const Digit);
  Digit           operator%=(const Digit);
  Digit           operator&=(const Digit);
  Natural&        operator|=(const Digit);
  Natural&        operator^=(const Digit);
  Natural&        operator>>=(size_t);
  Natural&        operator<<=(size_t);

  Natural&        operator++();
  Natural&        operator--();
  const Natural   operator++(int);
  const Natural   operator--(int);


  Natural(const binder_arguments<Natural, Natural, Natural_plus_tag>&);
  Natural& operator=(const binder_arguments<Natural, Natural, Natural_plus_tag>&);

  Natural(const binder_arguments<Natural, Natural, Natural_minus_tag>&);
  Natural& operator=(const binder_arguments<Natural, Natural, Natural_minus_tag>&);

  Natural(const binder_arguments<Natural, Natural, Natural_multiplies_tag>&);
  Natural& operator=(const binder_arguments<Natural, Natural, Natural_multiplies_tag>&);

  Natural(const binder_arguments<Natural, Natural, Natural_divides_tag>&);
  Natural& operator=(const binder_arguments<Natural, Natural, Natural_divides_tag>&);

  Natural(const binder_arguments<Natural, Natural, Natural_modulus_tag>&);
  Natural& operator=(const binder_arguments<Natural, Natural, Natural_modulus_tag>&);


  Natural(const binder_arguments<Natural, size_t, Natural_lshift_tag>&);
  Natural& operator=(const binder_arguments<Natural, size_t, Natural_lshift_tag>&);

  Natural(const binder_arguments<Natural, size_t, Natural_rshift_tag>&);
  Natural& operator=(const binder_arguments<Natural, size_t, Natural_rshift_tag>&);

  Natural(const binder_arguments<Natural, Natural, Natural_and_tag>&);
  Natural& operator=(const binder_arguments<Natural, Natural, Natural_and_tag>&);

  Natural(const binder_arguments<Natural, Natural, Natural_or_tag>&);
  Natural& operator=(const binder_arguments<Natural, Natural, Natural_or_tag>&);

  Natural(const binder_arguments<Natural, Natural, Natural_xor_tag>&);
  Natural& operator=(const binder_arguments<Natural, Natural, Natural_xor_tag>&);

  Natural(const binder_arguments<Natural, Natural, Natural_not_tag>&);
  Natural& operator=(const binder_arguments<Natural, Natural, Natural_not_tag>&);


  Natural(const binder_arguments<Natural, Digit, Natural_minus_tag>&);
  Natural& operator=(const binder_arguments<Natural, Digit, Natural_minus_tag>&);

  Natural(const binder_arguments<Natural, Digit, Natural_plus_tag>&);
  Natural& operator=(const binder_arguments<Natural, Digit, Natural_plus_tag>&);

  Natural(const binder_arguments<Natural, Digit, Natural_multiplies_tag>&);
  Natural& operator=(const binder_arguments<Natural, Digit, Natural_multiplies_tag>&);
  Natural& operator+=(const binder_arguments<Natural, Digit, Natural_multiplies_tag>&);
  Natural& operator-=(const binder_arguments<Natural, Digit, Natural_multiplies_tag>&);


  friend Digit        operator%(const Natural&, const Digit);

  inline friend bool  operator==(const Natural&, const Natural&);
  inline friend bool  operator!=(const Natural&, const Natural&);
  inline friend bool  operator<(const Natural&, const Natural&);
  inline friend bool  operator<=(const Natural&, const Natural&);
  inline friend bool  operator>(const Natural&, const Natural&);
  inline friend bool  operator>=(const Natural&, const Natural&);

  inline friend bool  operator==(const Natural&, const Digit);
  inline friend bool  operator!=(const Natural&, const Digit);
  inline friend bool  operator<(const Natural&, const Digit);
  inline friend bool  operator<=(const Natural&, const Digit);
  inline friend bool  operator>(const Natural&, const Digit);
  inline friend bool  operator>=(const Natural&, const Digit);


  Natural(const binder_arguments<Natural, Natural, Natural_square_root_tag>&);
  Natural& operator=(const binder_arguments<Natural, Natural, Natural_square_root_tag>&);


  friend void         div(const Natural&, const Natural&, Natural&, Natural&);
  friend void         div(const Natural&, const Digit, Natural&, Digit&);
  friend void         sqrt(const Natural&, Natural&, Natural&);
  friend void         swap(Natural&, Natural&);
  inline friend bool  sign(const Natural&);


  void    split(const size_t, Natural&, Natural&) const;
  Digit   highest() const;
  Digit   lowest() const;
  size_t  length() const;
  void    setbit(const size_t);
  void    clearbit(const size_t);
  bool    testbit(const size_t) const;
  bool    odd() const;
  bool    even() const;
  void    rand(size_t);
  bool    scan(istream&);


  inline friend rep print(const Natural&);
  friend char*      Ntoa(Natural, char*, const Digit = 10);

  friend ostream&   operator<<(ostream&, Natural);
  friend istream&   operator>>(istream&, Natural&);

  friend Natural    abs(const Natural&, const Natural&);
  friend int        abs(const Natural&, const Natural&, Natural&);
  friend Natural    gcd(Natural, Natural);

  friend class   FFT;

#ifndef _Old_STD_
  bool isprime() const;
#endif
};

inline binder_arguments<Natural, Natural, Natural_plus_tag>
 operator+(const Natural&, const Natural&);
inline binder_arguments<Natural, Natural, Natural_minus_tag>
 operator-(const Natural&, const Natural&);
inline binder_arguments<Natural, Natural, Natural_multiplies_tag>
 operator*(const Natural&, const Natural&);
inline binder_arguments<Natural, Natural, Natural_divides_tag>
 operator/(const Natural&, const Natural&);
inline binder_arguments<Natural, Natural, Natural_modulus_tag>
 operator%(const Natural&, const Natural&);
inline binder_arguments<Natural, size_t, Natural_lshift_tag>
 operator<<(const Natural&, const size_t&);
inline binder_arguments<Natural, size_t, Natural_rshift_tag>
 operator>>(const Natural&, const size_t&);
inline binder_arguments<Natural, Natural, Natural_and_tag>
 operator&(const Natural&, const Natural&);
inline binder_arguments<Natural, Natural, Natural_or_tag>
 operator|(const Natural&, const Natural&);
inline binder_arguments<Natural, Natural, Natural_xor_tag>
 operator^(const Natural&, const Natural&);
inline binder_arguments<Natural, Natural, Natural_not_tag>
 operator~(const Natural&);
inline binder_arguments<Natural, Digit, Natural_minus_tag>
 operator-(const Natural&, const Digit&);
inline binder_arguments<Natural, Digit, Natural_plus_tag>
 operator+(const Natural&, const Digit&);
inline binder_arguments<Natural, Digit, Natural_plus_tag>
 operator+(const Digit&, const Natural&);
inline binder_arguments<Natural, Digit, Natural_multiplies_tag>
 operator*(const Natural&, const Digit&);
inline binder_arguments<Natural, Digit, Natural_multiplies_tag>
 operator*(const Digit&, const Natural&);
inline Natural  operator/(const Natural&, const Digit);
inline Digit    operator&(const Natural&, const Digit);
inline Natural  operator|(const Natural&, const Digit);
inline binder_arguments<Natural, Natural, Natural_square_root_tag>
 sqrt(const Natural&);

Natural atoN(const char*, const Digit = 10);

ostream& operator<<(ostream&, const Natural::rep&);

inline Digit  log2(const Natural&);
Natural       lcm(const Natural&, const Natural&);
Natural       pow(const Natural&, Natural);
Natural       pow(const Natural&, Digit);
Natural       root(const Natural&, const Digit);
Natural       factorial(const Digit);
Natural       bin_coefficient(const Digit, const Digit);
Natural       fibonacci(Digit);

#ifndef _Old_STD_
Natural       euler(Natural);
#endif



///////////////////////// Inline-Implementation ////////////////////

@<comparison \verb#operator==# for \Natural s@>
@<comparison \verb#operator!=# for \Natural s@>
@<comparison \verb#operator<# for \Natural s@>
@<comparison \verb#operator<=# for \Natural s@>
@<comparison \verb#operator># for \Natural s@>
@<comparison \verb#operator>=# for \Natural s@>
@<comparison \verb#operator==# of a \Natural\ with a \Digit\@>
@<comparison \verb#operator!=# of a \Natural\ with a \Digit\@>
@<comparison \verb#operator<# of a \Natural\ with a \Digit\@>
@<comparison \verb#operator<=# of a \Natural\ with a \Digit\@>
@<comparison \verb#operator># of a \Natural\ with a \Digit\@>
@<comparison \verb#operator>=# of a \Natural\ with a \Digit\@>
@<protected constructor \Natural\ without the initialization of the elements@>
inline bool sign(const Natural& a)
// Algorithm:  c := sign(a)
// Input:      a in Natural.
// Output:     c in bool such that if a = 0 then c = false else c = true ||
{
  return (*a.p != 0);
}
@<memberfunction \Natural.highest@>
@<memberfunction \Natural.lowest@>
@<memberfunction \Natural.length@>
@<memberfunction \Natural.odd@>
@<memberfunction \Natural.even@>
inline const Digit* Natural::first() const
// Algorithm:  c := a.first()
// Input:      a in Natural.
// Output:     c in [a.p, a.p+L(a)[ such that c = a.p ||
{
  return p;
}

inline Digit* Natural::last() const
// Algorithm:  c := a.last()
// Input:      a in Natural.
// Output:     c in [a.p, a.p+L(a)[ such that c = a.p+L(a)-1 ||
{
  return p+size-1;
}
@<memberfunction \Natural.rootsize@>
@<memberfunction \Natural.normalize@>
@<fast division of a \Natural\ by a power of $2^\beta$@>
@<multiplication of a \Natural\ by a power of $2^\beta$@>
@<memberfunction \Natural.sub with 3 arguments@>
@<memberfunction \Natural.mod\_div@>
@<memberfunction \Natural.NumberOfDecimals@>
@<memberfunction \Natural.NumberOfDigits@>
@<memberfunction \Natural.RestoreSize@>
@<assign \verb#operator*=# for \Natural s@>
@<assign \verb#operator/=# for \Natural s@>
@<assign \verb#operator%=# for \Natural s@>
@<assign \verb#operator/=# of a \Natural\ with a \Digit\@>
@<assign \verb#operator%=# of a \Natural\ with a \Digit\@>
@<shift \verb#operator>># of a \Natural\@>
@<shift \verb#operator<<# of a \Natural\@>
@<assign \verb#operator=# for \Natural s@>
@<assign \verb#operator|=# of a \Natural\ with a \Digit\@>
@<assign \verb#operator&=# of a \Natural\ with a \Digit\@>
@<assign \verb#operator^=# of a \Natural\ with a \Digit\@>
@<prefix incrementation of a \Natural\@>
@<prefix decrementation of a \Natural\@>
@<postfix incrementation of a \Natural\@>
@<postfix decrementation of a \Natural\@>
@<bitwise \verb#operator~# of a \Natural\@>
@<additive \verb#operator+# for \Natural s@>
@<additive \verb#operator-# for \Natural s@>
@<multiplicative \verb#operator*# for \Natural s@>
@<multiplicative \verb#operator/# for \Natural s@>
@<multiplicative \verb#operator%# for \Natural s@>
@<bitwise \verb#operator&# for \Natural s@>
@<bitwise \verb#operator|# for \Natural s@>
@<bitwise \verb#operator^# for \Natural s@>
@<additive \verb#operator-# of a \Natural\ with a \Digit\@>
@<additive \verb#operator+# of a \Natural\ with a \Digit\@>
@<multiplicative \verb#operator*# of a \Natural\ with a \Digit\@>
@<multiplicative \verb#operator/# of a \Natural\ with a \Digit\@>
@<bitwise \verb#operator&# of a \Natural\ with a \Digit\@>
@<bitwise \verb#operator|# of a \Natural\ with a \Digit\@>
@<binary logarithm for \Natural s@>
@<function \verb#print# of a \Natural\@>
@<simple call for square root of a \Natural\@>
#endif
@}

@O natural.cpp
@{///////////////////////////////
//
// Piologie V1.2.1
// multi-precision arithmetic
// Natural
//
// Sebastian Wedeniwski
// 03/18/1998
//

#include "natural.h"

#ifndef _Old_STD_
# include "primes.h"
# include <list>
#endif
#include <string.h>


@<trade off points for the Karatsuba algorithm@>
@<trade off points for the FFT algorithm@>
@<trade off points for the newton-iteration of the division algorithm@>
@<trade off points for the newton-iteration of the square root algorithm@>

//#define FASTER_BINSQRT      // binary square root without shifting
                            // need BETA/2 times more memory! (Bug!)


@<macro definitions for internal conditions@>
@<macro definitions for internal memory manipulations@>
@<new operator@>


@<class \FFT\@>

@<memberfunction \FFT.size@>
@<memberfunction \FFT.setmodulo@>
#if defined(_DigitAsm_) && _M_IX86 >= 300 && defined(_MSC_VER)
void FFT::digitmulmod(const Digit a, const Digit b, const Digit m, Digit& c) const
// Algorithm:  f.digitmulmod(a, b, m, c)
// Input:      f in FFT, a,b,m in Digit where m > 0, a,b <= m.
// Output:     c in Digit such that c = a*b - f.m*[(a*b)/f.m] ||
{
  __asm {
    mov eax,a
    mov ebx,c
    mul b
    div m
    mov [ebx],edx
  }
}
#elif defined(_DigitAsm_) && (defined (__i386__) || defined (__i486__)) && defined (__GNUC__)
inline void FFT::digitmulmod(const Digit a, const Digit b, const Digit m, Digit& c) const
// Algorithm:  f.digitmulmod(a, b, m, c)
// Input:      f in FFT, a,b,m in Digit where m > 0, a,b <= m.
// Output:     c in Digit such that c = a*b - f.m*[(a*b)/f.m] ||
{
  __asm__ ("mull %2; divl %3"
    : "=&d" (c)
    : "%a" (a), "rm" (b), "rm" (m)
    : "%eax", "cc");
}
#else
@<memberfunction \FFT.digitmulmod@>
#endif


/////////////////// Natural arithmetic /////////////////////////

@<definition of static variables for memory management of \Natural s@>
@<memberfunction \Natural.get\_memory@>
@<protected constructors \Natural\ for the optimal allocation@>
@<memberfunction \Natural.trailing\_zeros@>
@<memberfunction \Natural.setsize@>
@<memberfunction \Natural.copy@>
@<memberfunction \Natural.enlarge@>
@<memberfunction \Natural.inc@>
@<memberfunction \Natural.dec@>
@<memberfunction \Natural.add\_with\_inc with 3 arguments@>
@<memberfunction \Natural.add\_with\_inc with 4 arguments@>
@<memberfunction \Natural.add\_no\_inc with 3 arguments@>
@<memberfunction \Natural.add\_no\_inc with 4 arguments@>
@<memberfunction \Natural.sub with 4 arguments@>
@<memberfunction \Natural.sub\_no\_dec with 3 arguments@>
@<internal function subpos with 3 arguments@>
@<internal function subpos with 4 arguments@>
@<memberfunction \Natural.abs@>
@<memberfunction \Natural.mul@>
@<protected memberfunction \Natural.muladd@>
@<conventional squaring algorithm@>
@<conventional multiplication algorithm@>
@<multiplication with equal size@>
@<multiplication with different size@>
@<default constructor \Natural\@>
@<copy constructor \Natural\@>
@<overloaded constructor \Natural\ for ASCII-string conversion@>
@<destructor \Natural\@>
@<assign \verb#operator+=# for \Natural s@>
@<assign \verb#operator-=# for \Natural s@>
@<assign \verb#operator&=# for \Natural s@>
@<assign \verb#operator|=# for \Natural s@>
@<assign \verb#operator^=# for \Natural s@>
@<compares two \Natural s@>
@<function swap for \Natural s@>
@<splits a \Natural\@>
@<addition of two \Natural s@>
@<subtraction of two \Natural s@>
@<multiplication of two \Natural s@>
@<squaring of a \Natural\@>
@<division of two \Natural s with remainder@>
@<division of two \Natural s@>
@<bitwise and of two \Natural s@>
@<bitwise inclusive or of two \Natural s@>
@<bitwise exclusive or of two \Natural s@>
@<bitwise not of a \Natural\@>
@<multiplication of a \Natural\ by a power of 2@>
@<division of a \Natural\ by a power of 2@>
@<addition of a \Natural\ with a \Digit\@>
@<subtraction of a \Natural\ with a \Digit\@>
@<multiplication of a \Natural\ by a \Digit\@>
@<division of a \Natural\ by a \Digit\@>
@<assign \verb#operator+=# of a \Natural\ with a \Digit\@>
@<assign \verb#operator-=# of a \Natural\ with a \Digit\@>
@<assign \verb#operator*=# of a \Natural\ with a \Digit\@>
@<assign \verb#operator>>=# of a \Natural\@>
@<assign \verb#operator<<=# of a \Natural\@>
@<assign \verb#operator=# of a \Natural\ with a \Digit\@>
@<multiplicative \verb#operator%# of a \Natural\ with a \Digit\@>
@<multiplication and addition of a \Natural\ by a \Digit\@>
@<multiplication and subtraction of a \Natural\ by a \Digit\@>
@<internal function \texttt{sqrtsub}@>
@<square root of a \Natural\@>
@<square root of a \Natural\ with newton iteration@>
@<square root and remainder of a \Natural\@>
@<calculates the root of a \Natural\@>
@<sets a bit in a \Natural\@>
@<clears a bit in a \Natural\@>
@<tests a bit in a \Natural\@>
@<calculates a \Natural\ random number@>
@<converts a string to a \Natural\@>
@<converts a \Natural\ to a string@>
@<puts a \Natural\ on output stream@>
@<gets a \Natural\ from input stream@>
@<puts internal representation of a \Natural\ on output stream@>
@<gets internal representation of a \Natural\ from input stream@>
@<distance of two \Natural s@>
#ifndef _Old_STD_
@<internal fermat test@>
@<primality test@>
#endif
@<greatest common divisor of two \Natural s@>
@<least common multiple of two \Natural s@>
@<calculates the function factorial@>
@<calculates the function binomial coefficient@>
@<calculates the power of a \Natural\ by a \Digit\@>
@<calculates the power of a \Natural\ by a \Natural\@>
@<calculates the n-th Fibonacci number@>
#ifndef _Old_STD_
@<euler function@>
#endif


///////////////////////// FFT class ////////////////////////////

const Digit*  FFT::moduli    = FFT::init_moduli();
const Digit*  FFT::primroots  = FFT::init_primroots();
Digit         FFT::m          = 0;
size_t        FFT::n1         = 0;
size_t        FFT::n2         = 0;
Digit*        FFT::trans_tmp  = 0;
Digit*        FFT::omega      = 0;
Digit*        FFT::omega2     = 0;
size_t*       FFT::order      = 0;

@<memberfunction \FFT.init\_moduli@>
@<memberfunction \FFT.init\_primroots@>
@<memberfunction \FFT.max\_size@>
@<memberfunction \FFT.pow@>
@<memberfunction \FFT.digitinv@>
@<memberfunction \FFT.init\_omega@>
@<transpose and copy a matrix@>
@<brevorder an array@>
@<inner loop of fft without multiplication@>
@<inner loop of fft@>
@<inner loop of inverse fft@>
@<memberfunction \FFT.innerfft3@>
@<memberfunction \FFT.innerfftinv3@>
@<memberfunction \FFT.fft@>
@<memberfunction \FFT.fftinv@>
@<multiply matrix components by a power of a primitive root of unity@>
@<five step FFT algorithm@>
@<five step FFT algorithm for inversion@>
@<fast fourier transform@>
@<inverse fast fourier transform@>
@<chinese remainder theorem for \FFT-class@>
@<componentwise squaring@>
@<componentwise multiplication@>
@<memberfunction \FFT.result@>
@<squaring of a \Natural\ with FFT@>
@<multiplication of two \Natural s with FFT@>
@<brevorder initialization@>
@<default constructor \FFT\@>
@<destructor \FFT\@>
@}




\section{\Integer}
%=================

@O integer.h
@{///////////////////////////////
//
// Piologie V1.2.1
// multi-precision arithmetic
// Integer
//
// Sebastian Wedeniwski
// 03/18/1998
//


#ifndef _Include_Integer_H_
#define _Include_Integer_H_

#include "natural.h"

struct Integer_plus_tag {};
struct Integer_negate_tag {};
struct Integer_minus_tag {};
struct Integer_multiplies_tag {};
struct Integer_divides_tag {};
struct Integer_modulus_tag {};
struct Integer_square_root_tag {};
struct Integer_lshift_tag {};
struct Integer_rshift_tag {};
struct Integer_and_tag {};
struct Integer_or_tag {};
struct Integer_xor_tag {};
struct Integer_not_tag {};

@<class \Integer\@>

inline binder_arguments<Integer, Integer, Integer_negate_tag>
 operator-(const Integer&);
inline binder_arguments<Integer, Integer, Integer_plus_tag>
 operator+(const Integer&, const Integer&);
inline binder_arguments<Integer, Integer, Integer_minus_tag>
 operator-(const Integer&, const Integer&);
inline binder_arguments<Integer, Integer, Integer_multiplies_tag>
 operator*(const Integer&, const Integer&);
inline binder_arguments<Integer, Integer, Integer_divides_tag>
 operator/(const Integer&, const Integer&);
inline binder_arguments<Integer, Integer, Integer_modulus_tag>
 operator%(const Integer&, const Integer&);
inline binder_arguments<Integer, size_t, Integer_lshift_tag>
 operator<<(const Integer&, const size_t&);
inline binder_arguments<Integer, size_t, Integer_rshift_tag>
 operator>>(const Integer&, const size_t&);
inline binder_arguments<Integer, Integer, Integer_and_tag>
 operator&(const Integer&, const Integer&);
inline binder_arguments<Integer, Integer, Integer_or_tag>
 operator|(const Integer&, const Integer&);
inline binder_arguments<Integer, Integer, Integer_xor_tag>
 operator^(const Integer&, const Integer&);
inline binder_arguments<Integer, Integer, Integer_not_tag>
 operator~(const Integer&);
inline binder_arguments<Integer, Natural, Integer_plus_tag>
 operator+(const Natural&, const Integer&);
inline binder_arguments<Integer, Natural, Integer_plus_tag>
 operator+(const Integer&, const Natural&);
inline binder_arguments<Natural, Integer, Integer_minus_tag>
 operator-(const Natural&, const Integer&);
inline binder_arguments<Integer, Natural, Integer_minus_tag>
 operator-(const Integer&, const Natural&);
inline binder_arguments<Integer, Natural, Integer_multiplies_tag>
 operator*(const Natural&, const Integer&);
inline binder_arguments<Integer, Natural, Integer_multiplies_tag>
 operator*(const Integer&, const Natural&);
inline binder_arguments<Natural, Integer, Integer_divides_tag>
 operator/(const Natural&, const Integer&);
inline binder_arguments<Integer, Natural, Integer_divides_tag>
 operator/(const Integer&, const Natural&);
inline binder_arguments<Integer, Digit, Integer_plus_tag>
 operator+(const Digit&, const Integer&);
inline binder_arguments<Integer, Digit, Integer_plus_tag>
 operator+(const Integer&, const Digit&);
inline binder_arguments<Integer, Digit, Integer_multiplies_tag>
 operator*(const Digit&, const Integer&);
inline binder_arguments<Integer, Digit, Integer_multiplies_tag>
 operator*(const Integer&, const Digit&);

inline Integer  operator/(const Integer&, const Digit);
inline Integer  operator%(const Integer&, const Digit);
inline Digit    operator&(const Integer&, const Digit);
inline Integer  operator|(const Integer&, const Digit);


inline bool     operator==(const Integer&, const Integer&);
inline bool     operator!=(const Integer&, const Integer&);
inline bool     operator<(const Integer&, const Integer&);
inline bool     operator<=(const Integer&, const Integer&);
inline bool     operator>(const Integer&, const Integer&);
inline bool     operator>=(const Integer&, const Integer&);

inline bool     operator==(const Integer&, const Digit);
inline bool     operator!=(const Integer&, const Digit);
inline bool     operator<(const Integer&, const Digit);
inline bool     operator<=(const Integer&, const Digit);
inline bool     operator>(const Integer&, const Digit);
inline bool     operator>=(const Integer&, const Digit);

inline binder_arguments<Integer, Integer, Integer_square_root_tag>
 sqrt(const Integer&);


inline const Natural& abs(const Integer&);
inline Digit          log2(const Integer&);
Integer               root(const Integer&, const Digit);
inline int            units(Integer&);
void                  gcd(Integer, Integer, Integer&, Integer&, Integer&);

Integer               atoI(const char*, const Digit = 10);
char*                 Itoa(const Integer&, char*, const Digit = 10);

inline Integer::rep   print(const Integer&);
inline ostream&       operator<<(ostream&, const Integer::rep&);
inline ostream&       operator<<(ostream&, const Integer&);


///////////////////////// Inline-Implementation ////////////////////

@<absolute value of an \Integer\@>
@<sign of an \Integer\@>
@<negation of an \Integer\@>
@<protected constructor \Integer\ without the initialization of the elements@>
@<default constructor \Integer\@>
@<overloaded constructor \Integer\ for \Natural\@>
@<copy constructor \Integer\@>
@<overloaded constructor \Integer\ for ASCII-string conversion@>
@<destructor \Integer\@>
@<comparison \verb#operator==# of an \Integer\ with a \Digit\@>
@<comparison \verb#operator!=# of an \Integer\ with a \Digit\@>
@<comparison \verb#operator<# of an \Integer\ with a \Digit\@>
@<comparison \verb#operator<=# of an \Integer\ with a \Digit\@>
@<comparison \verb#operator># of an \Integer\ with a \Digit\@>
@<comparison \verb#operator>=# of an \Integer\ with a \Digit\@>
@<assign \verb#operator+=# for \Integer s@>
@<assign \verb#operator-=# for \Integer s@>
@<assign \verb#operator*=# for \Integer s@>
@<assign \verb#operator/=# for \Integer s@>
@<assign \verb#operator%=# for \Integer s@>
@<assign \verb#operator&=# for \Integer s@>
@<assign \verb#operator|=# for \Integer s@>
@<assign \verb#operator^=# for \Integer s@>
@<assign \verb#operator+=# of an \Integer\ with a \Natural\@>
@<assign \verb#operator-=# of an \Integer\ with a \Natural\@>
@<assign \verb#operator*=# of an \Integer\ with a \Natural\@>
@<assign \verb#operator/=# of an \Integer\ with a \Natural\@>
@<assign \verb#operator%=# of an \Integer\ with a \Natural\@>
@<splits an \Integer\@>
@<comparison \verb#operator==# for \Integer s@>
@<comparison \verb#operator!=# for \Integer s@>
@<comparison \verb#operator<# for \Integer s@>
@<comparison \verb#operator<=# for \Integer s@>
@<comparison \verb#operator># for \Integer s@>
@<comparison \verb#operator>=# for \Integer s@>
@<negation \verb#operator-# of an \Integer\@>
@<additive \verb#operator+# for \Integer s@>
@<additive \verb#operator-# for \Integer s@>
@<additive \verb#operator+# of an \Integer\ with a \Natural\@>
@<additive \verb#operator-# of an \Integer\ with a \Natural\@>
@<multiplication of an \Integer\ by a \Digit\@>
@<multiplication of an \Integer\ by a \Natural\@>
@<multiplication of two \Integer s@>
@<multiplicative \verb#operator*# for \Integer s@>
@<multiplicative \verb#operator*# of an \Integer\ with a \Natural\@>
@<bitwise not of an \Integer\@>
@<multiplication of an \Integer\ by a power of 2@>
@<division of an \Integer\ by a power of 2@>
@<assign \verb#operator*=# of an \Integer\ with a \Digit\@>
@<assign \verb#operator/=# of an \Integer\ with a \Digit\@>
@<assign \verb#operator%=# of an \Integer\ with a \Digit\@>
@<assign \verb#operator>>=# of an \Integer\@>
@<assign \verb#operator<<=# of an \Integer\@>
@<shift \verb#operator>># of an \Integer\@>
@<shift \verb#operator<<# of an \Integer\@>
@<assign \verb#operator=# for \Integer s@>
@<assign \verb#operator=# for an \Integer\ with a \Natural\@>
@<assign \verb#operator=# of an \Integer\ with a \Digit\@>
@<assign \verb#operator|=# of an \Integer\ with a \Digit\@>
@<postfix incrementation of an \Integer\@>
@<postfix decrementation of an \Integer\@>
@<multiplicative \verb#operator/# for \Integer s@>
@<multiplicative \verb#operator%# for \Integer s@>
@<multiplicative \verb#operator/# of an \Integer\ with a \Natural\@>
@<bitwise \verb#operator~# of an \Integer\@>
@<bitwise \verb#operator&# for \Integer s@>
@<bitwise \verb#operator|# for \Integer s@>
@<bitwise \verb#operator^# for \Integer s@>
@<additive \verb#operator+# of an \Integer\ with a \Digit\@>
@<additive \verb#operator-# of an \Integer\ with a \Digit\@>
@<multiplicative \verb#operator*# of an \Integer\ with a \Digit\@>
@<multiplicative \verb#operator/# of an \Integer\ with a \Digit\@>
@<multiplicative \verb#operator%# of an \Integer\ with a \Digit\@>
@<bitwise \verb#operator&# of an \Integer\ with a \Digit\@>
@<bitwise \verb#operator|# of an \Integer\ with a \Digit\@>
@<binary logarithm for \Integer s@>
@<units of an \Integer\@>
@<simple call for square root of an \Integer\@>
@<function \verb#print# of an \Integer\@>
@<puts internal representation of an \Integer\ on output stream@>
@<puts an \Integer\ on output stream@>

#endif
@}

@D private memberfunctions of \Integer\
@{Natural&  abs();
void      neg();
void      neg(const Integer&);
void      add(const Integer&, const Integer&);
void      sub(const Integer&, const Integer&);
void      mul(const Integer&, const Integer&);
void      div(const Integer&, const Integer&);
void      sqrt(const Integer&);

void      lshift(const Integer&, const size_t);
void      rshift(const Integer&, const size_t);
void      bitwise_and(const Integer&, const Integer&);
void      bitwise_or(const Integer&, const Integer&);
void      bitwise_xor(const Integer&, const Integer&);
void      bitwise_not(const Integer&);

void      add(const Integer&, const Natural&);
void      sub(const Integer&, const Natural&);
void      sub(const Natural&, const Integer&);
void      mul(const Integer&, const Natural&);
void      div(const Integer&, const Natural&);
void      div(const Natural&, const Integer&);

void      add(const Integer&, const Digit);
void      sub(const Integer&, const Digit);
void      mul(const Integer&, const Digit);
void      muladd(const Integer&, const Digit);
void      mulsub(const Integer&, const Digit);
@}

@D public memberfunctions and friends of \Integer\
@{inline Integer& operator=(const Integer&);
inline Integer& operator=(const Natural&);
Integer&        operator+=(const Integer&);
Integer&        operator-=(const Integer&);
Integer&        operator*=(const Integer&);
Integer&        operator/=(const Integer&);
Integer&        operator%=(const Integer&);
Integer&        operator&=(const Integer&);
Integer&        operator|=(const Integer&);
Integer&        operator^=(const Integer&);

Integer&        operator+=(const Natural&);
Integer&        operator-=(const Natural&);
Integer&        operator*=(const Natural&);
Integer&        operator/=(const Natural&);
Integer&        operator%=(const Natural&);

Integer&        operator+=(const Digit);
Integer&        operator-=(const Digit);
Integer&        operator*=(const Digit);
Integer&        operator/=(const Digit);
Integer&        operator%=(const Digit);
Integer&        operator>>=(const size_t);
Integer&        operator<<=(const size_t);
inline Digit    operator=(const Digit);
Digit           operator&=(const Digit);
Integer&        operator|=(const Digit);

const Integer&  operator++();
const Integer&  operator--();
Integer         operator++(int);
Integer         operator--(int);


Integer(const binder_arguments<Integer, Integer, Integer_negate_tag>&);
Integer& operator=(const binder_arguments<Integer, Integer, Integer_negate_tag>&);

Integer(const binder_arguments<Integer, Integer, Integer_plus_tag>&);
Integer& operator=(const binder_arguments<Integer, Integer, Integer_plus_tag>&);

Integer(const binder_arguments<Integer, Integer, Integer_minus_tag>&);
Integer& operator=(const binder_arguments<Integer, Integer, Integer_minus_tag>&);

Integer(const binder_arguments<Integer, Integer, Integer_multiplies_tag>&);
Integer& operator=(const binder_arguments<Integer, Integer, Integer_multiplies_tag>&);

Integer(const binder_arguments<Integer, Integer, Integer_divides_tag>&);
Integer& operator=(const binder_arguments<Integer, Integer, Integer_divides_tag>&);

Integer(const binder_arguments<Integer, Integer, Integer_modulus_tag>&);
Integer& operator=(const binder_arguments<Integer, Integer, Integer_modulus_tag>&);


Integer(const binder_arguments<Integer, size_t, Integer_lshift_tag>&);
Integer& operator=(const binder_arguments<Integer, size_t, Integer_lshift_tag>&);

Integer(const binder_arguments<Integer, size_t, Integer_rshift_tag>&);
Integer& operator=(const binder_arguments<Integer, size_t, Integer_rshift_tag>&);

Integer(const binder_arguments<Integer, Integer, Integer_and_tag>&);
Integer& operator=(const binder_arguments<Integer, Integer, Integer_and_tag>&);

Integer(const binder_arguments<Integer, Integer, Integer_or_tag>&);
Integer& operator=(const binder_arguments<Integer, Integer, Integer_or_tag>&);

Integer(const binder_arguments<Integer, Integer, Integer_xor_tag>&);
Integer& operator=(const binder_arguments<Integer, Integer, Integer_xor_tag>&);

Integer(const binder_arguments<Integer, Integer, Integer_not_tag>&);
Integer& operator=(const binder_arguments<Integer, Integer, Integer_not_tag>&);


Integer(const binder_arguments<Integer, Natural, Integer_plus_tag>&);
Integer& operator=(const binder_arguments<Integer, Natural, Integer_plus_tag>&);

Integer(const binder_arguments<Integer, Natural, Integer_minus_tag>&);
Integer& operator=(const binder_arguments<Integer, Natural, Integer_minus_tag>&);

Integer(const binder_arguments<Natural, Integer, Integer_minus_tag>&);
Integer& operator=(const binder_arguments<Natural, Integer, Integer_minus_tag>&);

Integer(const binder_arguments<Integer, Natural, Integer_multiplies_tag>&);
Integer& operator=(const binder_arguments<Integer, Natural, Integer_multiplies_tag>&);

Integer(const binder_arguments<Integer, Natural, Integer_divides_tag>&);
Integer& operator=(const binder_arguments<Integer, Natural, Integer_divides_tag>&);

Integer(const binder_arguments<Natural, Integer, Integer_divides_tag>&);
Integer& operator=(const binder_arguments<Natural, Integer, Integer_divides_tag>&);


Integer(const binder_arguments<Integer, Digit, Integer_minus_tag>&);
Integer& operator=(const binder_arguments<Integer, Digit, Integer_minus_tag>&);
Integer(const binder_arguments<Digit, Integer, Integer_minus_tag>&);
Integer& operator=(const binder_arguments<Digit, Integer, Integer_minus_tag>&);

Integer(const binder_arguments<Integer, Digit, Integer_plus_tag>&);
Integer& operator=(const binder_arguments<Integer, Digit, Integer_plus_tag>&);

Integer(const binder_arguments<Integer, Digit, Integer_multiplies_tag>&);
Integer& operator=(const binder_arguments<Integer, Digit, Integer_multiplies_tag>&);
Integer& operator+=(const binder_arguments<Integer, Digit, Integer_multiplies_tag>&);
Integer& operator-=(const binder_arguments<Integer, Digit, Integer_multiplies_tag>&);

Integer(const binder_arguments<Integer, Integer, Integer_square_root_tag>&);
Integer& operator=(const binder_arguments<Integer, Integer, Integer_square_root_tag>&);

friend void       div(const Integer&, const Integer&, Integer&, Integer&);
friend void       div(const Integer&, const Digit, Integer&, Integer&);
friend void       swap(Integer&, Integer&);
friend void       sqrt(const Integer&, Integer&, Integer&);
inline friend int sign(const Integer&);
friend Integer    pow(const Integer&, const Digit);
friend Integer    pow(const Integer&, const Integer&);
friend istream&   operator>>(istream&, Integer&);


void split(const size_t, Integer&, Integer&) const;
void setbit(const size_t);
void clearbit(const size_t);
bool testbit(const size_t) const;
void rand(const size_t);

bool scan(istream&);
@}


@O integer.cpp
@{///////////////////////////////
//
// Piologie V1.2.1
// multi-precision arithmetic
// Integer
//
// Sebastian Wedeniwski
// 03/18/1998
//

#include "integer.h"


@<macro definitions for internal conditions@>


/////////////////// Integer Arithmetic /////////////////////////

@<addition of two \Integer s@>
@<addition of an \Integer\ with a \Natural\@>
@<subtraction of two \Integer s@>
@<subtraction of an \Integer\ with a \Natural\@>
@<subtraction of a \Natural\ with an \Integer\@>
@<division of two \Integer s@>
@<square root of an \Integer\@>
@<assign \verb#operator+=# of an \Integer\ with a \Digit\@>
@<assign \verb#operator-=# of an \Integer\ with a \Digit\@>
@<assign \verb#operator&=# of an \Integer\ with a \Digit\@>
@<prefix incrementation of an \Integer\@>
@<prefix decrementation of an \Integer\@>
@<addition of an \Integer\ with a \Digit\@>
@<subtraction of an \Integer\ with a \Digit\@>
@<multiplication and addition of an \Integer\ by a \Digit\@>
@<multiplication and subtraction of an \Integer\ by a \Digit\@>
@<division of two \Integer s with remainder@>
@<division of an \Integer\ by a \Natural\@>
@<division of a \Natural\ by an \Integer\@>
@<division of an \Integer\ by a \Digit\@>
@<function swap for \Integer s@>
@<positive square root and remainder of an \Integer\@>
@<converts a string to an \Integer\@>
@<converts an \Integer\ to a string@>
@<gets internal representation of an \Integer\ from input stream@>
@<gets an \Integer\ from input stream@>
@<bitwise and of two \Integer s@>
@<bitwise inclusive or of two \Integer s@>
@<bitwise exclusive or of two \Integer s@>
@<sets a bit in an \Integer\@>
@<clears a bit in an \Integer\@>
@<tests a bit in an \Integer\@>
@<calculates an \Integer\ random number@>
@<power of an \Integer\ by a \Digit\@>
@<power of an \Integer\ by an \Integer\@>
@<calculates the root of an \Integer\@>
@<extented greatest common divisor of two \Integer s@>
@}



\section{\Rational}
%==================

@O rational.h
@{///////////////////////////////
//
// Piologie V1.2.1
// multi-precision arithmetic
// Rational
//
// Sebastian Wedeniwski
// 03/18/1998
//


#ifndef _Include_Rational_H_
#define _Include_Rational_H_

#include "integer.h"

struct Rational_plus_tag {};
struct Rational_negate_tag {};
struct Rational_minus_tag {};
struct Rational_multiplies_tag {};
struct Rational_divides_tag {};
struct Rational_modulus_tag {};
struct Rational_square_root_tag {};
struct Rational_lshift_tag {};
struct Rational_rshift_tag {};

//#ifdef _Old_STD_
//template <class I, class N>
//#else
//template <class I, class N = I>
//#endif
class Rational {
public:
  @<output variables for representation of a \Rational\@>

private:
  Integer num;
  Natural den;

  void add(const Rational&, const Rational&);
  void sub(const Rational&, const Rational&);
  void mul(const Rational&, const Rational&);
  void sqr(const Rational&);
  void div(const Rational&, const Rational&);
  void lshift(const Rational&, size_t);
  void rshift(const Rational&, size_t);

public:
  Rational(const Digit = 0);
  Rational(const Integer&);
  Rational(const Integer&, const Natural&);
  Rational(const Rational&);
  ~Rational();

  const Integer&   numerator() const;
  const Natural&   denominator() const;

  Digit            operator=(const Digit);
  Rational&        operator=(const Integer&);
  Rational&        operator=(const Rational&);
  Rational&        operator+=(const Rational&);
  Rational&        operator-=(const Rational&);
  Rational&        operator*=(const Rational&);
  Rational&        operator/=(const Rational&);
  Rational&        operator<<=(const size_t);
  Rational&        operator>>=(const size_t);
  
  const Rational&  operator++();
  const Rational&  operator--();
  Rational         operator++(int);
  Rational         operator--(int);


  Rational(const binder_arguments<Rational, Rational, Rational_negate_tag>&);
  Rational& operator=(const binder_arguments<Rational, Rational, Rational_negate_tag>&);

  Rational(const binder_arguments<Rational, Rational, Rational_plus_tag>&);
  Rational& operator=(const binder_arguments<Rational, Rational, Rational_plus_tag>&);

  Rational(const binder_arguments<Rational, Rational, Rational_minus_tag>&);
  Rational& operator=(const binder_arguments<Rational, Rational, Rational_minus_tag>&);

  Rational(const binder_arguments<Rational, Rational, Rational_multiplies_tag>&);
  Rational& operator=(const binder_arguments<Rational, Rational, Rational_multiplies_tag>&);

  Rational(const binder_arguments<Rational, Rational, Rational_divides_tag>&);
  Rational& operator=(const binder_arguments<Rational, Rational, Rational_divides_tag>&);

  Rational(const binder_arguments<Rational, size_t, Rational_lshift_tag>&);
  Rational& operator=(const binder_arguments<Rational, size_t, Rational_lshift_tag>&);

  Rational(const binder_arguments<Rational, size_t, Rational_rshift_tag>&);
  Rational& operator=(const binder_arguments<Rational, size_t, Rational_rshift_tag>&);

  void rand(const size_t);
  bool scan(istream&);

  inline friend void  swap(Rational&, Rational&);
  friend istream&     operator>>(istream&, Rational&);
  friend Rational     inv(const Rational&);
};

inline binder_arguments<Rational, Rational, Rational_negate_tag>
 operator-(const Rational&);
inline binder_arguments<Rational, Rational, Rational_plus_tag>
 operator+(const Rational&, const Rational&);
inline binder_arguments<Rational, Rational, Rational_minus_tag>
 operator-(const Rational&, const Rational&);
inline binder_arguments<Rational, Rational, Rational_multiplies_tag>
 operator*(const Rational&, const Rational&);
inline binder_arguments<Rational, Rational, Rational_divides_tag>
 operator/(const Rational&, const Rational&);
inline binder_arguments<Rational, size_t, Rational_lshift_tag>
 operator<<(const Rational&, const size_t&);
inline binder_arguments<Rational, size_t, Rational_rshift_tag>
 operator>>(const Rational&, const size_t&);


inline bool operator==(const Rational&, const Rational&);
inline bool operator!=(const Rational&, const Rational&);
inline bool operator<(const Rational&, const Rational&);
inline bool operator<=(const Rational&, const Rational&);
inline bool operator>(const Rational&, const Rational&);
inline bool operator>=(const Rational&, const Rational&);

inline bool operator==(const Rational&, const Digit);
inline bool operator!=(const Rational&, const Digit);
inline bool operator<(const Rational&, const Digit);
inline bool operator<=(const Rational&, const Digit);
inline bool operator>(const Rational&, const Digit);
inline bool operator>=(const Rational&, const Digit);

inline const Integer& numerator(const Rational&);
inline const Natural& denominator(const Rational&);
inline Rational       abs(const Rational&);
inline int            sign(const Rational&);
inline Rational::rep  print(const Rational&);
inline ostream&       operator<<(ostream&, const Rational::rep&);
inline ostream&       operator<<(ostream&, const Rational&);
Integer               ceil(const Rational&);
Integer               floor(const Rational&);
inline Integer        round(const Rational&);
Integer               trunc(const Rational&);
inline Rational       pow(const Rational&, const Digit);
Rational              pow(const Rational&, const Integer&);


@<class \Rational\@>


///////////////////////// Inline-Implementation ////////////////////

@<squaring of a \Rational\@>
@<default constructor \Rational\@>
@<overloaded constructor \Rational\ for \Integer\@>
@<copy constructor \Rational\@>
@<destructor \Rational\@>
@<quotient field numerator (memberfunction)@>
@<quotient field denominator (memberfunction)@>
@<assign \verb#operator=# for a \Rational\ with a \Digit\@>
@<assign \verb#operator=# for a \Rational\ with a \Integer\@>
@<assign \verb#operator=# for \Rational s@>
@<assign \verb#operator+=# for \Rational s@>
@<assign \verb#operator-=# for \Rational s@>
@<assign \verb#operator*=# for \Rational s@>
@<assign \verb#operator/=# for \Rational s@>
@<assign \verb#operator<<=# of a \Rational\@>
@<assign \verb#operator>>=# of a \Rational\@>
@<prefix incrementation of a \Rational\@>
@<prefix decrementation of a \Rational\@>
@<postfix incrementation of a \Rational\@>
@<postfix decrementation of a \Rational\@>
@<negation \verb#operator-# of a \Rational\@>
@<additive \verb#operator+# for \Rational s@>
@<additive \verb#operator-# for \Rational s@>
@<multiplicative \verb#operator*# for \Rational s@>
@<multiplicative \verb#operator/# for \Rational s@>
@<shift \verb#operator<<# of a \Rational\@>
@<shift \verb#operator>># of a \Rational\@>
@<calculates a \Rational\ random number@>
@<function swap for \Rational s@>
@<comparison \verb#operator==# for \Rational s@>
@<comparison \verb#operator!=# for \Rational s@>
@<comparison \verb#operator<# for \Rational s@>
@<comparison \verb#operator<=# for \Rational s@>
@<comparison \verb#operator># for \Rational s@>
@<comparison \verb#operator>=# for \Rational s@>
@<comparison \verb#operator==# of a \Rational\ with a \Digit\@>
@<comparison \verb#operator!=# of a \Rational\ with a \Digit\@>
@<comparison \verb#operator<# of a \Rational\ with a \Digit\@>
@<comparison \verb#operator<=# of a \Rational\ with a \Digit\@>
@<comparison \verb#operator># of a \Rational\ with a \Digit\@>
@<comparison \verb#operator>=# of a \Rational\ with a \Digit\@>
@<quotient field numerator@>
@<quotient field denominator@>
@<absolute value of a \Rational\@>
@<sign of a \Rational\@>
@<power of a \Rational\ by a \Digit\@>
@<function \verb#print# of a \Rational\@>
@<puts internal representation of a \Rational\ on output stream@>
@<puts a \Rational\ on output stream@>
@<function \verb#round# of a \Rational\@>

#endif
@}


@O rational.cpp
@{///////////////////////////////
//
// Piologie V1.2.1
// multi-precision arithmetic
// Rational
//
// Sebastian Wedeniwski
// 03/18/1998
//

#include "rational.h"


/////////////////// Rational Arithmetic ////////////////////////

@<addition of two \Rational s@>
@<subtraction of two \Rational s@>
@<multiplication of two \Rational s@>
@<division of two \Rational s@>
@<multiplication of a \Rational\ by a power of 2@>
@<division of a \Rational\ by a power of 2@>
@<overloaded constructor \Rational\ for \Integer\ and \Natural\@>
@<gets internal representation of a \Rational\ from input stream@>
@<gets a \Rational\ from input stream@>
@<inversion of a \Rational\@>
@<function \verb#ceil# of a \Rational\@>
@<function \verb#floor# of a \Rational\@>
@<function \verb#trunc# of a \Rational\@>
@<power of a \Rational\ by an \Integer\@>
@}



\section{Primes}
%===============

@O primes.h
@{///////////////////////////////
//
// Piologie V1.2.1
// multi-precision arithmetic
// Primes
//
// Sebastian Wedeniwski
// 03/18/1998
//


#ifndef _Include_Primes_H_
#define _Include_Primes_H_

#include "natural.h"

@<prime factorization of a \Natural\@>
#endif
@}



\section{\PiClass}
%=================

@O pi.h
@{///////////////////////////////
//
// Piologie V1.2.1
// multi-precision arithmetic
// Pi-Calculation
//
// Sebastian Wedeniwski
// 03/18/1998
//


#ifndef _Include_Pi_H_
#define _Include_Pi_H_

#include "integer.h"

@<class \Fixed\@>

inline ostream& operator<<(ostream&, Fixed);

class Pi {
private:
  Fixed pi;

  void stoermer(const size_t, Natural&);
  void schoenhage(const size_t, Natural&);
  void chudnovsky(size_t, const size_t, Integer&, Integer&, Integer&) const;
  void chudnovsky2(const size_t, const size_t, Integer&, Integer&) const;

public:
  Pi(const size_t);

  inline friend ostream& operator<<(ostream&, const Pi&);
};

class Zeta3 {
private:
  Fixed zeta;

  void linear(const size_t, Natural&) const;
  void series(size_t, const size_t, Integer&, Integer&, Integer&) const;
  void series2(const size_t, const size_t, Integer&, Integer&) const;

public:
  Zeta3(const size_t);

  inline friend ostream& operator<<(ostream&, const Zeta3&);
};

/////////////////// Inline-Implementation ///////////////////////

@<inline implementation of class \Fixed\@>
@<puts \PiClass\ on output stream@>
@<puts Zeta3-class on output stream@>

#endif
@}

@O pi.cpp
@{///////////////////////////////
//
// Piologie V1.2.1
// multi-precision arithmetic
// Pi-Calculation
//
// Sebastian Wedeniwski
// 03/18/1998
//

#include "pi.h"

@<macro definitions for internal conditions@>

@<puts \Fixed\ on output stream@>
@<Pi-Calculation with Stoermer-Algorithms@>
@<Pi-Calculation with Sch\"onhage-Algorithms@>
@<Pi-Calculation with Chudnovsky-Algorithms@>
@<select $\pi$-Algorithm@>
@<Zeta(3)-Series-Calculation@>
@<Zeta(3)-Calculation@>
@<select $\zeta(3)$-Algorithm@>
@}



\section{Makefile}
%=================

@O makefile -t
@{###############################
##
## Piologie V1.2.1
## multi-precision arithmetic
## Makefile
##
## Sebastian Wedeniwski
## 03/18/1998
##

#
# include the configuration file
#

include config

#
# delete a file:
#
# DELETE = del       # Windows / DOS / OS/2
DELETE = rm -f     # UNIX / MKS-Tools / GNU-Tools

#
# copy a file:
#
# COPY = copy       # Windows / DOS / OS/2
COPY = cp         # UNIX / MKS-Tools / GNU-Tools


MP_OBJS = digit.$(OBJ) natural.$(OBJ) integer.$(OBJ) rational.$(OBJ) pi.$(OBJ)

all: piologie.$(LIB) test


piologie.$(LIB): $(MP_OBJS)
	$(DELETE) $@@
	$(MAKE_LIB)

digit.$(SUFFIX):
	$(COPY) digit.cpp $@@

digit.$(OBJ): digit.$(SUFFIX) digit.h
	$(COMPILE_ONLY) $(CFLAGS) digit.$(SUFFIX)

natural.$(SUFFIX):
	$(COPY) natural.cpp $@@

natural.$(OBJ): natural.$(SUFFIX) natural.h digit.h
	$(COMPILE_ONLY) $(CFLAGS) natural.$(SUFFIX)

integer.$(SUFFIX):
	$(COPY) integer.cpp $@@

integer.$(OBJ): integer.$(SUFFIX) integer.h natural.h digit.h
	$(COMPILE_ONLY) $(CFLAGS) integer.$(SUFFIX)

rational.$(SUFFIX):
	$(COPY) rational.cpp $@@

rational.$(OBJ): rational.$(SUFFIX) rational.h integer.h natural.h digit.h
	$(COMPILE_ONLY) $(CFLAGS) rational.$(SUFFIX)

pi.$(SUFFIX):
	$(COPY) pi.cpp $@@

pi.$(OBJ): pi.$(SUFFIX) pi.h integer.h natural.h digit.h
	$(COMPILE_ONLY) $(CFLAGS) pi.$(SUFFIX)


test.$(SUFFIX):
	$(COPY) test.cpp $@@

test: test.$(SUFFIX) piologie.$(LIB)
	$(COMPILE_ONLY) $(CFLAGS) $@@.$(SUFFIX)
	$(LINK)

cfrac.$(SUFFIX):
	$(COPY) cfrac.cpp $@@

cfrac: cfrac.$(SUFFIX) piologie.$(LIB)
	$(COMPILE_ONLY) $(CFLAGS) $@@.$(SUFFIX)
	$(LINK)

check.$(SUFFIX):
	$(COPY) check.cpp $@@

check: check.$(SUFFIX) piologie.$(LIB)
	$(COMPILE_ONLY) $@@.$(SUFFIX)
	$(LINK)

zeta.$(SUFFIX):
	$(COPY) zeta.cpp $@@

zeta: zeta.$(SUFFIX) piologie.$(LIB)
	$(COMPILE_ONLY) $(CFLAGS) $@@.$(SUFFIX)
	$(LINK)


dvi: piologie.dvi manual.dvi
	cd doc
	cd src
	nuweb -o piologie.w
	latex piologie.tex
	nuweb -o piologie.w
	makeindex piologie.idx
	latex piologie.tex
	latex manual.tex
	latex manual.tex
	cd ..
	cd ..

ps: piologie.ps manual.ps dvi
	cd doc
	cd src
	dvips piologie.dvi
	dvips manual.dvi
	cd ..
	cd ..

clean:
	$(DELETE) *.$(OBJ)


#
# Supported Compiler
#

ap: clean
	$(COPY) config.ap config; make

borland: clean
	$(COPY) config.i386_bc config; make

dec: clean
	$(COPY) config.dec config; make

edg: clean
	$(COPY) config.edg config; make

gnu: clean
	$(COPY) config.gnu config; make

gnu28: clean
	$(COPY) config.gnu28 config; make

gnu_mips4: clean
	$(COPY) config.gnu_mips4 config; make

gnu_sparc: clean
	$(COPY) config.gnu_sparc config; make

hp: clean
	$(COPY) config.hp config; make

hpa: clean
	$(COPY) config.hpa config; make

i386_ibm: clean
	$(COPY) config.i386_ibm config
	make

kai: clean
	$(COPY) config.kai config; make

ppc_ibm: clean
	$(COPY) config.ppc_ibm config
	make

sgi: clean
	$(COPY) config.sgi config; make

sgi_8000: clean
	$(COPY) config.sgi_8000 config; make

sun: clean
	$(COPY) config.sun config; make

visual: clean
	$(COPY) config.i386_vc config; make

watcom: clean
	$(COPY) config.i386_wc config; make
@}


\subsection{Standardeinstellung}
%-------------------------------

@O config
@{#
# Default: GNU C++ 2.8.0:
#

# The C++ compiler:
CC           = g++

# The flags for the C++ compiler:
CFLAGS       = -O3

# Suffix of the object file names:
OBJ          = o

# Suffix of the library file name:
LIB          = a

# Command to compile only:
COMPILE_ONLY = $(CC) -c

# Flag to specify the output file name:
OUTPUT       = -o $@@

# Command to make the library:
MAKE_LIB     = ar cr $@@ $(MP_OBJS); ranlib $@@

# Command to call the linker:
LINK         = $(CC) $(CFLAGS) $@@.$(OBJ) piologie.$(LIB) -o $@@

# Suffix of the C++ file names:
SUFFIX       = cpp
@}


\subsection{Apogee \cpp}
%-----------------------

@O config.ap
@{#
# Apogee C++ Release AC3.0:
#
CC           = apCC
CFLAGS       = -fast -O5 -D_Old_STD_ -D_Unknown_Apogee_Bug_
OBJ          = o
LIB          = a
COMPILE_ONLY = $(CC) -c
OUTPUT       = -o $@@
MAKE_LIB     = ar cr $@@ $(MP_OBJS); ranlib $@@
LINK         = $(CC) $(CFLAGS) $@@.$(OBJ) piologie.$(LIB) -o $@@
SUFFIX       = cpp
@}


\subsection{Borland \cpp}
%------------------------

@O config.i386_bc
@{#
# Pentium, Borland C++ 5.0:
#
CC           = bcc32
CFLAGS       = -O2 -OS -Oc -w- -D_Old_STD_
OBJ          = obj
LIB          = lib
COMPILE_ONLY = $(CC) -c
OUTPUT       = 
MAKE_LIB     = tlib $@@ -+digit.obj -+natural.obj -+integer.obj -+pi.obj
LINK         = $(CC) $@@.$(OBJ) piologie.$(LIB)
SUFFIX       = cpp
@}


\subsection{DEC \cpp}
%--------------------

@O config.dec
@{#
# DEC Alpha 150 Mhz, DEC C++ 5.0:
#
CC           = cxx
CFLAGS       = -O2 -D_Old_STD_
OBJ          = o
LIB          = a
COMPILE_ONLY = $(CC) -c
OUTPUT       = -o $@@
MAKE_LIB     = ar cr $@@ $(MP_OBJS); ranlib $@@
LINK         = $(CC) $(CFLAGS) $@@.$(OBJ) piologie.$(LIB) -o $@@
SUFFIX       = cc
@}


\subsection{Edison Design Group \cpp\ front end}
%-----------------------------------------------

@O config.edg
@{#
# EDG C++ front end 2.33:
#
CC           = eccp
CFLAGS       = -O2 -DSTATIC_VS_INLINE=static -D_Old_STD_
OBJ          = o
LIB          = a
COMPILE_ONLY = $(CC) -c
OUTPUT       = -o $@@
MAKE_LIB     = ar cr $@@ $(MP_OBJS); ranlib $@@
LINK         = $(CC) $(CFLAGS) $@@.$(OBJ) piologie.$(LIB) -o $@@
SUFFIX       = cpp
@}


\subsection{GNU \cpp}
%--------------------

@O config.gnu
@{#
# GNU C++ 2.6.1 - 2.7.2:
#
CC           = g++
CFLAGS       = -O3 -D_Old_STD_
OBJ          = o
LIB          = a
COMPILE_ONLY = $(CC) -c
OUTPUT       = -o $@@
MAKE_LIB     = ar cr $@@ $(MP_OBJS); ranlib $@@
LINK         = $(CC) $(CFLAGS) $@@.$(OBJ) piologie.$(LIB) -o $@@
SUFFIX       = cpp
@}

@O config.gnu28
@{#
# GNU C++ 2.8.0:
#
CC           = g++
CFLAGS       = -O3
OBJ          = o
LIB          = a
COMPILE_ONLY = $(CC) -c
OUTPUT       = -o $@@
MAKE_LIB     = ar cr $@@ $(MP_OBJS); ranlib $@@
LINK         = $(CC) $(CFLAGS) $@@.$(OBJ) piologie.$(LIB) -o $@@
SUFFIX       = cpp
@}

@O config.gnu_mips4
@{#
# MIPS R8000, GNU C++ 2.7.2:
#
CC           = g++
CFLAGS       = -O3 -mips4 -D_Old_STD_
OBJ          = o
LIB          = a
COMPILE_ONLY = $(CC) -c
OUTPUT       = -o $@@
MAKE_LIB     = ar cr $@@ $(MP_OBJS); ranlib $@@
LINK         = $(CC) $(CFLAGS) $@@.$(OBJ) piologie.$(LIB) -o $@@
SUFFIX       = cpp
@}

@O config.gnu_sparc
@{#
# SuperSPARC, GNU C++ 2.7.2:
#
CC           = g++
CFLAGS       = -O3 -msupersparc -D_Old_STD_
OBJ          = o
LIB          = a
COMPILE_ONLY = $(CC) -c
OUTPUT       = -o $@@
MAKE_LIB     = ar cr $@@ $(MP_OBJS); ranlib $@@
LINK         = $(CC) $(CFLAGS) $@@.$(OBJ) piologie.$(LIB) -o $@@
SUFFIX       = cpp
@}


\subsection{HP \cpp}
%--------------------

@O config.hp
@{#
# HP PA, HP C++ A.10.22:
#
CC           = CC
CFLAGS       = +O4 +a1 -DSTATIC_VS_INLINE=static -D_Old_STD_
OBJ          = o
LIB          = a
COMPILE_ONLY = $(CC) -c
OUTPUT       = -o $@@
MAKE_LIB     = ar cr $@@ $(MP_OBJS); ranlib $@@
LINK         = $(CC) $(CFLAGS) $@@.$(OBJ) piologie.$(LIB) -o $@@
SUFFIX       = cpp
@}


@O config.hpa
@{#
# HP PA-RISC 2.0, HP aC++ A.01.00:
#
CC           = aCC
CFLAGS       = +O4 +DA2.0 -D_Old_STD_
OBJ          = o
LIB          = a
COMPILE_ONLY = $(CC) -c
OUTPUT       = -o $@@
MAKE_LIB     = ar cr $@@ $(MP_OBJS); ranlib $@@
LINK         = $(CC) $(CFLAGS) $@@.$(OBJ) piologie.$(LIB) -o $@@
SUFFIX       = cpp
@}

\subsection{IBM C Set++ for AIX 3.1.1}
%-------------------------------------

@O config.ppc_ibm
@{#
# PowerPC, IBM C Set++:
#
CC           = xlC
CFLAGS       = -O3 -D_Old_STD_
OBJ          = o
LIB          = a
COMPILE_ONLY = $(CC) -c
OUTPUT       = -o $@@
MAKE_LIB     = ar cr $@@ $(MP_OBJS); ranlib $@@
LINK         = $(CC) $(CFLAGS) $@@.$(OBJ) piologie.$(LIB) -o $@@
SUFFIX       = C
@}


\subsection{IBM Visal Age \cpp}
%------------------------------

@O config.i386_ibm
@{#
# Pentium, IBM Visual Age C++ 3.0:
#
CC           = icc
CFLAGS       = /G5 /Gf+ /Gi+ /Gu+ /O+ /Om- /D_Old_STD_
OBJ          = obj
LIB          = lib
COMPILE_ONLY = $(CC) /c
OUTPUT       =
MAKE_LIB     = ilib /nologo $@@ $(MP_OBJS),,
LINK         = $(CC) $@@.$(OBJ) piologie.$(LIB)
SUFFIX       = cpp
@}


\subsection{KAI \cpp}
%------------------------------

@O config.kai
@{#
# KAI C++ 3.2:
#
CC           = KCC
CFLAGS       = -O3 -DSTATIC_VS_INLINE=static
OBJ          = o
LIB          = a
COMPILE_ONLY = $(CC) -c
OUTPUT       = -o $@@
MAKE_LIB     = ar cr $@@ $(MP_OBJS); ranlib $@@
LINK         = $(CC) $(CFLAGS) $@@.$(OBJ) piologie.$(LIB) -o $@@
SUFFIX       = cpp
@}


\subsection{Microsoft Visual \cpp}
%---------------------------------

@O config.i386_vc
@{#
# Pentium, Microsoft Visual C++ 5.0:
#
CC           = cl
CFLAGS       = /nologo /ML /W3 /GX /O2
OBJ          = obj
LIB          = lib
COMPILE_ONLY = $(CC) /c
OUTPUT       = 
MAKE_LIB     = link -lib /nologo $(MP_OBJS) /out:$@@
LINK         = link /nologo $@@ piologie.$(LIB)
SUFFIX       = cpp
@}


\subsection{SGI \cpp}
%--------------------

@O config.sgi
@{#
# MIPS R4000, SGI MIPSpro C++ 7.1:
#
CC           = CC
CFLAGS       = -O2 -mips4 -D_Old_STD_
OBJ          = o
LIB          = a
COMPILE_ONLY = $(CC) -c
OUTPUT       = -o $@@
MAKE_LIB     = ar cr $@@ $(MP_OBJS)
LINK         = $(CC) $(CFLAGS) $@@.$(OBJ) piologie.$(LIB) -o $@@
SUFFIX       = cpp
@}

@O config.sgi_8000
@{#
# MIPS R8000, SGI MIPSpro C++ 7.1:
#
CC           = CC
CFLAGS       = -O2 -mips4 -64 -D_Old_STD_
OBJ          = o
LIB          = a
COMPILE_ONLY = $(CC) -c
OUTPUT       = -o $@@
MAKE_LIB     = ar cr $@@ $(MP_OBJS)
LINK         = $(CC) $(CFLAGS) $@@.$(OBJ) piologie.$(LIB) -o $@@
SUFFIX       = cpp
@}


\subsection{SUN WorkShop \cpp}
%-----------------------------

@O config.sun
@{#
# SUN WorkShop C++ 4.2:
#
CC           = CC
CFLAGS       = -fast -O5 -native -DDEFINE_VS_CONST -D_Old_STD_
OBJ          = o
LIB          = a
COMPILE_ONLY = $(CC) -c
OUTPUT       = -o $@@
MAKE_LIB     = ar cr $@@ $(MP_OBJS); ranlib $@@
LINK         = $(CC) $(CFLAGS) $@@.$(OBJ) piologie.$(LIB) -o $@@
SUFFIX       = cpp
@}


\subsection{Watcom \cpp}
%-----------------------

@O config.i386_wc
@{#
# Pentium, Watcom C++ 11.0:
#
CC           = wpp386
CFLAGS       = /5r /oneatx /zp4 /D_Old_STD_
OBJ          = obj
LIB          = lib
COMPILE_ONLY = $(CC)
OUTPUT       = 
MAKE_LIB     = wlib -b -c -n -q piologie.$(LIB) +$(MP_OBJS)
LINK         = wlink f $@@ l piologie.$(LIB)
SUFFIX       = cpp
@}
