\section{Eine einfache Festkommaklasse}
%======================================

F"ur die Berechnung von Konstanten gen"ugt es, eine Festkomma-Arithemtik einzuf"uhren, weil wir keinen
direkten Gebrauch eines flexiblen Exponenten brauchen, wie er zum Beispiel bei der Gleitkomma-Arithmetik n"otig ist.
Stattdessen verwenden wir einen impliziten Exponenten, der ja nach Objekt an einer anderen
Stelle liegt. Ein Beispiel: Das Objekt \texttt{Fixed(n)} mit $n\in\Size$ reserviert $n$ Dezimalen des abgeleiteten
\Natural s f"ur die Nachkommastellen.\\
Die Klasse \Fixed\ ist jedoch im Gegensatz zu den Klassen \Natural\ oder \Integer\ momentan noch nicht zur direkten Nutzung
durch den Benutzer vorgesehen. Sie ist von der Klasse \Natural\ abgeleitet und besteht haupts"achlich aus dem Konstruktor
und der Streamausgabe:
@D class \Fixed\
@{class Fixed : private Natural {
private:
  const size_t prec,dec;

public:
  Fixed(const size_t);
  Fixed(const Fixed&);

  void      output(ostream&);
  size_t    precision() const;
  size_t    decimals() const;
  Natural&  value() const;
};
@| Fixed @}

@D inline implementation of class \Fixed\
@{inline Natural& Fixed::value() const
{
  return (Natural&)*this;
}

inline Fixed::Fixed(const size_t n)
 : prec(NumberOfDecimals(n+ALPHA_WIDTH)), dec(n)
{
  RestoreSize();
}

inline Fixed::Fixed(const Fixed& a)
 : Natural(a.value()), prec(a.prec), dec(a.dec)
{
}

inline size_t Fixed::precision() const
{
  return prec;
}

inline size_t Fixed::decimals() const
{
  return dec;
}

inline ostream& operator<<(ostream& out, Fixed a)
// Algorithm:  o := operator<<(o, a)
// Input:      o in ostream, a in Fixed.
// Output:     o in ostream ||
//
// Note:       puts Fixed a on output stream.
{
  a.output(out);
  return out;
}
@| Fixed Fixed::Fixed Fixed::precision Fixed::decimals operator<< @}


\subsection{Streamausgabe}
%-------------------------

Die Streamausgabe entspricht bis auf die interne Festkommadarstellung der Ausgabe eines \Natural s aus dem
Kapitel \ref{Dezimalsystem}:
@D puts \Fixed\ on output stream
@{void Fixed::output(ostream& out)
// Algorithm:  a.output(o)
// Input:      o in ostream, a in Fixed.
// Output:     o in ostream ||
//
// Note:       puts Fixed a on output stream.
{
  const size_t block_size = 500;
  size_t n = decimals();
  size_t m = precision();
  static Natural c = pow(Natural(ALPHA), Digit(block_size));
  const size_t sB = length();
  enlarge((sB > m)? 3 : 3+m-sB);
  if (sB > m) {
    fast_rshift(m);
    out << value();
    if (n) out << '.';
    value() = 0; fast_append(m); normalize();
  } else out << "0.";
  while (n >= block_size*ALPHA_WIDTH) {
    value() *= c;
    out.width(block_size*ALPHA_WIDTH);
    out.fill('0');
    if (length() > m) {
      fast_rshift(m);
      out << value();
      value() = 0; fast_append(m); normalize();
      if (value() != 0) {
        Digit* pT;
        const size_t sz = trailing_zeros(pT);
        fast_rshift(sz); m -= sz;
      }
    } else out << '0';
    n -= block_size*ALPHA_WIDTH;
  }
  Digit* pE = last();
  Digit* pT = pE-m;
  while (n) {
    value() *= ALPHA;
    if (n <= ALPHA_WIDTH) {
       out.width(n);
       out.fill('0');
       Digit d = highest();
       for (n = ALPHA_WIDTH-n; n; --n) d /= 10;
       out << d;
       break;
    }
    out.width(ALPHA_WIDTH);
    out.fill('0');
    if (length() > m) {
      out << *pT;
      *pT = 0;
      if (*pE == 0) { --pE; fast_rshift(1); --m; }
      normalize();
    } else out << '0';
    n -= ALPHA_WIDTH;
  }
}
@| operator<< @}
\textbf{Laufzeit.} Sei \texttt{$o\in$ostream} und $f\in\Fixed$, dann \texttt{$f$.output($o$)$\in
\Ord\left(\frac{f\mathtt{.precision()}}{2}
\cdot\left\lfloor\frac{\beta f\mathtt{.precision()}}{\log_2\alpha}\right\rfloor\right)$}.\\

\textbf{Bemerkung.} Die Umwandlung ins Dezimalsystem wird blockweise durchgef"uhrt, um die Ausgabe zu beschleunigen.



\section{$\pi$-Berechnung}
%=========================

Zu den Klassikern der Mathematikgeschichte geh"ort das Problem der Quadratur des Kreises mit Zirkel und Lineal.
Die antiken Mathematiker haben sich  erfolglos darum bem"uht; ungel"ost schleppte sich die Aufgabe durch die 
Jahrhunderte fort bis weit in die Neuzeit hinein, genaugenommen bis ins Jahr 1882, als F. Lindemann zeigen konnte,
da"s $\pi$ transzendent "uber den rationalen Zahlen ist. Da jegliche Konstruktionen mit Zirkel und Lineal in der Algebra
aber quadratischen und  linearen Gleichungen entsprechen, ist die Konstruktion also unm"oglich.\\
Weil man parallel zu den geometrischen L"osungen auch nach numerischen gesucht hatte, machte schon 200 Jahre vor der
endg"ultigen Absage an die Konstruktion G. W. Leibniz einen entscheidenden Fortschritt in der $\pi$-Berechnung mit der
Entdeckung der Reihe $\frac{\pi}{4} = \arctan(1) = \sum_{n = 0}^\infty\frac{(-1)^n}{2n + 1}$.
Daraufhin haben viele Mathematiker durch trickreiches Anwenden der Additionstheoreme der Tangensfunktion eine immense
Anzahl von neuen Arkustangens-Reihen gebastelt (siehe Anhang \ref{Pi-Gleichungen}). Genaueres zu diesen Reihen findet man
in dem Artikel \cite{Lehm38} von D. H. Lehmer.\\
Im Laufe der Zeit sind noch andere merk- und denkw"urdige Beziehungen zur Zahl $\pi$ hinzugekommen, wie zum Beispiel die,
da"s die Wahrscheinlichkeit, da"s zwei nat"urliche Zahlen teilerfremd sind, gerade $\frac{6}{\pi^2}$ betr"agt, oder 
die Entstehung von neuen Multiplikationsalgorithmen (A. Karatsuba, siehe Kapitel \ref{Karatsuba-Multiplikation})
oder auch die Formel $e^{\pi i} = -1$ mit der Eulerschen Zahl $e = \lim_{n\to\infty}(1 + \frac{1}{n})^n$
und der imagin"aren Einheit $i^2 = -1$.\\
Heute wird die Kreiszahlermittlung gerne f"ur Benchmark beziehungsweise
Prozessor- oder Arithme\-tik-Testsysteme verwendet, da sich schon kleine Fehler in der Berechnung sofort bemerkbar machen.


\subsection{Historische Zusammenfassung "uber die $\pi$-Berechnung}
%------------------------------------------------------------------

\begin{enumerate}
\item Durch handschriftliche Berechnung:\\
Die ersten ernsthaften Versuche, die Zahl $\pi$ genauer zu bestimmen, gehen auf \textbf{Archimedes} (287-212 v. Chr.)
zur"uck. Er f"uhrte Untersuchungen an regelm"a"sigen Vielecken durch und konnte $\pi$ auf 2 Dezimalen genau 
eingrenzen ($3\frac{10}{71} < \pi < 3\frac{1}{7}$).\\
\textbf{Ptolem"aus} (150)\dotfill auf 4 Dezimalen genau\\
mit dem Bruch $\frac{377}{120}$.\\
\textbf{Tsu Ch'ung} (480)\dotfill auf 6 Dezimalen genau\\ 
mit dem Bruch $\frac{355}{113}$.\\
\textbf{Ludolph von Coelen} (1539-1610)\dotfill auf 32 Dezimalen genau\\
Er machte es zu seiner Lebensaufgabe, die Umf"ange von ein- und umbeschriebenen Vielecken mit $2^{62}$ Seiten zu berechnen.\\
\textbf{Abraham Sharp} (1653-1742)\dotfill auf 72 Dezimalen genau\\
mit der Formel $\frac{\pi}{2} = \sqrt{3}\sum_{n = 0}^\infty\frac{(-1)^n}{3^n(2n + 1)}$.\\
\textbf{John Machin} (1706)\dotfill auf 100 Dezimalen\\
mit der Formel $\frac{\pi}{4} = 4 \arctan\left(\frac{1}{5}\right) - \arctan\left(\frac{1}{239}\right)$.\\
\textbf{De Lagny} (1719)\dotfill auf 127 Dezimalen (davon 112 korrekt)\\
\textbf{Vega} (1754-1802)\dotfill auf 140 Dezimalen\\
\textbf{Zacharias Dase} (1844)\dotfill auf 200 Dezimalen\\
Deutscher Rechenk"unstler, konnte hundertstellige Zahlen in etwa acht Stunden im Kopf multiplizieren.
Er benutzte die Formel
$\frac{\pi}{4} = \arctan\left(\frac{1}{2}\right) + \arctan\left(\frac{1}{5}\right) + \arctan\left(\frac{1}{8}\right)$
und ben"otigte knapp zwei Monate zur Berechnung der 200 Stellen von $\pi$.\\
\textbf{William Rutherford} (1853)\dotfill auf 400 Dezimalen\\
\textbf{William Shanks} (1853)\dotfill auf 707 Dezimalen (davon 527 korrekt)\\
Es vergingen 92 Jahre, bis der Rechenfehler entdeckt wurde.\\

\item Mittels elektronischer Rechenanlagen:\\
\textbf{D. F. Ferguson} (1947)\dotfill auf 808 Dezimalen\\
mit der Formel
$\frac{\pi}{4} = 3 \arctan\left(\frac{1}{4}\right) + \arctan\left(\frac{1}{20}\right) + \arctan\left(\frac{1}{1985}\right)$.\\
\textbf{John von Neumann et al.} (1949)\dotfill auf 2037 Dezimalen\\
berechnet in 70 Stunden mit Machins Formel auf dem R"ohrenmonstrum ENIAC (18 000 Vakuumr"ohren).\\
\textbf{S. C. Nicholson, J. Jeenel} (1954)\dotfill auf 3092 Dezimalen\\
\textbf{G. E. Felton} (1957)\dotfill 10 000 Dezimalen (davon 7480 korrekt)\\
berechnet auf einem Ferranti PEGASUS in 33 Stunden.\\
\textbf{Fran\c{c}ois Genuys} (1958)\dotfill 10 000 Dezimalen\\
berechnet auf einem IBM 704 in 100 Minuten.\\
\textbf{Jean Guilloud} (1959)\dotfill 16 167 Dezimalen\\
\textbf{John W. Wrench jr., Daniel Shanks} (1961)\dotfill 100 265 Dezimalen\\
berechnet auf einem IBM 7090 in weniger als 9 Stunden mit der Formel von St"ormer\\
$\frac{\pi}{4} = 6\arctan\left(\frac{1}{8}\right) + 2\arctan\left(\frac{1}{57}\right) + \arctan\left(\frac{1}{239}\right)$.\\
\textbf{J. Guilloud, J. Filliatre} (1966)\dotfill 250 000 Dezimalen\\
\textbf{J. Guilloud, M. Dichampt} (1967)\dotfill 500 000 Dezimalen\\
berechnet auf einer CDC 6600.\\
\textbf{Jean Guilloud, Martine Bouyer} (1973)\dotfill 1 001 250 Dezimalen\\
berechnet auf einer CDC 7600 in knapp einem Tag Rechenzeit mit der Formel von C.~F.~Gau"s\\
$\frac{\pi}{4} = 12 \arctan\left(\frac{1}{18}\right) + 8 \arctan\left(\frac{1}{57}\right) - 5 \arctan\left(\frac{1}{239}\right)$.\\
\textbf{Y. Tamura} (1982)\dotfill 2 097 144 Dezimalen\\
\textbf{Y. Tamura, Y. Kanada} (1982)\dotfill 4 194 288 Dezimalen\\
\textbf{Y. Tamura, Y. Kanada} (1982)\dotfill 8 388 576 Dezimalen\\
\textbf{Kannada, Tamura, Yoshino, Ushiro} (1983)\dotfill 16 777 216 Dezimalen\\
berechnet auf einer HITAC M-280H in 30 Stunden mit der quadratisch konvergenten Iteration von P.~Brent und E.~Salamin
$a_0 = 1, b_0 = \frac{\sqrt{2}}{2}, c_0 = \frac{1}{4};\\
a_{n+1} = \frac{a_n + b_n}{2}, b_{n+1} = \sqrt{b_na_n}, c_{n+1} = c_n - 2^n(a_{n+1} - a_n)^2$\qquad f"ur $n\in\N$.\\
Es gilt $\lim_{n\to\infty}\frac{(a_n + b_n)^2}{4c_n} = \pi$.\\
\textbf{R. William Gosper} (1985)\dotfill 17 526 200 Dezimalen\\
berechnet auf einer Symbolics 3670 mit der Reihe von Srinivasa Ramanujan\\
$\frac{1}{\pi} = \frac{2\sqrt{2}}{9801}\sum_{n = 0}^\infty\frac{(4n)!}{(n!)^4}\frac{1103 + 26390n}{396^{4n}}$,
wobei noch heute kein Beweis existiert, ob diese Summe tat\-s"ach\-lich gegen $\frac{1}{\pi}$ konvergiert.\\
\textbf{David H. Bailey} (1986)\dotfill 29 360 111 Dezimalen\\
berechnet auf einem CRAY-2-Supercomputer der amerikanischen Luft- und 
Raum\-fahrt\-be\-h"or\-de NASA in weniger als 28 Stunden.\\
\textbf{Y. Kanada, Y. Tamura} (1986)\dotfill 33 554 414 Dezimalen\\
\textbf{Y. Kanada, Y. Tamura} (1986)\dotfill 67 108 839 Dezimalen\\
\textbf{Y. Kanada, Y. Tamura, Y. Kubo, et al.} (1987)\dotfill 134 214 700 Dezimalen\\
berechnet auf einem SX-2-Supercomputer der Firma NEC.\\
\textbf{Y. Kanada, Y. Tamura} (1988)\dotfill 201 326 551 Dezimalen\\
\textbf{G. V. Chudnovsky und D. V. Chudnovsky} (1989)\dotfill 480 000 000 Dezimalen\\
\textbf{G. V. Chudnovsky und D. V. Chudnovsky} (1989)\dotfill 525 229 270 Dezimalen\\
\textbf{Y. Kanada, Y. Tamura} (1989)\dotfill 536 870 898 Dezimalen\\
\textbf{G. V. Chudnovsky und D. V. Chudnovsky} (1989)\dotfill 1 011 196 691 Dezimalen\\
berechnet auf einer IBM 3090 mit ihrer eigenen Formel\\
$\frac{1}{\pi} = \frac{1}{53360\sqrt{640320}}\sum_{n = 0}^\infty(-1)^n\frac{(6n)!}{(3n)!(n!)^3}\frac{13591409 + 545140134n}{640320^{3n}}$\\
\textbf{Y. Kanada, Y. Tamura} (1989)\dotfill 1 073 741 799 Dezimalen\\
\textbf{G. V. Chudnovsky und D. V. Chudnovsky} (1991)\dotfill 2 260 000 000 Dezimalen\\
\textbf{D. Takahashi, Y. Kanada} (1995)\dotfill 3 221 225 466 Dezimalen\\
berechnet auf einer HITAC S-3800/480 in knapp 37 Stunden mit dem quartisch konvergenten
Algorithmus der Gebr"uder Borwein $a_0 = 6 - 4\sqrt{2}, b_0 = \sqrt{2} - 1,\\
b_{n+1} = \frac{1 - \sqrt[4]{1 - b_n^4}}{1 + \sqrt[4]{1 - b_n^4}},
a_{n+1} = a_n(1 + b_{n+1})^4 - 2^{2n + 3}b_{n+1}(1 + b_{n+1} + b_{n+1}^2)$
\qquad f"ur $n\in\N$.\\
\textbf{G. V. Chudnovsky und D. V. Chudnovsky} (1994)\dotfill 4 044 000 000 Dezimalen\\
\textbf{D. Takahashi, Y. Kanada} (1995)\dotfill 4 294 967 286 Dezimalen\\
\textbf{D. Takahashi, Y. Kanada} (1995)\dotfill 6 442 450 938 Dezimalen\\
berechnet auf einer HITAC S-3800/480 in 112 Stunden mit dem quartisch konvergenten Algorithmus der Gebr"uder Borwein.\\
\textbf{Y. Kanada, D. Takahashi} (1997)\dotfill 51 539 600 000 Dezimalen\\
berechnet auf einer HITACHI SR2201 mit 1024 Prozessoren in 29 Stunden mit dem quartisch konvergenten Algorithmus der Gebr"uder Borwein.
\end{enumerate}


\subsection{$\pi$ beschrieben durch den Arkustangens}
%----------------------------------------------------

Wir greifen nun eine ganz bestimmte Formel zur $\pi$-Berechnung bis etwa 10 000 Dezimalen heraus:

\subsubsection{Arkustangens-Reihe von St"ormer}

\[\pi = 24\arctan\left(\frac{1}{8}\right) + 8\arctan\left(\frac{1}{57}\right) + 4\arctan\left(\frac{1}{239}\right).\]
Um die Arkustangens-Formel von St"ormer einsetzen zu k"onnen, ben"otigen wir noch die Arkustangens-Reihe, die im Jahre 1671 von Gregory entdeckt wurde:
\begin{eqnarray*}
\arctan\left(\frac{1}{x}\right) & = & \sum_{n \geq 0}\frac{(-1)^n}{(2n + 1)x^{2n+1}}\\
& = & \sum_{n \geq 0}\frac{1}{(4n + 1)x^{4n+1}} - \sum_{n \geq 0}\frac{1}{(4n + 3)x^{4n+3}}\\
& = & \sum_{n \geq 0}\frac{(4n + 3) - (4n + 1)x^2}{(4n + 1)(4n + 3)x^{4n+3}}.\\
\end{eqnarray*}
Um die Konvergenz etwas zu beschleunigen, fassen wir immer zwei aufeinanderfolgende Summanden aus der alternierenden Reihe
zusammen.\\
Die Implementierung sieht dann folgenderma"sen aus:
@D Pi-Calculation with Stoermer-Algorithms
@{void Pi::stoermer(const size_t n, Natural& pi)
// Algorithm:  p.stoermer(n, a)
// Input:      p in Pi, a in Natural, n in size_t
//             where n < 3(GAMMA_LOW+1)/BETA, BETA >= 32.
// Output:     a in Natural such that |a-pi*2^(BETA*n)| < 1 ||
{
  const size_t sz = BETA*n;
  Natural t;
  Natural a = Digit(8*24);
  Natural b = Digit(57*8);
  Natural c = Digit(239*4);
  pi        = 0;
  Digit   k = 0;

  a <<= sz; b <<= sz; c <<= sz;
  do {
    a >>= 12;
    t = a * (63*k+191);
    if (b != 0) {
      b /= Digit(57*57*57*57);
      t += b * Digit(3248*k+9746);
      if (c != 0) {
        c /= Digit(239UL*239*239*239);
        t += c * Digit(57120*k+171362);
      }
    }
    pi += t /= (k+1)*(k+3);
    k += 4;
  } while (t != 0);
}
@| Pi::stoermer stoermer @}
\textbf{Laufzeit.} Sei $p\in\PiClass$, $x\in\Natural$ und $t\in\Size$, dann\smallskip\\
\mbox{}\hfill\texttt{$p$.stoermer($t$,$x$)$\in\frac{t\beta}{3\log_{10}\alpha}
\cdot\Ord(\texttt{operator/=}:\Natural\times\Digit\to\Natural)$}.\hfill\mbox{}\\

Dieser Algorithmus ist f"ur einige tausend Dezimalstellen kompakt und sehr schnell, weil der am langsamsten konvergierende
Summand eine Zweierpotenz ist und dadurch unserer internen Darstellung angepa"st ist. Genauer gesagt liegt unsere
Genauigkeitsschranke bei $2^{\frac{\beta}{2}}\cdot \log_{10}(8)$ Stellen, da danach ein "Uberlauf im
\Digit bereich bei \texttt{(k+1)*(k+3)} auftritt. In Zahlen ausgedr"uckt hei"st das, da"s wir bei einem 32-Bit-System
59184 dezimale Nachkommastellen ermitteln k"onnen. Doch f"ur diese Gr"o"sen existieren weitaus schneller konvergierende
Algorithmen.


\subsection{Ramanujan-Reihen}
%----------------------------

Srinivasa Ramanujan hat sehr viele Reihen zur Berechnung von $\pi$ angegeben (siehe Anhang \ref{Ramanujan-Pi-Gleichungen}, Seite
\pageref{Ramanujan-Pi-Gleichungen}f). Die Chudnovsky Br"uder formulierten 1987 aus der Verallgemeinerung dieser Reihen die am
schnellsten konvergierende Reihe, die mit 32-Bit \Digit s auskommt:

\begin{eqnarray*}
\frac{1}{\pi} & = & \frac{12}{C^{\frac{3}{2}}}\sum_{n\geq0}(-1)^n\frac{(6n)!}{(3n)!(n!)^3}\frac{A + Bn}{C^{3n}}\\
 &=& \frac{12}{C^{\frac{3}{2}}}\sum_{n\geq0}(-1)^n\left(\prod_{k=1}^n\frac{6k(6k-1)(6k-2)(6k-3)(6k-4)(6k-5)}{3k(3k-1)(3k-2)}\right)\frac{A + Bn}{(n!)^3\cdot C^{3n}}\\
 &=& \frac{12}{C^{\frac{3}{2}}}\sum_{n\geq0}(-1)^n\left(\prod_{k=1}^n8(6k-1)(6k-3)(6k-5)\right)\frac{A + Bn}{(n!)^3\cdot C^{3n}}\\
 &=& \frac{12}{C^{\frac{3}{2}}}\sum_{n\geq0}(-1)^n\frac{\left(\prod_{k=1}^n(6k-1)(2k-1)(6k-5)\right)}{(n!)^3}\frac{A + Bn}{26680\cdot C^{2n}}\\
 && \mbox{mit $A = 13591409, B = 545140134, C = 640320$}.
\end{eqnarray*}

Diese Formel liefert etwa 14,1643 dezimale Stellen pro Term.\\

Bei der Berechnung dieser linear konvergierenden Reihe verwenden wir einen rekursiven Algorithmus, der die schnelle Multiplikation
ausnutzt. Kurz gesagt werden alle rationalen Zahlen (die Terme) miteinander addiert ohne ihren gemeinsamen Teiler zu entfernen.
Eine genaue Beschreibung dieses Algorithmus einschlie"slich einiger Beispiele f"uhren Bruno Haible und Thomas Papanikolaou in
\cite{HaPa97} durch.

@D Pi-Calculation with Chudnovsky-Algorithms
@{void Pi::chudnovsky(size_t n, const size_t m, Integer& p, Integer& q, Integer& t) const
// Algorithm:  p.chudnovsky(n, m, p, q, t)
// Input:      p in Pi, p,q,t in Integer, n,m in size_t where n < 2^BETA/6, n < m.
// Output:     p,q,t in Integer ||
{
  CONDITION(n < m && n < ((size_t(1) << (BETA-1))/3));

  const Digit A = Digit(13591409);
  const Digit B = Digit(545140134);
  const Digit C = Digit(640320);
  switch (m-n) {
    case 1: {
      @<special case to evaluate one term@>
      break;
    }
    case 2: {
      @<special case to evaluate two terms@>
      break;
    }
    case 3: {
      @<special case to evaluate three terms@>
      break;
    }
    default: {
      Integer p1,q1,t1;
      Integer p2,q2,t2;

      const size_t l = (n+m)/2;
      chudnovsky(n, l, p1, q1, t1);
      chudnovsky(l, m, p2, q2, t2);
      t = t1*q2; t += t1 = t2*p1;
      p = p1*p2; q = q1*q2;
    }
  }
}
@| Pi::chudnovsky chudnovsky @}
\textbf{Laufzeit.} Sei $p\in\PiClass$, $u,v,w\in\Integer$ und $t_0,t_1\in\Size$, dann\smallskip\\
\mbox{}\hfill\texttt{$p$.chudnovsky($t_0$,$t_1$,$u$,$v$,$w$)$\in
\Ord(\log_2(t_1-t_0)^2\cdot\Ord(\texttt{mul}:\Integer^2\to\Integer)$}.\hfill\mbox{}

@D Pi-Calculation with Chudnovsky-Algorithms
@{void Pi::chudnovsky2(const size_t n, const size_t m, Integer& q, Integer& t) const
// Algorithm:  p.chudnovsky(n, m, q, t)
// Input:      p in Pi, q,t in Integer, n,m in size_t where n < m.
// Output:     q,t in Integer ||
{
  CONDITION(n < m);

  Integer p1,q1,t1;
  Integer q2,t2;

  const int l = (n+m)/2;
  chudnovsky(n, l, p1, q1, t1);
  if (m-l <= 3) {
    Integer p2;
    chudnovsky(l, m, p2, q2, t2);
  } else chudnovsky2(l, m, q2, t2);
  t = t1*q2; t += t1 = t2*p1;
  q = q1*q2;
}
@| Pi::chudnovsky2 chudnovsky2 @}
\textbf{Laufzeit.} Sei $p\in\PiClass$, $u,v,w\in\Integer$ und $t_0,t_1\in\Size$, dann\smallskip\\
\mbox{}\hfill\texttt{$p$.chudnovsky2($t_0$,$t_1$,$v$,$w$) $\sim$ $p$.chudnovsky($t_0$,$t_1$,$u$,$v$,$w$)}.\hfill\mbox{}\\

Zur schnellen Berechnung der Terme und Verringerung der Rekursionstiefe berechnen wir bis zu drei Terme explizit durch die
folgenden drei Programmteile:

@D special case to evaluate one term
@{if (n <= GAMMA_LOW/4) p = (6*n-5)*(2*n-1);
else { p = 6*n-5; p *= 2*n-1; }
p *= 6*n-1; p = -p;
if (n <= GAMMA_LOW) q = n*n;
else { q = n; q *= n; }
q *= n; q *= 230*C; q *= 116*C;
t = B; t *= n; t += A; t *= p;
@}

@D special case to evaluate two terms
@{Integer s1,s2;
if (n <= GAMMA_LOW/4) {
  p = (6*n-5)*(2*n-1);
  s1 = p *= 6*n-1; s1 = -s1;
  p *= 6*n+5; p *= (2*n+1)*(6*n+1);
} else {
  p = 6*n-5; p *= 2*n-1;
  s1 = p *= 6*n-1; s1 = -s1;
  p *= 6*n+5; p *= 2*n+1; p *= 6*n+1;
}
if (n < GAMMA_LOW) {
  const size_t k = n+1;
  q = k*k; q *= k; q *= 230*C;
  s2 = q *= 116*C; q *= n*n;
} else {
  const size_t k = n+1;
  q = k; q *= k; q *= k; q *= 230*C;
  s2 = q *= 116*C; q *= n; q *= n;
}
q *= n; q *= 230*C; q *= 116*C;
t = B; t *= n; t += A;
const Integer s3 = t;
t += B;
const Integer s4 = t*p;
t = s1*s2*s3;
t += s4;
@}

@D special case to evaluate three terms
@{Integer s1,s2;
if (n <= GAMMA_LOW/6) {
  p = (6*n+1)*(2*n-1); p *= (6*n-1)*(6*n-5);
  s1 = p *= (2*n+1)*(6*n+5);
  p *= (6*n+7)*(2*n+3);
} else {
  p = 6*n-5; p *= 2*n-1; p *= 6*n-1;
  p *= 6*n+5; p *= 2*n+1;
  s1 = p *= 6*n+1;
  p *= 6*n+7; p *= 2*n+3;
}
p *= 6*n+11; p = -p;
if (n < GAMMA_LOW-1) {
  const size_t k = n+1;
  q = k*k; q *= k; q *= 230*C; q *= 116*C;
  const size_t k2 = n+2;
  q *= k2*k2; q *= k2; q *= 230*C;
  s2 = q *= 116*C; q *= n*n;
} else {
  const size_t k = n+1;
  q = k; q *= k; q *= k; q *= 230*C; q *= 116*C;
  const size_t k2 = n+2;
  q *= k2; q *= k2; q *= k2; q *= 230*C;
  s2 = q *= 116*C; q *= n; q *= n;
}
q *= n; q *= 230*C; q *= 116*C;
t = B; t *= n; t += A;
const Integer s3 = t;
t += B;
const Integer s4 = t;
t += B;
const Integer s5 = t*p;
t = s2*s3;
if (n <= GAMMA_LOW/4) t *= (6*n-5)*(2*n-1);
else { t *= 6*n-5; t *= 2*n-1; }
t *= 6*n-1; t = -t;
s2 = s1*s4;
n += 2;
if (n <= GAMMA_LOW) s2 *= n*n;
else { s2 *= n; s2 *= n; }
s2 *= n; s2 *= 230*C; s2 *= 116*C;
t += s2; t += s5;
@}



%
% Result: the following programms are slower.
%
% void Pi::linear(const size_t n, Natural& z) const
% {
% /*
%   Natural a = 1;
%   Digit k = 1;
% 
%   z = 5; z <<= n*BETA;
%   a <<= n*BETA;
% 
%   while (true) {
%     a >>= 9; a /= k; a /= k; a /= k;
%     if (a == 0) break;
%     z += a * Digit(42*k+5);
% 
%     const Digit i = 2*k+1;
%     a *= i; a *= i; a *= i;
%     ++k;
%   }
% 
%   z >>= 4;
% */
%   const Digit A = 13591409;
%   const Digit B = 545140134;
%   const Digit C = 640320;
%   Natural t,a = 1;
%   Natural c = A+B;
%   Digit k = 1;
% 
%   z = A; z <<= n*BETA;
%   a <<= n*BETA;
% 
%   while (true) {
%     if (k <= GAMMA_LOW) a /= k*k;
%     else { a /= k; a /= k; }
%     a /= k; a /= 23*10*C; a /= 29*4*C;
%     if (a == 0) break;
%     if (k <= GAMMA_LOW/4) a *= (6*k-5)*(2*k-1);
%     else { a *= 6*k-5; a *= 2*k-1; }
%     a *= 6*k-1;
%     t = a*c;
%     if (k&1) z -= t;
%     else z += t;
% 
%     c += B;
%     ++k;
%   }
%   a = 2208*C; a *= 290*C; a <<= n*BETA*2;
%   t = sqrt(a); t <<= n*BETA;
%   z = t / z; z /= 12;
% }



\subsection{Das arithmetisch-geometrische Mittel}
%------------------------------------------------

Die bisherigen Formeln haben den Vorteil, da"s sie ausschlie"slich elementare Grundoperationen benutzen, aber den Nachteil,
da"s sie nur linear konvergieren. Das bedeutet also zum Beispiel, da"s sich die Laufzeit vervierfacht, wenn wir die
gew"unschte Stellenanzahl verdoppeln.\\
Die neuesten Methoden zur Berechnung von $\pi$ beruhen auf den Untersuchungen, die C. F. Gau"s zum arithmetischen und
geometrischen Mittel zweier Zahlen machte:\\

Seien $a_0 = 1, b_0 = \frac{1}{2}\sqrt{2}, c_0 = \frac{1}{4}$ und
\begin{eqnarray*}
a_{n+1} &=& \frac{a_n + b_n}{2},\\
b_{n+1} &=& \sqrt{b_na_n},\\
c_{n+1} &=& c_n - 2^n(a_{n+1} - a_n)^2\quad\mbox{ f"ur $n\in\N$},
\end{eqnarray*}
dann gilt
\[\lim_{n\to\infty}\frac{(a_n + b_n)^2}{4c_n} = \pi.\]

Eine ausf"uhrliche Analyse und Betrachtung dieser Gleichung sowie theoretische Hintergr"unde findet man im Buch
\cite{BoBo87} von J. M. Borwein und P. B. Borwein.\\
Zusammenfassend gesagt beruht dieser Algorithmus auf einer Schleife, die bei jedem Durchlauf die Anzahl der korrekten
Stellen ann"ahernd verdoppelt. Dies ergibt zum Beispiel bereits bei der neunten Iteration ungef"ahr eine Millionen Stellen.\\

Bei der Umsetzung dieser Formel verwenden wir einige Optimierungsvorschl"age von A. Sch"onhage aus \cite{Scho94}
(Seite 266):
@D Pi-Calculation with Sch\"onhage-Algorithms
@{void Pi::schoenhage(const size_t n, Natural& pi)
// Algorithm:  p.schoenhage(n, a)
// Input:      p in Pi, a in Natural, n in size_t.
// Output:     a in Natural such that |a-pi*2^(BETA*n)| < 1 ||
{
  const size_t m = size_t(log2(BETA*n+BETA)-1.68025927);
  const size_t sz = BETA*n;
 
  Integer s,t,a(1),A(1),b(1),B(1),c(1);
  a <<= sz; A <<= sz; B <<= sz-1; c <<= sz-2;
  for (size_t k = 0; k <= m; ++k) {
    t = A+B; t >>= 2;
    s = B << sz; b = sqrt(s);
    a += b; a >>= 1;
    A = a*a; A >>= sz;
    B = A-t; B <<= 1;
    t = A-B; t <<= k; c -= t;
  }
  A <<= sz;
  div(A, c, t, a);
  pi = abs(t);
}
@| Pi::schoenhage schoenhage @}
\textbf{Laufzeit.} Sei $p\in\PiClass$, $x\in\Natural$ und $t\in\Size$, dann\smallskip\\
\mbox{}\hfill\texttt{$p$.schoenhage($t$,$x$)$\in
\Ord(\log_2(t)\cdot\Ord(\texttt{sqrt}:\Natural\to\Natural)$}.\hfill\mbox{}\\

%  Natural t,a = 1,b = 1,c = 1;
%  a <<= sz; b <<= sz-1; c <<= sz-2;
%  b <<= sz; ::sqrt(b, b);
%  for (size_t k = 0; k < m; ++k) {
%    t = a; a += b; a >>= 1;
%    ::mul(b, t, b); ::sqrt(b, b);
%    t -= a; ::sqr(t, t); t >>= sz;
%    t <<= k; c -= t;
%  }
%  a += b; ::sqr(a, b); c <<= 2;
%  ::div(b, c, pi, a);



\subsection{Auswahl der Methoden}
%--------------------------------

Im Konstruktor der Klasse \PiClass\ w"ahlen wir nun zwischen den vorgestellten Algorithmen denjenigen aus, der die
gew"unschte Anzahl an dezimalen Nachkommastellen von $\pi$ am schnellsten berechnet:

@D select $\pi$-Algorithm
@{Pi::Pi(const size_t n)
// Algorithm:  a := Pi(n)
// Input:      n in size_t.
// Output:     a in Pi such that |a-pi*10^n| < 1 ||
 : pi(n)
{
  const size_t sz = pi.precision();

  if (BETA < 32) schoenhage(sz, pi.value());
  else if (sz < (3*GAMMA_LOW+3)/BETA && n < 10000)
    stoermer(sz, pi.value());
  else if (n >= 100000000 || n >= ((size_t(1) << (BETA-1))/3))
    schoenhage(sz, pi.value());
  else {
    const Digit A = Digit(13591409);
    const Digit C = Digit(640320);

    Integer p1,q1,t1;
    Integer q2,t2;

    const size_t k = size_t(n/14.1643);
    const size_t l = (k+1)/2;
    chudnovsky(1, l, p1, q1, t1);
    chudnovsky2(l, k, q2, t2);
    Integer t = t1*q2;
    t += t1 = t2*p1;
    p1 = q1*q2;
    t += p1*A;
    t2 = Digit(2208*C); t2 *= Digit(290*C); t2 <<= BETA*sz*2;
    t1 = sqrt(t2); t2 = p1*t1;
    pi.value() = abs(t2) / abs(t);
    pi.value() /= 12;
  }
}
@| Pi::Pi Pi @}
\textbf{Laufzeit.} Sei $p\in\PiClass$, $x\in\Natural$, $u,v,w\in\Integer$ und $t\in\Size$, dann \[\texttt{Pi(}t\texttt{)} \sim 
\left\{\begin{array}{r@@{\;,\quad}l}
p\texttt{.schoenhage(}t,x\texttt{)} & \beta < 32\\
p\texttt{.stoermer(}t,x\texttt{)} & \frac{t\beta\log_{10}(2)}{\log_{10}(\alpha)} < \frac{3}{\beta}(\gamma_{low}+3) \wedge t<10000\\
p\texttt{.schoenhage(}t,x\texttt{)} & t\geq10^8\vee t\geq\frac{2^\beta}{6}\\
p\texttt{.chudnovsky(}0,t,u,v,w\texttt{)} & \mbox{ sonst}\\
\end{array}\right..\]

Die Ausgabe der Zahl $\pi$ erfolgt durch die Weiterleitung an die Festkommaklasse \Fixed:

@D puts \PiClass\ on output stream
@{inline ostream& operator<<(ostream& out, const Pi& a)
// Algorithm:  o := o << a
// Input:      o in ostream, a in Pi.
// Output:     o in ostream ||
//
// Note:       puts Pi a on output stream.
{
  return out << a.pi;
}
@| operator<< @}
\textbf{Laufzeit.} Sei \texttt{$o\in$ostream}, $p\in\PiClass$ und $f\in\Fixed$, dann
\texttt{$o$<<$p\sim\ o$<<$f$}.



\section{Ap\'{e}ry's Konstante}
%==============================

Die Ap\'{e}ry's Konstante\index{Ap\'{e}ry's Konstante} ist die Riemannsche Zetafunktion an der Stelle 3, $\zeta(3)$, wobei die Riemannsche Zetafunktion
folgenderma"sen definiert ist:
\[\zeta(x) := \sum_{n\geq1}\frac{1}{n^x}\mbox{f"ur $x\in\N,\, x\geq2.$}\]
$\zeta(3)\approx1.202056903\ldots$ wird erst seit 1979, aber wohl verdient als Ap\'{e}ry's Konstante bezeichnet, weil es ihm in diesem Jahr gelungen
ist, die Irrationalit"at dieser Konstanten zu beweisen.\\
Theodor Amdeberhan und Doron Zeilberger stellten in \cite{AmdZei97} die folgende schnell konvergente Reihe vor:

\begin{eqnarray*}
\zeta(3) &=& \frac{1}{2}\sum_{n\geq1}(-1)^{n-1}\frac{205n^2-160n+32}{n^5\binkoef{2n}{n}^5}\\
&=& \frac{1}{2}\sum_{n\geq1}(-1)^{n-1}\frac{(205n^2-160n+32)(n!)^{10}}{n^5((2n)!)^5}\\
&=& \frac{1}{2}\sum_{n\geq1}(-1)^{n-1}\frac{205n^2-160n+32}{n^5}\left(\prod_{k=1}^n\frac{k\cdot k}{k(n+k)}\right)^5\\
&=& \frac{1}{2}\sum_{n\geq1}(-1)^{n-1}\frac{205n^2-160n+32}{n^5}\left(\prod_{k=1}^n\frac{k\cdot k}{2k(2k-1)}\right)^5\\
&=& \frac{1}{2}\sum_{n\geq1}(-1)^{n-1}\frac{(205n^2-160n+32)(n!)^5}{n^5\cdot 32^n\cdot\prod_{k=1}^{n}(2k-1)^5}\\
&=& \frac{1}{2}\sum_{n\geq0}(-1)^{n}\frac{(205(n+1)^2-160(n+1)+32)((n+1)!)^5}{(n+1)^5\cdot 32^{n+1}\cdot\prod_{k=1}^{n+1}(2k-1)^5}\\
&=& \frac{1}{64}\sum_{n\geq0}(-1)^{n}\frac{(205n^2+250n+77)(n!)^5}{32^{n}\cdot\prod_{k=0}^{n}(2k+1)^5}
\end{eqnarray*}

Eine direkte Implementierung der linear konvergienden Reihe ist bis zu 3000 dezimalen Stellen effizient und
schnell realisiert:
@D Zeta(3)-Calculation
@{void Zeta3::linear(const size_t n, Natural& z) const
// Algorithm:  b.linear(n, a)
// Input:      b in Zeta3, a in Natural, n in size_t where b.decimals()/574 <= GAMMA.
// Output:     a in Natural such that |a-Zeta(3)*2^(BETA*n)| < 1 ||
{
  CONDITION((zeta.decimals()/1.4)/410 <= GAMMA);

  Natural a = 1;
  Natural c = 77;
  Natural t;

  a <<= n*BETA-5;
  z = 0;
  Digit k = 1;
  do {
    t = a * c;
    if (k&1) z += t;
    else z -= t;

    c += 410*k+45;
    if (k <= GAMMA_LOW) { a *= k*k; a *= k*k; }
    else { a *= k; a *= k; a *= k; a *= k; }
    a *= k;
    const Digit i = 2*k+1;
    if (i <= GAMMA_LOW) { a /= i*i; a /= i*i; }
    else { a /= i; a /= i; a /= i; a /= i; }
    a /= 32*i;
    ++k;
  } while (a != 0);
  z >>= 1;
}
@| Zeta3::linear @}
\textbf{Laufzeit.} Sei $z\in\texttt{Zeta3}$, $x\in\Natural$ und $t\in\Size$, dann\smallskip\\
\mbox{}\hfill\texttt{$z$.linear($t$,$x$)$\in\Ord(\texttt{operator/=}:\Natural\times\Digit\to\Natural)$}.\hfill\mbox{}\\

Genauso wie bei der $\pi$-Berechnung verwenden wir nun auch hier den rekursiven Algorithmus, der von Bruno Haible und Thomas Papanikolaou in
\cite{HaPa97} beschrieben wird:

@D Zeta(3)-Series-Calculation
@{void Zeta3::series(size_t n, const size_t m, Integer& p, Integer& q, Integer& t) const
// Algorithm:  z.series(n, m, p, q, t)
// Input:      z in Zeta3, p,q,t in Integer, n,m in size_t where n < 2^BETA/64-32, n < m.
// Output:     p,q,t in Integer ||
{
  CONDITION(n < m && n < ((size_t(1) << (BETA-1))/64)-32);

  if (n+1 == m) {
    if (n <= GAMMA_LOW) { p = n*n; p *= n*n; }
    else { p = n; p *= n; p *= n; p *= n; }
    p *= n; p = -p;
    if (n < GAMMA/207) t = 205*n+250;
    else { t = n; t *= 205; t += 250; }
    t *= n; t += 77; t *= p;
    n = 2*n+1;
    if (n <= GAMMA_LOW) { q = n*n; q *= n*n; }
    else { q = n; q *= n; q *= n; q *= n; q *= n; }
    q *= 32*n;
  } else {
    Integer p1,q1,t1;
    Integer p2,q2,t2;

    const int l = (n+m)/2;
    series(n, l, p1, q1, t1);
    series(l, m, p2, q2, t2);
    t = t1*q2; t += t1 = t2*p1;
    p = p1*p2; q = q1*q2;
  }
}
@| Zeta3::series @}
\textbf{Laufzeit.} Sei $z\in\texttt{Zeta3}$, $u,v,w\in\Integer$ und $t_0,t_1\in\Size$, dann\smallskip\\
\mbox{}\hfill\texttt{$z$.series($t_0$,$t_1$,$u$,$v$,$w$)$\in
\Ord(\log_2(t_1-t_0)^2\cdot\Ord(\texttt{mul}:\Integer^2\to\Integer)$}.\hfill\mbox{}

@D Zeta(3)-Series-Calculation
@{void Zeta3::series2(const size_t n, const size_t m, Integer& q, Integer& t) const
// Algorithm:  z.series2(n, m, q, t)
// Input:      z in Zeta3, q,t in Integer, n,m in size_t where n < m.
// Output:     q,t in Integer ||
{
  CONDITION(n < m);

  Integer p1,q1,t1;
  Integer q2,t2;

  const int l = (n+m)/2;
  series(n, l, p1, q1, t1);
  if (m-l <= 1) {
    Integer p2;
    series(l, m, p2, q2, t2);
  } else series2(l, m, q2, t2);
  t = t1*q2; t += t1 = t2*p1;
  q = q1*q2;
}
@| Zeta3::series2 @}
\textbf{Laufzeit.} Sei $z\in\texttt{Zeta3}$, $u,v,w\in\Integer$ und $t_0,t_1\in\Size$, dann\smallskip\\
\mbox{}\hfill\texttt{$z$.series2($t_0$,$t_1$,$v$,$w$) $\sim$ $p$.series($t_0$,$t_1$,$u$,$v$,$w$)}.\hfill\mbox{}\\

Im Konstruktor der Klasse \texttt{Zeta3} w"ahlen wir bei einer kleineren Anzahl an dezimalen Nachkommastellen von $\zeta(3)$
die lineare Implementation \texttt{linear} aus, anderenfalls die rekursive Implementation \texttt{series}:

@D select $\zeta(3)$-Algorithm
@{Zeta3::Zeta3(const size_t n)
// Algorithm:  a := Zeta3(n)
// Input:      n in size_t.
// Output:     a in Zeta3 such that |a-zeta(3)*10^n| < 1 ||
 : zeta(n)
{
  const size_t sz = zeta.precision();

  if (n <= 4000 && n/574 <= GAMMA) linear(sz, zeta.value());
  else {
    Integer p1,q1,t1;
    Integer q2,t2;

    const size_t k = (n >= 100000)? size_t(n/3.0102348) : n/3;
    const size_t l = (k+1)/2;
    series(1, l, p1, q1, t1);
    series2(l, k, q2, t2);
    Integer t = t1*q2;
    t += t1 = t2*p1;
    t2 = q1*q2;
    t += t2*77; t2 <<= 5;
    t <<= BETA*sz-1;
    zeta.value() = abs(t) / abs(t2);
  }
}
@| Zeta3::Zeta3 Zeta3 @}

Die Ausgabe der Zahl $\zeta(3)$ erfolgt durch die Weiterleitung an die Festkommaklasse \Fixed:

@D puts Zeta3-class on output stream
@{inline ostream& operator<<(ostream& out, const Zeta3& a)
// Algorithm:  o := o << a
// Input:      o in ostream, a in Zeta3.
// Output:     o in ostream ||
//
// Note:       puts Zeta3 a on output stream.
{
  return out << a.zeta;
}
@| operator<< @}
\textbf{Laufzeit.} Sei \texttt{$o\in$ostream}, $z\in\texttt{Zeta3}$ und $f\in\Fixed$, dann
\texttt{$o$<<$z\sim\ o$<<$f$}.
