
#include <math.h>

#include "fft.h"


#define sumdiff(a,b,s,d)    { s=a+b; d=a-b; }
#define sumdiff_05(a,b,s,d) { s=0.5*(a+b); d=0.5*(a-b); }
/* NEVER call like sumdiff(a,b,a,b) (i.e. input=output) */

#define sumdiff2(s,d)       { double t; t=s-d; s+=d; d=t; }

#define cmult(c,s,u,v) { double t; t=u*s+v*c; u*=c; u-=v*s; v=t; }

void scramble(double *fr, double *fi, long n);
void scramble1(double *fr, long n);
void interlace(double *f, long n);


void real_fft(double *f, int ldn)
/*
 * ordering on output:
 * f[0]     = re[0] 
 * f[1]     = re[1] 
 *         ... 
 * f[n/2-1] = re[n/2-1] 
 * f[n/2]   = re[n/2](==nyquist freq) 
 *
 * f[n/2+1] = -im[1]  (wrt. compl fft with is=-1)
 * f[n/2+2] = -im[2] 
 *         ... 
 * f[n-1]   = -im[n/2-1]
 *
 * corresponding real and imag parts (with the exception of
 * zero and nyquist freq) are found in f[i] and f[n/2+i]
 */
{
    long i,j;
    long n,n2,n4;
    double ph0;
    double *re,*im;

    n=(1<<ldn);
    n2=(n>>1);
    n4=(n>>2);

    re=f;
    im=f+n2;

    /* re gets even indexed elements, im gets odd indexed: */
    interlace(f,n);  

    fft(re,im,ldn-1,+1);

    ph0=+1.0*M_PI/n2;
    for (i=1,j=n2-1; i<=n4; ++i,--j)
    {
        double evr,evi,odr,odi;
        double c,s,w;

        sumdiff(re[i],re[j],evr,odr);
        sumdiff(im[i],im[j],odi,evi);

        w=ph0*i;
	c=cos(w);
	s=sin(w);
	
        cmult(s,-c,odr,odi);
 
        sumdiff_05(evr,odr,re[i],re[j]);
        sumdiff_05(odi,evi,im[i],im[j]);
    }

    sumdiff2(re[0],im[0]);
}
/* --------------------------- */



void fft(double *fr, double *fi, int ldn, int is)
/*
 * radix 2 fft a la cooley tukey
 *
 */
{
    int    n2,ldm,m,mh,j,r;
    int    t1, t2;
    double pi,phi,c,s;
    double ur,vr, ui,vi;

    n2=1<<ldn;

    pi=is*M_PI;

    scramble(fr,fi,n2);

    for (ldm=1; ldm<=ldn; ++ldm)
    {
        m=(1<<ldm);            /* m=2^ldm */
	mh=(m>>1);             /* mh=m/2 */

        phi=pi/(double)(mh);

        for (j=0; j<mh; ++j)
        {
	    double w=phi*(double)j;

	    c=cos(w);
	    s=sin(w);

            for (r=0; r<n2; r+=m)
            {
		/*
		 *  u=f[t1]
		 *  v=f[t2]*exp(+-2*pi*i*j/m)
		 *  f[t1]= (u+v)
		 *  f[t2]= (u-v)
		 */

                t1=r+j;
                t2=t1+mh;

                vr=fr[t2]*c-fi[t2]*s;
                vi=fr[t2]*s+fi[t2]*c;

                ur=fr[t1];
                fr[t1]+=vr;
                fr[t2]=ur-vr;

                ui=fi[t1];
                fi[t1]+=vi;
                fi[t2]=ui-vi;
            }
        }
    }
} 
/* --------------------------- */


#define SWAP(x,y)    {double tmp=x; x=y; y=tmp;}

void scramble(double *fr, double *fi, long n)
{
    long m,j;
    for (m=1,j=0; m<n-1; m++)
    {
	long k;
	{for(k=n>>1; (!((j^=k)&k)); k>>=1);}

        if (j>m)
        {
            SWAP(fr[m],fr[j]);
            SWAP(fi[m],fi[j]);
        }
    }
}
/* --------------------------- */


void scramble1(double *fr, long n)
{
    long m,j;
    for (m=1,j=0; m<n-1; m++)
    {
	long k;
	for(k=n>>1; (!((j^=k)&k)); k>>=1);

	if (j>m)  SWAP(fr[m],fr[j]);
   }
}
/* --------------------------- */



void interlace(double *f, long n)
/*
 * put part of data with even indices 
 * sorted into the lower half,
 * odd part into the higher half
 */
{
    scramble1(f,n);

    scramble1(f,n/2);
    scramble1(f+n/2,n/2);
}    
/* --------------------------- */
