
#include <math.h>

#include "fxt.h"
#include "fxtaux.h"

#define FFT(re,im,ldn,is)    four_step_fft(re,im,ldn,is)
#define FFT0(re,im,ldn,is)   four_step_fft0(re,im,ldn,is)


void interlace(double *f, long n);
void inverse_interlace(double *f, long n);

void complex_real_pre_processing(double *re, double *im, ulong n2);
void real_complex_post_processing(double *re, double *im, ulong n2);


void
wrap_real_complex_fft(double *f, ulong ldn)
//
// ordering on output:
// f[0]     = re[0]
// f[1]     = re[1]
//         ...
// f[n/2-1] = re[n/2-1]
// f[n/2]   = re[n/2](==nyquist freq)
//
// f[n/2+1] = -im[1]  (wrt. compl fft with is=-1)
// f[n/2+2] = -im[2]
//         ...
// f[n-1]   = -im[n/2-1]
//
// corresponding real and imag parts (with the exception of
// zero and nyquist freq) are found in f[i] and f[n/2+i]
//
// note that the order of imaginary parts
// is reversed wrt. fht_real_complex_fft()
//
{
    const ulong n=(1<<ldn);
    const ulong n2=(n>>1);

    double *re=f;
    double *im=f+n2;

    // re gets even indexed elements, im gets odd indexed:
    interlace(f,n);

    FFT(re,im,ldn-1,+1);

    real_complex_post_processing(re,im,n2);
}
// =============== end WRAP_REAL_COMPLEX_FFT ===========



void
wrap_real_complex_fft0(double *f, ulong ldn)
//
// version for zero padded data
//
// ordering on output:
// see wrap_real_complex_fft()
//
{
    const ulong n=(1<<ldn);
    const ulong n2=(n>>1);
    const ulong n4=(n>>2);

    double *re=f;
    double *im=f+n2;

    // re gets even indexed elements, im gets odd indexed:
    for(ulong i=0,j=1; i<n4; i++,j+=2)
    {
        im[i]=re[j];
        re[i]=re[j-1];
    }

    for(ulong i=n4; i<n2; i++)  re[i]=0.0;  // im[i] already zero

    FFT0(re,im,ldn-1,+1);

    real_complex_post_processing(re,im,n2);
}
// =============== end WRAP_REAL_COMPLEX_FFT0 ===========



void
wrap_complex_real_fft(double *f, ulong ldn)
//
// ordering on input:
// like the output of wrap_real_complex_fft()
//
{
    const ulong n=(1<<ldn);
    const ulong n2=(n>>1);

    double *re=f;
    double *im=f+n2;

    complex_real_pre_processing(re,im,n2);

    FFT(re,im,ldn-1,-1);

    inverse_interlace(f,n);
}
//=========== end WRAP_COMPLEX_REAL_FFT ===========



void
interlace(double *f, long n)
//
// put part of data with even indices
// sorted into the lower half,
// odd part into the higher half
//
{
    scramble(f,n);

    scramble(f,n/2);
    scramble(f+n/2,n/2);
}
//============= end INTERLACE ============


void
inverse_interlace(double *f, long n)
//
// inverse of interlace()
//
{
    scramble(f,n/2);
    scramble(f+n/2,n/2);

    scramble(f,n);
}
//============= end INVERSE_INTERLACE ============



void
complex_real_pre_processing(double *re, double *im, ulong n2)
{
    const ulong n4=(n2>>1);
    const double ph0=-1.0*M_PI/n2;

    for (ulong i=1,j=n2-1; i<=n4; ++i,--j)
    {
        double evr,evi,odr,odi;

        sumdiff4(re[i],re[j],evr,odr);
        sumdiff4(im[i],im[j],odi,evi);

        double c,s;
        sincos(&c,&s,ph0*i);
        cmult4(s,-c,odr,odi);

        sumdiff4(evr,odr,re[j],re[i]);
        sumdiff4(-odi,evi,im[i],im[j]);
    }

    sumdiff2(re[0],im[0]);
}
// =============== end COMPLEX_REAL_PRE_PROCESSING ===========


void
real_complex_post_processing(double *re, double *im, ulong n2)
{
    const ulong n4=(n2>>1);
    const double ph0=+1.0*M_PI/n2;

    for (ulong i=1,j=n2-1; i<=n4; ++i,--j)
    {
        double evr,evi,odr,odi;

        sumdiff4(re[i],re[j],evr,odr);
        sumdiff4(im[i],im[j],odi,evi);

        double c,s;
        sincos(&c,&s,ph0*i);
        cmult4(s,-c,odr,odi);

        sumdiff4_05(evr,odr,re[i],re[j]);
        sumdiff4_05(odi,evi,im[i],im[j]);
    }

    sumdiff2(re[0],im[0]);
}
// =============== end REAL_COMPLEX_POST_PROCESSING ===========

