\section{Zahlendarstellung}
%==========================

\subsection{Darstellung negativer Zahlen durch das $2^n$-Komplement}
%-------------------------------------------------------------------

\label{2n-komplement}
Arithmetik-Systeme wie das INTARI (\cite{Scho94}) verwenden zur Darstellung negativer Zahlen das $2^n$-Kom\-ple\-ment.
Dabei erhalten sie die
negative Zahl dadurch, da"s die Bin"arziffern der entsprechenden positiven Zahl komplementiert werden und an der letzten
Stelle eine Eins aufaddiert wird. Hierbei mu"s w"ahrend der Operationen darauf geachtet werden, da"s eine f"uhrende Eins
eine negative Zahl repr"asentiert.\medskip\\
\textbf{Beispiel.} $(\gamma) = -1, (0, \gamma) = 2^\beta-1$\medskip\\
Um dies zu gew"ahren, m"ussen wir zum Beispiel die Funktion \texttt{normalize} (siehe Kapitel \ref{normalize},
Seite \pageref{normalize}) aus unserer \Natural-Arithmetik folgenderma"sen "uberladen:\\

\small \begin{minipage}{\linewidth}
\begin{verbatim}
void normalize()
{
  const Digit HIBIT = 1 << (BETA-1);
  while (*p == 0 && size > 1 && (p[1] & HIBIT) == 0) { ++p; --size; }
  while (*p == GAMMA && size > 1 && (p[1] & HIBIT)) { ++p; --size; }
}
\end{verbatim}
\end{minipage} \normalsize\\[3ex]

Ein Vorteil dieses Ansatzes ist, da"s unsere bisherigen Funktionen weitgehend unver"andert und ohne
Fallunterscheidungen implementiert werden k"onnen.\medskip\\
\textbf{Beispiel.} $(\gamma) + (0, \gamma) \equiv (0, \gamma-1)\pmod{2^\beta},\\
(-1) \cdot \gamma = (\gamma) \cdot (0, \gamma)
 = (\gamma-1, 1) - (\gamma)\cdot2^\beta \equiv (\gamma, 1) = -\gamma\pmod{2^{2\beta}}$.
\medskip\\
Auch wenn dies die Art ist, wie der Computer intern Zahlen darstellt und festverdrahtet miteinander operieren l"a"st,
hat sie f"ur uns den Nachteil, da"s wir uns wie bei der \Natural-Arithmetik "uberwiegend auf der elementaren Ebene
befinden und nur Bruchteile aus der geerbten Klasse
\Natural\ "ubernehmen, womit wir auch beim Austauschen von Algorithmen unflexibler sind.



\subsection{Darstellung durch Betrag und Vorzeichen}
%---------------------------------------------------

Unsere \Integer-Klasse gestaltet sich in dieser Art:\index{\Integer}

@D class \Integer\
@{class Integer : public Natural {
public:
  @<output variables for representation of an \Integer\@>
private:
  int sgn;

  Integer(const size_t, char);

  @<private memberfunctions of \Integer\@>

public:
  Integer(const Digit = 0);
  Integer(const Natural&);
  Integer(const Integer&);
#ifndef _Old_STD_
explicit
#endif
  Integer(const char*, const Digit = 10);
  ~Integer();

  @<public memberfunctions and friends of \Integer\@>
};
@| Integer @}

Hierbei gilt f"ur die hinzugekommene Variable:
\[a.\texttt{sgn} = \mbox{sign}(a) = \left\{\begin{array}{r@@{\;,\quad}l}
1 & a > 0\\
0 & a = 0\\
-1 & a < 0
\end{array}\right.\qquad\mbox{f"ur $a\in\Integer$}.\]

Dadurch erf"ullen wir nun die Eigenschaft:\label{Integer}
\[\Integer := \{x\mid x\in\Natural\}\cup\{-x\mid x\in\Natural\} = \Natural\cup-\Natural\subset\Z.\]

\textbf{Bemerkung.} Wegen der vorzeichenlosen \Digit-Definition ist eine negative Wertzuweisung nicht direkt m"oglich:
\small
\begin{tabbing}
\verb|Integer a = "-1";|\\
\verb|cout << a << endl;|  \`\textit{// $a = -1$.}\\
\\
\verb|a = -Integer(1);|\\
\verb|cout << a << endl;|  \`\textit{// $a = -1$.}\\
\\
\verb|a = -1;|\\
\verb|cout << a << endl;|  \`\textit{// $a = 2^\beta - 1$.}\\
\end{tabbing}
\normalsize


\subsubsection{Diagnosemakro}

Wir ben"otigen das folgende Diagnosemakro, um die Bedingung an die interne Darstellung eines \Integer s zu "uberpr"ufen.

@d condition of an \Integer\
@{# define INTEGERCONDITION(a)                            \
  CONDITION((a).highest() == 0 && sign(a) == 0 ||       \
            (a).highest() != 0 &&                       \
            (sign(a) == 1 || sign(a) == -1));

# define INTEGER_FOR_CHECK(a, b)                        \
  Integer (a) = (b);
@| INTEGERCONDITION INTEGER_FOR_CHECK @}


\section{Elementare Funktionen und Operationen}
%==============================================

Alle Operationen auf \Integer s werden wir wegen der internen Betragsdarstellung grunds"atzlich nur in elementarer Weise
auf die abgeleiteten \Natural-Operationen abbilden und somit jede Funktion in verschiedene F"alle unterteilen.
Zus"atzlich mu"s noch jedes Mal die hinzugekommene Variable \texttt{sgn} aktualisiert werden.


\subsection{Konstruktoren und Destruktor}
%----------------------------------------

\subsubsection{Gesch"utzt}

@d protected constructor \Integer\ without the initialization of the elements
@{inline Integer::Integer(const size_t a, char b)
 : Natural(a, b)
// Algorithm:  c := Integer(a, b)
// Input:      b in char, a in size_t where a >= 1.
// Output:     c in Integer such that L(c) = R(c) = a;
//             map Integer(a, b) to Natural(a, b) ||
//
// Note:       This constructor don't fulfill the conditions for Integers.
//
// internal constructor without the initialization of the elements.
{
}
@| Integer::Integer Integer @}
\textbf{Laufzeit.} Sei $t\in\Size$ und $c\in\mathtt{char}$, dann \texttt{Integer($t$,$c$) $\sim$ Natural($t$,$c$)}.


\subsubsection{"Offentlich}

@d default constructor \Integer\
@{inline Integer::Integer(const Digit a)
 : Natural(a)
// Algorithm:  c := Integer(a)
// Input:      a in Digit.
// Output:     c in Integer such that c = a ||
{
  sgn = (a != 0);
}
@| Integer::Integer Integer @}
\textbf{Laufzeit.} Sei $a\in\Digit$, dann \texttt{Integer($a$) $\sim$ Natural($a$)}.

@d overloaded constructor \Integer\ for \Natural\
@{inline Integer::Integer(const Natural& a)
 : Natural(a)
// Algorithm:  c := Integer(a)
// Input:      a in Natural.
// Output:     c in Integer such that c = a ||
{
  sgn = (a != 0);
}
@| Integer::Integer Integer @}
\textbf{Laufzeit.} Sei $x\in\Natural$, dann \texttt{Integer($x$) $\sim$ Natural($x$)}.

@d copy constructor \Integer\
@{inline Integer::Integer(const Integer& a)
: Natural(::abs(a))
// Algorithm:  c := Integer(a)
// Input:      a in Integer.
// Output:     c in Integer such that c = a ||
{
  sgn = a.sgn;
}
@| Integer::Integer Integer @}
\textbf{Laufzeit.} Sei $u\in\Integer$, dann \texttt{Integer($u$) $\sim$ Natural(abs($u$))}.

@d overloaded constructor \Integer\ for ASCII-string conversion
@{inline Integer::Integer(const char* a, const Digit b)
 : Natural(a + (*a == '-'), b)
// Algorithm:  c := Integer(a, b)
// Input:      a in String, b in Digit.
// Output:     c in Integer such that c = a ||
{
  if (*a == '-') sgn = -1;
  else sgn = (::abs(*this) != 0);
}
@| Integer::Integer Integer @}
\textbf{Laufzeit.} Sei $s\in\String$ und $a\in\Digit$, dann \texttt{Integer($s$,$a$) $\sim$ Natural($s$,$a$)}.

@d destructor \Integer\
@{inline Integer::~Integer()
{
}
@| Integer::~Integer ~Integer @}
\textbf{Laufzeit.} \verb|~Integer()|$\sim$\verb|~Natural()|.



\subsection{Absolutbetrag}
%-------------------------

@D absolute value of an \Integer\
@{inline const Natural& abs(const Integer& a)
// Algorithm:  c := abs(a)
// Input:      a in Integer.
// Output:     c in Natural such that c = |a| ||
{
  return (const Natural&)a;
}

inline Natural& Integer::abs()
// Algorithm:  c := a.abs()
// Input:      a in Integer.
// Output:     c in Natural such that c = |a| ||
{
  return (Natural&)*this;
}
@}
\textbf{Laufzeit.} Sei $u\in\Integer$, dann \texttt{abs($u$) $\in\Ord(1)$}.\\
\textbf{Laufzeit.} Sei $u\in\Integer$, dann \texttt{$u$.abs() $\in\Ord(1)$}.



\subsection{Vorzeichen/Einheiten}
%--------------------------------

@d sign of an \Integer\
@{inline int sign(const Integer& a)
// Algorithm:  c := sign(a)
// Input:      a in Integer.
// Output:     c in int such that if a = 0 then c = 0
//             else if a > 0 then c = 1 else c = -1 ||
{
  return a.sgn;
}
@| sign @}
\textbf{Laufzeit.} Sei $u\in\Integer$, dann \texttt{sign($u$)$\in\Ord(1)$}.

@d units of an \Integer\
@{inline int units(Integer& a)
// Algorithm:  c := units(a)
// Input:      a in Integer.
// Output:     a in Integer, c in int such that a := |a|,
//             if a >= 0 then c = 1 else c = -1 ||
{
  if (sign(a) >= 0) return 1;
  else { a = -a; return -1; }
}
@| units @}
\textbf{Laufzeit.} Sei $u\in\Integer$, dann \texttt{units($u$)$\in\Ord(1)$}.



\subsection{Vertauschen zweier \Integer s}
%-----------------------------------------

@D function swap for \Integer s
@{void swap(Integer& a, Integer& b)
// Algorithm:  swap(a, b)
// Input:      a,b in Integer.
// Output:     a,b in Integer such that t := a, a := b, b := t
//             where t in Integer ||
{
  INTEGERCONDITION(a);
  INTEGERCONDITION(b);

  swap(a.abs(), b.abs());
  int t = a.sgn; a.sgn = b.sgn; b.sgn = t;

  INTEGERCONDITION(a);
  INTEGERCONDITION(b);
}
@| swap @}
\textbf{Laufzeit.} Seien $u,v\in\Integer$, dann \texttt{swap($u$,$v$) $\sim$ swap(abs($u$),abs($v$))}.


\subsection{Kopierfunktionen}
%----------------------------

@d assign \verb#operator=# for \Integer s
@{inline Integer& Integer::operator=(const Integer& a)
// Algorithm:  c := c = a
// Input:      a,c in Integer.
// Output:     c in Integer such that c = a ||
{
  sgn = a.sgn;
  abs() = ::abs(a);
  return *this;
}
@| Integer::operator= operator= @}
\textbf{Laufzeit.} Seien $u,v\in\Integer$, dann \texttt{$u$=$v$ $\sim$ abs($u$)=abs($v$)}.

@d assign \verb#operator=# for an \Integer\ with a \Natural\
@{inline Integer& Integer::operator=(const Natural& a)
// Algorithm:  c := c = a
// Input:      a in Natural,c in Integer.
// Output:     c in Integer such that c = a ||
{
  sgn = (a > 0);
  abs() = a;
  return *this;
}
@| Integer::operator= operator= @}
\textbf{Laufzeit.} Sei $x\in\Natural$ und $u\in\Integer$, dann \texttt{$u$=$x$ $\sim$ abs($u$)=$x$}.

@d assign \verb#operator=# of an \Integer\ with a \Digit\
@{inline Digit Integer::operator=(const Digit a)
// Algorithm:  c := b = a
// Input:      a in Digit, b in Integer.
// Output:     b in Integer, c in Digit such that b = a, c = a ||
{
  sgn = (a > 0);
  abs() = a;
  return a;
}
@| Integer::operator= operator= @}
\textbf{Laufzeit.} Sei $u\in\Integer$ und $a\in\Digit$, dann \texttt{$u$=$a$ $\sim$ abs($u$)=$a$}.


\subsection{Streamausgabe}
%-------------------------

@d puts an \Integer\ on output stream
@{inline ostream& operator<<(ostream& out, const Integer& a)
// Algorithm:  o := o << a
// Input:      o in ostream, a in Integer.
// Output:     o in ostream ||
//
// Note:       puts Integer a on output stream.
{
  if (sign(a) < 0) {
    const int b = out.width();
    if (b > 0) { out.width(0); out << '-'; out.width(b-1); }
    else out << '-';
  }
  return out << abs(a);
}
@| operator<< @}
\textbf{Laufzeit.} Sei \texttt{$o\in$ostream} und $u\in\Integer$, dann \texttt{$o$<<$u \sim\ o$<< abs($u$)}.


\subsubsection{Interne Darstellung}

@d puts internal representation of an \Integer\ on output stream
@{inline ostream& operator<<(ostream& out, const Integer::rep& a)
// puts internal representation of Integer a on output stream.
{
  if (a.sgn == -1) out << '-';
  return out << print(a.nat);
}
@| operator<< @}
\textbf{Laufzeit.} Sei \texttt{$o\in$ostream} und \texttt{$r\in\Integer$::rep}, dann \texttt{$o$<<$r\in\Ord(\Length(r))$}.\\

@d output variables for representation of an \Integer\
@{struct rep {
  const int   sgn;
  const       Natural& nat;
  rep(const int a, const Natural& b)
    : sgn(a), nat(b) {}
};
@}

@d function \verb#print# of an \Integer\
@{inline Integer::rep print(const Integer& a)
// Algorithm:  o := o << print(a)
// Input:      o in ostream, a in Integer.
// Output:     o in ostream ||
//
// Note:       puts internal representation of Integer a on an output stream.
{
  return Integer::rep(sign(a), abs(a));
}
@| print @}


\subsection{Streameingabe}
%-------------------------

@d gets an \Integer\ from input stream
@{istream& operator>>(istream& in, Integer& a)
// Algorithm:  i := i >> a
// Input:      i in istream.
// Output:     i in istream, a in Integer ||
//
// Note:       gets Integer a from input stream.
{
  INTEGERCONDITION(a);

  if (!in.good()) return in;
  char ch = 0;
  if (in.get(ch) && ch != '-') in.putback(ch);
  in >> a.abs();
  a.sgn = (abs(a) != 0);
  if (ch == '-') a.neg();

  INTEGERCONDITION(a);

  return in;
}
@| operator>> @}
\textbf{Laufzeit.} Sei \texttt{$i\in$istream} und $u\in\Integer$, dann \texttt{$i$>>$u \sim\ i$>> abs($u$)}.


\subsubsection{Interne Darstellung}

@D gets internal representation of an \Integer\ from input stream
@{bool Integer::scan(istream& in)
// Algorithm:  b := a.scan(i)
// Input:      a in Integer, i in istream.
// Output:     a in Integer, i in istream, b in bool ||
//
// Note:       gets Integer a as an internal representation from input stream
//             if b is true.
{
  if (!in.good()) return false;
  char c = 0;
  if (in.get(c) && c != '-') in.putback(c);
  const bool b = Natural::scan(in);
  sgn = (::abs(*this) != 0);
  if (c == '-') neg();
  return b;
}
@| Integer::scan scan @}
\textbf{Laufzeit.} Sei \texttt{$i\in$ostream}, $u\in\Integer$, dann
 \texttt{$u$.scan($i$)$\sim\ $abs($u$).scan($i$)}.


\subsubsection{Konvertierung in ein beliebiges Stellenwertsystem}

@D converts an \Integer\ to a string
@{char* Itoa(const Integer& a, char* b, const Digit c)
// Algorithm:  c := Itoa(a, c, b)
// Input:      a in Natural, b in Digit, c in String
//             where 2 <= b <= 36, sizeof(c) > BETA*L(a)/log2(b).
// Output:     c in String such that c = a ||
//
// Note:       conversion Integer to string.
{
  INTEGERCONDITION(a);

  if (sign(a) >= 0) Ntoa(abs(a), b, c);
  else { *b = '-'; Ntoa(abs(a), b+1, c); }
  return b;
}
@| Itoa @}
\textbf{Laufzeit.} Sei $u\in\Integer$, $s\in\String$ und $a\in\Digit$, dann
\[\texttt{Itoa($u$,$s$,$a$) $\sim$ Ntoa(abs($u$),$s$,$a$)}.\]


\subsubsection{Konvertierung aus einem beliebigen Stellenwertsystem}

@D converts a string to an \Integer\
@{Integer atoI(const char* a, const Digit b)
// Algorithm:  c := atoI(a, b)
// Input:      a in String, b in Digit where 2 <= b <= 36.
// Output:     c in Integer such that c = a ||
//
// Note:       conversion string to Integer; return 0 by conversion error.
{
  const bool d = (*a == '-');
  Integer c(atoN(a + d, b));
  if (d) c = -c;

  INTEGERCONDITION(c);

  return c;
}
@| atoI @}
\textbf{Laufzeit.} Sei $s\in\String$ und $a\in\Digit$, dann \texttt{atoI($s$,$a$) $\sim$ atoN($s$,$a$)}.


\subsection{Vergleichsoperatoren}
%--------------------------------

@d comparison \verb#operator==# for \Integer s
@{inline bool operator==(const Integer& a, const Integer& b)
// Algorithm:  c := a == b
// Input:      a,b in Integer.
// Output:     c in bool such that if a = b then c = true else c = false ||
{
  return (sign(a) == sign(b) && abs(a) == abs(b));
}
@| operator== @}
\textbf{Laufzeit.} Seien $u,v\in\Integer$, dann \texttt{$u$==$v$ $\sim$ abs($u$) == abs($v$)}.

@d comparison \verb#operator!=# for \Integer s
@{inline bool operator!=(const Integer& a, const Integer& b)
// Algorithm:  c := a != b
// Input:      a,b in Integer.
// Output:     c in bool such that if a = b then c = false else c = true ||
{
  return (sign(a) != sign(b) || abs(a) != abs(b));
}
@| operator!= @}
\textbf{Laufzeit.} Seien $u,v\in\Integer$, dann \texttt{$u$!=$v$ $\sim$ abs($u$) != abs($v$)}.

@D comparison \verb#operator<# for \Integer s
@{inline bool operator<(const Integer& a, const Integer& b)
// Algorithm:  c := a < b
// Input:      a,b in Integer.
// Output:     c in bool such that if a < b then c = true else c = false ||
{
  return (sign(a) < sign(b) || sign(a) == sign(b)
   && ((sign(a) > 0)? abs(a) < abs(b) : abs(b) < abs(a)));
}
@| operator< @}
\textbf{Laufzeit.} Seien $u,v\in\Integer$, dann \texttt{$u$<$v$ $\sim$ abs($u$) < abs($v$)}.

@D comparison \verb#operator<=# for \Integer s
@{inline bool operator<=(const Integer& a, const Integer& b)
// Algorithm:  c := a <= b
// Input:      a,b in Integer.
// Output:     c in bool such that if a <= b then c = true else c = false ||
{
  return !(b < a);
}
@| operator<= @}
\textbf{Laufzeit.} Seien $u,v\in\Integer$, dann \texttt{$u$<=$v$ $\sim$ $v$<$u$}.

@D comparison \verb#operator># for \Integer s
@{inline bool operator>(const Integer& a, const Integer& b)
// Algorithm:  c := a > b
// Input:      a,b in Integer.
// Output:     c in bool such that if a > b then c = true else c = false ||
{
  return (b < a);
}
@| operator> @}
\textbf{Laufzeit.} Seien $u,v\in\Integer$, dann \texttt{$u$>$v$ $\sim$ $v$<$u$}.

@D comparison \verb#operator>=# for \Integer s
@{inline bool operator>=(const Integer& a, const Integer& b)
// Algorithm:  c := a >= b
// Input:      a,b in Integer.
// Output:     c in bool such that if a >= b then c = true else c = false ||
{
  return !(a < b);
}
@| operator>= @}
\textbf{Laufzeit.} Seien $u,v\in\Integer$, dann \texttt{$u$>=$v$ $\sim$ $u$<$v$}.



\subsubsection{\Digit-Vergleiche}

@d comparison \verb#operator==# of an \Integer\ with a \Digit\
@{inline bool operator==(const Integer& a, const Digit b)
// Algorithm:  c := a == b
// Input:      a in Integer, b in Digit.
// Output:     c in bool such that if a = b then c = true else c = false ||
{
  return (sign(a) >= 0 && abs(a) == b);
}
@| operator== @}
\textbf{Laufzeit.} Sei $u\in\Integer$ und $a\in\Digit$, dann \texttt{$u$==$a \sim$ abs($u$) ==$a$}.

@d comparison \verb#operator!=# of an \Integer\ with a \Digit\
@{inline bool operator!=(const Integer& a, const Digit b)
// Algorithm:  c := a != b
// Input:      a in Integer, b in Digit.
// Output:     c in bool such that if not a = b then c = true else c = false ||
{
  return (sign(a) < 0 || abs(a) != b);
}
@| operator!= @}
\textbf{Laufzeit.} Sei $u\in\Integer$ und $a\in\Digit$, dann \texttt{$u$!=$a \sim$ abs($u$) !=$a$}.

@d comparison \verb#operator<# of an \Integer\ with a \Digit\
@{inline bool operator<(const Integer& a, const Digit b)
// Algorithm:  c := a < b
// Input:      a in Integer, b in Digit.
// Output:     c in bool such that if a < b then c = true else c = false ||
{
  return (sign(a) < 0 || abs(a) < b);
}
@| operator< @}
\textbf{Laufzeit.} Sei $u\in\Integer$ und $a\in\Digit$, dann \texttt{$u$<$a \sim$ abs($u$) <$a$}.

@d comparison \verb#operator<=# of an \Integer\ with a \Digit\
@{inline bool operator<=(const Integer& a, const Digit b)
// Algorithm:  c := a <= b
// Input:      a in Integer, b in Digit.
// Output:     c in bool such that if a <= b then c = true else c = false ||
{
  return (sign(a) <= 0 || abs(a) <= b);
}
@| operator<= @}
\textbf{Laufzeit.} Sei $u\in\Integer$ und $a\in\Digit$, dann \texttt{$u$<=$a \sim$ abs($u$) <=$a$}.

@d comparison \verb#operator># of an \Integer\ with a \Digit\
@{inline bool operator>(const Integer& a, const Digit b)
// Algorithm:  c := a > b
// Input:      a in Integer, b in Digit.
// Output:     c in bool such that if a > b then c = true else c = false ||
{
  return (sign(a) > 0 && abs(a) > b);
}
@| operator> @}
\textbf{Laufzeit.} Sei $u\in\Integer$ und $a\in\Digit$, dann \texttt{$u$>$a \sim$ abs($u$) >$a$}.

@d comparison \verb#operator>=# of an \Integer\ with a \Digit\
@{inline bool operator>=(const Integer& a, const Digit b)
// Algorithm:  c := a >= b
// Input:      a in Integer, b in Digit.
// Output:     c in bool such that if a >= b then c = true else c = false ||
{
  return (sign(a) >= 0 && abs(a) >= b);
}
@| operator>= @}
\textbf{Laufzeit.} Sei $u\in\Integer$ und $a\in\Digit$, dann \texttt{$u$>=$a \sim$ abs($u$) >=$a$}.



\subsection{Additive Operationen}
%--------------------------------

\subsubsection{Inkrementierung}

@D prefix incrementation of an \Integer\
@{const Integer& Integer::operator++()
// Algorithm:  c := ++a
// Input:      a in Integer.
// Output:     a,c in Integer such that a := a+1, c := a ||
{
  INTEGERCONDITION(*this);

  const int sT = sgn;
  if (sT == 0) *this = 1;
  else if (sT > 0) ++abs();
  else if (::abs(*this) == 1) *this = 0;
  else --abs();

  INTEGERCONDITION(*this);

  return *this;
}
@| Integer::operator++ operator++ @}
\textbf{Laufzeit.} Sei $u\in\Integer$, dann \texttt{++$u$ $\sim$ ++abs($u$)}.

@d postfix incrementation of an \Integer\
@{inline Integer Integer::operator++(int)
// Algorithm:  c := a++
// Input:      a in Integer.
// Output:     a,c in Integer such that c := a, a := a+1 ||
{
  Integer a(*this);
  ++(*this);
  return a;
}
@| Integer::operator++ operator++ @}
\textbf{Laufzeit.} Sei $u\in\Integer$, dann \texttt{$u$++ $\sim$ abs($u$)++}.


\subsubsection{Decrementierung}

@D prefix decrementation of an \Integer\
@{const Integer& Integer::operator--()
// Algorithm:  c := --a
// Input:      a in Integer.
// Output:     a,c in Integer such that a := a-1, c := a ||
{
  INTEGERCONDITION(*this);

  const int sT = sgn;
  if (sT == 0) { abs() = 1; sgn = -1; }
  else if (sT < 0) ++abs();
  else if (::abs(*this) == 1) *this = 0;
  else --abs();

  INTEGERCONDITION(*this);

  return *this;
}
@| Integer::operator-- operator-- @}
\textbf{Laufzeit.} Sei $u\in\Integer$, dann \texttt{--$u$ $\sim$ --abs($u$)}.

@d postfix decrementation of an \Integer\
@{inline Integer Integer::operator--(int)
// Algorithm:  c := a--
// Input:      a in Integer.
// Output:     a,c in Integer such that c := a, a := a-1 ||
{
  Integer a(*this);
  --(*this);
  return a;
}
@| Integer::operator-- operator-- @}
\textbf{Laufzeit.} Sei $u\in\Integer$, dann \texttt{$u$-- $\sim$ abs($u$)--}.



\subsubsection{Negation}

@d negation of an \Integer\
@{inline void Integer::neg()
// Algorithm:  a.neg()
// Input:      a in Integer.
// Output:     a in Integer such that a := -a ||
{
  sgn = -sgn;
}
@| Integer::neg neg @}
\textbf{Laufzeit.} Sei $u\in\Integer$, dann \texttt{$u$.neg()$\in\Ord(1)$}.

@d negation of an \Integer\
@{inline void Integer::neg(const Integer& a)
// Algorithm:  b.neg(a)
// Input:      a in Integer.
// Output:     b in Integer such that b = -a ||
{
  *this = a; neg();
}
@| Integer::neg neg @}
\textbf{Laufzeit.} Seien $u,v\in\Integer$, dann \texttt{$v$.neg($u$)$\in\Ord(\Length(|u|))$}.

@D negation \verb#operator-# of an \Integer\
@{inline Integer::Integer(const binder_arguments<Integer, Integer,
                                               Integer_negate_tag>& a)
{
  get_memory(a.x.length()+DELTA);
  neg(a.x);
}

inline Integer& Integer::operator=(const binder_arguments<Integer, Integer,
                                                          Integer_negate_tag>& a)
{
  if (this == &a.x) neg();
  else neg(a.x);
  return *this;
}

inline binder_arguments<Integer, Integer, Integer_negate_tag>
 operator-(const Integer& a)
// Algorithm:  c := -a
// Input:      a in Integer.
// Output:     c in Integer such that c = -a ||
{
  return binder_arguments<Integer, Integer, Integer_negate_tag>(a, a);
}
@| Integer::Integer Integer Integer::operator= operator= operator- @}
\textbf{Laufzeit.} Seien $u,v\in\Integer$, dann 
\begin{tabular}{lcl}
$u$ = -$u$ & $\sim$ & $u$.neg(),\\
-$u$ & $\sim$ & $v$.neg($u$).
\end{tabular}



\subsubsection{Addition}

@D addition of two \Integer s
@{void Integer::add(const Integer& a, const Integer& b)
// Algorithm:  c.add(a, b)
// Input:      a,b in Integer.
// Output:     c in Integer such that c = a+b ||
{
  INTEGERCONDITION(a);
  INTEGERCONDITION(b);

  const int sA = a.sgn;
  const int sB = b.sgn;
  if (sA == 0) *this = b;
  else if (sB == 0) *this = a;
  else if (sA == sB) {
    abs() = ::abs(a) + ::abs(b);
    sgn = sA;
  } else sgn = sA*::abs(::abs(a), ::abs(b), abs());
  
  INTEGERCONDITION(*this);
}
@| Integer::add add @}
\textbf{Laufzeit.} Seien $u,v,w\in\Integer$, dann \[w\texttt{.add(}u,v\texttt{)} \sim 
\left\{\begin{array}{r@@{\;,\quad}l}
w\texttt{.add(abs(}u\texttt{),abs(}v\texttt{))} & \mathtt{sign}(u) = \mathtt{sign}(v)\\
\texttt{abs(abs(}u\texttt{),abs(}v\texttt{),abs(}w\texttt{))} & \mathtt{sign}(u) \not= \mathtt{sign}(v)
\end{array}\right..\]

@D addition of an \Integer\ with a \Natural\
@{void Integer::add(const Integer& a, const Natural& b)
// Algorithm:  c.add(a, b)
// Input:      a in Integer, b in Natural.
// Output:     c in Integer such that c = a+b ||
{
  INTEGERCONDITION(a);

  const int sA = a.sgn;
  if (sA == 0) *this = b;
  else if (sA == 1) {
    abs() = ::abs(a) + b;
    sgn = 1;
  } else sgn = -::abs(::abs(a), b, abs());
  
  INTEGERCONDITION(*this);
}
@| Integer::add add @}
\textbf{Laufzeit.} Seien $u,v\in\Integer$ und $x\in\Natural$, dann \[v\texttt{.add(}u,x\texttt{)} \sim 
\left\{\begin{array}{r@@{\;,\quad}l}
v\texttt{.add(abs(}u\texttt{),}x\texttt{)} & \mathtt{sign}(u) \geq0\\
\texttt{abs(abs(}u\texttt{),}x\texttt{,abs(}v\texttt{))} & \mathtt{sign}(u) < 0
\end{array}\right..\]

@D addition of an \Integer\ with a \Digit\
@{void Integer::add(const Integer& a, const Digit b)
// Algorithm:  c.add(a, b)
// Input:      a in Integer, b in Digit where not &a = &c.
// Output:     c in Integer such that c = a+b ||
{
  INTEGERCONDITION(a);
  CONDITION(&a != this);

  const int sA = a.sgn;
  sgn = sA;
  if (sA > 0) Natural::add(::abs(a), b);
  else if (sA == 0) *this = b; 
  else if (::abs(a) > b) Natural::sub(::abs(a), b);
  else *this = b - a.highest();

  INTEGERCONDITION(*this);
}
@| Integer::add add @}
\textbf{Laufzeit.} Sei $u,v\in\Integer$ und $a\in\Digit$, dann $v\texttt{.add($u$,$a$)} \sim
\left\{\begin{array}{r@@{\;,\quad}l}
\texttt{abs($v$).add($u$,$a$)} & \mathtt{sign}(u)\geq 0\\
\texttt{abs($v$).sub($u$,$a$)} & \mathtt{sign}(u) < 0
\end{array}\right.$.

@D additive \verb#operator+# for \Integer s
@{inline Integer::Integer(const binder_arguments<Integer, Integer,
                                               Integer_plus_tag>& a)
{
  get_memory(max(a.x.length(), a.y.length())+DELTA);
  add(a.x, a.y);
}

inline Integer& Integer::operator=(const binder_arguments<Integer, Integer,
                                                          Integer_plus_tag>& a)
{
  add(a.x, a.y);
  return *this;
}

inline binder_arguments<Integer, Integer, Integer_plus_tag>
 operator+(const Integer& a, const Integer& b)
// Algorithm:  c := a+b
// Input:      a,b in Integer.
// Output:     c in Integer such that c = a+b ||
{
  return binder_arguments<Integer, Integer, Integer_plus_tag>(a, b);
}
@| Integer::Integer Integer Integer::operator= operator= operator+ @}
\textbf{Laufzeit.} Seien $u,v,w\in\Integer$, dann \texttt{$u$+$v$ $\sim$ $w$.add($u$,$v$)}.

@D additive \verb#operator+# of an \Integer\ with a \Natural\
@{inline Integer::Integer(const binder_arguments<Integer, Natural,
                                               Integer_plus_tag>& a)
{
  get_memory(max(a.x.length(), a.y.length())+DELTA);
  add(a.x, a.y);
}

inline Integer& Integer::operator=(const binder_arguments<Integer, Natural,
                                                          Integer_plus_tag>& a)
{
  add(a.x, a.y);
  return *this;
}

inline binder_arguments<Integer, Natural, Integer_plus_tag>
 operator+(const Integer& a, const Natural& b)
// Algorithm:  c := a+b
// Input:      a in Integer, b in Natural.
// Output:     c in Integer such that c = a+b ||
{
  return binder_arguments<Integer, Natural, Integer_plus_tag>(a, b);
}

inline binder_arguments<Integer, Natural, Integer_plus_tag>
 operator+(const Natural& a, const Integer& b)
// Algorithm:  c := a+b
// Input:      a in Natural, b in Integer.
// Output:     c in Integer such that c = a+b ||
{
  return binder_arguments<Integer, Natural, Integer_plus_tag>(b, a);
}
@| Integer::Integer Integer Integer::operator= operator= operator+ @}
\textbf{Laufzeit.} Seien $u,v\in\Integer$ und $x\in\Natural$, dann \texttt{$u$+$x$ $\sim$ $v$.add($u$,$x$)}.\\
\textbf{Laufzeit.} Seien $u,v\in\Integer$ und $x\in\Natural$, dann \texttt{$x$+$u$ $\sim$ $v$.add($u$,$x$)}.

@D additive \verb#operator+# of an \Integer\ with a \Digit\
@{inline Integer::Integer(const binder_arguments<Integer, Digit,
                                               Integer_plus_tag>& a)
{
  get_memory(a.x.length()+DELTA);
  add(a.x, a.y);
}

inline Integer& Integer::operator=(const binder_arguments<Integer, Digit,
                                                          Integer_plus_tag>& a)
{
  if (this == &a.x) return *this += a.y;
  else { add(a.x, a.y); return *this; }
}

inline binder_arguments<Integer, Digit, Integer_plus_tag>
 operator+(const Integer& a, const Digit& b)
// Algorithm:  c := a+b
// Input:      a in Integer, Digit b.
// Output:     c in Integer such that c = a+b ||
{
  return binder_arguments<Integer, Digit, Integer_plus_tag>(a, b);
}

inline binder_arguments<Integer, Digit, Integer_plus_tag>
 operator+(const Digit& a, const Integer& b)
// Algorithm:  c := a+b
// Input:      a in Digit, Integer b.
// Output:     c in Integer such that c = a+b ||
{
  return binder_arguments<Integer, Digit, Integer_plus_tag>(b, a);
}
@| Integer::Integer Integer Integer::operator= operator= operator+ @}
\textbf{Laufzeit.} Seien $u,v\in\Integer$ und $a\in\Digit$, dann \texttt{$u$+$a$ $\sim$ $v$.add($u$,$a$)}.


\subsubsection{Zuweisungsoperatoren}

@d assign \verb#operator+=# for \Integer s
@{inline Integer& Integer::operator+=(const Integer& a)
// Algorithm:  c := c += a
// Input:      a,c in Integer.
// Output:     c in Integer such that c := c+a ||
{
  add(a, *this);
  return *this;
}
@| Integer::operator+= operator+= @}
\textbf{Laufzeit.} Seien $u,v\in\Integer$, dann \texttt{$u$+=$v$ $\sim$ $u$.add($u$,$v$)}.

@D assign \verb#operator+=# of an \Integer\ with a \Natural\
@{inline Integer& Integer::operator+=(const Natural& a)
// Algorithm:  c := c += a
// Input:      a in Natural, c in Integer.
// Output:     c in Integer such that c := c+a ||
{
  add(*this, a);
  return *this;
}
@| Integer::operator+= operator+= @}
\textbf{Laufzeit.} Sei $u\in\Integer$ und $x\in\Natural$, dann \texttt{$u$+=$x$ $\sim$ $u$.add($u$,$x$)}.

@D assign \verb#operator+=# of an \Integer\ with a \Digit\
@{Integer& Integer::operator+=(const Digit a)
// Algorithm:  c := c += a
// Input:      a Digit,c in Integer.
// Output:     c in Integer such that c := c+a ||
{
  INTEGERCONDITION(*this);

  const int sT = sgn;
  if (sT > 0) abs() += a;
  else if (sT == 0) *this = a; 
  else if (::abs(*this) > a) abs() -= a;
  else *this = a - highest();

  INTEGERCONDITION(*this);

  return *this;
}
@| Integer::operator+= operator+= @}
\textbf{Laufzeit.} Sei $u\in\Integer$ und $a\in\Digit$, dann $u\texttt{+=}a \sim
\left\{\begin{array}{r@@{\;,\quad}l}
\texttt{abs(}u\texttt{)+=a} & \mathtt{sign}(u)\geq 0\\
\texttt{abs(}u\texttt{)-=a} & \mathtt{sign}(u) < 0
\end{array}\right.$.



\subsubsection{Subtraktion}

@D subtraction of two \Integer s
@{void Integer::sub(const Integer& a, const Integer& b)
// Algorithm:  c.sub(a, b)
// Input:      a,b in Integer.
// Output:     c in Integer such that c = a-b ||
{
  INTEGERCONDITION(a);
  INTEGERCONDITION(b);

  const int sA = a.sgn;
  const int sB = b.sgn;
  if (sA == 0) neg(b);
  else if (sB == 0) *this = a;
  else if (sA != sB) {
    abs() = ::abs(a) + ::abs(b);
    sgn = sA;
  } else sgn = sA*::abs(::abs(a), ::abs(b), abs());
  
  INTEGERCONDITION(*this);
}
@| Integer::sub sub @}
\textbf{Laufzeit.} Seien $u,v,w\in\Integer$, dann \[w\texttt{.sub(}u,v\texttt{)} \sim 
\left\{\begin{array}{r@@{\;,\quad}l}
\texttt{abs(}w\texttt{).add(abs(}u\texttt{),abs(}v\texttt{))} & \mathtt{sign}(u) \not= \mathtt{sign}(v)\\
\texttt{abs(abs(}u\texttt{),abs(}v\texttt{),abs(}w\texttt{))} & \mathtt{sign}(u) = \mathtt{sign}(v)
\end{array}\right..\]

@D subtraction of an \Integer\ with a \Natural\
@{void Integer::sub(const Integer& a, const Natural& b)
// Algorithm:  c.sub(a, b)
// Input:      a in Integer, b in Natural.
// Output:     c in Integer such that c = a-b ||
{
  INTEGERCONDITION(a);

  const int sA = a.sgn;
  if (sA == 0) { *this = b; neg(); }
  else if (sA == -1) {
    abs() = ::abs(a) + b;
    sgn = -1;
  } else sgn = ::abs(::abs(a), b, abs());
  
  INTEGERCONDITION(*this);
}
@| Integer::sub sub @}
\textbf{Laufzeit.} Seien $u,v\in\Integer$ und $x\in\Natural$, dann \[v\texttt{.sub(}u,x\texttt{)} \sim 
\left\{\begin{array}{r@@{\;,\quad}l}
\texttt{abs(}v\texttt{).add(abs(}u\texttt{),}x\texttt{)} & \mathtt{sign}(u) < 0\\
\texttt{abs(abs(}u\texttt{),}x\texttt{,abs(}v\texttt{))} & \mathtt{sign}(u) \geq 0
\end{array}\right..\]

@D subtraction of a \Natural\ with an \Integer\
@{void Integer::sub(const Natural& a, const Integer& b)
// Algorithm:  c.sub(a, b)
// Input:      a in Natural, b in Integer.
// Output:     c in Integer such that c = a-b ||
{
  INTEGERCONDITION(b);

  const int sB = b.sgn;
  if (sB == 0) *this = a;
  else if (sB == -1) {
    abs() = a + ::abs(b);
    sgn = 1;
  } else sgn = ::abs(a, ::abs(b), abs());
  
  INTEGERCONDITION(*this);
}
@| Integer::sub sub @}
\textbf{Laufzeit.} Seien $u,v\in\Integer$ und $x\in\Natural$, dann \[v\texttt{.sub(}x,u\texttt{)} \sim 
\left\{\begin{array}{r@@{\;,\quad}l}
\texttt{abs(}v\texttt{).add(}x\texttt{,abs(}u\texttt{))} & \mathtt{sign}(u) < 0\\
\texttt{abs(}x\texttt{,abs(}u\texttt{),abs(}v\texttt{))} & \mathtt{sign}(u) \geq 0
\end{array}\right..\]

@D subtraction of an \Integer\ with a \Digit\
@{void Integer::sub(const Integer& a, const Digit b)
// Algorithm:  c.sub(a, b)
// Input:      a,c in Integer, b in Digit where not &a = &c.
// Output:     c in Integer such that c = a-b ||
{
  INTEGERCONDITION(a);
  CONDITION(&a != this);

  const int sA = a.sgn;
  sgn = sA;
  if (sA < 0) Natural::add(::abs(a), b);
  else if (sA == 0) neg(b);
  else if (::abs(a) > b) Natural::sub(::abs(a), b);
  else {
    *this = b - a.highest();
    neg();
  }

  INTEGERCONDITION(*this);
}
@| Integer::sub sub @}
\textbf{Laufzeit.} Sei $u,v\in\Integer$ und $a\in\Digit$, dann $v\texttt{.sub($u$,$a$)} \sim
\left\{\begin{array}{r@@{\;,\quad}l}
\texttt{abs($v$).sub($u$,$a$)} & \mathtt{sign}(u)\geq 0\\
\texttt{abs($v$).add($u$,$a$)} & \mathtt{sign}(u) < 0
\end{array}\right.$.


@D additive \verb#operator-# for \Integer s
@{inline Integer::Integer(const binder_arguments<Integer, Integer,
                                               Integer_minus_tag>& a)
{
  get_memory(max(a.x.length(), a.y.length())+DELTA);
  sub(a.x, a.y);
}

inline Integer& Integer::operator=(const binder_arguments<Integer, Integer,
                                                          Integer_minus_tag>& a)
{
  sub(a.x, a.y);
  return *this;
}

inline binder_arguments<Integer, Integer, Integer_minus_tag>
 operator-(const Integer& a, const Integer& b)
// Algorithm:  c := a-b
// Input:      a,b in Integer.
// Output:     c in Integer such that c = a-b ||
{
  return binder_arguments<Integer, Integer, Integer_minus_tag>(a, b);
}
@| Integer::Integer Integer Integer::operator= operator= operator- @}
\textbf{Laufzeit.} Seien $u,v,w\in\Integer$, dann \texttt{$u$-$v$ $\sim$ $w$.sub($u$,$v$)}.

@D additive \verb#operator-# of an \Integer\ with a \Natural\
@{inline Integer::Integer(const binder_arguments<Integer, Natural,
                                               Integer_minus_tag>& a)
{
  get_memory(max(a.x.length(), a.y.length())+DELTA);
  sub(a.x, a.y);
}

inline Integer& Integer::operator=(const binder_arguments<Integer, Natural,
                                                          Integer_minus_tag>& a)
{
  sub(a.x, a.y);
  return *this;
}

inline Integer::Integer(const binder_arguments<Natural, Integer,
                                               Integer_minus_tag>& a)
{
  get_memory(max(a.x.length(), a.y.length())+DELTA);
  sub(a.x, a.y);
}

inline Integer& Integer::operator=(const binder_arguments<Natural, Integer,
                                                          Integer_minus_tag>& a)
{
  sub(a.x, a.y);
  return *this;
}

inline binder_arguments<Integer, Natural, Integer_minus_tag>
 operator-(const Integer& a, const Natural& b)
// Algorithm:  c := a-b
// Input:      a in Integer, b in Natural.
// Output:     c in Integer such that c = a-b ||
{
  return binder_arguments<Integer, Natural, Integer_minus_tag>(a, b);
}

inline binder_arguments<Natural, Integer, Integer_minus_tag>
 operator-(const Natural& a, const Integer& b)
// Algorithm:  c := a-b
// Input:      a in Natural, b in Integer.
// Output:     c in Integer such that c = a-b ||
{
  return binder_arguments<Natural, Integer, Integer_minus_tag>(a, b);
}
@| Integer::Integer Integer Integer::operator= operator= operator- @}
\textbf{Laufzeit.} Seien $u,v\in\Integer$ und $x\in\Natural$, dann \texttt{$u$-$x$ $\sim$ $v$.sub($u$,$x$)}.\\
\textbf{Laufzeit.} Seien $u,v\in\Integer$ und $x\in\Natural$, dann \texttt{$x$-$u$ $\sim$ $v$.sub($x$,$u$)}.

@D additive \verb#operator-# of an \Integer\ with a \Digit\
@{inline Integer::Integer(const binder_arguments<Integer, Digit,
                                               Integer_minus_tag>& a)
{
  get_memory(a.x.length()+DELTA);
  sub(a.x, a.y);
}

inline Integer& Integer::operator=(const binder_arguments<Integer, Digit,
                                                          Integer_minus_tag>& a)
{
  if (this == &a.x) return *this -= a.y;
  else { sub(a.x, a.y); return *this; }
}

inline Integer::Integer(const binder_arguments<Digit, Integer,
                                               Integer_minus_tag>& a)
{
  get_memory(a.y.length()+DELTA);
  sub(a.y, a.x);
  neg();
}

inline Integer& Integer::operator=(const binder_arguments<Digit, Integer,
                                                          Integer_minus_tag>& a)
{
  if (this == &a.y) *this -= a.x;
  else sub(a.y, a.x);
  neg();
  return *this;
}

inline binder_arguments<Integer, Digit, Integer_minus_tag>
 operator-(const Integer& a, const Digit& b)
// Algorithm:  c := a-b
// Input:      a in Integer, Digit b.
// Output:     c in Integer such that c = a-b ||
{
  return binder_arguments<Integer, Digit, Integer_minus_tag>(a, b);
}

inline binder_arguments<Digit, Integer, Integer_minus_tag>
 operator-(const Digit& a, const Integer& b)
// Algorithm:  c := a-b
// Input:      a in Digit, Integer b.
// Output:     c in Integer such that c = a+b ||
{
  return binder_arguments<Digit, Integer, Integer_minus_tag>(a, b);
}
@| Integer::Integer Integer Integer::operator= operator= operator- @}
\textbf{Laufzeit.} Seien $u,v\in\Integer$ und $a\in\Digit$, dann \texttt{$u$+$a$ $\sim$ $v$.sub($u$,$a$)}.


\subsubsection{Zuweisungsoperatoren}

@d assign \verb#operator-=# for \Integer s
@{inline Integer& Integer::operator-=(const Integer& a)
// Algorithm:  c := c -= a
// Input:      a,c in Integer.
// Output:     c in Integer such that c := c-a ||
{
  sub(*this, a);
  return *this;
}
@| Integer::operator-= operator-= @}
\textbf{Laufzeit.} Seien $u,v,w\in\Integer$, dann \texttt{$u$-=$v$ $\sim$ $w$.sub($u$,$v$)}.


@d assign \verb#operator-=# of an \Integer\ with a \Natural\
@{inline Integer& Integer::operator-=(const Natural& a)
// Algorithm:  c := c -= a
// Input:      a in Natural, c in Integer.
// Output:     c in Integer such that c := c-a ||
{
  sub(*this, a);
  return *this;
}
@| Integer::operator-= operator-= @}
\textbf{Laufzeit.} Seien $u,v\in\Integer$ und $x\in\Natural$, dann \texttt{$u$-=$x$ $\sim$ $v$.sub($u$,$x$)}.

@D assign \verb#operator-=# of an \Integer\ with a \Digit\
@{Integer& Integer::operator-=(const Digit a)
// Algorithm:  c := c -= a
// Input:      a Digit,c in Integer.
// Output:     c in Integer such that c := c-a ||
{
  INTEGERCONDITION(*this);

  const int sT = sgn;
  if (sT < 0) abs() += a;
  else if (sT == 0) neg(a);
  else if (::abs(*this) > a) abs() -= a;
  else {
    *this = a - highest();
    neg();
  }

  INTEGERCONDITION(*this);

  return *this;
}
@| Integer::operator-= operator-= @}
\textbf{Laufzeit.} Sei $u\in\Integer$ und $a\in\Digit$, dann $u\texttt{-=}a \sim
\left\{\begin{array}{r@@{\;,\quad}l}
\texttt{abs(}u\texttt{)-=a} & \mathtt{sign}(u)\geq 0\\
\texttt{abs(}u\texttt{)+=a} & \mathtt{sign}(u) < 0
\end{array}\right.$.



\subsection{Schiebeoperationen}
%------------------------------

\subsubsection{Linksverschiebung}

@D multiplication of an \Integer\ by a power of 2
@{inline void Integer::lshift(const Integer& a, const size_t b)
// Algorithm:  c.lshift(a, b)
// Input:      a in Integer, b in size_t.
// Output:     c in Integer such that c = a*2^b ||
{
  abs() = ::abs(a) << b;
  sgn = a.sgn;
}
@| lshift @}
\textbf{Laufzeit.} Seien $u,v\in\Integer$ und $t\in\Size$, dann\smallskip\\
\mbox{}\hfill\texttt{$v$.lshift($u$,$t$) $\sim$ abs($v$)=abs($u$)<<$t$}.\hfill\mbox{}

@D shift \verb#operator<<# of an \Integer\
@{inline Integer::Integer(const binder_arguments<Integer, size_t,
                                               Integer_lshift_tag>& a)
{
  get_memory(a.x.length()+a.y%BETA+DELTA);
  lshift(a.x, a.y);
}

inline Integer& Integer::operator=(const binder_arguments<Integer, size_t,
                                                          Integer_lshift_tag>& a)
{
  lshift(a.x, a.y);
  return *this;
}

inline binder_arguments<Integer, size_t, Integer_lshift_tag>
 operator<<(const Integer& a, const size_t& b)
// Algorithm:  c := a << b
// Input:      a in Integer, b in size_t.
// Output:     c in Integer such that c = a*2^b ||
{
  return binder_arguments<Integer, size_t, Integer_lshift_tag>(a, b);
}
@| Integer::Integer Integer Integer::operator= operator= operator<< @}
\textbf{Laufzeit.} Seien $u,v\in\Integer$ und $t\in\Size$, dann \texttt{$u$<<$t$ $\sim$ $v$.lshift($u$,$t$)}.


\subsubsection{Zuweisungsoperator}

@d assign \verb#operator<<=# of an \Integer\
@{inline Integer& Integer::operator<<=(const size_t a)
// Algorithm:  c := c <<= a
// Input:      a in size_t, c in Integer.
// Output:     c in Integer such that c := c*2^a ||
{
  abs() <<= a;
  return *this;
}
@| Integer::operator<<= operator<<= @}
\textbf{Laufzeit.} Sei $u\in\Integer$ und $t\in\Size$, dann \texttt{$u$<<=$t$ $\sim$ abs($u$)<<=$t$}.


\subsubsection{Rechtsverschiebung}

@d division of an \Integer\ by a power of 2
@{inline void Integer::rshift(const Integer& a, const size_t b)
// Algorithm:  c.rshift(a, b)
// Input:      a in Integer, b in size_t.
// Output:     c in Integer such that c = sign(a)*[|a|/2^b] ||
{
  abs() = ::abs(a) >> b;
  if (::abs(*this) == 0) sgn = 0;
  else sgn = a.sgn;
}
@| Integer::rshift rshift @}
\textbf{Laufzeit.} Seien $u,v\in\Integer$ und $t\in\Size$, dann\smallskip\\
\mbox{}\hfill\texttt{$v$.rshift($u$,$t$) $\sim$ abs($v$)=abs($u$)>>$t$}.\hfill\mbox{}

@D shift \verb#operator>># of an \Integer\
@{inline Integer::Integer(const binder_arguments<Integer, size_t,
                                               Integer_rshift_tag>& a)
{
  get_memory(a.x.length()+DELTA);
  rshift(a.x, a.y);
}

inline Integer& Integer::operator=(const binder_arguments<Integer, size_t,
                                                          Integer_rshift_tag>& a)
{
  rshift(a.x, a.y);
  return *this;
}

inline binder_arguments<Integer, size_t, Integer_rshift_tag>
 operator>>(const Integer& a, const size_t& b)
// Algorithm:  c := a >> b
// Input:      a in Integer, b in size_t.
// Output:     c in Integer such that c = sign(a)*[|a|/2^b] ||
{
  return binder_arguments<Integer, size_t, Integer_rshift_tag>(a, b);
}
@| Integer::Integer Integer Integer::operator= operator= operator>> @}
\textbf{Laufzeit.} Seien $u,v\in\Integer$ und $t\in\Size$, dann \texttt{$u$>>$t$ $\sim$ $v$.rshift($u$,$t$)}.


\subsubsection{Zuweisungsoperator}

@D assign \verb#operator>>=# of an \Integer\
@{inline Integer& Integer::operator>>=(const size_t a)
// Algorithm:  c := c >>= a
// Input:      a in size_t, c in Integer.
// Output:     c in Integer such that c := sign(c)*[|c|/2^a] ||
{
  abs() >>= a;
  if (::abs(*this) == 0) sgn = 0;
  return *this;
}
@| Integer::operator>>= operator>>= @}
\textbf{Laufzeit.} Sei $u\in\Integer$ und $t\in\Size$, dann \texttt{$u$>>=$t$ $\sim$ abs($u$)>>=$t$}.



\subsection{Bitweise Verkn"upfungen}
%-----------------------------------

\label{integer-logic}
Wir verwenden bei den bitweisen Verkn"upfungen die Darstellung des $2^n$-Komplements (siehe Abschnitt
\ref{2n-komplement}, Seite \pageref{2n-komplement}) f"ur negative Zahlen, um die gleichen Eigenschaften 
wie die eingebauten Typen (zum Beispiel \texttt{int}) zu besitzen. Dadurch unterst"utzen wir die
folgenden drei Rechenregeln:
\begin{eqnarray}
-a &=& \verb!~!(a-1)\label{logic-rule-1}\\
&=& \verb!~!a+1\label{logic-rule-2}\\
\verb!~!(a \verb! & ! b) &=& \verb!~!a \verb! | ! \verb!~!b\label{logic-rule-3}\\
\verb!~!(a \verb! | ! b) &=& \verb!~!a \verb! & ! \verb!~!b\label{logic-rule-4}
\end{eqnarray}


\subsubsection{Bitweise UND-Verknp"upfung}

@D bitwise and of two \Integer s
@{void Integer::bitwise_and(const Integer& a, const Integer& b)
// Algorithm:  c.bitwise_and(a, b)
// Input:      a,b in Integer.
// Output:     c in Integer such that c = a and b ||
{
  INTEGERCONDITION(a);
  INTEGERCONDITION(b);

  const int sA = a.sgn;
  const int sB = b.sgn;
  if (sA >= 0) {
    if (sB >= 0) {
      abs() = ::abs(a) & ::abs(b);
      sgn = (::abs(*this) != 0);
    } else {
      @<case 1 of bitwise\_and@>
    }
  } else if (sB < 0) {
    @<case 2 of bitwise\_and@>
  } else {
    @<case 3 of bitwise\_and@>
  }

  INTEGERCONDITION(*this);
}
@| Integer::bitwise_and bitwise_and @}
\textbf{Laufzeit.} Seien $u,v,w\in\Integer$, dann\smallskip\\
\mbox{}\hfill\texttt{$w$.bitwise\_and($u$,$v$) $\sim$ abs($w$)=abs($u$)\&abs($v$)}.\hfill\smallskip\\

Falls die beiden Eingabeargumente der Funtion \verb!bitwise_and! positiv sind, so werden die Argumente einfach
an die Funktion \verb!bitwise_and! f"ur \Natural s weitergeleitet. Andernfalls m"ussen wir die folgenden beiden F"alle
betrachten:
\begin{enumerate}
\item Es gilt die folgende Rechenregel, falls genau ein Argument negativ ist:
\begin{eqnarray*}
\verb!a & -b! &\refequ{=}{(\ref{logic-rule-2})}& \verb!a & (~b+1)!
\end{eqnarray*}

Und wir erhalten das Programm:
@D case 1 of bitwise\_and
@{const Natural c = ::abs(a);
Natural::bitwise_not(::abs(b));
++abs();
const size_t sC = c.length();
const size_t sT = length();
abs() &= c;
if (sT < sC) { copy(c, sC-sT); sgn = 1; }
else sgn = (::abs(*this) != 0);@}

Wir vertauschen die Aufgaben von \texttt{a} und \texttt{b}, falls nur das Argument \texttt{a} negativ ist.
@D case 3 of bitwise\_and
@{const Natural c = ::abs(b);
Natural::bitwise_not(::abs(a));
++abs();
const size_t sC = c.length();
const size_t sT = length();
abs() &= c;
if (sT < sC) { copy(c, sC-sT); sgn = 1; }
else sgn = (::abs(*this) != 0);@}

\item Falls beide Argumente negativ sind, wird der Rechenaufwand ein wenig gr"o"ser:
\begin{eqnarray*}
\verb!-a & -b! &\refequ{=}{(\ref{logic-rule-1})}& \verb!~(a-1) & ~(b-1)!\\
&\refequ{=}{(\ref{logic-rule-4})}& \verb!~((a-1) | (b-1))!\\
&\refequ{=}{(\ref{logic-rule-2})}& \verb!-(((a-1) | (b-1))+1)!\\
&=& \verb!-(((a-1) | (b-1))+1)!
\end{eqnarray*}
Der Programm-Code ist aber sehr elementar:
@D case 2 of bitwise\_and
@{Natural c = b;
--c;
*this = a;
--abs(); abs() |= c; ++abs();@}
\end{enumerate}


@D bitwise \verb#operator&# for \Integer s
@{inline Integer::Integer(const binder_arguments<Integer, Integer,
                                               Integer_and_tag>& a)
{
  get_memory(min(a.x.length(), a.y.length())+DELTA);
  bitwise_and(a.x, a.y);
}

inline Integer& Integer::operator=(const binder_arguments<Integer, Integer,
                                                          Integer_and_tag>& a)
{
  bitwise_and(a.x, a.y);
  return *this;
}

inline binder_arguments<Integer, Integer, Integer_and_tag>
 operator&(const Integer& a, const Integer& b)
// Algorithm:  c := a & b
// Input:      a,b in Integer.
// Output:     c in Integer such that c = a and b ||
{
  return binder_arguments<Integer, Integer, Integer_and_tag>(a, b);
}
@| Integer::Integer Integer Integer::operator= operator= operator& @}
\textbf{Laufzeit.} Seien $u,v,w\in\Integer$, dann \texttt{$u$\&$v$ $\sim$ $w$.bitwise\_and($u$,$v$)}.

@d bitwise \verb#operator&# of an \Integer\ with a \Digit\
@{inline Digit operator&(const Integer& a, const Digit b)
// Algorithm:  c := a & b
// Input:      a in Integer, b in Digit.
// Output:     c in Integer such that c = a and b ||
{
  return (sign(a) >= 0)? abs(a)&b : ((~a.lowest())+1)&b;
}
@| operator& @}
\textbf{Laufzeit.} Sei $u\in\Integer$ und $a\in\Digit$, dann \texttt{$u$\&$a$ $\sim$ abs($u$)\&$a$}.


\subsubsection{Zuweisungsoperator}

@d assign \verb#operator&=# for \Integer s
@{inline Integer& Integer::operator&=(const Integer& a)
// Algorithm:  c := c &= a
// Input:      a,c in Integer.
// Output:     c in Integer such that c := c and a ||
{
  bitwise_and(*this, a);
  return *this;
}
@| Integer::operator&= operator&= @}
\textbf{Laufzeit.} Seien $u,v\in\Integer$, dann \texttt{$u$\&=$v$ $\sim$ $u$.bitwise\_and($u$,$v$)}.

@D assign \verb#operator&=# of an \Integer\ with a \Digit\
@{Digit Integer::operator&=(const Digit b)
// Algorithm:  c := a &= b
// Input:      a in Integer, b in Digit.
// Output:     a in Integer, c in Digit such that a := a and b, c = a ||
{
  INTEGERCONDITION(*this);

  Digit c;
  if (sgn >= 0) {
    c = abs() &= b;
    sgn = (c != 0);
  } else *this = c = ((~lowest())+1)&b;

  INTEGERCONDITION(*this);

  return c;
}
@| Integer::operator&= operator&= @}
\textbf{Laufzeit.} Sei $u\in\Integer$ und $a\in\Digit$, dann \texttt{$u$\&=$a$ $\sim$ abs($u$)\&=$a$}.


\subsubsection{Bitwise inklusive ODER-Verknp"upfung}

@D bitwise inclusive or of two \Integer s
@{void Integer::bitwise_or(const Integer& a, const Integer& b)
// Algorithm:  c.bitwise_or(a, b)
// Input:      a,b in Integer.
// Output:     c in Integer such that c = a or b ||
{
  INTEGERCONDITION(a);
  INTEGERCONDITION(b);

  const int sA = a.sgn;
  const int sB = b.sgn;
  if (sA >= 0) {
    if (sB >= 0) {
      abs() = ::abs(a) | ::abs(b);
      sgn = (::abs(*this) != 0);
    } else {
      @<case 1 of bitwise\_or@>
    }
  } else if (sB < 0) {
    @<case 2 of bitwise\_or@>
  } else {
    @<case 3 of bitwise\_or@>
  }

  INTEGERCONDITION(*this);
}
@| Integer::bitwise_or bitwise_or @}
\textbf{Laufzeit.} Seien $u,v,w\in\Integer$, dann\smallskip\\
\mbox{}\hfill\texttt{$w$.bitwise\_or($u$,$v$) $\sim$ abs($w$)=abs($u$)|abs($v$)}.\hfill\mbox{}

Falls die beiden Eingabeargumente der Funtion \verb!bitwise_or! positiv sind, so werden die Argumente einfach
an die Funktion \verb!bitwise_or! f"ur \Natural s weitergeleitet. Ansonsten m"ussen wir die folgenden beiden F"alle
betrachten:
\begin{enumerate}
\item Es gilt die folgende Rechnung, falls genau ein Argument negativ ist:
\begin{eqnarray*}
\verb!a | -b! &\refequ{=}{(\ref{logic-rule-1})}& \verb!a | ~(b-1)!\\
&\refequ{=}{(\ref{logic-rule-3})}& \verb!~(~a & (b-1))!\\
&\refequ{=}{(\ref{logic-rule-2})}& \verb!-(~a & (b-1))-1!\\
\end{eqnarray*}

Und wir erhalten das Programm:
@D case 1 of bitwise\_or
@{Natural c = b;
--c;
Natural::bitwise_not(::abs(a));
const size_t sC = c.length();
const size_t sT = length();
abs() &= c;
if (sT < sC) { copy(c, sC-sT); sgn = -1; }
else { const int b = (::abs(*this) != 0); sgn = -b; }
--(*this);@}

Wir vertauschen die Aufgaben von \texttt{a} und \texttt{b}, falls nur das Argument \texttt{a} negativ ist.
@D case 3 of bitwise\_or
@{Natural c = a;
--c;
Natural::bitwise_not(::abs(b));
const size_t sC = c.length();
const size_t sT = length();
abs() &= c;
if (sT < sC) { copy(c, sC-sT); sgn = -1; }
else { const int b = (::abs(*this) != 0); sgn = -b; }
--(*this);@}

\item Falls beide Argumente negativ sind, so f"uhren wir die folgende Rechnung durch:
\begin{eqnarray*}
\verb!-a | -b! &\refequ{=}{(\ref{logic-rule-1})}& \verb!~(a-1) | ~(b-1)!\\
&\refequ{=}{(\ref{logic-rule-3})}& \verb!~((a-1) & (b-1))!\\
&\refequ{=}{(\ref{logic-rule-2})}& \verb!-(((a-1) & (b-1))+1)!
\end{eqnarray*}
und erhalten das Programm:
@D case 2 of bitwise\_or
@{Natural c = b;
--c;
*this = a;
--abs(); abs() &= c; ++abs();@}
\end{enumerate}


@D bitwise \verb#operator|# for \Integer s
@{inline Integer::Integer(const binder_arguments<Integer, Integer,
                                               Integer_or_tag>& a)
{
  get_memory(max(a.x.length(), a.y.length())+DELTA);
  bitwise_or(a.x, a.y);
}

inline Integer& Integer::operator=(const binder_arguments<Integer, Integer,
                                                          Integer_or_tag>& a)
{
  bitwise_or(a.x, a.y);
  return *this;
}

inline binder_arguments<Integer, Integer, Integer_or_tag>
 operator|(const Integer& a, const Integer& b)
// Algorithm:  c := a | b
// Input:      a,b in Integer.
// Output:     c in Integer such that c = a or b ||
{
  return binder_arguments<Integer, Integer, Integer_or_tag>(a, b);
}
@| Integer::Integer Integer Integer::operator= operator= operator| @}
\textbf{Laufzeit.} Seien $u,v,w\in\Integer$, dann \texttt{$u$|$v$ $\sim$ $w$.bitwise\_or($u$,$v$)}.

@d bitwise \verb#operator|# of an \Integer\ with a \Digit\
@{inline Integer operator|(const Integer& a, const Digit b)
// Algorithm:  c := a | b
// Input:      a in Integer, b in Digit.
// Output:     c in Integer such that c = a or b ||
{
  return Integer(a) |= b;
}
@| operator| @}
\textbf{Laufzeit.} Sei $u\in\Integer$ und $a\in\Digit$, dann \texttt{$u$|$a$ $\sim$ $u$|=$a$}.



\subsubsection{Zuweisungsoperator}

@d assign \verb#operator|=# for \Integer s
@{inline Integer& Integer::operator|=(const Integer& a)
// Algorithm:  c := c |= a
// Input:      a,c in Integer.
// Output:     c in Integer such that c := c or a ||
{
  bitwise_or(*this, a);
  return *this;
}
@| Integer::operator|= operator|= @}
\textbf{Laufzeit.} Seien $u,v\in\Integer$, dann \texttt{$u$|=$v$ $\sim$ $u$.bitwise\_or($u$,$v$)}.

@d assign \verb#operator|=# of an \Integer\ with a \Digit\
@{inline Integer& Integer::operator|=(const Digit a)
// Algorithm:  c := c |= a
// Input:      c in Integer, a in Digit.
// Output:     c in Integer such that c := c or a ||
{
  if (sgn >= 0) abs() |= a;
  else *this |= Integer(a);
  return *this;
}
@| Integer::operator|= operator|= @}
\textbf{Laufzeit.} Sei $u\in\Integer$ und $a\in\Digit$, dann \texttt{$u$|=$a$ $\sim$ abs($u$)|=$a$}.


\subsubsection{Bitwise exklusive ODER-Verknp"upfung}

@D bitwise exclusive or of two \Integer s
@{void Integer::bitwise_xor(const Integer& a, const Integer& b)
// Algorithm:  c.bitwise_xor(a, b)
// Input:      a,b in Integer.
// Output:     c in Integer such that c = a xor b ||
{
  INTEGERCONDITION(a);
  INTEGERCONDITION(b);

  if (a.sgn >= 0 && b.sgn >= 0) {
    abs() = ::abs(a) ^ ::abs(b);
    sgn = (::abs(*this) != 0);
  } else {
    Integer s,t;
    s.bitwise_not(a); t.bitwise_not(b);
    s &= b; t &= a;
    bitwise_or(s, t);
  }

  INTEGERCONDITION(*this);
}
@| Integer::bitwise_xor bitwise_xor @}
\textbf{Laufzeit.} Seien $u,v,w\in\Integer$, dann\smallskip\\
\mbox{}\hfill\texttt{$w$.bitwise\_xor($u$,$v$) $\sim$ abs($w$)=abs($u$)\^{}abs($v$)}.\hfill\mbox{}

@D bitwise \verb#operator^# for \Integer s
@{inline Integer::Integer(const binder_arguments<Integer, Integer,
                                               Integer_xor_tag>& a)
{
  get_memory(max(a.x.length(), a.y.length())+DELTA);
  bitwise_xor(a.x, a.y);
}

inline Integer& Integer::operator=(const binder_arguments<Integer, Integer,
                                                          Integer_xor_tag>& a)
{
  bitwise_xor(a.x, a.y);
  return *this;
}

inline binder_arguments<Integer, Integer, Integer_xor_tag>
 operator^(const Integer& a, const Integer& b)
// Algorithm:  c := a ^ b
// Input:      a,b in Integer.
// Output:     c in Integer such that c = a xor b ||
{
  return binder_arguments<Integer, Integer, Integer_xor_tag>(a, b);
}
@| Integer::Integer Integer Integer::operator= operator= operator^ @}
\textbf{Laufzeit.} Seien $u,v,w\in\Integer$, dann \texttt{$u$\^{}$v$ $\sim$ $w$.bitwise\_xor($u$,$v$)}.


\subsubsection{Zuweisungsoperator}

@d assign \verb#operator^=# for \Integer s
@{inline Integer& Integer::operator^=(const Integer& a)
// Algorithm:  c := c ^= a
// Input:      a,c in Integer.
// Output:     c in Integer such that c := c xor a ||
{
  bitwise_xor(*this, a);
  return *this;
}
@| Integer::operator^= operator^= @}
\textbf{Laufzeit.} Seien $u,v\in\Integer$, dann \texttt{$u$\^{}=$v$ $\sim$ $u$.bitwise\_xor($u$,$v$)}.


\subsubsection{Bitwise Negation}

@d bitwise not of an \Integer\
@{inline void Integer::bitwise_not(const Integer& a)
// Algorithm:  b.not(a)
// Input:      a in Integer.
// Output:     b in Integer such that b = not a = -a-1 ||
{
  *this = a; neg(); --*this;
}
@| Integer::bitwise_not bitwise_not @}
\textbf{Laufzeit.} Seien $u,v\in\Integer$, dann\smallskip\\
\mbox{}\hfill\texttt{$v$.bitwise\_not($u$) $\sim$ $u$=$v$}.\hfill\mbox{}

@D bitwise \verb#operator~# of an \Integer\
@{inline Integer::Integer(const binder_arguments<Integer, Integer,
                                               Integer_not_tag>& a)
{
  get_memory(a.x.length()+DELTA);
  bitwise_not(a.x);
}

inline Integer& Integer::operator=(const binder_arguments<Integer, Integer,
                                                          Integer_not_tag>& a)
{
  bitwise_not(a.x);
  return *this;
}

inline binder_arguments<Integer, Integer, Integer_not_tag>
 operator~(const Integer& a)
// Algorithm:  c := ~a
// Input:      a in Integer.
// Output:     c in Integer such that c = not a ||
{
  return binder_arguments<Integer, Integer, Integer_not_tag>(a, a);
}
@| Integer::Integer Integer Integer::operator= operator= Integer::operator~ operator~ @}
\textbf{Laufzeit.} Seien $u,v\in\Integer$, dann \verb|~|\texttt{$u$ $\sim$ $v$.bitwise\_not($u$)}.



\subsection{Bitoperationen}
%--------------------------

Bei den Bitoperationen gilt genauso wie bei den bitweisen Verkn"upfungen (siehe Abschnitt \ref{integer-logic},
Seite \pageref{integer-logic}) die Darstellung des $2^n$-Komplements f"ur negative Zahlen.

@D sets a bit in an \Integer\
@{void Integer::setbit(const size_t a)
// Algorithm:  c.setbit(a)
// Input:      a in size_t, c in Integer.
// Output:     c in Integer such that c := c or 2^a ||
{
  INTEGERCONDITION(*this);
  INTEGER_FOR_CHECK(_t, *this);

  if (sgn >= 0) Natural::setbit(a);
  else {
    const size_t b = a/BETA;
    if (b <= length()) {
      Digit* pT;
      const size_t c = trailing_zeros(pT);
      if (c < b) Natural::clearbit(a);
      else if (c == b) *pT = ((*pT-1) & ~(Digit(1)<<(a%BETA)))+1;
      else {
        pT += c-b;                    // not very good!
        *pT = ~(Digit(1) << (a%BETA))+1;
        dec(pT);
      }
    }
  }

  CONDITION(*this == (_t | Integer(1) << a));
  INTEGERCONDITION(*this);
}
@| Integer::setbit setbit @}
\textbf{Laufzeit.} Sei $u\in\Integer$ und $t\in\Size$,
 dann \texttt{$u$.setbit($t$)$\sim abs($u$).setbit($t$)$}.

@D clears a bit in an \Integer\
@{void Integer::clearbit(const size_t a)
// Algorithm:  c.clearbit(a)
// Input:      a in size_t, c in Integer.
// Output:     c in Integer such that c := c and not(2^a) ||
{
  INTEGERCONDITION(*this);
  INTEGER_FOR_CHECK(_t, *this);

  if (sgn >= 0) Natural::clearbit(a);
  else {
    const size_t b = a/BETA;
    Digit* pT;
    const size_t c = trailing_zeros(pT);
    if (c < b) Natural::setbit(a);
    else if (c == b) *pT = ((*pT-1) | (Digit(1)<<(a%BETA)))+1;
  }

  CONDITION(*this == (_t & ~(Integer(1) << a)));           
  INTEGERCONDITION(*this);
}
@| Integer::clearbit clearbit @}
\textbf{Laufzeit.} Sei $u\in\Integer$ und $t\in\Size$,
 dann \texttt{$u$.clearbit($t$)$\sim abs($u$).clearbit($t$)$}.

@D tests a bit in an \Integer\
@{bool Integer::testbit(const size_t a) const
// Algorithm:  c := b.testbit(a)
// Input:      a in size_t, b in Integer.
// Output:     c in bool such that if b and 2^a then c = true else c = false ||
{
  INTEGERCONDITION(*this);

  if (sgn >= 0) return Natural::testbit(a);
  const size_t b = a/BETA;
  Digit* pT;
  const size_t c = trailing_zeros(pT);
  if (c < b) return !Natural::testbit(a);
  else if (c > b) return false;
  else return (((*pT-1) & (Digit(1)<<(a%BETA))) == 0);
}
@| Integer::testbit testbit @}
\textbf{Laufzeit.} Sei $u\in\Integer$ und $t\in\Size$,
 dann \texttt{$u$.testbit($t$)$\sim abs($u$).testbit($t$)$}.



\subsection{Multiplikation}
%--------------------------

@D multiplication of an \Integer\ by a \Digit\
@{inline void Integer::mul(const Integer& a, const Digit b)
// Algorithm:  c.mul(a, b)
// Input:      a in Integer, b in Digit.
// Output:     c in Integer such that c = a*b ||
{
  if (b) {
    abs() = ::abs(a) * b;
    sgn = a.sgn;
  } else *this = 0;
}
@| Integer::mul mul @}
\textbf{Laufzeit.} Seien $u,v\in\Integer$ und $a\in\Digit$, dann\smallskip\\
\mbox{}\hfill\texttt{$v$.mul($u$,$a$) $\sim$ abs($v$)=abs($u$)*$a$}.\hfill\mbox{}

@D multiplication of an \Integer\ by a \Natural\
@{inline void Integer::mul(const Integer& a, const Natural& b)
// Algorithm:  c.mul(a, b)
// Input:      a in Integer, b in Natural.
// Output:     c in Integer such that c = a*b ||
{
  abs() = ::abs(a) * b;
  sgn = a.sgn;
}
@| Integer::mul mul @}
\textbf{Laufzeit.} Seien $u,v\in\Integer$ und $x\in\Natural$, dann\smallskip\\
\mbox{}\hfill\texttt{$v$.mul($u$,$x$) $\sim$ abs($v$)=abs($u$)*$x$}.\hfill\mbox{}

@d multiplication of two \Integer s
@{inline void Integer::mul(const Integer& a, const Integer& b)
// Algorithm:  c.mul(a, b)
// Input:      a,b in Integer.
// Output:     c in Integer such that c = a*b ||
{
  abs() = ::abs(a) * ::abs(b);
  sgn = a.sgn*b.sgn;
}
@| Integer::mul mul @}
\textbf{Laufzeit.} Seien $u,v,w\in\Integer$, dann\smallskip\\
\mbox{}\hfill\texttt{$w$.mul($u$,$v$) $\sim$ abs($w$)=abs($u$)*abs($v$)}.\hfill\mbox{}

@D multiplicative \verb#operator*# for \Integer s
@{inline Integer::Integer(const binder_arguments<Integer, Integer,
                                               Integer_multiplies_tag>& a)
{
  get_memory(a.x.length()+a.y.length()+DELTA);
  mul(a.x, a.y);
}

inline Integer& Integer::operator=(const binder_arguments<Integer, Integer,
                                                          Integer_multiplies_tag>& a)
{
  mul(a.x, a.y);
  return *this;
}

inline binder_arguments<Integer, Integer, Integer_multiplies_tag>
 operator*(const Integer& a, const Integer& b)
// Algorithm:  c := a*b
// Input:      a,b in Integer.
// Output:     c in Integer such that c = a*b ||
{
  return binder_arguments<Integer, Integer, Integer_multiplies_tag>(a, b);
}
@| Integer::Integer Integer Integer::operator= operator= operator* @}
\textbf{Laufzeit.} Seien $u,v,w\in\Integer$, dann \texttt{$u$*$v$ $\sim$ $w$.mul($u$,$v$)}.

@D multiplicative \verb#operator*# of an \Integer\ with a \Natural\
@{inline Integer::Integer(const binder_arguments<Integer, Natural,
                                               Integer_multiplies_tag>& a)
{
  get_memory(a.x.length()+a.y.length()+DELTA);
  mul(a.x, a.y);
}

inline Integer& Integer::operator=(const binder_arguments<Integer, Natural,
                                                          Integer_multiplies_tag>& a)
{
  mul(a.x, a.y);
  return *this;
}

inline binder_arguments<Integer, Natural, Integer_multiplies_tag>
 operator*(const Integer& a, const Natural& b)
// Algorithm:  c := a*b
// Input:      a in Integer, b in Natural.
// Output:     c in Integer such that c = a*b ||
{
  return binder_arguments<Integer, Natural, Integer_multiplies_tag>(a, b);
}

inline binder_arguments<Integer, Natural, Integer_multiplies_tag>
 operator*(const Natural& a, const Integer& b)
// Algorithm:  c := a*b
// Input:      a in Natural, b in Integer.
// Output:     c in Integer such that c = a*b ||
{
  return binder_arguments<Integer, Natural, Integer_multiplies_tag>(b, a);
}
@| Integer::Integer Integer Integer::operator= operator= operator* @}
\textbf{Laufzeit.} Seien $u,v\in\Integer$ und $x\in\Natural$, dann \texttt{$u$*$x$ $\sim$ $v$.mul($u$,$x$)}.

@D multiplicative \verb#operator*# of an \Integer\ with a \Digit\
@{inline Integer::Integer(const binder_arguments<Integer, Digit,
                                               Integer_multiplies_tag>& a)
{
  get_memory(a.x.length()+DELTA);
  mul(a.x, a.y);
}

inline Integer& Integer::operator=(const binder_arguments<Integer, Digit,
                                                          Integer_multiplies_tag>& a)
{
  mul(a.x, a.y);
  return *this;
}

inline Integer& Integer::operator+=(const binder_arguments<Integer, Digit,
                                                           Integer_multiplies_tag>& a)
{
  muladd(a.x, a.y);
  return *this;
}

inline Integer& Integer::operator-=(const binder_arguments<Integer, Digit,
                                                           Integer_multiplies_tag>& a)
{
  mulsub(a.x, a.y);
  return *this;
}

inline binder_arguments<Integer, Digit, Integer_multiplies_tag>
 operator*(const Integer& a, const Digit& b)
// Algorithm:  c := a*b
// Input:      a in Integer, b in Digit.
// Output:     c in Integer such that c = a*b ||
{
  return binder_arguments<Integer, Digit, Integer_multiplies_tag>(a, b);
}

inline binder_arguments<Integer, Digit, Integer_multiplies_tag>
 operator*(const Digit& a, const Integer& b)
// Algorithm:  c := a*b
// Input:      a in Digit, b in Integer.
// Output:     c in Integer such that c = a*b ||
{
  return binder_arguments<Integer, Digit, Integer_multiplies_tag>(b, a);
}
@| Integer::Integer Integer Integer::operator= operator= operator* @}
\textbf{Laufzeit.} Sei $u\in\Integer$ und $a\in\Digit$, dann \texttt{$u$*$a$ $\sim$ $u$*=$a$}.


\subsubsection{Zuweisungsoperatoren}

@d assign \verb#operator*=# for \Integer s
@{inline Integer& Integer::operator*=(const Integer& a)
// Algorithm:  c := c *= a
// Input:      a,c in Integer.
// Output:     c in Integer such that c := c*a ||
{
  sgn *= a.sgn;
  abs() *= ::abs(a);
  return *this;
}
@| Integer::operator*= operator*= @}
\textbf{Laufzeit.} Seien $u,v\in\Integer$, dann \texttt{$u$*=$v$ $\sim$ abs($u$)*=abs($v$)}.

@d assign \verb#operator*=# of an \Integer\ with a \Natural\
@{inline Integer& Integer::operator*=(const Natural& a)
// Algorithm:  c := c *= a
// Input:      a in Natural, c in Integer.
// Output:     c in Integer such that c := c*a ||
{
  if (a != 0) abs() *= a;
  else *this = 0;
  return *this;
}
@| Integer::operator*= operator*= @}
\textbf{Laufzeit.} Sei $u\in\Integer$ und $x\in\Natural$, dann \texttt{$u$*=$x$ $\sim$ abs($u$)*=$x$}.

@d assign \verb#operator*=# of an \Integer\ with a \Digit\
@{inline Integer& Integer::operator*=(const Digit a)
// Algorithm:  c := c *= a
// Input:      a in Digit, c in Integer.
// Output:     c in Integer such that c := c*a ||
{
  if (a) abs() *= a;
  else *this = 0;
  return *this;
}
@| Integer::operator*= operator*= @}
\textbf{Laufzeit.} Sei $u\in\Integer$ und $a\in\Digit$, dann \texttt{$u$*=$a$ $\sim$ abs($u$)*=$a$}.


\subsubsection{Gleichzeitige Multiplikation und Addition}

@D multiplication and addition of an \Integer\ by a \Digit\
@{void Integer::muladd(const Integer& a, const Digit b)
// Algorithm:  c.muladd(a, b)
// Input:      a,c in Integer, b in Digit.
// Output:     c in Integer such that c := c + a*b ||
{
  INTEGERCONDITION(*this);
  INTEGERCONDITION(a);

  const int sT = sgn;
  if (sT == 0) mul(a, b);
  else if (sT == a.sgn) Natural::muladd(::abs(a), b);
  else if (a.length()+1 < length()) Natural::mulsub(::abs(a), b);
  else {
    Integer c = a*b;
    *this += c;
  }

  INTEGERCONDITION(*this);
}
@| Integer::muladd muladd @}
\textbf{Laufzeit.} Seien $u,v\in\Integer$ und $a\in\Digit$, dann
\texttt{$u$.muladd($v$,$a$) $\sim$ $u$.muladd(abs($v$),$a$)}.


\subsubsection{Gleichzeitige Multiplikation und Subtraktion}

@D multiplication and subtraction of an \Integer\ by a \Digit\
@{void Integer::mulsub(const Integer& a, const Digit b)
// Algorithm:  c.mulsub(a, b)
// Input:      a,c in Integer, b in Digit.
// Output:     c in Integer such that c := c - a*b ||
{
  INTEGERCONDITION(*this);
  INTEGERCONDITION(a);

  const int sT = sgn;
  if (sT == 0) { mul(a, b); neg(); }
  else if (sT != a.sgn) Natural::muladd(::abs(a), b);
  else if (a.length()+1 < length()) Natural::mulsub(::abs(a), b);
  else {
    Integer c = a*b;
    *this -= c;
  }

  INTEGERCONDITION(*this);
}
@| Integer::mulsub mulsub @}
\textbf{Laufzeit.} Seien $u,v\in\Integer$ und $a\in\Digit$, dann
\texttt{$u$.mulsub($v$,$a$) $\sim$ $u$.mulsub(abs($v$),$a$)}.



\subsection{Bin"arer Logarithmus}
%--------------------------------

@d binary logarithm for \Integer s
@{inline Digit log2(const Integer& a)
// Algorithm:  b := log2(a)
// Input:      a in Integer.
// Output:     b in Digit
//             such that if |a| > 0 then b = [log2(|a|)] else b = 0 ||
{
  return log2(abs(a));
}
@| log2 @}
\textbf{Laufzeit.} Sei $u\in\Integer$, dann \texttt{log2($u$) $\sim$ log2(abs($u$))}.



\subsection{Potenzieren}
%-----------------------

@D power of an \Integer\ by a \Digit\
@{Integer pow(const Integer& a, const Digit b)
// Algorithm:  c := pow(a, b)
// Input:      a in Integer, b in Digit.
// Output:     c in Integer such that c = a^b ||
{
  INTEGERCONDITION(a);

  Integer c(pow(abs(a), b));
  c.sgn = (abs(c) != 0);
  if (a.sgn == -1 && (b&1)) c.neg();

  INTEGERCONDITION(c);

  return c;
}
@| pow @}
\textbf{Laufzeit.} Sei $u\in\Integer$ und $a\in\Digit$, dann \texttt{pow($u$,$a$) $\sim$ pow(abs($u$),$a$)}.

@D power of an \Integer\ by an \Integer\
@{Integer pow(const Integer& a, const Integer& b)
// Algorithm:  c := pow(a, b)
// Input:      a,b in Integer.
// Output:     c in Integer such that c = sign(a)^b*[|a|^b] ||
{
  INTEGERCONDITION(a);
  INTEGERCONDITION(b);

  if (b < 0 && abs(a) != 1) return Digit(0);
  else {
    Integer c;
    c.abs() = pow(abs(a), abs(b));
    c.sgn = (abs(c) != 0);
    if (a.sgn == -1 && (b&1)) c.neg();

    INTEGERCONDITION(c);

    return c;
  }
}
@| pow @}
\textbf{Laufzeit.} Seien $u,v\in\Integer$, dann \texttt{pow($u$,$v$) $\sim$ pow(abs($u$),abs($v$))}.


\subsection{Division}
%--------------------

@D division of two \Integer s with remainder
@{void div(const Integer& a, const Integer& b, Integer& c, Integer& d)
// Algorithm:  div(a, b, c, d)
// Input:      a,b,c,d in Integer where not b = 0, not c = d;
// Output:     c,d in Integer such that c = sign(a*b)*[|a/b|], d = a-c*b ||
{
  INTEGERCONDITION(a);
  INTEGERCONDITION(b);
  INTEGER_FOR_CHECK(_a, a);
  INTEGER_FOR_CHECK(_b, b);

  div(abs(a), abs(b), c.abs(), d.abs());
  const int sA = a.sgn;
  const int sB = b.sgn;
  c.sgn = (abs(c) != 0)? sA*sB : 0;
  d.sgn = (abs(d) != 0)? sA : 0;

  CONDITION(d+c*_b == _a);
  INTEGERCONDITION(c);
  INTEGERCONDITION(d);
}
@| div @}
\textbf{Laufzeit.} Seien $u,v,w_0,w_1\in\Integer$, dann\smallskip\\
\mbox{}\hfill\texttt{div($u$,$v$,$w_0$,$w_1$) $\sim$ div(abs($u$),abs($v$),abs($w_0$),abs($w_1$))}.\hfill\mbox{}

@D division of two \Integer s
@{void Integer::div(const Integer& a, const Integer& b)
// Algorithm:  c.div(a, b)
// Input:      a,b,c in Integer where not b = 0;
// Output:     c in Integer such that c = sign(a*b)*[|a/b|] ||
{
  INTEGERCONDITION(a);
  INTEGERCONDITION(b);

  Natural::div(::abs(a), ::abs(b));
  sgn = (::abs(*this) != 0)? a.sgn*b.sgn : 0;

  INTEGERCONDITION(*this);
}
@| Integer::div div @}
\textbf{Laufzeit.} Seien $u,v,w\in\Integer$, dann\smallskip\\
\mbox{}\hfill\texttt{$w$.div($u$,$v$) $\sim$ abs($w$).div(abs($u$),abs($v$))}.\hfill\mbox{}

@D multiplicative \verb#operator/# for \Integer s
@{inline Integer::Integer(const binder_arguments<Integer, Integer,
                                                 Integer_divides_tag>& a)
{
  get_memory(a.x.length()+DELTA);
  div(a.x, a.y);
}

inline Integer& Integer::operator=(const binder_arguments<Integer, Integer,
                                                          Integer_divides_tag>& a)
{
  div(a.x, a.y);
  return *this;
}

inline binder_arguments<Integer, Integer, Integer_divides_tag>
 operator/(const Integer& a, const Integer& b)
// Algorithm:  c := a/b
// Input:      a,b in Integer where not b = 0.
// Output:     c in Integer such that c = sign(a*b)*[|a/b|] ||
{
  return binder_arguments<Integer, Integer, Integer_divides_tag>(a, b);
}
@| Integer::Integer Integer Integer::operator= operator= operator/ @}
\textbf{Laufzeit.} Seien $u,v,w\in\Integer$, dann \texttt{$u$/$v$ $\sim$ $w$.div($u$,$v$)}.

@D multiplicative \verb#operator%# for \Integer s
@{inline Integer::Integer(const binder_arguments<Integer, Integer,
                                               Integer_modulus_tag>& a)
{
  Integer t(a.x.length()+DELTA, ' ');
  get_memory(a.y.length()+DELTA);
  ::div(a.x, a.y, t, *this);
}

inline Integer& Integer::operator=(const binder_arguments<Integer, Integer,
                                                          Integer_modulus_tag>& a)
{
  Integer t(a.x.length()+DELTA, ' ');
  ::div(a.x, a.y, t, *this);
  return *this;
}

inline binder_arguments<Integer, Integer, Integer_modulus_tag>
 operator%(const Integer& a, const Integer& b)
// Algorithm:  c := a%b
// Input:      a,b in Integer where not b = 0.
// Output:     c in Integer such that c = a - sign(a*b)*[|a/b|]*b ||
{
  return binder_arguments<Integer, Integer, Integer_modulus_tag>(a, b);
}
@| Integer::Integer Integer Integer::operator= operator= operator% @}
\textbf{Laufzeit.} Seien $u,v,w_0,w_1\in\Integer$, dann \texttt{$u$\%$v$ $\sim$ div($u$,$v$,$w_0$,$w_1$)}.

@D division of an \Integer\ by a \Natural\
@{void Integer::div(const Integer& a, const Natural& b)
// Algorithm:  c.div(a, b)
// Input:      a,c in Integer, b in Natural where not b = 0;
// Output:     c in Integer such that c = sign(a)*[|a|/b] ||
{
  INTEGERCONDITION(a);

  Natural::div(::abs(a), b);
  sgn = (::abs(*this) != 0)? a.sgn : 0;

  INTEGERCONDITION(*this);
}
@| Integer::div div @}
\textbf{Laufzeit.} Seien $u,v\in\Integer$ und $x\in\Natural$, dann\smallskip\\
\mbox{}\hfill\texttt{$v$.div($u$,$x$) $\sim$ abs($v$).div(abs($u$),$x$)}.\hfill\mbox{}

@D division of a \Natural\ by an \Integer\
@{void Integer::div(const Natural& a, const Integer& b)
// Algorithm:  c.div(a, b)
// Input:      a in Natural, b,c in Integer where not b = 0;
// Output:     c in Integer such that c = sign(b)*[a/|b|] ||
{
  INTEGERCONDITION(b);

  Natural::div(a, ::abs(b));
  sgn = (::abs(*this) != 0)? b.sgn : 0;

  INTEGERCONDITION(*this);
}
@| Integer::div div @}
\textbf{Laufzeit.} Seien $u,v\in\Integer$ und $x\in\Natural$, dann\smallskip\\
\mbox{}\hfill\texttt{$v$.div($x$,$u$) $\sim$ abs($v$).div($x$,abs($u$))}.\hfill\mbox{}

@D multiplicative \verb#operator/# of an \Integer\ with a \Natural\
@{inline Integer::Integer(const binder_arguments<Integer, Natural,
                                                 Integer_divides_tag>& a)
{
  get_memory(a.x.length()+DELTA);
  div(a.x, a.y);
}

inline Integer& Integer::operator=(const binder_arguments<Integer, Natural,
                                                          Integer_divides_tag>& a)
{
  div(a.x, a.y);
  return *this;
}

inline Integer::Integer(const binder_arguments<Natural, Integer,
                                                 Integer_divides_tag>& a)
{
  get_memory(a.x.length()+DELTA);
  div(a.x, a.y);
}

inline Integer& Integer::operator=(const binder_arguments<Natural, Integer,
                                                          Integer_divides_tag>& a)
{
  div(a.x, a.y);
  return *this;
}

inline binder_arguments<Integer, Natural, Integer_divides_tag>
 operator/(const Integer& a, const Natural& b)
// Algorithm:  c := a/b
// Input:      a in Integer, b in Natural where not b = 0.
// Output:     c in Integer such that c = sign(a)*[|a|/b] ||
{
  return binder_arguments<Integer, Natural, Integer_divides_tag>(a, b);
}

inline binder_arguments<Natural, Integer, Integer_divides_tag>
 operator/(const Natural& a, const Integer& b)
// Algorithm:  c := a/b
// Input:      a in Natural, b in Integer where not b = 0.
// Output:     c in Integer such that c = sign(b)*[a/|b|] ||
{
  return binder_arguments<Natural, Integer, Integer_divides_tag>(a, b);
}
@| Integer::Integer Integer Integer::operator= operator= operator/ @}
\textbf{Laufzeit.} Seien $u,v\in\Integer$ und $x\in\Natural$, dann \texttt{$u$/$x$ $\sim$ $v$.div($u$,$x$)}.\\
\textbf{Laufzeit.} Seien $u,v\in\Integer$ und $x\in\Natural$, dann \texttt{$x$/$u$ $\sim$ $v$.div($x$,$u$)}.


@D division of an \Integer\ by a \Digit\
@{void div(const Integer& a, const Digit b, Integer& c, Integer& d)
// Algorithm:  div(a, b, c, d)
// Input:      a in Integer, b in Digit where not b = 0.
// Output:     c,d in Integer such that c = sign(a)*[|a|/b], d = a - c*b ||
{
  INTEGERCONDITION(a);

  Digit r;
  div(abs(a), b, c.abs(), r);
  d = r;
  if (sign(a) == -1) d.neg();
  c.sgn = (abs(c) != 0)? a.sgn : 0;

  INTEGERCONDITION(c);
}
@| div @}
\textbf{Laufzeit.} Seien $u,v,w\in\Integer$ und $a,b\in\Digit$, dann\smallskip\\
\mbox{}\hfill\texttt{div($u$,$a$,$v$,$w$) $\sim$ div(abs($u$),$a$,abs($v$), $b$)}.\hfill\mbox{}
 
@d multiplicative \verb#operator/# of an \Integer\ with a \Digit\
@{inline Integer operator/(const Integer& a, const Digit b)
// Algorithm:  c := a/b
// Input:      a in Integer, b in Digit where not b = 0.
// Output:     c in Integer such that c = sign(a)*[|a|/b] ||
{
  return Integer(a) /= b;
}
@| operator/ @}
\textbf{Laufzeit.} Sei $u\in\Integer$ und $a\in\Digit$, dann \texttt{$u$/$a$ $\sim$ $u$/=$a$}.

@d multiplicative \verb#operator%# of an \Integer\ with a \Digit\
@{inline Integer operator%(const Integer& a, const Digit b)
// Algorithm:  c := a%b
// Input:      a in Integer, b in Digit where not b = 0.
// Output:     c in Digit such that c = a - sign(a)*[|a|/b]*b ||
{
  Integer c(abs(a) % b);
  if (sign(a) == -1) c = -c;
  return c;
}
@| operator% @}
\textbf{Laufzeit.} Sei $u\in\Integer$ und $a\in\Digit$, dann \texttt{$u$\%$a$ $\sim$ abs($u$)\%$a$}.


\subsubsection{Zuweisungsoperatoren}

@d assign \verb#operator/=# for \Integer s
@{inline Integer& Integer::operator/=(const Integer& a)
// Algorithm:  c := c /= a
// Input:      a,c in Integer where not a = 0.
// Output:     c in Integer such that c := sign(a*c)*[|c/a|] ||
{
  abs() /= ::abs(a);
  sgn = (::abs(*this) != 0)? sgn*a.sgn : 0;
  return *this;
}
@| Integer::operator/= operator/= @}
\textbf{Laufzeit.} Seien $u,v\in\Integer$, dann \texttt{$u$/=$v$ $\sim$ abs($u$)/=abs($v$)}.

@d assign \verb#operator%=# for \Integer s
@{inline Integer& Integer::operator%=(const Integer& a)
// Algorithm:  c := c %= a
// Input:      a,c in Integer where not a = 0.
// Output:     c in Integer such that c := c - sign(a*c)*[|c/a|]*a ||
{
  abs() %= ::abs(a);
  if (::abs(*this) == 0) sgn = 0;
  return *this;
}
@| Integer::operator%= operator%= @}
\textbf{Laufzeit.} Seien $u,v\in\Integer$, dann \texttt{$u$\%=$v$ $\sim$ abs($u$)\%=abs($v$)}.

@d assign \verb#operator/=# of an \Integer\ with a \Natural\
@{inline Integer& Integer::operator/=(const Natural& a)
// Algorithm:  c := c /= a
// Input:      c in Integer, a in Natural where not a = 0.
// Output:     c in Integer such that c := sign(c)*[|c|/a] ||
{
  abs() /= a;
  if (::abs(*this) == 0) sgn = 0;
  return *this;
}
@| Integer::operator/= operator/= @}
\textbf{Laufzeit.} Sei $u\in\Integer$ und $x\in\Natural$, dann \texttt{$u$/=$x$ $\sim$ abs($u$)/=$x$}.

@d assign \verb#operator%=# of an \Integer\ with a \Natural\
@{inline Integer& Integer::operator%=(const Natural& a)
// Algorithm:  c := c %= a
// Input:      c in Integer, a in Natural where not a = 0.
// Output:     c in Integer such that c := c - sign(c)*[|c|/a]*a ||
{
  abs() %= a;
  if (::abs(*this) == 0) sgn = 0;
  return *this;
}
@| Integer::operator%= operator%= @}
\textbf{Laufzeit.} Sei $u\in\Integer$ und $x\in\Natural$, dann \texttt{$u$\%=$x$ $\sim$ abs($u$)\%=$x$}.

@d assign \verb#operator/=# of an \Integer\ with a \Digit\
@{inline Integer& Integer::operator/=(const Digit a)
// Algorithm:  c := c /= a
// Input:      c in Integer, a in Digit where not a = 0.
// Output:     c in Integer such that c := sign(c)*[|c|/a] ||
{
  if (::abs(*this) < a) *this = 0;
  else abs() /= a;
  return *this;
}
@| Integer::operator/= operator/= @}
\textbf{Laufzeit.} Sei $u\in\Integer$ und $a\in\Digit$, dann \texttt{$u$/=$a$ $\sim$ abs($u$)/=$a$}.

@d assign \verb#operator%=# of an \Integer\ with a \Digit\
@{inline Integer& Integer::operator%=(const Digit a)
// Algorithm:  c := c %= a
// Input:      c in Integer, a in Digit where not a = 0.
// Output:     c in Integer such that c := c - sign(c)*[|c|/a]*a ||
{
  abs() %= a;
  if (::abs(*this) == 0) sgn = 0;
  return *this;
}
@| Integer::operator%= operator%= @}
\textbf{Laufzeit.} Sei $u\in\Integer$ und $a\in\Digit$, dann \texttt{$u$\%=$a$ $\sim$ abs($u$)\%=$a$}.


\subsection{Splitting}
%---------------------

@d splits an \Integer\
@{inline void Integer::split(const size_t b, Integer& c, Integer& d) const
// Algorithm:  a.split(b, c, d)
// Input:      a,c,d in Integer, b in size_t where not c = d.
// Output:     c,d in Integer such that c = sign(a)*[|a|/2^(BETA*b)],
//             d = a - c*2^(BETA b) ||
{
  Natural::split(b, c.abs(), d.abs());
  c.sgn = (::abs(c) != 0)? sgn : 0;
  d.sgn = (::abs(d) != 0)? sgn : 0;
}
@| Integer::split split @}
\textbf{Laufzeit.} Seien $u,v,w\in\Natural$ und $t\in\Size$, dann
\[\texttt{$u$.split($t$,$v$,$w$) $\sim$ abs($u$).split($t$),abs($v$),abs($w$)}.\]



\subsection{Wurzelberechnung}
%----------------------------

\subsubsection{Quadratwurzel}


@D square root of an \Integer\
@{void Integer::sqrt(const Integer& a)
// Algorithm:  b.sqrt(a)
// Input:      a,b in Integer where a >= 0.
// Output:     b in Integer such that b = [sqrt(a)] ||
{
  INTEGERCONDITION(a);

  const int sA = a.sgn;
  if (sA == -1) errmsg(6, "(sqrt)");
  Natural::sqrt(::abs(a));
  sgn = sA;

  INTEGERCONDITION(*this);
}
@| Integer::sqrt sqrt @}
\textbf{Laufzeit.} Sei $u\in\Integer$, dann \texttt{sqrt($u$) $\sim$ sqrt(abs($u$))}.


@D positive square root and remainder of an \Integer\
@{void sqrt(const Integer& a, Integer& b, Integer& c)
// Algorithm:  sqrt(a, b, c)
// Input:      a in Integer where a >= 0.
// Output:     b,c in Integer such that b = [sqrt(a)], c = a - b^2 ||
{
  INTEGERCONDITION(a);
  INTEGER_FOR_CHECK(_a, a);

  const int sA = a.sgn;
  if (sA == -1) a.errmsg(6, "(sqrt)");
  else if (sA == 0) b = c = 0;
  else {
    sqrt(abs(a), b.abs(), c.abs());
    b.sgn = 1;
    c.sgn = (abs(c) != 0);
  }

  CONDITION(c+b*b == _a);
  INTEGERCONDITION(b);
  INTEGERCONDITION(c);
}
@| sqrt @}
\textbf{Laufzeit.} Seien $u,v,w\in\Integer$, dann\smallskip\\
\mbox{}\hfill\texttt{sqrt($u$,$v$,$w$) $\sim$ sqrt(abs($u$),abs($v$),abs($w$))}.\hfill\mbox{}

@D simple call for square root of an \Integer\
@{inline Integer::Integer(const binder_arguments<Integer, Integer,
                                               Integer_square_root_tag>& a)
{
  get_memory(a.x.length()/2+DELTA);
  sqrt(a.x);
}

inline Integer& Integer::operator=(const binder_arguments<Integer, Integer,
                                                          Integer_square_root_tag>& a)
{
  sqrt(a.x);
  return *this;
}

inline binder_arguments<Integer, Integer, Integer_square_root_tag>
 sqrt(const Integer& a)
// Algorithm:  b := sqrt(a)
// Input:      a in Integer where a >= 0.
// Output:     b in Integer such that b = [sqrt(a)] ||
{
  return binder_arguments<Integer, Integer, Integer_square_root_tag>(a, a);
}
@| Integer::Integer Integer Integer::operator= operator= sqrt @}
\textbf{Laufzeit.} Sei $u\in\Integer$ und $x\in\Natural$, dann \texttt{sqrt($u$) $\sim$ sqrt(abs($u$),$x$)}.


\subsubsection{Beliebige Wurzel}

@D calculates the root of an \Integer\
@{Integer root(const Integer& a, const Digit b)
// Algorithm:  c := root(a, b)
// Input:      a in Integer, b in Digit where 2|b or a >= 0.
// Output:     c in Integer such that c = sign(a)^b*[|a|^(1/b)] ||
{
  INTEGERCONDITION(a);

  const int sA = sign(a);
  if ((b&1) && sA < 0) return -Integer(root(abs(a), b));
  if (sA < 0) a.errmsg(6, "(root)");
  return root(abs(a), b);
}
@| root @}
\textbf{Laufzeit.} Sei $u\in\Integer$ und $a\in\Digit$, dann \texttt{root($u$,$a$) $\sim$ root(abs($u$),$a$)}.



\subsection{Zufallszahl}
%-----------------------

@D calculates an \Integer\ random number
@{void Integer::rand(const size_t n)
// Algorithm:  a.rand(n)
// Input:      n in size_t.
// Output:     a in Integer such that |a| < 2^n (random number) ||
{
  INTEGERCONDITION(*this);

  Natural::rand(n);
  if (::abs(*this) == 0) sgn = 0;
  else sgn = (::rand()&1)? 1 : -1;

  INTEGERCONDITION(*this);
}
@| Integer::rand rand @}
\textbf{Laufzeit.} Sei $u\in\Integer$ und $a\in\Digit$, dann \texttt{$u$.rand($a$) $\sim$ abs($u$).rand($a$)}.
