\section{Fibonacci-Zahlen}
%=========================

Die Fibonacci-Zahlen sind durch die folgende Rekursion definiert:\index{Fibonacci-Zahlen}\label{Fibonacci-Zahlen}
\[F_0 = 0, F_1 = 1\mbox{ und }F_n = F_{n-1} + F_{n-2}\quad\mbox{ f"ur }n\geq2.\]
\label{Fibonacci-Algorithmus}

Eine direkte Umsetzung dieser Definition haben wir bereits im Kapitel \ref{additiver Fibonacci-Algorithmus} auf der
Seite \pageref{additiver Fibonacci-Algorithmus} kennengelernt.\\
In den folgenden vier Schritten werden wir uns dagegen einen wesentlich schnelleren Algorithmus herleiten:
\begin{enumerate}
\item \textbf{Behauptung.}\begin{eqnarray}\small\left(\begin{array}{cc}
1 & 1\\
1 & 0
\end{array}\right)^n
&=& \left(\begin{array}{cc}
F_{n+1} & F_n\\
F_n & F_{n-1}
\end{array}\right)\normalsize\mbox{ f"ur alle }n\geq1.\label{Fibonacci-Matrix}\end{eqnarray}
\textbf{Beweis.} Durch vollst"andige Induktion "uber n.\\
Induktionsanfang f"ur $n=1$:
$\small\left(\begin{array}{cc}
1 & 1\\
1 & 0
\end{array}\right)
 = \left(\begin{array}{cc}
F_2 & F_1\\
F_1 & F_0
\end{array}\right)\normalsize\quad\checkmark$.\\
Induktionsschritt: Sei (\ref{Fibonacci-Matrix}) unsere Induktionsvoraussetzung f"ur $n$, dann\\
$\small\left(\begin{array}{cc}
1 & 1\\
1 & 0
\end{array}\right)^{n+1}
 = \left(\begin{array}{cc}
F_{n+1} & F_n\\
F_n & F_{n-1}
\end{array}\right)
\left(\begin{array}{cc}
1 & 1\\
1 & 0
\end{array}\right)
 = \left(\begin{array}{cc}
F_{n+1}+F_n & F_{n+1}\\
F_n+F_{n-1} & F_n
\end{array}\right)
 = \left(\begin{array}{cc}
F_{n+2} & F_{n+1}\\
F_{n+1} & F_n
\end{array}\right)\normalsize$\\
\qed
\item \textbf{Behauptung.}\begin{eqnarray}
F_{2n-1} &=& F_n^2+F_{n-1}^2,\label{quadrat Fibonacci-odd}\\
F_{2n} &=& F_n^2+2F_nF_{n-1} = (F_n+F_{n-1})^2-F_{n-1}^2.\label{quadrat Fibonacci-even}
\end{eqnarray}
\textbf{Beweis.} Nach (\ref{Fibonacci-Matrix}) gilt:
\begin{eqnarray*}
\small\left(\begin{array}{cc}
F_{2n+1} & F_{2n}\\
F_{2n} & F_{2n-1}
\end{array}\right)
&=& \left(\begin{array}{cc}
1 & 1\\
1 & 0
\end{array}\right)^{2n}
= \left(\begin{array}{cc}
F_{n+1} & F_n\\
F_n & F_{n-1}
\end{array}\right)^2\\
&=& \left(\begin{array}{cc}
F_{n+1}^2+F_n^2 & F_n^2+F_{n-1}F_n\smallskip\\
F_n^2+2F_nF_{n-1} & F_n^2+F_{n-1}^2
\end{array}\right)\normalsize
\end{eqnarray*}\qed
\item \textbf{Folgerung.}
$F_n = \left\{\begin{array}{r@@{\;,\quad}l}
F_{\frac{n-1}{2}}^2 + F_{\frac{n+1}{2}}^2 & 2\nmid n\\
F_{\frac{n}{2}}^2 + 2F_{\frac{n}{2}}F_{\frac{n}{2}-1} & 2\mid n
\end{array}\right..$
\item Die Implementation der Folgerung w"urde nun drei Quadraturen ben"otigen. Die Anzahl der
Quadraturen l"a"st sich jedoch durch die folgende Beziehung auf zwei reduzieren:\\
\textbf{Behauptung.}\begin{eqnarray}
F_n\cdot F_{n-1} &=& F_n^2-F_{n-1}^2+(-1)^n\qquad\mbox{ f"ur alle }n\geq1.\label{quadrat Fibonacci-both}
\end{eqnarray}
\textbf{Beweis.} Durch vollst"andige Induktion "uber n.\\
Induktionsanfang f"ur $n\in\{1,2\}$:
\begin{eqnarray*}
F_1\cdot F_0 &=& 0 = 1^2-0^2-1 = F_1^2-F_0^2-1\quad\checkmark\\
F_2\cdot F_1 &=& 1 = 1^2-1^2+1 = F_2^2-F_1^2+1\quad\checkmark.\\
\end{eqnarray*}
Induktionsschritt: Sei (\ref{quadrat Fibonacci-both}) unsere Induktionsvoraussetzung f"ur $n$, dann\\
\begin{eqnarray*}
F_{n+2}\cdot F_{n+1} &=& (F_{n+1}+F_{n})\cdot(F_{n}+F_{n-1})\\
&=& (2F_{n}+F_{n-1})\cdot(F_{n}+F_{n-1})\\
&=& 2F_{n}^2+3F_{n}F_{n-1}+F_{n-1}^2\\
&\refequ{=}{(IS)}& 2F_{n}^2+2F_{n}F_{n-1}+F_{n-1}^2+(F_{n}^2-F_{n-1}^2+(-1)^n)\\
&=& 3F_{n}^2+2F_{n}F_{n-1}+(-1)^n\\
&=& 4F_{n}^2+4F_{n}F_{n-1}+F_{n-1}^2-F_{n}^2-2F_{n}F_{n-1}-F_{n-1}^2+(-1)^n\\
&=& (2F_{n}+F_{n-1})^2-(F_{n}+F_{n-1})^2+(-1)^n\\
&=& F_{n+2}^2-F_{n+1}^2+(-1)^n
\end{eqnarray*}\qed

\item Die Implementation erhalten wir dann schlie"slich durch die Bin"ardarstellung von $n$:
@D fast Fibonacci algorithm
@{Digit a = Digit(1) << log2(n);
do {
  k = j*j; t = i*i;
  j = k+t;                    // (4.2)
  --k; i = t-k;
  i <<= 1; i += t;            // (4.4)
  while (n&a) {
    j += i; swap(i, j);
    a >>= 1;
    if (a == 0) return i;
    k = j*j; t = i*i;
    j = k+t;                  // (4.2)
    i = t-k;
    --i; i <<= 1; i += t;     // (4.4)
  }
} while (a >>= 1);
return i;@}
\end{enumerate}



\section{Der gr"o"ste gemeinsame Teiler}
%=======================================

\label{groesste gemeinsame Teiler}
\textbf{Definition}. Sei $n\in\N$, so setze f"ur alle $a_k\in\Z, 0\leq k\leq n$:
\begin{eqnarray*}
\lcm(a_k)_{k=0}^n & := & \min\{x\in\N\mid a_k|x, 0\leq k\leq n\}\\
\gcd(a_k)_{k=0}^n & := & \max\{x\in\N\mid x|a_k, 0\leq k\leq n\},
\end{eqnarray*}
wobei die Konvention $\gcd(0^n) := 0$ gilt.


\subsection{Einschr"ankung auf zwei Argumente}
%---------------------------------------------
\label{ggT-Einschraenkung auf 2 Argumente}

Es gilt die Eigenschaft:
\[ \gcd(a, b, c) = \gcd(a, \gcd(b, c)) = \gcd(\gcd(a, b), c). \]

Dies ist auch f"ur eine beliebige Anzahl an Argumenten m"oglich, was programmtechnisch mit den Makros aus der
Standarddatei \texttt{<stdarg.h>} realisiert wird. Der Vorteil dieser Programmierweise ist, da"s sie nicht nur komfortabel,
sondern auch durch interne Sortiermethoden schneller ist.\\

\textbf{Beispiel.} Der Aufruf
\[\gcd(\gcd(F_{21}, F_{17}), F_{14})\]
mit den Fermat-Zahlen $F_n := 2^{2^n} + 1$\index{Fermat-Zahlen}\label{Fermat-Zahlen}
ben"otigt auf dem GNU-Compiler 2.7.2 72,6 Sekunden und ist im Vergleich zur "aquivalenten L"osung
\[\gcd(F_{21}, \gcd(F_{17}, F_{14}))\]
mit 0,6 Sekunden deutlich langsamer.\\

Beliebig viele Argumente k"onnen folgenderma"sen implementiert werden:\\

\small\begin{minipage}{\linewidth}
\begin{verbatim}
Natural gcd(const Natural& a, const Natural& b, ...)
{
  va_list pData;
  va_start(pData, b);
  /*
      gcd Algorithm
  */
  va_end(pData);
}
\end{verbatim}
\end{minipage} \normalsize\\[3ex]

Die H"urde ist dabei, die unbekannten Argumente zu "ubernehmen, weil der Zugriff ja "uber Makros erfolgt.
So mu"s  bei jedem Aufruf des Makros \verb|va_arg(pData, Natural)| der Typ des erwarteten Arguments angegeben werden, da
man normalerweise nicht feststellen kann, ob wirklich ein Argument diesen Typs "ubergeben wird.\\
Damit w"urde zum Beispiel die folgende Konstruktion fehlerhafte Ergebnisse liefern:\\

\small \begin{minipage}{\linewidth}
\begin{verbatim}
Digit a = 641;
Natural b = Natural(1) << 32;
cout << "gcd=" << gcd(a, ++b) << endl;
\end{verbatim}
\end{minipage} \normalsize\\[3ex]

Hier w"are es notwendig, stattdessen \texttt{gcd(Natural(a), Natural(b))} aufzurufen, was dem Anwender ein zus"atzliches
Ma"s an Aufmerksamkeit abverlangt und gegen unsere Konvention der bedingungslosen Argument"ubergabe aus Kapitel
\ref{Benutzerforderungen} verst"o"st.\\

Aus diesem Grund werden wir uns in dieser Arbeit lediglich mit der Berechnung des gr"o"sten gemeinsamen Teilers von zwei
Argumenten befassen.



\subsection{Euklidischer Algorithmus f"ur \Digit s}
%--------------------------------------------------

Der bekannteste Algorithmus zur Ermittlung des gr"o"sten gemeinsamen Teilers stammt von Euklid aus der Zeit
um 300 vor Christus:\index{Algorithmus von!Euklid}\\

\small \begin{minipage}{\linewidth}
\begin{verbatim}
Digit gcd(Digit a, Digit b)
// Algorithm:  c := gcd(a, b)
// Input:      a,b in Digit.
// Output:     c in Digit such that c = gcd(a, b) ||
{
  while (b) {
    a %= b;
    if (a == 0) return b;
    b %= a;
  }
  return a;
}
\end{verbatim}
\end{minipage} \normalsize\\[3ex]
Hierbei nutzen wir die folgende rekursive Eigenschaft des gr"o"sten gemeinsamen Teilers aus:
\[\gcd : \N^2\to\N : (a, b)\mapsto\left\{ \begin{array}{r@@{\;,\quad}l}
\gcd(b, a\bmod b) & a\geq b\\
\gcd(a, b\bmod a) & a < b\\
a & b = 0\\
b & a = 0
\end{array}\right..\]


\subsection{Bin"arer Algorithmus von Stein}
%------------------------------------------

Josef Stein entdeckte 1961 (\cite{Stein67}) einen bin"aren Algorithmus, der auf den folgenden rekursiven Eigenschaften
beruht:
\[\gcd : \N^2\to\N : (a, b)\mapsto\left\{ \begin{array}{r@@{\;,\quad}l}
2\gcd(\frac{a}{2}, \frac{b}{2}) & 2\mid a, 2\mid b\\
\gcd(\frac{a}{2}, b) & 2\mid a, 2\nmid b\\
\gcd(a, \frac{b}{2}) & 2\nmid a, 2\mid b\\
\gcd(|a - b|, b) & 2\nmid a, 2\nmid b
\end{array}\right..\]

Hierf"ur werden nur Subtraktionen und Divisionen durch 2 ben"otigt, die schneller als eine beliebige Division von einem
Rechner verarbeitet werden. Durch diese Eigenschaft ist das folgende Programm im \Digit-Bereich deutlich schneller als der
herk"ommliche Euklidische Algorithmus aus dem vorherigen Abschnitt, obwohl mehr Iterationsschritte durchgef"uhrt werden:

@D greatest common divisor of two \Digit s
@{Digit gcd(Digit a, Digit b)
// Algorithm:  c := gcd(a, b)
// Input:      a,b in Digit.
// Output:     c in Digit such that c = gcd(a, b) ||
{
  if (a == 0) return b;
  if (b == 0) return a;

  Digit i,j;
  for (i = 0; (a&1) == 0; ++i) a >>= 1;
  for (j = 0; (b&1) == 0; ++j) b >>= 1;
  while (a != b)
    if (a > b) {
      a -= b;
      do a >>= 1; while ((a&1) == 0);
    } else {
      b -= a;
      do b >>= 1; while ((b&1) == 0);
    }
  return a << ((i > j)? j : i);
}
@| gcd @}
\textbf{Laufzeit.} Seien $a,b\in\Digit$, dann \texttt{gcd($a$,$b$)$\in\Ord(1)$}.\\

Selbst bei zwei \Digit s weist der bin"are Algorithmus ein besseres Verhalten als der Euklidische auf, obwohl der Code nun
erheblich anw"achst:

@D greatest common divisor of two double \Digit s
@{void NumberBase::gcd(Digit a, Digit b, Digit c, Digit d, Digit& x, Digit& y) const
// Algorithm:  n.gcd(a, b, c, d; x, y)
// Input:      n in NumberBase, a,b,c,d in Digit.
// Output:     x,y in Digit such that x*2^BETA+y = gcd(a*2^BETA+b, c*2^BETA+d) ||
{
  if (a == 0 && b == 0) { x = c; y = d; }
  else if (c == 0 && d == 0) { x = a; y = b; }
  else {
    Digit i = 0;
    Digit j = 0;
    @<find power of 2 in the binary gcd algorithm for double \Digit s@>
    while (b != d && a != c)
      if (a > c) {
        a -= c + (b < d); b -= d;
        j = 0;
        do { ++j; b >>= 1; } while ((b&1) == 0);
        b |= a << (BETA-j); a >>= j;
      } else {
        c -= a + (d < b); d -= b;
        j = 0;
        do { ++j; d >>= 1; } while ((d&1) == 0);
        d |= c << (BETA-j); c >>= j;
      }
    if (b != d || a != c) {
      if (a == c) {
        if (b > d) {
          b -= d;
          digitmod(c, d, b, a);
          a = ::gcd(a, b);
        } else {
          d -= b;
          digitmod(a, b, d, c);
          a = ::gcd(c, d);
        }
      } else {
        if (a > c) {
          a -= c;
          digitmod(c, d, a, b);
          a = ::gcd(a, b);
        } else {
          c -= a;
          digitmod(a, b, c, d);
          a = ::gcd(c, d);
        }
      }
      if (i >= BETA) { x = a << (i-BETA); y = 0; }
      else if (i) { x = a >> (BETA-i); y = a << i; }
      else { x = 0; y = a; }
    } else if (i >= BETA) { x = b << (i-BETA); y = 0; }
    else if (i) { x = a << i | b >> (BETA-i); y = b << i; }
    else { x = a; y = b; }
  }
}
@| gcd @}
\textbf{Laufzeit.} Sei $n\in\NumberBase$ und $a_0,a_1,b_0,b_1,d_0,d_1\in\Digit$, dann
 \texttt{$n$.gcd($a_0$,$a_1$,$b_0$,$b_1$,$d_0$,$d_1$)$\in\Ord(1)$}.\\

Hierbei wird zun"achst die gr"o"ste Zweierpotenz aus den Eingabe-\Digit s herausgezogen, um dann sp"ater wieder zum
Resultat hinzumultipliziert zu werden:

@D find power of 2 in the binary gcd algorithm for double \Digit s
@{if (b == 0) {
  b = a; a = 0;
  for (i = BETA; (b&1) == 0; ++i) b >>= 1;
} else if ((b&1) == 0) {
  do { ++i; b >>= 1; } while ((b&1) == 0);
  b |= a << (BETA-i); a >>= i;
}
if (d == 0) {
  d = c; c = 0;
  for (j = BETA; (d&1) == 0; ++j) d >>= 1;
} else if ((d&1) == 0) {
  do { ++j; d >>= 1; } while ((d&1) == 0);
  d |= c << (BETA-j); c >>= j;
}
if (j < i) i = j;@}



\subsection{Euklidischer Algorithmus f"ur \Natural s}
%----------------------------------------------------

Bei der Berechnung des gr"o"sten gemeinsamen Teilers f"ur \Natural s verwenden wir zuerst den Euklidischen Algorithmus f"ur
Langzahlen und wechseln dann bei einer L"ange kleiner oder gleich zwei "uber zum bin"aren Algorithmus f"ur \Digit s.
@D greatest common divisor of two \Natural s
@{Natural gcd(Natural a, Natural b)
// Algorithm:  c := gcd(a, b)
// Input:      a,b in Natural.
// Output:     c in Natural such that c = max{x in Natural : x|a, x|b} ||
{
  Natural t;

  if (a < b) swap(a, b);
  while (b != 0) {
    const size_t sA = a.size;
    const Digit* pA = a.p;
    const Digit* pB = b.p;
    if (sA == 2) {
      Digit x,y;
      const size_t sB = b.size;
      if (sB == 2) a.gcd(pA[0], pA[1], pB[0], pB[1], x, y);
      else a.gcd(pA[0], pA[1], 0, *pB, x, y);
      if (x == 0) return y;
      Digit* pT = t.setsize(2);
      pT[0] = x; pT[1] = y;
      return t;
    } else if (sA == 1) return gcd(*pA, *pB);
    div(a, b, t, a);
    swap(a, b);
  }
  return a;
}
@| gcd @}
\textbf{Laufzeit.} Seien $x,y\in\Natural$, dann \texttt{gcd($x$,$y$)$\in\Ord(n^2)$}.\\

\textbf{Bemerkung.} Es existieren heute schon weitaus schnellere Algorithmen wie zum Beispiel der von K. Weber (\cite{Web95}).\\
Diese werden wir jedoch erst in einer sp"ateren Arbeit behandeln und implementieren.


\subsection{Der erweiterte Euklidische Algorithmus}
%--------------------------------------------------

In diesem Abschnitt nutzen wir die folgende Eigenschaft der ganzen Zahlen aus:\\
Es sei $n\in\N$ und $a_i\in\N_+ (0\leq i\leq n)$, so existieren $b_i\in\Z$ mit
$\gcd(a_i)_{i=0}^n = \sum_{j=0}^na_jb_j$ f"ur $0\leq i\leq n$.\\

Da uns aber nur der gr"o"ste gemeinsame Teiler von zwei Argumenten interessiert (siehe 
Kapitel~\ref{ggT-Einschraenkung auf 2 Argumente}), implementieren wir hier nur den Fall $n = 2$:\\

F"ur alle $a,b\in\Z$, existieren $x,y,z\in\Z$ mit $\gcd(a,b) = z = ax + by$.
@D extented greatest common divisor of two \Integer s
@{void gcd(Integer a, Integer b, Integer& x, Integer& y, Integer& z)
// Algorithm:  gcd(a, b, x, y, z)
// Input:      a,b in Integer.
// Output:     x,y,z in Integer such that z = a*x + b*y = gcd(a, b) ||
{
  if (&x == &y || &x == &z || &y == &z) z.errmsg(5, "(gcd)");
  x = 0; y = 1;
  if (b == 0) z = a;
  else {
    const int as = sign(a);
    const int bs = sign(b);
    if (as == -1) a = -a;
    if (bs == -1) b = -b;
    Integer u(1);
    Integer v(Digit(0));
    Integer t;

    while (true) {
      div(a, b, z, a);
      if (a == 0) { z = b; break; }
      t = x*z; u -= t;
      t = y*z; v -= t;

      div(b, a, z, b);
      if (b == 0) { z = a; x = u; y = v; break; }
      t = u*z; x -= t;
      t = v*z; y -= t;
    }
    if (as == -1) x = -x;
    if (bs == -1) y = -y;
  }
}
@| gcd @}
\textbf{Laufzeit.} Seien $u_0,u_1,v,w_0,w_1\in\Integer$, dann
 \texttt{gcd($u_0$,$u_1$,$v$,$w_0$,$w_1$)$\in\Ord(n^2)$}.


\subsection{Das kleinste gemeinsame Vielfache}
%---------------------------------------------

Um das kleinste gemeinsame Vielfache zu ermitteln, m"ussen wir nur die folgende Beziehung zwischen dem gr"o"sten
gemeinsamen Teiler und dem kleinsten gemeinsamen Vielfachen verwenden:
\[ab = \gcd(a, b)\cdot\lcm(a, b) \qquad\mbox{f"ur } a, b\in\N.\]

@D least common multiple of two \Natural s
@{Natural lcm(const Natural& a, const Natural& b)
// Algorithm:  c := lcm(a, b)
// Input:      a,b in Natural.
// Output:     c in Natural such that c = min{x in Natural : a|x, b|x} ||
{
  if (a == 0) return b;
  if (b == 0) return a;

  Natural q,r;
  div(a, gcd(a, b), q, r);
  return q*b;
}
@| lcm @}
\textbf{Laufzeit.} Seien $x,y\in\Natural$, dann \texttt{lcm($a$,$b$) $\sim$ gcd($a$,$b$)}.



\section{Kongruenzen}
%====================

\index{Modulare Arithmetik}\label{Modulare Arithmetik}
\textbf{Definition.} F"ur $a,b,m\in\Z : a\equiv b\pmod{m} \Leftrightarrow a-b\in m\Z \Leftrightarrow$
es existiert ein $k\in\Z a = b + k\cdot m$.


\subsection{Zahlendarstellung}
%-----------------------------

Es werden vorerst nur einige Eigenschaften der modularen Arithmetik zum besseren Verst"andnis der Primfaktorisierung
eingef"uhrt. In einer sp"ateren Arbeit werden wir diese Arithmetik noch ausf"uhrlicher betrachten.

\small\begin{verbatim}
template<class T>
class Modulo : public NumberBase {
private:
  T m,x;

public:
  Modulo(const T&, const T&);
  Modulo(const Modulo&);
  ~Modulo();
  //...
};
\end{verbatim}\normalsize


\subsection{Elementare modulare Operationen}
%-------------------------------------------

\subsubsection{Addition}

\small\begin{verbatim}
template<class T>
inline Modulo<T>& Modulo<T>::operator+=(const T& a)
// Algorithm:  b := b += a
// Input:      a in Modulo<T>.
// Output:     b in Modulo<T> such that b := b+a ||
{
  x += a % m;
  if (x > m) x -= m;
  return *this;
}
\end{verbatim}\normalsize

\subsubsection{Subtraktion}

\small\begin{verbatim}
template<class T>
inline Modulo<T>& Modulo<T>::operator-=(T a)
// Algorithm:  b := b -= a
// Input:      a in Modulo<T>.
// Output:     b in Modulo<T> such that b := b-a ||
{
  a %= m;
  if (x >= a) x -= a;
  else x += m-a;
  return *this;
}
\end{verbatim}\normalsize


\subsubsection{Multiplikation}

\small\begin{verbatim}
template<class T>
inline Modulo<T>& Modulo<T>::operator*=(const T& a)
// Algorithm:  b := b *= a
// Input:      a in Modulo<T>.
// Output:     b in Modulo<T> such that b := b*a ||
{
  x *= a;
  x %= m;
  return *this;
}
\end{verbatim}\normalsize


\subsection{Berechnung des modularen Inversen}
%---------------------------------------------

\label{modulares Inverses}
Um zu gegebenem $a\in\Z$ sein Inverses $x$ mit $ax\equiv1\pmod{m}$ zu finden, wird der Euklidische
Algorithmus\index{Algorithmus von!Euklid}\index{modulare Inverse} durchgef"uhrt. Dieser liefert dann
\[ax + my = \gcd(a, m)\qquad\Rightarrow\qquad ax\equiv \gcd(a, m)\pmod{m}.\]
$x$ ist das Inverse von $a$ in der Restklasse mod $m$, wenn noch zus"atzlich $\gcd(a, m) = 1$ gilt.

\small\begin{verbatim}
template<class T>
T Modulo<T>::inv() const
// Algorithm:  b := a.inv()
// Input:      a in Modulo<T> where gcd(a.x, a.m) = 1.
// Output:     b in Modulo<T> such that b = a^{-1} ||
{
  T a = x;
  T b = m;
  T q = 1;
  T p = 0;

  do {
    T t = a/b;
    a -= t*b;
    q += t*p;
    if (a == 0) return m - p;
    t = b/a;
    b -= t*a;
    p += t*q;
  } while (b > 0);
  return q;
}
\end{verbatim}\normalsize


\subsubsection{Dividieren}

Die Division erzeugen wir durch das Multiplizieren mit dem inversen Element, falls das multiplikative Inverse existiert.
\small\begin{verbatim}
template<class T>
inline Modulo<T>& Modulo<T>::operator/=(const T& a)
// Algorithm:  b := b /= a
// Input:      a in Modulo<T> where gcd(a.x, a.m) = 1.
// Output:     b in Modulo<T> such that b := b*a^{-1} ||
{
  Modulo<T> b(m, a);
  return *this *= b.inv();
}
\end{verbatim}\normalsize



\subsection{Simultane Kongruenzen}
%---------------------------------

\index{Chinesischer Restsatz}\label{Chinesischer Restsatz}
\textbf{Chinesischer Restsatz.} Sind $\{m_i\}_{i = 1}^n$ paarweise teilerfremde ganze Zahlen und
$m = \prod_{i = 1}^n m_i$ und $b, \{a_i\}_{i = 1}^n$ beliebige ganze Zahlen, dann existiert genau ein $x$ mit
\[ b\leq x < b+m \mbox{ und } x\equiv a_i\pmod{m_i}\qquad\mbox{f"ur alle } 1\leq i\leq n.\]

Wenn wir den Chinesischen Restsatz nur f"ur den speziellen Fall mit zwei Kongruenzen l"osen wollen, so kann man die
L"osung einfach explizit angeben:\\
\textbf{Gegeben.} $x \equiv a_1\pmod{m_1}$ und $x \equiv a_2\pmod{m_2}, \gcd(m_1, m_2) = 1$.\\
\textbf{Gesucht.} $x$.\\
Sei $c\cdot m_1 \equiv 1\pmod{m_2}$, so
$x \equiv (a_2 - a_1)\cdot c\cdot m_1 + a_1\pmod{m_1m_2}$.\\

Die direkte Umsetzung sieht dann so aus:
\index{\texttt{chines}}\label{chines}
\small\begin{verbatim}
template<class T>
Set<T> chines(const T m1, const T m2, Set<T>& s1, Set<T>& s2)
{
  Set<T> s;
  Modulo<T> a(m1, m2);
  Modulo<T> b(m2, m1);
  const T c = b.inv();

  if (s1.first(a.x))
    do
      if (s2.first(b.x))
        do {
          b -= a.x;
          b *= c;
          s += a.m*b.x + a.x;
        } while (s2.next(b.x));
    while (s1.next(a.x));
  return s;
}
\end{verbatim}\normalsize

Hierbei gilt:
\[l_1\subseteq\Z/m_1\Z, l_1\subseteq\Z/m_2\Z, \gcd(m_1, m_2) = 1\]
und schlie"slich:
\[l\subseteq\Z/(m_1\cdot m_2)\Z, {\mbox{f"ur alle $a\in l$ gilt }}
a\equiv a_1\pmod{m_1}, a\equiv a_2\pmod{m_2}\Rightarrow a_1\in l_1, a_2\in l_2.\]

\textbf{Beispiel.} Das folgende Beispiel dient zur L"osung eines linearen Kongruenzsystems:
\[x\equiv20\pmod{70}, x\equiv30\pmod{101} \Rightarrow x = 2050.\]
\small\begin{verbatim}
Set<Digit> s1;
Set<Digit> s2;

s1 += 20;
s2 += 30;

Set<Digit> s = chines(70, 101, s1, s2);

cout << "s=" << s << endl;
\end{verbatim}\normalsize



\section{Quadratische Reste / Jacobi-Symbol}
%===========================================

\index{quadratischer Rest}\index{quadratischer Nichtrest}
\textbf{Definition.} Seien $x,m\in\Z$. Wir nennen $x$ einen quadratischen Rest mod $m$, falls ein $a\in\Z$ existiert mit
$a^2 \equiv x\pmod{m}$, andernfalls einen quadratischen Nichtrest.\\
Sei $p\geq 3$ Primzahl, $x\in\Z$, so ist das Legendre-Symbol\index{Legendre-Symbol} folgenderma"sen definiert:
\[\jacobi{x}{p} := \left\{ \begin{array}{r@@{\;,\quad}l}
1 & \mbox{ex. } a\in\Z\ \mbox{mit } a^2 \equiv x\pmod{p}.\\
0 & p\mid x.\\
-1 & \mbox{f"ur alle } a\in\Z\  a^2 \not\equiv x\pmod{p}.
\end{array}\right..\]


\subsubsection{Einfaches Durchtesten}

Ein erster Ansatz, der nur die Tatsache ausnutzt, da"s in der Einheitengruppe $(\Z/p\Z)^*$ genausoviele quadratische
Reste wie  Nichtreste existieren, und die Symmetrie
\[(p - x)^2 = p^2 - 2px + x^2\equiv x^2\pmod{p}\]
ausnutzt, sieht wie folgt aus:
\index{\texttt{root}}\label{root}
\small\begin{verbatim}
void root(const Digit p)
// Algorithm:  root(p)
// Input:      p prim ||
{
  char* r = new char[p];
  if (!r) { cerr << "Out of Memory!\n"; return; }

  Digit i;
  for (i = 0; i < p; ++i) r[i] = 0;
  i = 1;                        // Null auslassen
  for (Digit j = 3; j <= p; j += 2) {
    r[i] = 1; i += j;
    if (i >= p) i -= p;
  }

  // Falls r[i] = 1, so ist i Quadratischer Rest mod p
  delete[] r;
}
\end{verbatim}\normalsize
Hierbei wird f"ur gro"se Zahlen $p$ viel Speicher (genau $p$ Bytes) alloziiert, und es w"achst die Laufzeit linear zu $p$.


\subsubsection{Das Legendre-Symbol}

\index{Quadratisches Reziprozit\"atsgesetz}
\textbf{Quadratisches Reziprozit"atsgesetz.} Seien $p, q\geq 3$ Primzahlen, so gilt:
\[\jacobi{p}{q} = (-1)^{\frac{p-1}{2}\cdot\frac{q-1}{2}} \jacobi{q}{p}
= \left\{ \begin{array}{r@@{\;,\quad}l}
\jacobi{q}{p} & p \equiv 1\pmod{4} \mbox{ oder } q \equiv 1\pmod{4}\smallskip\\
-\jacobi{q}{p} & p \equiv 3\pmod{4} \mbox{ und } q \equiv 3\pmod{4}
\end{array}\right..\]

Desweiteren gelten die \textbf{Erg"anzungss"atze}:\index{Erg\"anzungss\"atze}\label{Ergaenzungssaetze}
\[\jacobi{2}{p} = (-1)^{\frac{p^2-1}{8}}
= \left\{ \begin{array}{r@@{\;,\quad}l}
1 & p \equiv \pm1\pmod{8}\\
-1 & p \equiv \pm3\pmod{8}
\end{array}\right.,\]

die bis $\pm101$ in \cite{Ries85} aufgef"uhrt werden.\bigskip\\
Anhand dieser Formeln l"a"st sich das Legendre-Symbol effizient implementieren:
\index{\texttt{legendre}}
\small\begin{verbatim}
int Modulo<T>::legendre() const
// Algorithm:  b := a.legendre()
// Input:      a in Modulo<T> where a.m prim and a.m <= GAMMA_LOW.
// Output:     b in int such that b := (a.x//a.m) ||
{
  if (x == 0) return 0;
  int ret = 0;
  T b = m;
  for (T a = x; a > 1; a %= b) {
    for (int c = 0; (a&1) == 0; c++) a >>= 1;
    if (c&1 && ((b*b-1)&15) == 8) ++ret;
    if (a == 1) break;
    if (((a-1)*(b-1)&7) == 4) ++ret;
    swap(a, b);
  }
  return 1 - (ret&1)*2;
}
\end{verbatim}\normalsize


\subsubsection{Das Jacobi-Symbol}

Diese Implementierung ist aber noch nicht brauchbar, weil hierbei Bedingungen an den Benutzer gestellt werden, was gegen
unsere Anfangsforderung (Kapitel \ref{Benutzerforderungen}) verst"o"st. Das Programm wird schneller und allgemeing"ultig,
wenn wir stattdessen das Jacobi-Symbol verwenden:\\
\textbf{Definition.} Seien $n,m\in\N$ mit $\gcd(n, m) = 1$, $m$ ungerade und
$m = \prod_{i = 0}^k p_i^{a_i}$ f"ur $k\in\N, a_i\in\N_+$ die Primfaktorzerlegung, so setze:
\[\begin{array}{r@@{\quad:=\quad}l}
\underbrace{\jacobi{n}{m}}_{\mbox{Jacobi-Symbol}}
& \prod_{i = 0}^k\underbrace{\jacobi{n}{p_i}}_{\mbox{Legendre-Symbol}}
\end{array}\]
\index{\texttt{jacobi}}
\small\begin{verbatim}
template<class T>
int Modulo<T>::jacobi() const
// Algorithm:  b := a.jacobi()
// Input:      a in Modulo<T>.
// Output:     b in int such that b := (a.x//a.m) ||
{
  if (x == 0) return 0;
  int ret = 0;
  T b = m;
  for (T a = x; a > 1; a %= b) {
    for (int c = 0; (a&1) == 0; c++) a >>= 1;
    if (c&1 && (b&7) != 1 && (b&7) != 7) ++ret;
    if (a == 1) break;
    if ((a&3) != 1 && (b&3) != 1) ++ret;
    swap(a, b);
  }
  return 1 - (ret&1)*2;
}
\end{verbatim}
\normalsize

Hierbei setzen wir noch $\jacobi{n}{m} := 0$ f"ur den Fall $\gcd(n, m) > 1$.\\

\textbf{Bemerkung.} Auch das Jacobi-Symbol kann wie das Legendre-Symbol bei Primzahlen eine Aussage "uber den quadratischen
Rest machen. Bei einer Nicht-Primzahl ist es lediglich eine Rechenerweiterung des Legendre-Symbols.\\

\textbf{Beispiel.} $\jacobi{5}{9} = \jacobi{2}{3}^2 = 1$, aber 5 ist kein Quadrat mod 9.\\
Wenn jedoch $\jacobi{n}{m} = -1$ gilt, so ist $n$ quadratischer Nichtrest mod $m$.



\section{Primzahlen und die Primfaktorzerlegung}
%===============================================

\textbf{Definition.} $\Primes$ sei die Menge aller Primzahlen.\label{Primes}\medskip\\
\textbf{Definition.} Eine Zahl $n\in\Z$ hei"st \textbf{unzerlegbar},\index{unzerlegbar}\label{unzerlegbar} falls f"ur
alle $x\in\Z-\Z^*$ aus $x\mid n$ stets $x\in\{\pm1\}$ oder $x\in\{\pm n\}$ folgt.\medskip\\
\textbf{Definition.} Eine Zahl $n\in\Z$ hei"st \textbf{Primzahl},\index{Primzahl}\label{Primzahl} falls f"ur alle
$x,y\in\Z$ aus $n\mid x\cdot y$ stets $n\mid x$ oder $n\mid y$ folgt.\medskip\\
\textbf{Bemerkung.} F"ur den Ring $\Z$ ist die Eigenschaft "'unzerlegbar"{} und die Bezeichnung "'Primzahl"{}
gleichbedeutend.\\

Alle Methoden, die wir im folgenden betrachten werden, beruhen auf dem\index{Fundamentalsatz der Zahlentheorie}\medskip\\
\textbf{Fundamentalsatz der Zahlentheorie.} Jede nat"urliche Zahl $n\ge 2$ l"a"st sich eindeutig als das Produkt von
Primzahlen darstellen:
\[n = \prod_{i = 0}^t p_i^{a_i}\qquad\mbox{f"ur $t\in\N, a_i\in\N_+, 0\leq i\leq t$.}\]


\subsection{Siebmethode von Eratosthenes}
%----------------------------------------
\index{Algorithmus von!Eratosthenes}\label{Siebalgorithmen}

\subsubsection{Primzahlenermittlung}

Doch bevor wir genauer auf die Primfaktorzerlegung einer nat"urlichen Zahl eingehen, ben"otigen wir zun"achst die
Ermittlung einer Menge endlich vieler Primzahlen. Hierzu beschrieb bereits um 230 vor Christus der griechische Gelehrte
Eratosthenes von Kyrene einen Algorithmus zur Auffindung aller Primzahlen zwischen 2 und $n\in\N$, mit welchem alle
zusammengesetzten Zahlen aus der Menge $\{2,\ldots, n\}$ entfernt werden:
\begin{enumerate}
\item Schreibe alle Zahlen von 2 bis $n$ auf.
\item Beginne mit der Zahl 2 und streiche alle Vielfachen heraus, die gr"o"ser oder gleich 4 sind.
\item Fahre mit der n"achstgr"o"seren nicht durchgestrichenen Zahl $p$ fort und streiche wieder alle Vielfachen, die
      gr"o"ser oder gleich $p^2$ sind, heraus.
\item F"uhre Schritt 3 solange durch, bis $p^2\leq n$ ist.
\end{enumerate}

Eine effiziente Implementation des Siebverfahrens kann demnach so aussehen:\index{Siebmethode}
\small\begin{verbatim}
void Sieb1(const Digit N)
// Algorithm:  Sieb1(N)
// Input:      N in Digit where N >= 2 ||
{
  const Digit n = N/2;
  char* primes = new char[n];
  if (!primes) { cerr << "Out of Memory!\n"; return; }

  Digit i;
  for (i = 0; i < n; ++i) primes[i] = 1;
  for (i = 0; i < n; ++i)
    if (primes[i]) {
      const Digit p = 2*i+3;
      for (Digit k = p+i; k < n; k += p) primes[k] = 0;
    }

  // Primzahlausgabe:
  cout << 2;
  for (i = 0; i < n; ++i)
    if (primes[i]) cout << ',' << 2*i+3;
  cout << endl;

  delete[] primes;
}
\end{verbatim}\normalsize

Hierbei sind nur die ungeraden Zahlen ab drei im Speicher abgelegt, weil die Eigenschaft ausgenutzt wird, da"s alle
geraden Zahlen bis auf die Primzahl zwei zerlegbar sind.\\
Noch effizienter arbeiten wir, wenn wir keine Elemente wiederholt aussieben, wie es aber im vorigen Algorithmus
\texttt{Sieb1} geschieht. Dort werden n"amlich alle Zahlen, die aus zwei verschiedenen Primzahlen zusammengesetzt sind,
doppelt ausgesiebt.\\
Eine Implementation, die dies vermeidet und die auszusiebende Menge noch weiter verkleinert, stellte Xuedong Luo
in \cite{Luo89} vor. Bei dieser sind alle ungeraden und nicht durch drei teilbaren Zahlen ab der Primzahl f"unf im
Speicher abgelegt:

\small\begin{verbatim}
void Sieb2(const Digit N)
// Algorithm:  Sieb2(N)
// Input:      N in Digit where N >= 3 ||
{
  const Digit n = N/3;
  char* primes = new char[n];
  if (!primes) { cerr << "Out of Memory!\n"; return; }

  Digit i;
  for (i = 0; i < n; ++i) primes[i] = 1;
  Digit c = GAMMA;
  Digit k = 1;
  Digit q = 2*(sqrt(N)/3);

  for (i = 3; i <= q; i += 2) {
    k += 3;
    Digit j = c += 2*k;
    k += 3;
    while (j < n) {
      primes[j] = 0; j += i;
      if (j >= n) break;
      primes[j] = 0; j += k;
    }

    i += 2;
    if (i > q) break;
    j = c += ++k; ++k;

    while (j < n) {
      primes[j] = 0; j += k;
      if (j >= n) break;
      primes[j] = 0; j += i;
    }
  }

  // Primzahlausgabe:
  cout << 2 << ',' << 3;
  k = 5;
  for (i = 0; i < n; ++i) {
    if (primes[i]) cout << ',' << k;
    k += 2;
    if (++i > n) break;
    if (primes[i]) cout << ',' << k;
    k += 4;
  }
  cout << endl;

  delete[] primes;
}
\end{verbatim}\normalsize

Um den Speicher nun optimal ausnutzen zu k"onnen, werden wir jede Zahl nicht mehr durch ein Byte, sondern nur noch durch
ein Bit repr"asentieren. Hierf"ur m"ussen wir kaum eine Ver"anderung durchf"uhren und reduzieren unseren Speicherverbrauch
auf $\frac{1}{\mathtt{CHAR\_BIT}}$. Allerdings ist diese "Anderung inkonsequent gegen"uger unserer Forderung nach
schnellstm"oglicher Laufzeit, da dieser bitweise Zugriff mehr Zeit ben"otigt als der byteweise. Doch ist es in diesem Fall
sinnvoll, den Speicherplatz auf Kosten der Geschwindigkeit zu minimieren, weil es sich bei der Primzahlzerlegung um eine
endliche Aufgabe (\texttt{SieveSize}) handelt, die aufgrund unserer anderen optimalen Vorgaben in gen"ugend schneller Zeit
bearbeitet werden kann.

@D generating prime numbers
@{const Digit n = 8*SieveSize;
unsigned char* p = primes = NOTHROW_NEW unsigned char[SieveSize];
if (!primes) errmsg(2, "(constructor)");
firstPrime();
const unsigned char* pE = p+SieveSize;
do *p++ = (unsigned char)~0; while (p != pE);
p -= SieveSize;

Digit c = GAMMA;
Digit k = 1;
Digit q = 2*(sqrt(Digit(24*SieveSize))/3);

for (Digit i = 3; i <= q; i += 2) {
  k += 3;
  Digit j = c += 2*k;
  k += 3;

  while (j < n) {
    p[j/8] &= (unsigned char)~(1 << (j%8)); j += i;
    if (j >= n) break;
    p[j/8] &= (unsigned char)~(1 << (j%8)); j += k;
  }

  i += 2;
  if (i > q) break;
  j = c += ++k; ++k;

  while (j < n) {
    p[j/8] &= (unsigned char)~(1 << (j%8)); j += k;
    if (j >= n) break;
    p[j/8] &= (unsigned char)~(1 << (j%8)); j += i;
  }
}
@}

Die Ausgabe aller ermittelten Primzahlen erfolgt dann "uber die beiden Funktionen
\begin{enumerate}
\item
@D getting first prime
@{inline Digit NumberBase::firstPrime() const
// Algorithm:  c := n.firstPrime()
// Input:      n in NumberBase.
// Output:     c in Digit such that c = 2 ||
{
  primPos = primes;
  idx = 1;
  return primNumber = 2;
}
@| NumberBase::firstPrime firstPrime @}
\textbf{Laufzeit.} Sei $n\in\NumberBase$, dann \texttt{$n$.firstPrime()$\in\Ord(1)$}.

\item
@D getting next prime
@{bool NumberBase::nextPrime(Digit& a) const
// Algorithm:  c := n.nextPrime(a)
// Input:      n in NumberBase.
// Output:     a in Digit, c in bool
//             such that if a prime then c = true else c = false ||
{
  if (primNumber == 2) { a = 3; primNumber = 5; }
  else {
    Digit p;
    do {
      if (primPos - primes == SieveSize) return false;
      p = *primPos & idx;
      if (p) a = primNumber;
      if (idx == 1 << (CHAR_BIT-1)) { idx = 1; ++primPos; primNumber += 4; }
      else {
        if (idx == 2 || idx == 8 || idx == 32) primNumber += 2;
        primNumber += 2;
        idx *= 2;
      }
    } while (p == 0);
  }
  return true;
}
@| NumberBase::nextPrime nextPrime @}
\textbf{Laufzeit.} Sei $n\in\NumberBase$ und $a\in\Digit$, dann \texttt{$n$.nextPrime($a$)$\in\Ord(1)$}.
\end{enumerate}

Die letzte Primzahl in unserem Speicher (beziehungsweise unserer endlichen Menge) l"a"st sich auch direkt berechnen:
@D getting last prime
@{Digit NumberBase::lastPrime() const
// Algorithm:  c := n.lastPrime()
// Input:      n in NumberBase.
// Output:     c in Digit such that c prime ||
{
  static Digit lprim = 0;
  if (lprim) return lprim;
  Digit i = SieveSize;
  while (primes[--i] == 0);
  const Digit p = log2(Digit(primes[i]));
  i *= 24; i += 3*p + 5;
  return lprim = i - (p&1);
}
@| NumberBase::lastPrime lastPrime @}
\textbf{Laufzeit.} Sei $n\in\NumberBase$, dann \texttt{$n$.lastPrime() $\sim$ log2($\gamma$)}.\\


\subsubsection{Anzahl der Primzahlen}

Es m"ussen alle Primzahlen im Speicher gez"ahlt werden, um eine genaue Aussage "uber die Anzahl der Primzahlen aus unserer
endlichen Menge zu machen. Hierbei treffen wir die folgende Konvention:\\
Sei $n\in\NumberBase$, dann
\[n\texttt{.NumberOfPrimes} : \Digit\to\Digit : a\mapsto\left\{\begin{array}{r@@{\;,\quad}l}
\pi(a) & a\leq n\texttt{.lastPrime()}\\
0 & a>n\texttt{.lastPrime()}
\end{array}\right.\]
mit der \textbf{Primzahlfunktion}\index{Primzahlfunktion}\label{Primzahlfunktion}
\[\pi : \N\to\N : n\mapsto|\{p\in\Primes\mid p\leq n\}| = \sum_{\Primes\ni p\leq n}1.\]

@D getting the number of primes
@{size_t NumberBase::NumberOfPrimes(Digit n) const
// Algorithm:  c := n.NumberOfPrimes(a)
// Input:      n in NumberBase, a in Digit.
// Output:     c in Digit
//             such that if a <= n.lastPrime() then c = pi(a) else c = 0 ||
{
  if (n < 5) return size_t(n/2 + (n == 3));
  n -= 5;
  if (n >= 24*SieveSize) return 0;
  const unsigned char* q = primes + n/24;
  n %= 24;
  size_t i = 2;
  for (unsigned char j = char(1 << (n/3 + ((n%6) == 2))); j >= 1; j /= 2)
    if (*q & j) ++i;

  for (unsigned char* p = primes; p < q; ++p) {
    for (unsigned char j = 1; j; j *= 2)
      if (*p & j) ++i;
  }
  return i;
}
@| NumberBase::NumberOfPrimes @}
\textbf{Laufzeit.} Sei $n\in\NumberBase$ und $a\in\Digit$, dann\smallskip\\
\mbox{}\hfill\texttt{$n$.NumberOfPrimes($a$)}$\in\left\{\begin{array}{r@@{\;,\quad}l}
\Ord(1) & a<5 \mbox{ oder } a\geq 24\cdot\mathtt{SieveSize}\\
\Ord(a) & \mathrm{sonst}
\end{array}\right.$.\hfill\mbox{}\\


Wenn wir nun $N=10^7$ betrachten, dann erhalten wir die folgenden Laufzeiten auf einen Pentium 100 MHz mit dem
Watcom Compiler 10.6:\medskip\\

\hspace*{\fill}
\begin{tabular}{ccc}
Algorithmus & Laufzeit & Speicherverbrauch\\
\hline
Sieb1 & 2.47 s & 4.77 MB\\
Sieb2 & 0.95 s & 3.18 MB\\
Sieb3 & 2.63 s & 407 kB\\
\end{tabular}\hfill\mbox{}



\subsection{Primzahltest von Brillhart und Selfridge}
%----------------------------------------------------
\index{Primzahltest}

Wir werden in diesem Abschnitt Methoden vorstellen, die entscheiden k"onnen, ob eine gegebene nat"urliche Zahl eine
Primzahl ist.

\subsubsection{Kleiner Satz von Fermat}
\index{Satz!von Fermat}
Zur "Uberpr"ufung der Primzahleigenschaft verwenden wir die folgende Beziehung:
\[\mbox{Ist } p\in\Primes, a\in\N\mbox{ und } \gcd(a, p) = 1, \mbox{so gilt } a^{p-1}\equiv1\pmod{p}.\]

Dadurch haben wir einen einfachen Primzahltest.\index{Primzahltest}
Ist n"amlich zum Beispiel $2^{n-1}\not\equiv1\pmod{n}$, so wissen wir wegen der Kontraposition des kleinen Satzes von
Fermat, ohne die Zerlegung von $n$ zu kennen, da"s $n$ zusammengesetzt ist.\\
Die Implementation dieses Satzes beruht auf dem Potenzieralgorithmus aus Kapitel \ref{Potenzieren}.
Hier werden die Operationen nun aber modular ausgef"uhrt:

% Desweiteren w"are auch f"ur $a\not=0$ ein effizienter Algorithmus zur Ermittlung des modularen Inversen mittels
% $a^{p-2}\equiv a^{-1}\pmod{p}$ m"oglich.

@D internal fermat test
@{static bool fermat(const Natural& m, Digit a, Natural b)
// Algorithm:  c := fermat(m, a, b)
// Input:      m in Natural where m > 0, b in Natural, a in Digit.
// Output:     c in bool
//             such that if 1 = a^b (mod m) then c = true else c = false ||
{
  if (m.length() == 1) a %= m.highest();
  if (b == 1) return (a == 1);
  else if (b == 0) return true;
  if (a == 0) return false;

  Natural t, c = a;
  while (b.even()) {
    b >>= 1; t = c*c; div(t, m, t, c);
  }
  Natural d = c;
  b >>= 1;
  while (b != 0) {
    t = c*c; div(t, m, t, c);
    if (b.odd()) { t = d*c; div(t, m, t, d); }
    b >>= 1;
  }
  return (d == 1);
}
@| fermat @}
\textbf{Laufzeit.} Seien $x,y,z\in\Natural$ und $a\in\Digit$, dann \texttt{fermat($x$,$a$,$y$) $\sim$ pow($a$,$y$,$z$)}.\\

\textbf{Bemerkung.} Dieser Algorithmus l"a"st sich allgemeiner programmieren, wozu jedoch eine modulare Arithmetik
ben"otigt wird, die wir hier noch nicht ber"ucksichtigen k"onnen.


\subsubsection{Primzahltest}

Anhand des kleinen Satzes von Fermat kann nur "uberpr"uft werden, ob eine gegebene nat"urliche Zahl zusammengesetzt ist,
nicht jedoch, ob es sich dabei um eine Primzahl handelt. Denn aus $a^{p-1}\equiv1\pmod{p}$ folgt nicht zwingend
$p\in\Primes$, wie dieses Beispiel zeigt:\\
Es gilt zwar $4^{15-1}\equiv1\pmod{15}$, aber $15 = 3\cdot5$ ist zerlegbar und damit keine Primzahl.\\

Ein allgemeing"ultiger Primzahltest f"ur eine gegebene Zahl $n\in\N$ stammt von J. Brillhart und J. L. Selfridge
(\cite{BrSe67}), bei dem die Faktorisierung von $n-1$ vorausgesetzt wird.\\
Sei $n\geq3$ ungerade und
\[n-1 = \prod_{i = 0}^t p_i^{a_i} \qquad\mbox{f"ur } t\in\N, a_i\in\N_+,0\leq i\leq t\]
die  Primfaktorzerlegung der Zahl $n-1$, so gilt die folgende Aussage:\\
Falls f"ur jeden Primteiler $p_i$ von $n-1$ ein $x\in\N$ existiert mit den Eigenschaften
\[\gcd(x, n) = 1,\quad x^{n-1}\equiv1\pmod{n},\quad x^{(n-1)/p_i}\not\equiv1\pmod{n}
\qquad\mbox{f"ur } 1\leq i\leq t,\]
dann ist $n$ eine Primzahl.

@D primality test
@{bool Natural::isprime() const
// Algorithm:  c := a.isprime()
// Input:      a in Natural.
// Output:     c in bool such that
//             if a is a prime element then c = true else c = false ||
//
// Note:       Error possible: "the primality could not be determined".
{
  static const bool primes[30] = { 0, 0, 1, 1, 0, 1, 0, 1, 0, 0,
                                   0, 1, 0, 1, 0, 0, 0, 1, 0, 1,
                                   0, 0, 0, 1, 0, 0, 0, 0, 0, 1 };
  static const bool t[30] = { 1, 0, 1, 1, 1, 1, 1, 0, 1, 1, 1, 0, 1, 0, 1,
                              1, 1, 0, 1, 0, 1, 1, 1, 0, 1, 1, 1, 1, 1, 0 };
  const size_t sT = size;
  const Digit x = highest();
  if (sT == 1 && x < 30) return primes[x];
  if (t[*this%30]) return false;              // not prime

  if (sT == 1) {
    const Digit y = lastPrime();
    if (x == y) return true;
    else if (x < y) {
      Digit q = firstPrime();
      while (nextPrime(q) && q < x);
      if (q == x) return true;
      else if (q > x) return false;
    }
  }

  Natural b = *this;
  --b;
  if (!fermat(*this, 3, b)) return false;     // Fermat:  3^b != 1 (mod *this).

  list<Natural> p;
  factoring(b, p);
  Digit k = firstPrime();
  while (true) {
    if (!fermat(*this, k, b)) return false;
    list<Natural>::iterator i = p.begin();
    while (true) {
      const Natural c = b / *i;
      const Digit k2 = k;
      while (fermat(*this, k, c))
        if (!nextPrime(k)) errmsg(0, "the primality could not be determined");
      if (k != k2) break;

      list<Natural>::const_iterator j = i;
      do
        if (++i == p.end()) return true;
      while (*j == *i);
    }
  }
}
@| Natural::isprime isprime @}
\textbf{Laufzeit.} Sei $x\in\Natural$ und $C$ ein Container "uber $\Natural$, dann\smallskip\\
\mbox{}\hfill\texttt{$x$.isprime() $\sim$ factoring($x-1$,$C$)}.\hfill\mbox{}



\subsection{Primfaktorzerlegung}
%-------------------------------

Der einfachste Algorithmus zur Primfaktorzerlegung von $n\in\N$ ist das sukzessive Dividieren durch alle Primzahlen.
Hierbei werden alle m"oglichen Faktoren von zwei bis $\sqrt{n}$ durchprobiert, wozu eine Laufzeit von $\Ord(\sqrt{n})$
ben"otigt wird.


\subsubsection{Verbesserungsvorschlag}

Sei $\N\ni p\leq\sqrt{n}$, so betrachten wir beim sukzessiven Dividieren den Divisionsrest $r := n\bmod p$. Dadurch ist $n$
genau dann durch $p$ teilbar, wenn $r=0$ ist. Bevor wir aber zur n"achsten gr"o"seren Zahl $\N\ni q>p$ wechseln, um mit
der Iteration des sukzessiven Dividierens fortzufahren, wollen wir uns mit der folgenden Frage besch"aftigen:

\[\mbox{Sei }n\in\N\mbox{ und }\N\ni p<\sqrt{n}.\mbox{ F"ur welche }x\in\N_+\mbox{ gilt }n \bmod(p)\leq n\bmod(p+x) ?\]

Damit wollen wir ein Intervall $I := [p, p+x]$ bestimmen, in dem der Divisionsrest auf jeden Fall ungleich Null
ist und folglich $n$ nicht durch $p$ teilbar sein kann.\\

Es gilt $0\leq a - b\lfloor\frac{a}{b}\rfloor\leq a - b\lfloor\frac{a}{b+x}\rfloor< b + x$.\\
\textbf{Gesucht.} $a - b\lfloor\frac{a}{b}\rfloor\leq a - (b+x)\lfloor\frac{a}{b+x}\rfloor$\\
Eine m"ogliche Bedingung: $\lfloor\frac{a}{b+x}\rfloor = \lfloor\frac{a-b}{b+x}\rfloor$.\\

\textbf{Beispiel.} Sei $n = 9887$ und $p = 2$, so erhalten wir $1\leq x\leq4941$ und wir m"ussen die n"achste Division
nicht mit $q = 3$ durchf"uhren, sondern erst mit der Primzahl $q = p+4941 = 4943$ fortfahren. Wir sind mit dieser einen
Division jedoch bereits fertig, weil wir nur bis $\lfloor\sqrt{n}\rfloor = 99$ nach Teilern suchen m"ussen und mit 4943
diese Grenze schon weit "uberschritten haben. Daher k"onnen wir folgern, da"s $n$ eine Primzahl ist.\\

\textbf{Beispiel.} Angenommen, wir haben alle Primzahlen bis 23 gegeben und wir wollen $n = 998$ faktorisieren, dann
ben"otigen wir mit dem sukzessiven Dividieren 2 Quadratwurzel- und 9 Divisionsoperationen. Dagegen ben"otigen wir
nur 2 Quadratwurzel- und 2 Divisions\-operationen, wenn wir zus"atzlich den Divisionsrest betrachten, weil bereits aus
$499\bmod(2)\leq499\bmod(2+x)$ f"ur $1\leq x\leq20$ die Primzahleigenschaft von 499 folgt.\\

Offensichtlich gilt die folgende Gleichung:\\
Sei $x,y,z\in\N, y > x, \gcd(z, x) = 1$,\\
so erf"ullt $y \in x + \{1,\ldots,z - l - 1\}$ mit $x \equiv l \pmod{z}$ die Gleichung
$\lfloor\frac{x}{z}\rfloor = \lfloor\frac{y}{z}\rfloor$.\\
Das ist der einfache Fall, uns interessiert aber eine Aufl"osung nach $z$.


\subsection{Faktorisierung durch sukzessives Dividieren.}
\index{Faktorisierungsalgorithmus!sukzessives Dividieren}

Bereits beim Siebalgorithmus (siehe Kapitel \ref{Siebalgorithmen}) ist aufgefallen, da"s es keine handliche Formel zur
Erzeugung beliebig vieler Primzahlen gibt. Deswegen m"ussen wir eine Teilmenge der nat"urlichen Zahlen konstruieren, die
alle Primzahlen enth"alt, aber m"oglichst klein ist, um die Anzahl der durchzuf"uhrenden Divisionen f"ur die
Faktorisierung gering zu halten.\\
Die triviale Menge hierzu w"are nat"urlich $\N$. Eine weitaus bessere Menge aber
 -- mit der Eigenschaft, da"s alle Primzahlen au"ser 2 ungerade sind -- ist
\[\{p\in\N\mid\ p\equiv1\pmod{2}\}\dot{\cup}\{2\},\]
denn hier sind alle Vielfachen von zwei ausgesiebt, womit die Anzahl der durchzuf"uhrenden Divisionen um 50\% reduziert ist.
Werden nun zus"atzlich auch alle Vielfachen von 3 aussortiert, so erhalten wir die Menge
\[\{p\in\N\mid\ p\equiv\pm1\pmod{6}\}\dot{\cup}\{2, 3\},\]
und es bleiben $(1 - \frac{1}{2})(1 - \frac{1}{3})\cdot100 = 33\%$
der urspr"uglichen Divisionen "ubrig.\\
Wenn wir zus"atzlich noch die Vielfachen von 5 herausnehmen, dann erhalten wir die Menge
\label{Pseudo-Primes}
\[\Primes' := \{p\in\N\mid\ p\equiv q\pmod{30}, q\in\{1, 7, 11, 13, 17, 19, 23, 29\}\}\dot{\cup}\{2, 3, 5\}\]
und wir gelangen auf 26,7\% der urspr"unglichen Divisionsanzahl. Diese Vorgehensweise k"onnten wir noch weiter fortsetzen,
doch wird der Prozentsatz der insgesamt eingesparten Divisionen immer geringer, und es mu"s auch bedacht werden, da"s
unsere konstruierte Menge umso unhandlicher und komplexer wird, je mehr Elemente wir aussieben. Schlie"slich versagt
dieses Verfahren f"ur sehr gro"se Zahlen, weil trotz der Einsparungen alle Primzahlvorg"anger bis
$\lfloor\sqrt{n}\rfloor$ durchprobiert werden m"ussen.


\subsubsection{Vorgehensweise}

Sei $a\in\Natural$ die zu faktorisierende Zahl, so testen wir zuerst alle Primzahlen unserer endlichen Menge
\texttt{primes}, die kleiner oder gleich $\lfloor\sqrt{a}\rfloor$ sind:
@D successive division for less factors
@{if (a == 0) return;
while (a.even()) {                      // less factors
  p.push_back(2);
  a >>= 1;
}
if (a == 1) return;
Natural t;
Digit i;
Digit prim = a.firstPrime();            // didn't need 2
while (a.nextPrime(prim)) {
  while (true) {
    div(a, prim, t, i);
    if (i) break;
    p.push_back(prim);
    if (t == 1) return;
    swap(a, t);
  }
  if (t < prim) { p.push_back(a); return; }
}
@}

Falls alle Primzahlen durchlaufen sind und die Faktorisierung noch nicht abgeschlossen ist, setzen wir mit der gr"o"seren
Menge $\Primes'$ das sukzessive Dividieren bis $\lfloor\sqrt{a}\rfloor$ fort:
@D successive division for greater factors
@{Natural s = prim;
const Digit n[8] = { 4, 2, 4, 2, 4, 6, 2, 6 };
i = 0;
switch (s % 30) {
  case 1:  ++i;
  case 29: ++i;
  case 23: ++i;
  case 19: ++i;
  case 17: ++i;
  case 13: ++i;
  case 11: ++i;
}
Natural q,r;
t = root(a, 3);
for (s += n[i]; s.length() == 1 && s <= t; s += n[i&7]) {
  if (a%s.highest() == 0) {
    p.push_back(s);
    a /= s.highest();
    if (a.isprime() == 1) { p.push_back(a); return; }
    t = root(a, 3);
  }
  ++i;
}
while (s <= t) {
  div(a, s, q, r);
  if (r == 0) {
    p.push_back(s);
    if (q.isprime() == 1) { p.push_back(q); return; }
    swap(a, q);
    t = root(a, 3);
  }
  ++i;
  s += n[i&7];
}
@}

Danach wissen wir, da"s $a$ ein Produkt h"ochstens zweier Primzahlen ist. Deshalb wird ab jetzt unser Interesse
darin bestehen, eine gegebene nat"urliche Zahl in zwei Faktoren (falls "uberhaupt m"oglich) zu zerlegen.


\subsection{Faktorisierungsalgorithmus von Fermat}
%-------------------------------------------------
\index{Faktorisierungsalgorithmus!von Fermat}

Ist $n$ eine ungerade nat"urliche Zahl und Differenz zweier Quadratzahlen, so l"a"st sie sich zerlegen in

\[n = pq = \left(\frac{p + q}{2}\right)^2 - \left(\frac{p - q}{2}\right)^2 = x^2 - y^2 = (x + y)(x - y).\]

Ausgehend von $\lfloor\sqrt{n}\rfloor$, suchen wir eine iterative L"osung f"ur $r = x^2 - y^2 - n$. Falls $r = 0$ ist,
haben wir unsere Zerlegung gefunden, ansonsten wird $x$, solange $r > 0$ ist, schrittweise verkleinert
oder $y$ f"ur $r < 0$ dementsprechend vergr"o"sert.

\small\begin{verbatim}
void fermat(const Natural& n, Natural& a, Natural& b)
// Algorithm:  fermat(n; a, b)
// Input:      n in Natural where n is odd.
// Output:     a,b in Natural such that ab = n ||
{
  Natural r = sqrt(n);
  Natural x = 2*r+1;
  Natural y = 1;
  r *= r;
  while (r < n) { r += x; r -= y; x += 2; y += 2; }
  r -= n;
  while (r != 0) {
    while (r >= y) { r -= y; y += 2; }
    if (r != 0)
      do { r += x; r -= y; x += 2; y += 2; } while (r < y);
  }
  a = (x-y)/2;
  b = (x+y-2)/2;
}
\end{verbatim}\normalsize

Dieser Algorithmus zeichnet sich besonders dadurch aus, da"s er lediglich mit Additionen und Subtraktionen auskommt.
Noch geschickter ist es aber, nur alle Quadratzahlen von $x$ zu durchlaufen und jedesmal zu "uberpr"ufen, ob $x-n$ eine
Quadratzahl ist. Die n"achste Quadratzahl l"a"st sich dann einfach durch Addieren ermitteln: $(x+1)^2 = x^2 + 2x + 1$.
Zus"atzlich betrachten wir $y\equiv z\pmod{16}$ und wissen, da"s $y$ keine Quadratzahl ist, falls
$z\not\in\{0, 1, 4, 9\}$ ist.\\
Dennoch kann dieser Ansatz in der Praxis schlechter als das sukzessive Dividieren sein, das die Ordnung
$\Ord(n^\frac{1}{2})$ hat, n"amlich wenn zum Beispiel $p\approx n^\frac{1}{3}$ und $q\approx n^\frac{2}{3}$ ist.
Dann gilt n"amlich
\[\frac{(\sqrt{n} - \sqrt[3]{n})^2}{2\sqrt[3]{n}}
 = \frac{n^\frac{2}{3}(\sqrt[6]{n} - 1)^2}{2\sqrt[3]{n}}\in\Ord(n^\frac{2}{3}).\]



\subsection{Faktorisierungsalgorithmus von Lehman}
%-------------------------------------------------
\index{Faktorisierungsalgorithmus!von Lehman}\label{factoring by lehman}
Der vorige Ansatz von Fermat kann verwendet werden, um eine Laufzeit von $\Ord(n^\frac{1}{3})$ zu erzielen.
Dies demonstrierte R. S. Lehman 1974 in \cite{Lehm74}:\\
Zun"achst zerlegt man die zu faktorisierende Zahl $n\in\N$ durch sukzessives Dividieren in 

\[m = \frac{n}{\prod_{i = 0}^t p_i}\qquad\mbox{mit allen Primteilern $p_i\leq\sqrt[3]{n}$ von $n$ f"ur $0\leq i\leq t$}.\]

Desweiteren pr"uft man, ob sich $m = pq$ zerlegen l"a"st mit
\[m^\frac{1}{3} < p\leq q < m^\frac{2}{3},\]
indem gepr"uft wird, ob $m = p^2 = q^2$ ist oder ob $(\lfloor\sqrt{4km}\rfloor + d)^2 - 4km$ eine
Quadratzahl f"ur $1\leq k\leq\lfloor\sqrt[3]{m}\rfloor,1\leq d\leq\left\lfloor\frac{\sqrt[6]{m}}{4\sqrt{k}}\right\rfloor+1$
ist. Falls ein solches Paar $(k, d)$ nicht existiert, so ist $m$ eine Primzahl. Andernfalls setzt man
\[a := \lfloor\sqrt{4km}\rfloor + d, b := \sqrt{a^2 - 4km},\]
und somit gilt $m = \gcd(a + b, m)\cdot\frac{m}{\gcd(a + b, m)} = \gcd(a + b, m)\cdot\gcd(a - b, m)$.\\

\textbf{Laufzeitbeispiel.} Wenn wir die Mersennezahl $2^{67}-1 = 193707721\cdot 761838257287$ faktorisieren
wollen, dann ben"otigt das sukzessive Divisionsverfahren 126 Sekunden mit dem GNU-Compiler auf einem Pentium 100 MHz
(639 Sekunden f"ur die volle ANSI-Version), w"ahrend unter denselben Bedingungen der Algorithmus von
R. S. Lehman mit lediglich 40 Sekunden (52 Sekunden f"ur die volle ANSI-Version) auskommt.
Dies liegt aber haupts"achlich daran, weil wir f"ur ihn mit unserem Wurzelzieh-Algorithmus aus
Kapitel \ref{Wurzelberechnung} eine gute Grundlage gesetzt haben und schon vorher berechnete Ergebnisse wiederverwenden,
wenn es lediglich um die Berechnung des Nachfolgers geht.\\
So gilt zum Beispiel f"ur ein $a\in\N$ die Gleichung $a = \lfloor\sqrt{a^2 + k}\rfloor$ f"ur $0\leq k\leq 2a$, da erst mit
$\lfloor\sqrt{a^2 + 2a + 1}\rfloor = \lfloor\sqrt{(a + 1)^2}\rfloor = a + 1$
der Nachfolger von $a$ erreicht wird.\\
Das Ganze f"uhrt nun zur folgenden Implementation:

@D prime factorization of a \Natural\
@{template <class Container>
void factoring(Natural a, Container& p)
// Algorithm:  factoring(a, c)
// Input:      a in Natural.
// Output:     c is a container over Natural with output-iterator
//             such that if a >= 1 then a = prod_{c.begin() <= i < c.end()} *i
//             else c.begin() = c.end(),
//             for all i in [c.begin(), c.end()[, j in [i, c.end()[ : *i <= *j ||
{
  p.erase(p.begin(), p.end());
  @<successive division for less factors@>
  if (a.isprime() == 1) { p.push_back(a); return; }	// greater
  @<successive division for greater factors@>
  Natural w;                                        // large factors
  sqrt(a, s, w);
  if (w == 0) { p.push_back(s); p.push_back(s); return; }
  s = root(a, 6);
  q = a << 2;
  t *= a; t <<= 2;

  Natural x,y,z,d = 4;
  Natural e = 2;
  r = s >> 2;
  const char c[32] = { 1, 1, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0,
                       1, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0 };
  Natural k = q;
  for (; k <= t; k += q) {
    if (e == 0) {
      d += 4;
      if (d > s) break;
      r = d >> 1;
      div(s, d, r, x);
    } else --e;
    sqrt(k, x, y);
    z = x << 1; ++x; ++z; z -= y;
    while (true) {
      if (c[z.lowest() & 31]) {
        sqrt(z, y, w);
        if (w == 0) {
          y = gcd(x+y, a);
          a /= y;
          if (y < a) { p.push_back(y); p.push_back(a); }
          else { p.push_back(a); p.push_back(y); }
          return;
        }
      }
      if (r == 0) break;
      --r; z = x << 1; ++x; ++z;
    }
  } 
  while (k <= t) {
    sqrt(k, x, z);
    y = x << 1; ++x; ++y; y -= z;
    if (c[y.lowest() & 31]) {
      sqrt(y, y, w);
      if (w == 0) {
        y = gcd(x+y, a);
        a /= y;
        if (y < a) { p.push_back(y); p.push_back(a); }
        else { p.push_back(a); p.push_back(y); }
        return;
      }
    } 
    k += q;
  }
  p.push_back(a);
}
@| factoring@}
\textbf{Laufzeit.} Sei $x\in\Natural$ und $C$ ein Container "uber $\Natural$, dann
\texttt{factoring($x$, $C$)$\in\Ord(x^{\frac{1}{3}})$}.



\subsection{Faktorisierung nach Gau"s/Legendre}
%----------------------------------------------
\index{Faktorisierungsalgorithmus!von Gau"s/Legendre}

Trotz dieses guten Programms k"onnen dennoch bereits drei"sigstellige Zahlen (wie zum Beispiel
$2^{101} - 1 = 7432339208719 \cdot 341117531003194129$) nur mit viel Rechenaufwand zerlegt werden.\\
Sehr gute und ausf"uhrliche Erkl"arungen zu den folgenden drei Methoden (Gau"s/Legendre, Dixon, Morrision
und Brillhart) zur L"osung dieses Faktorisierungsproblems k"onnen genauer im Buch von H. Riesel \cite{Ries85} nachgelesen
werden. Wir werden hier haupts"achlich die Implementation und den praktischen Nutzen betrachten.\\

Die eigentliche Idee von Gau"s besteht darin, quadratische Reste Modulo der zu faktorisierenden Zahl $a\in\N$ zu finden.
Dabei gilt folgendes:
\[y\equiv x^2\pmod{a}\quad\Longrightarrow\quad y\equiv x^2\pmod{p}
\mbox{\qquad f"ur alle Teiler $p$ von $a$}.\]
Wenn wir nun einen quadratischen Rest mod $a$ haben, so k"onnen wir mit Hilfe der Erg"anzungss"atze
(siehe Kapitel \ref{Ergaenzungssaetze}) die Anzahl der f"ur das
sukzessive Dividieren notwendigen Divisoren halbieren.\medskip\\
\textbf{Beispiel.} Sei $a\in\N$ die zu faktorisierende Zahl, $D := \{p\in\N\mid\ p\equiv\pm1\pmod{6}\}\dot{\cup}\{2, 3\}$
unsere Menge aller Divisoren und 5 ein quadratischer Rest mod $a$. So ist $\Z/5\Z\supseteq Q = \{1^2, 2^2\} = \{1, 4\}$ die
Menge aller quadratischen Reste mod 5 au"ser Null. Die Menge $Q$ kann dabei mit dem Programm
\texttt{root} (siehe Seite \pageref{root}) ermittelt werden. Wenden wir nun noch den Chinesischen Restsatz
\texttt{chines} (siehe Seite \pageref{chines}) an, so erhalten wir die Menge
\[D' = \{p\in\N\mid\ p\equiv q\pmod{30},\; q\in\{1, 11, 19, 29\}\} \dot{\cup} \{2, 3\}.\]

Bis heute ist es noch nicht gelungen, kleine quadratische Reste mod $a$ explizit angeben zu k"onnen.
C. F. Gau"s wendete deswegen den folgenden Trick an:
\[y\equiv x^2\pmod{a}\quad\Longleftrightarrow\quad x^2 - y = ka\qquad\mbox{f"ur ein $k\in\N$}.\]
W"ahlen wir nun $x$ nahe bei $\lfloor\sqrt{ka}\rfloor$, so l"a"st sich $y$ leichter faktorisieren, denn wir
brauchen nur alle Exponenten unserer Primteiler von $y$ Modulo 2 zu betrachten, weil das Produkt zweier quadratischer Reste
oder zweier Nichtreste mod $a$ ein quadratischer Rest mod $a$ ist.\\

\textbf{Beispiel.} Sei $a = 11111$, so ist 5 ein quadratischer Rest mod $a$,
weil
\[\lfloor\sqrt{11111}+1\rfloor - 5^3 = 106^2 - 5^3 = 11111.\]\\

\textbf{Bemerkung.} A. M. Legendre hat zur Auffindung der quadratischen Reste mod $a$ die Kettenbruchentwicklung von
$\sqrt{a}$ vorgeschlagen (siehe H. Riesel \cite{Ries85}).



\subsection{Faktorisierungalgorithmus von Dixon}
%-----------------------------------------------
\index{Faktorisierungsalgorithmus!von Dixon}

Es gilt die folgende Beziehung: Sei $a\in\N$ eine ungerade Zahl, die wir faktorisieren wollen, so ist der Hauptgedanke in
J. D. Dixons Faktorisierungalgorithmus (\cite{Dixon81}), zwei ganze Zahlen $x$ und $y$ mit $\gcd(x, a) = \gcd(y, a) = 1$ zu
finden, wobei $x^2 \equiv y^2 \pmod{a}$ und $x \not\equiv y \pmod{a}$ ist. Daraus folgt, da"s
$x^2 - y^2 = (x - y)(x + y)\equiv 0\pmod{a}$ und $x + y\not\equiv 0\pmod{a}$ gilt. Somit ist
$\gcd(x + y, a)$ ein nichttrivialer Teiler von $a$.



\subsection{Faktorisierungalgorithmus von Morrision und Brillhart}
%-----------------------------------------------------------------
\index{Faktorisierungsalgorithmus!von Morrision und Brillhart}

Die Grundgedanken von C. F. Gau"s, A. M. Legendre und J. D. Dixon wurden 1970 mit gro"sem Erfolg zur Faktorisierung der
siebten Fermatzahl 
\[F_7 = 2^{128} + 1 = 59649589127497217\cdot 5704689200685129054721\]
eingesetzt. Detaillierte
Informationen und genauere Arbeiten findet man in der Originalarbeit von M. A. Morrison und J. Brillhart in \cite{MoBr75}.
Wir werden hier ein paar Punkte herausnehmen und realisieren.\\
Sei $a\in\N$ eine ungerade Zahl, die wir faktorisieren wollen und $k\in\N_+$. Bevor wir nun die Kettenbruchentwicklung von
$\sqrt{ka}$ durchf"uhren, betrachten wir eine endliche Menge von Primzahlen
$P(k, a, m) = \{p_i\in\Primes\mid 1\leq i\leq m, (ka)^\frac{p_i-1}{2}\equiv x\pmod{p_i}, x\in\{0, 1\}\}\cup\{2\}$
mit $m\in\N$, die das Euler-Kriterium erf"ullen:

@O cfrac.cpp
@{#include <string.h>
#include <time.h>
#include "natural.h"

@<new operator@>

Natural cfrac(const Natural& a, const Digit m, const Digit c)
// Algorithm:  q := cfrac(a, b, c)
// Input:      a in Natural, b,c in Digit where b >= 50, c >= 1.
// Output:     b in Natural such that q | a ||
{
  struct prim_t {
    Natural* x;
    char*    c;
  };
  Natural s,s2,s3,r,v,y;
  Natural t = 1;
  const Digit pm = 3;
  Digit* p = NOTHROW_NEW Digit[m];
  if (!p) a.errmsg(2, "(cfrac)");
  p[0] = p[1] = 0;

  prim_t* d = NOTHROW_NEW prim_t[m+1];
  if (!d) a.errmsg(2, "(cfrac)");
  Digit i;
  for (i = 0; i <= m; ++i) {
    d[i].x = NOTHROW_NEW Natural[3];
    if (!d[i].x) a.errmsg(2, "(cfrac)");
    d[i].c = NOTHROW_NEW char[m];
    if (!d[i].c) a.errmsg(2, "(cfrac)");
    memset(d[i].c, 0, m);
  }
  for (Natural b = c*a; t == 1; b += a) {
    sqrt(b, r, v);
    Natural w = r;
    Natural z = 1;
    Natural u = 1;
    r <<= 1;
    Natural x = r;
    Digit prim = a.firstPrime();
    Digit n = 2;
    while (a.nextPrime(prim) && n < m) {
      Digit q = b % prim;
      Digit q2 = 1;
      for (i = prim/2; i > 1; i /= 2) {
        if (i&1) { q2 *= q; q2 %= prim; }
        q *= q; q %= prim;
      }
      q2 *= q; q2 %= prim;
      if (q2 <= 1) p[n++] = prim;
    }
@}

Nun suchen wir diese Zahlen $v$ in unserer Kettenbruchentwicklung, deren Primteiler alle aus $P$ sind. Um nicht $m$
Divisionen durchf"uhren zu m"ussen, setzen wir
\[T := \prod_{p\in P}p\mbox{ und betrachten }T\equiv s\pmod{v}.\]
Ist nun
$s = 0$, so haben wir ein gesuchtes $v$ gefunden, ansonsten quadrieren wir $s$ und betrachten $s^2\equiv s_2\pmod{v}$.
Ist nun $s_2 = 0$, so l"a"st sich $v$ in Primfaktoren aus $P$ mit maximal quadratischem Exponenten zerlegen. Je nach
Gr"o"senordnung unserer gegebenen Zahl $a$ k"onnen wir auf diese Weise fortfahren. Es ist jedoch sinnvoller, die
Zerlegung in zwei Bereiche aufzuteilen:
\begin{enumerate}
\item Kleine Primzahlen k"onnen einen beliebig hohen Exponenten haben.
\item Gr"o"sere Primzahlen d"urfen einen Exponeten haben, der h"ochstens gleich vier ist.
\end{enumerate}

@O cfrac.cpp
@{    Natural trial = 1;
    for (i = 2; i < pm; ++i);
    while (i < n) trial *= p[i++];

    Digit nd = 0;
    Digit iteration = 0;
    while (true) {
      ++iteration;
      t = v;
      if (t == 1) break;
      while ((t&3) == 0) t >>= 2;
      d[nd].c[1] = char(t.even());
      if (d[nd].c[1]) t >>= 1;
      for (i = 2; i < pm && t > 1; ++i) {
        const Digit k = p[i]*p[i];
        const Digit l = k*p[i];
        Digit m;
        while (true) {
          div(t, k*k, s, m);
          if (m) break;
          swap(t, s);
        }
        if (m%l == 0) {
          d[nd].c[i] = 1;
          t /= l;
        } else if (m%k == 0) {
          d[nd].c[i] = 0;
          t /= k;
        } else if (m%p[i] == 0) {
          d[nd].c[i] = 1;
          t /= p[i];
        } else d[nd].c[i] = 0;
      }
      if (t > 1) {
        div(trial, t, s, s2);
        if (s2 != 0) {
          s2 *= s2; s2 %= t;
          if (s2 != 0) { s2 *= s2; s2 %= t; }
        }
      }
@}

Falls wir nun $v$ komplett durch Primzahlen aus $P$ faktorisieren k"onnen, halten wir die vorkommenden Exponenten Modulo 2
im Vektor $c$ der Struktur \verb|prim_t| fest, wobei hier $c_0$ f"ur -1, $c_1$ f"ur 2 und so weiter stehen. Unser
Vektor $d$ enth"alt dadurch die einzelnen Vektoren $c$, mit denen er eine Matrix "uber $(\Z/2\Z)^m\times(\Z/2\Z)^m$ bildet,
sowie die faktorisierten Zahlen $v$ und die Quadratwurzel $w$ von $v$ mod $a$:

@O cfrac.cpp
@{      if (t == 1 || s2 == 0) {
        if (t > 1) {
          s = t;
          Digit j,k;
          do {
            k = p[i]*p[i];
            div(s, k, t, j);
            if (j == 0) {
              swap(s, t);
              --i;
            } else if (j%p[i] == 0) {
              d[nd].c[i] = 1;
              s /= p[i];
            }
            ++i;
          } while (s > k);
          if (s != 1) {
            while (s > p[i]) ++i;
            d[nd].c[i] = 1;
          }
        }
        d[nd].c[0] = char(iteration&1);
        d[nd].x[0] = w; d[nd].x[1] = v; d[nd].x[2] = 1;
@}

Nun f"ugen wir unseren Vektor $c$ in unsere Matrix ein. Dabei beachten wir, da"s die erste 1 in unserem Vektor $c$ nur
einmal in der Matrix vorkommen darf. Dadurch erzeugen wir durch Linearkombinationen sukzessive eine Diagonalform. Bei den
Linearkombinationen k"onnen wir unsere $v$-Eintr"age (quadratische Reste mod $a$) einfach modular miteinander
multiplizieren, w"ahrend unsere $w$-Eintr"age (quadratische Wurzeln mod $a$) weiterhin regul"ar miteinander multipliziert
werden m"ussen. Diese Zahlen w"urde rasch gro"se Dimensionen annehmen und somit die Laufzeit stark hemmen, wenn man nicht
zuvor den gr"o"sten gemeinsamen Teiler herausn"ahme, der auf jeden Fall quadratisch vorkommt:

@O cfrac.cpp
@{        Digit j;
        for (j = 0; j < n && d[nd].c[j] == 0; ++j);
        Digit k = 0;
        for (i = 0; i < nd && j < n; ++i) {
          while (d[i].c[k] == 0) ++k;
          if (k == j) {
          d[nd].c[k] = 0;
          for (Digit l = ++k; l < n; ++l) d[nd].c[l] ^= d[i].c[l];
            d[nd].x[0] *= d[i].x[0];
            d[nd].x[0] %= a;
            t = gcd(d[nd].x[1], d[i].x[1]);
            d[nd].x[1] *= d[i].x[1];
            d[nd].x[1] /= t*t;
            d[nd].x[2] *= t;
            d[nd].x[2] *= d[i].x[2];
            d[nd].x[2] %= a;
            while (j < n && d[nd].c[j] == 0) ++j;
          } else if (k > j) {
            prim_t tP = d[nd];
            Digit l;
            for (l = nd; l > i; --l) d[l] = d[l-1];
            d[l] = tP;
            ++i;
            while (true) {
              if (d[l].c[k]) {
                d[l].c[k] = 0;
                for (Digit l2 = k+1; l2 < n; ++l2) d[l].c[l2] ^= d[i].c[l2];
                d[l].x[0] *= d[i].x[0];
                d[l].x[0] %= a;
                t = gcd(d[l].x[1], d[i].x[1]);
                d[l].x[1] *= d[i].x[1];
                d[l].x[1] /= t*t;
                d[l].x[2] *= t;
                d[l].x[2] *= d[i].x[2];
                d[l].x[2] %= a;
              }
              if (++i > nd) break;
              while (d[i].c[++k] == 0);
            }
            for (i = 0; i < l; ++i)
              if (d[i].c[j]) {
              d[i].c[j] = 0;
              for (Digit l2 = j+1; l2 < n; ++l2) d[i].c[l2] ^= d[l].c[l2];
              d[i].x[0] *= d[l].x[0];
              d[i].x[0] %= a;
              t = gcd(d[l].x[1], d[i].x[1]);
              d[i].x[1] *= d[l].x[1];
              d[i].x[1] /= t*t;
              d[i].x[2] *= t;
              d[i].x[2] *= d[l].x[2];
              d[i].x[2] %= a;
            }
            i = ++nd;
          } else ++k;
        }
@}

Wenn wir nun einen linear abh"angigen Vektor ermittelt haben, dann brauchen wir nur noch zu "uberpr"ufen, ob die Relation
von J. D. Dixon zutrifft:

@O cfrac.cpp
@{        if (j == n) {	// Solution?
          Natural y = sqrt(d[nd].x[1]) * d[nd].x[2];
          y %= a;
          t = a-y;
          if (d[nd].x[0] != y && d[nd].x[0] != t) {
            y += d[nd].x[0];
            t = gcd(y, a);
            v = 2;
            break;
          } else memset(d[nd].c, 0, n);
        } else if (i == nd) {
          for (i = 0; i < nd; ++i)
            if (d[i].c[j]) {
              d[i].c[j] = 0;
              for (Digit l2 = j+1; l2 < n; ++l2) d[i].c[l2] ^= d[nd].c[l2];
              d[i].x[0] *= d[nd].x[0];
              d[i].x[0] %= a;
              t = gcd(d[nd].x[1], d[i].x[1]);
              d[i].x[1] *= d[nd].x[1];
              d[i].x[1] /= t*t;
              d[i].x[2] *= t;
              d[i].x[2] *= d[nd].x[2];
              d[i].x[2] %= a;
          }
          ++nd;
        }
      }
      div(x, v, s, t);			// Kettenbruchentwicklung
      y = r-t; z += w*s; z %= a;
      if (x >= y) {
        t = x-y; t *= s; u += t;
      } else {
        t = y-x; t *= s; u -= t;
      }
      swap(v, u); swap(x, y); swap(w, z);
    }
    for (i = 0; i < nd; ++i) memset(d[i].c, 0, n);

    cout << "Iterationen=" << iteration << endl;
    cout << "Matrix size=" << nd << endl;
  }
  for (i = 0; i <= m; ++i) {
    delete[] d[i].x;
    delete[] d[i].c;
  }
  delete[] d;
  delete[] p;
  return t;
}

int main()
{
  Natural a = 1;
  a <<= 128; ++a;

  clock_t start = clock();
  Natural c = cfrac(a, 400, 257);
  clock_t stop = clock();
  cout << "time [s] = " << double(stop-start)/CLOCKS_PER_SEC << endl;

  cout << c << '*' << a/c << endl;

  return 1;
}
@}

Somit haben wir jetzt eine Funktion, die gro"se zusammengesetzte Zahlen in zwei Faktoren aufteilt. Diese k"onnen wir
so lange rekursiv anwenden, bis wir die gesuchte Primfaktorisierung gefunden haben oder f"ur kleinere Zahlen zu den 
herk"ommlichen Faktorisierungsmethoden "uberwechseln.\\

Auf einen Pentium 100 MHz erreichen wir mit dem Watcom-Compiler 10.6 die folgenden Zeiten:

\hspace*{\fill}\begin{tabular}{l|c|c|c}
$a$ & $b$ & $m$ & Laufzeit \\
\hline
$2^{59} - 1$ & $a$ & 50 & 0.88 s\\
$2^{67} - 1$ & $a$ & 50 & 2.86 s\\
$2^{101} - 1$ & $a$ & 100 & 150.9 s\\
$2^{128} + 1$ & $257a$ & 400 & 2546.2 s
\end{tabular}\hfill\mbox{}



\subsection{Euler-Funktion}
%--------------------------

Sei $n\in\N$, so bezeichnet man mit der \textbf{Euler-Funktion}\index{Euler-Funktion}\label{Euler-Funktion}
$\varphi(n)$ die Anzahl aller nat"urlichen Zahlen, die kleiner $n$ und teilerfremd zu $n$ sind, das hei"st
\[\varphi : \N\to\N : n\mapsto|\{\N\in x\leq n\mid \gcd(x, n) = 1\}|.\]

F"ur die Primfaktorzerlegung 
\[n = \prod_{i = 0}^t p_i^{a_i}\qquad\mbox{f"ur $t\in\N, a_i\in\N_+, 0\leq i\leq t$}\]
gilt
\[\varphi(n) = n\prod_{i=0}^t\left(1-\frac{1}{p_i}\right).\]

Durch diese Eigenschaft l"a"st sich die Funktionswert $\varphi(n)$ direkt berechnen:
@D euler function
@{Natural euler(Natural a)
// Algorithm:  c := euler(a)
// Input:      a in Natural.
// Output:     c in Natural such that c = phi(a) ||
{
  list<Natural> p;
  factoring(a, p);
  Natural c,d;

  list<Natural>::iterator i = p.begin();
  while (i != p.end()) {
    div(a, *i, c, d); a -= c;
    list<Natural>::const_iterator j = i;
    while (++i != p.end() && *j == *i);
  }
  return a;
}
@| euler @}
\textbf{Laufzeit.} Sei $x\in\Natural$ und $C$ ein Container "uber $\Natural$, dann\smallskip\\
\mbox{}\hfill\texttt{euler($x$) $\sim$ factoring($x$,$C$)}.\hfill\mbox{}
