proc (NLT) = annlintest(y,arg2,arg3,arg4)
; ----------------------------------------------------------------------
; Library      times
; ----------------------------------------------------------------------
; See_also     annarchtest
; ----------------------------------------------------------------------
; Macro        annlintest
; ----------------------------------------------------------------------
; Description  This macro calculates the neural network test for neglected
;              nonlinearity proposed by Lee, White and Granger (1993). This 
;              statistic is evaluated from uncentered squared multiple 
;              correlation of an auxiliary regression in which we regress 
;              the residuals of a linear regression on the regressors of 
;              this regression and the principal components of a nonlinear
;              transformation of the regressors.  The first argument of the       
;              macro is the series y. The second argument is either a set
;              of regressors X if the series is regressed on X, or the number
;              of lags if the series is regressed on its lagged realizations.
;              The macro adds automatically a constant if the constant term 
;              is missing in X. If the series is regressed on its past 
;              realizations, then the second argument, i.e., the number of
;              lags, may be a vector. In that case, the corresponding linear 
;              models and statistics for neglected nonlinearity are computed.
;              The third optional argument is the number of hidden units of
;              the neural network, which should be greater than or equal to 3.
;              The fourth optional argument is the number of principal  
;              components used in the auxiliary regression. The number of
;              principal components should be less than the number of 
;              corresponding hidden units. The default third argument               
;              is the vector (10,20), the default fourth argument is the 
;              vector (2,3). If the series is regressed on a set of exogeneous
;              variables X, the macro returns the number of principal 
;              components used in the auxiliary regression, the value of the 
;              test, the 95% critical value for the null hypothesis of the
;              test, and the P value of the test. If the series is regressed
;              on its past realizations, the number of lagged explanatory 
;              variables is also displayed.
; ----------------------------------------------------------------------        
; Usage        q = annlintest (y, x{, units{, principal_components}})
; Input
;   Parameter  y
;   Definition n dimensional vector
;   Parameter  x  
;   Definition n x p matrix
;   Parameter  order
;   Definition matrix
; Output
;   Parameter  q
;   Definition vector of strings 
; ----------------------------------------------------------------------
; Example      ;We calculate here the test for neglected nonlinearity. As the  
;              ;second argument is equal to two, the series is regressed on its 
;              ;two lagged values. Since there is no third and fourth argument,
;              ;the number of hiden units is equal to the default numbers, i.e., 
;              ;10 and 20, and the default number of principal components is 
;              ;equal to 2 and 3, as displayed in the output.
;              library("times") ; loads the library times
;              randomize(1)  
;              y = gentar(2,1,0,0.5|-0.5,normal(400))
;              q = annlintest(y,2) 
;              q
; ----------------------------------------------------------------------
; Result   
; Contents of q
; [1,] Number of Principal  Number of  Statistic  95% Critical Value  P-Value  
; [2,]     Components          lags                                            
; [3,] _______________________________________________________________________ 
; [4,]  
; [5,]           2              2      11.78424         5.97378       0.00276 
; [6,]           3              2       6.51763         7.80251       0.08897 
;
; ----------------------------------------------------------------------
; Example      ;We calculate here the test, where y is regressed on its three 
;              ;lagged realizations. The number of hidden units is given by the 
;              ;vector hu, i.e., we consider the cases with 10, 20 and 30 
;              ;hidden units. The number of principal components is given in 
;              ;the vector pcomp, and are respectively equal to 2, 3, and 4, 
;              ;as displayed in the output.
;              library("times") 
;              randomize(1)  ; 
;              y = gentar(2,1,0,0.5|-0.5,normal(400))
;              hu = #(10,20,30)
;              pcomp = #(2,3,4)
;              q = annlintest(y,3,hu,pcomp) 
;              q
; ----------------------------------------------------------------------
; Result   
; Contents of q
; [1,] Number of Principal  Number of  Statistic  95% Critical Value  P-Value  
; [2,]     Components          lags                                            
; [3,] _______________________________________________________________________ 
; [4,]  
; [5,]           2              3       5.77143         5.97378       0.05582 
; [6,]           3              3      10.68780         7.80251       0.01354 
; [7,]           4              3       9.20524         9.47844       0.05617 
; ----------------------------------------------------------------------
; Example      ;We illustrate here the other use of the macro, in which the 
;              ;regressors are explicitely mentionned. As we choose as regressors
;              ;X the two lagged realizations of y, as number of hidden units 10,
;              ;and as number of principal components 2, the result is a particular 
;              ;case of the first example. As we did not include a constant in the 
;              ;regressors, this constant is automatically added as displayed
;              ;in the warning message after the results.
;              library("times") 
;              randomize(1)  ; 
;              y = gentar(2,1,0,0.5|-0.5,normal(400))
;              x1 = y[3:400]; y
;              x2 = y[1:398]~y[2:399]; two lagged values of y
;              q = annlintest(x1,x2,10,2)
;              q
; ----------------------------------------------------------------------
; Result   
; Contents of q
;
; [1,] Number of Principal   Statistic  95% Critical Value  P-Value  
; [2,]     Components                                                
; [3,] _____________________________________________________________ 
; [4,]  
; [5,]          2            11.78424         5.97378       0.00276 
; [6,]  
; [7,] A constant has been added to the matrix of regressors         
;
; ----------------------------------------------------------------------
; Reference
;   Lee, T-H., White, H. and C.W.J. Granger (1993): Testing for Neglected
;	      Nonlinearity in Time Series Models. A Comparison of Neural
;             Networks Methods and Alternative Tests.
;      	      Journal of Econometrics, vol 56, pp 269-290.
; ----------------------------------------------------------------------
;   Keywords
;      neural networks, tests, linearity
; ----------------------------------------------------------------------
;   Author        Gilles Teyssiere, 980614
; ---------------------------------------------------------------------- 
  targ2 = exist(arg2)
  targ3 = exist(arg3)
  targ4 = exist(arg4)
  if (rows(arg2) == rows(y))
    ty = y
    tx = arg2
    reg = 0
  else
    lags = arg2
    tnlags = rows(lags)
    reg = 1
  endif
  if (targ3 == 1); Number of hidden units of the network
    MQ = arg3
    error(min(MQ) .<3,"The number of hidden units should be >= 3")
  else
    MQ = #(10,20) ; default number of hidden units
  endif
  if (targ4 == 1); Number of principal components of the matrix Psi
    MQstar = arg4
  else
    MQstar = #(2,3) ; default number of principal components
  endif
  error(rows(MQ) != rows(MQstar),"The number of different principal components does not match the number of corresponding hidden units")
  nQ = rows(MQ)
  i = 1;
  while (i <= nQ)
    error (MQstar[i] > MQ[i],"The number of principal components should be less than the number of corresponding hidden units")
    i = i+1
  endo
  ;;
  if (reg == 0); simple case ty is regressed on tx  
    ; check whether the constant has been included in the regressors
    T = rows(ty)
    IdMx = matrix(T) 
    NColx = cols(tx)  
    i = 1;
    TestCst = 0
    while (i <= NColx)
      if (tx[,i] == IdMx)
        i = NColx
        TestCst = 1
      endif
      i = i+1
    endo
    if (TestCst == 0) 
      tx = IdMx~tx   
      NColx = NColx+1
    endif
    ; Normalisation in [0,1]    
    ty = (ty-min(y))/(max(y)-min(y))    
    tx = RescaleMat(tx) 
    te = ty - tx*inv(tx'*tx)*tx'*ty; ols residuals from the OLS of ty on tx
    NLT = matrix(nQ)
    i = 1
    while (i <= nQ)
      Psi = MakePsi(tx,MQ[i]); computation of the matrix Psi
      RegAux = SelectAuxReg(Psi,tx,MQstar[i]);regressors of the auxiliary regression
      NLT[i] = TrsqAuxReg(te,RegAux) 
      i = i+1
    endo   
    CritOrder = matrix(nQ)*0.95
    ;NLT = MQstar~NLT~qfc(CritOrder,MQstar)~(1-cdfc(NLT,MQstar))  
    ; Output display
    line1 ="Number of Principal   Statistic  95% Critical Value  P-Value "
    line2 ="    Components                                               "
    line3 ="_____________________________________________________________"
    line4 ="A constant has been added to the matrix of regressors        "
    st1 = string("%10.0f",MQstar)
    st2 = string("%20.5f",NLT)
    st3 = string("%16.5f",qfc(CritOrder,MQstar))
    st4 = string("%14.5f",(1-cdfc(NLT,MQstar)))
    st5 = st1+st2+st3+st4
    NLT = line1|line2|line3|""|st5    
    if (TestCst == 0) 
      NLT = line1|line2|line3|""|st5|""|line4
    endif
    ;;
  else ; y is regressed on its past realizations
    nT = rows(y)
    NLT = matrix(nQ*tnlags,3)
    i = 1
    iN = 1
    while (i <= tnlags) 
      k = lags[i]
      ty = y[k+1:nT]
      tx = matrix(nT-k)~y[1:nT-k]    
      j = 2
      while (j <= k)
        idx = nT-k+j-1
        tx = tx~y[j:idx]
        j = j+1
      endo         
      tx = RescaleMat(tx)  
      ty = (ty-min(ty))/(max(ty)-min(ty))
      te = ty - tx*inv(tx'*tx)*tx'*ty
      j = 1
      while (j <= nQ)
        Psi = MakePsi(tx,MQ[j])
        RegAux = SelectAuxReg(Psi,tx,MQstar[j])
        NLT[iN,1] = MQstar[j]
        NLT[iN,2] = k
        NLT[iN,3] = TrsqAuxReg(te,RegAux) 
        iN = iN+1
        j = j+1
      endo   
      i = i+1
    endo  
    
    CritOrder = matrix(nQ*tnlags)*0.95
    ;NLT = NLT~qfc(CritOrder,NLT[,1])~(1-cdfc(NLT[,3],NLT[,1]))  
    ; Output display
    line1 ="Number of Principal  Number of  Statistic  95% Critical Value  P-Value "
    line2 ="    Components          lags                                           "
    line3 ="_______________________________________________________________________"
    st1 = string("%11.0f",NLT[,1])
    st2 = string("%15.0f",NLT[,2])
    st3 = string("%14.5f",NLT[,3])
    st4 = string("%16.5f",qfc(CritOrder,NLT[,1]))
    st5 = string("%14.5f",(1-cdfc(NLT[,3],NLT[,1])))
    st6 = st1+st2+st3+st4+st5
    NLT = line1|line2|line3|""|st6
  endif
endp
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; procedure RescaleMat, rescales a matrix in [0,1]
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
proc (ResM) = RescaleMat(X) 
  NColx = cols(X)
  MaxX = max(X)
  MinX = min(X) 
  i = 1 ; case of the constant
  while (i <= NColx)
    if (MaxX[i] == MinX[i])
      MinX[i] = 0
      MaxX[i] = 1
    endif
    i = i+1
  endo        
  ResM = (X-MinX)/(MaxX-MinX)  
endp
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; procedure MakePsi, makes the matrix Psi
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
proc (Psi) = MakePsi(tX,q) 
  n = cols(tX)
  ; the random weights lambdas are between -2 and 2
  lambda = 2*(2*uniform(n,q) - matrix(n,q))   
  MA = tX*(lambda[,1])
  i = 2
  while (i <= q)
    MA = MA~(tX*lambda[,i])
    i = i+1
  endo
  Psi = 1/(1+exp(-MA)); Logistic transformation
endp
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; procedure SelectAuxReg, selects the regressors of the auxiliary regression
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
proc (RegAux) = SelectAuxReg(Psi,tX,qstar)  
  ;; part of the code is from the pca macro
  m = mean(Psi)
  n = cols(Psi)
  c = cov(Psi-m)
  {lam, gamma}=eigsm(c)  
  c = lam~trans(gamma) 
  c = sort(c, -1)                 
  gamma  = trans(c[,2:cols(c)])
  y  = (Psi-m)*gamma                 
  RegAux = y[,2:qstar+1]~tX
endp
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; procedure TrsqAuxReg, returns the T-R2 of the auxiliary regression
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
proc (Trsq) = TrsqAuxReg(y,x) 
  txy = x'*y  
  Trsq = rows(y)*txy'*inv(x'*x)*txy/(y'*y)
endp
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; end of macro annlintest
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

