proc(A,b,ctrl) = calibrIC(T,Sig,H,F,Q,R,typ,A0,b0,e,N,eps,itmax,expl,fact0,aus)
; -----------------------------------------------------------------
; Library        kalman
; -----------------------------------------------------------------
;  See_also      rICfil
; -----------------------------------------------------------------
;   Macro        calibrIC
; -----------------------------------------------------------------
;   Description  Auxiliary routine for rICfil
;                Calibrates the robust IC's for a given State Space model
;                to a given relative efficiency loss in terms of the MSE in the
;                ideal model.
;                The state-space model is assumed to be in the
;                following form:
;
;                y_t = H x_t + v_t
;
;                x_t = F x_t-1 + w_t
;
;                x_0 ~ (mu,Sig), v_t ~ (0,Q), w_t ~ (0,R)
;
;                All parameters are assumed known.
;
; -----------------------------------------------------------------
;   Keywords     robust Kalman filter, Kalman filter, Kalman smoother, filter
; -----------------------------------------------------------------
;   Usage         {A,b,ctrl}=calibrIC(T,Sig,H,F,Q,R,typ,A0,b0,e,N,eps,itmax,expl,fact0,aus)
;   Input
;     Parameter   T  
;     Definition      number of observations/states to be filtered by rIC
;     Parameter   Sig    
;     Definition      n x n covariance matrix of the initial state
;     Parameter   H    
;     Definition      m x n matrix
;     Parameter   F    
;     Definition      n x n matrix
;     Parameter   Q   
;     Definition      m x m variance-covariance matrix
;     Parameter   R
;     Definition      n x n variance-covariance matrix 
;     Parameter   typ
;     Definition     integer:    either 0 for simultaneous clipping,
;                                               1 - for AO-separate-clipping,
;                                               2 - for IO-separate-clipping ( !!! not too good - better: mIC to be done later)
;     Parameter   e
;     Definition     numeric; efficiency loss to attain
;     Parameter   A0  
;     Definition     p x p; starting value for A; if 0 is entered I^{-1} is taken
;                                 else if p>1 and dim(A0)==1 FI^{-1}*A0 is taken
;                                  and if p==1 and A0<0  , -A0/FI is taken     
;     Parameter   b0    
;     Definition     numeric; starting value for clipping height; if a nonpositive value is entered
;                        it is set to 1.6 sdv(component to be clipped)
;     Parameter    N
;     Definition      integer; MC-sample size / integration grid-points
;     Parameter    eps
;     Definition      numeric; exactitude
;     Parameter    itmax
;     Definition      integer; maximal number of Iterations
;     Parameter    expl
;     Definition      numeric (1.5<.<5) ; threshold for the changes in abs. value of A: 
;                        beyond this value convergence is uncertain  ~ 2
;     Parameter    fact0
;     Definition      numeric (1.1< .<1.5 )!; factor determining how fast we descend from b to b ~1.2
;     Parameter    aus
;     Definition      integer; 0: no output during execution, 1: some output, 2: more output, 3: a lot of output
;   Output                                                           
;     Parameter   A
;     Definition      T x n x n "vector" of A's
;     Parameter    b
;     Definition      T vector of clipping heights
;     Parameter    ctrl
;     Definition      T integer: tells if everything went well in step 1.. T (1 --- 0 else)
; -----------------------------------------------------------------
;   Author    P.Ruckdeschel 991010 
; -----------------------------------------------------------------
;   Example   to be looked up in rICfil
; -----------------------------------------------------------------
; Notes
;      Calles [within kalman.lib]  ICerzsep, ICerz.
; 
;      The Output produced if aus>=1 is not identical with the output parameters. 
;      (aus>=1) is for interactive use ;   success is controlled by "verbal" output.
;      (aus==0) is for other quantlets calling calibrIC:
;                       success is controlled by variable "ctrl".
; -----------------------------------------------------------------

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;preparations
  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  dimX = rows(F)                               ;get dimension of state vector
  dimY = rows(Q)                              ;get dimension of observation vector

;;;;;;;;;;;;;;;;;;;;;;;;error checks;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  Rs=svd(R)
  Qs=svd(Q)
  Sigs=svd(Sig)
  ;
  error((sum (Rs.l<0) > 0 )||(R<>R')||(cols(R) <> rows(R)), "Matrix R is not a Covariance.")
  error((sum (Sigs.l<0) > 0)||(Sig<>Sig')||(cols(Sig) <> rows(Sig)) , "Matrix Sig is not a Covariance.")
  error((sum (Qs.l<0) > 0)||(Q<>Q')||(cols(Q) <> rows(Q)) , "Matrix Q is not a Covariance..")
  error(cols(F) <> rows(F), "Matrix F is not square.")
  error(cols(R) <> rows(F), "Matrices F and R mismatch.")
  error(cols(R) <> cols(H), "Matrices R and H mismatch.")
  error(rows(Q) <> rows(H), "Matrices Q and H mismatch.")
  error(((typ<>0)&&(typ<>1)&&(typ<>2)), "wrong Clipping Type.")
  error(e<0, "wrong efficiency loss")
  error(((ceil(N)<>floor(N))|| (N<2) ), "wrong number of grid points / MC-Sample Size")
  error(((ceil(itmax)<>floor(itmax))|| (itmax<2)), "wrong maximal number of iterations")
  error(((expl>5)||(expl<1.5)), "wrong detection threshold for Explosions (expl)")
  error(((fact0>1.5)||(fact0<1.1)), "wrong decent - factor for b (fact0)")
  error(((ceil(aus)<>floor(aus))|| (aus<0) || (aus>5)), "wrong Output-Parameter (aus).")
  warning(e>0.5,"possibly IC does not exist.")
  A=matrix(T,dimX,dimX)*0
  b=matrix(T)*0
  ctrl=matrix(T)*0
  SAO=H'*(ginv(Q)*H)  ;   Cov.-matrix corresponding to "AO's"
  
  if (max(vec(abs(Sig)))>0)
      S=ginv(Sig)
  else
      S=0
  endif
  A0t=ginv(S+SAO)

;finding "good" starting values for A and b:


  switch
     case(dimX==1)
     switch
        case (A0<0)
                At=abs(A0)*A0t
        break
        case (A0==0)
                At=A0t
        break
        case (A0>0)
                At=A0
        break
        default
        break
     endsw
     break
   case (dimX>1) 
      if (dim(A0)==1)
          At=A0t*abs(A0)
      endif
      if (A0==0)
          At=A0t
      else
          At=A0
          Asvd=svd(At)
          error((sum(Asvd.l<0)>0)||(At<>At'), "A0 is no Lagrange A!")      
      endif
   break
   default
   break  
 endsw
  switch
      case (typ==0)
              Imax=1.6*sqrt(max(vec(abs(At))))
      break
      case (typ==1)
            Imax=1.6*sqrt(max(vec(abs( At*(SAO*At)) ) ) )
      break
      case (typ==2)
              Imax=1.6*sqrt(max(vec(abs( At*(ginv(Sig)*At)) ) ) )
      break
      default
      break
  endsw
  if (b0<=0)
       bt=Imax+(Imax<=0)
    else
       bt=b0
  endif    

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  ;start of the loop
  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
    
  i=0

  PreviousP = Sig                               ;set starting covariance matrix P_0|0 = Sigma
   
  del=eps+1
  while (((i<T)&&(del>eps))||(i<5))
      At
      A0t
      i=i+1
  ; Step of the Kalman recursion - the notation follows Brockwell&Davis
    
    pom = (F * (PreviousP * F') + R)              ;temporary variable, P_(t|t-1) in fact
    
    if (aus>0)
          "Zeitpunkt" i
    endif

      switch
          case (typ==0)  ; simultaneous clipping
              erg=ICerz(e,ginv(A0t),At,bt,N,eps,itmax,expl,fact0,aus)   
          break 
          case (typ==1)  ; separate clipping -- A
              erg=ICerzsep(e,SAO,ginv(pom),At,bt,N,eps,itmax,expl,fact0,aus)
          break 
          case (typ==2)  ; separate clipping -- IO
            erg=ICerzsep(e,ginv(pom),SAO,At,bt,N,eps,itmax,expl,fact0,aus)
          break 
          default
          break
    endsw
    ctrl[i]=erg.ctrl    
  
    if (ctrl[i]==0)
        bt=erg.b
        A[i,,]=reshape(erg.A,#(1,dimX,dimX))
        b[i]=erg.b
    else
        A[i,,]=reshape(At,#(1,dimX,dimX))
        b[i]=bt
    endif


    Delta = H * (pom * H') + Q
    DeltaInv = ginv(Delta)
    CurrentP = pom - (pom * H' )* DeltaInv * (H * pom)  
    
; Step of the loop 
    
    PreviousP = CurrentP 
    del=max(abs(vec(CurrentP-A0t)))
    A0t=CurrentP;  also Cov (IC)_id in the simultaneous case


; because of Sig t==1 has to be treated separately   
   
    if ((i==1)||(del>eps/1.5))
       At=A0t
       bt=-1  ; set to default
    else
       At=erg.A
    endif   
endo  
j=i
while (j<T)
   j=j+1
   A[j,,]=A[i,,]
endo
if (i<T)
     b[i+1:T]=b[i]
     ctrl[i+1:T]=ctrl[i]
endif

if (aus>0)
   A
   b
   ctrl
endif
warning(sum (ctrl)<>0, "rIC could not be calibrated for all t to the desired efficiency loss; try other control parameters or change e.")
endp
