proc(hrot)=denrotp(x, K, opt)
; -----------------------------------------------------------------
; Library        smoother
; -----------------------------------------------------------------
;  See_also      denbwsel denbwcrit
; -----------------------------------------------------------------
;   Keywords     kernel smoothing, kernel density estimation, 
;                bandwidth selection, Scott's rule of thumb
; -----------------------------------------------------------------
;   Macro        denrotp
; -----------------------------------------------------------------
;   Description  determines a rule-of-thumb bandwidth for
;                multivariate density estimation according to Scott.
; -----------------------------------------------------------------
;   Reference    Scott (1992): Multivariate Density Estimation:
;                Theory, Practice and Visualization
; -----------------------------------------------------------------
;   Usage        hrot = denrotp(x {,K {,opt}})
;   Input
;     Parameter  x  
;     Definition   n x 1 vector, the data.  
;     Parameter  K  
;     Definition   string, univariate kernel function on [-1,1] 
;                  or Gaussian kernel "gau". The rule-of-thumb is
;                  determined for the product kernel obtained from
;                  K. If K is not given, "qua" is used.
;     Parameter  opt  
;     Definition   string, if opt equals "robust", a robust estimate 
;                  for the standard deviations of the data is used.
;   Output
;     Parameter  hrot  
;     Definition   scalar, rule-of-thumb bandwidth.
; -----------------------------------------------------------------
;   Example   library("smoother")
;             x=read("nicfoo")
;             hrot=denrotp(x)
;             hrot
; -----------------------------------------------------------------
;   Result    Contents of hrot
;             [1,]  0.60341 
;             [2,]  0.41691 
; -----------------------------------------------------------------
;   Example   library("smoother")
;             x=read("nicfoo")
;             hrot=denrotp(x,"qua","robust")
;             hrot
; -----------------------------------------------------------------
;   Result    Contents of hrot
;             [1,]   0.6046 
;             [2,]  0.44921 
;-----------------------------------------------------------------
;   Author    Marlene Mueller, 990413
; ----------------------------------------------------------------- 
  if (exist(K)==0)
    K="qua"
  endif
  if (exist(opt)==0)
    opt="null"
  endif
;
  n = rows(x)
  p = cols(x)
;
  {tmp,cK,mK}=canbw(K)
  dK=((cK^p)/(mK^2))^(1/(p+4))
  {tmp,cG,mG}=canbw("gau")
  dG=((cG^p)/(mG^2))^(1/(p+4))
  fac=dK./dG
;
  if (opt=="robust")
    lambda = diff(quantile(x,0.25|0.75))
    qn     = diff(qfn(0.25|0.75))
    hrot  = fac.* (lambda'./qn) .* (n^(-1/(p+4)))
  else
    hrot  = fac.* sqrt(var(x)') .* (n^(-1/(p+4)))
  endif
endp   
