proc(dp)=dispdot(inX, inColorScheme, inFactor)
; -----------------------------------------------------------------------
; Library      graphic
; -----------------------------------------------------------------------
; See_also     dispbox plotdot
; -----------------------------------------------------------------------
; Macro        dispdot
; -----------------------------------------------------------------------
; Description  Generates a dotplot with tonal highlighting. An optional
;              factor allows multiple dotplots by group.
; -----------------------------------------------------------------------
; Usage        dp = dispdot(x , inColorScheme {, inFactor})
; Input
;   Parameter  x
;   Definition n x 1      vector (continuous variable)
;   Parameter  inColorScheme
;   Definition m x 3      vector (discrete) of rgb colors for the different 
;                         highlighting stages
;   Parameter  inFactor
;   Definition n x 1      vector (discrete variable)   
; Output
;   Parameter  dp
;   Definition composed graphical object
; ---------------------------------------------------------------------
; Notes        Displays up to 250 dashes representing the data. If some
;              dashes overlap, another color from inColorScheme is used.
;              The more dashes, the higher the index of the color. This
;              is called "tonal highlighting".
;
;              The colors are linearly assigned to the frequencies of 
;              overlappings.
;              If the maximal frequency of overlappings is less than
;              the number of colors provided, only this smaller number
;              of colors is used.
;
;              Obviously, if inColorScheme is of lenght one, tonal 
;              highlighting is off.
;
;              The output of grcolorscheme() is a valid input for 
;              inColorScheme.
;
;              Requires the library "stats"
;
;              IMPORTANT: The factor variable might be alphabetically reordered!
; -----------------------------------------------------------------------
; Example      library("graphic")
;              library("stats")
;              x = normal(4000)
;              m = matrix(1000)
;              factor = m | 2*m | 3*m | 4*m
;              bp1 = dispdot(x, grcolorscheme("greyscale"))
;              bp2 = dispdot(x, grcolorscheme("topographic"), factor)
;              dd = createdisplay(1, 2)
;              show(dd, 1, 1, bp1)
;              show(dd, 1, 2, bp2)
; -----------------------------------------------------------------------
; Result       Shows a dotplot of x on the left. On the right four
;              dotplots of the four groups within x defined by factor are
;              shown.
; -----------------------------------------------------------------------
; Keywords     graphic primitives, dotplot, tonal highlighting, color scheme
; -----------------------------------------------------------------------
; Author       Stephan Lauer 990224
; -----------------------------------------------------------------------
  
  // check if inX has multiple columns (forbidden)
  error(cols(inX).<>1, "dispdot : only one column of continuos data allowed")

  // check if inColorScheme has multiple columns (forbidden)
  error (cols(inColorScheme).<>3, "dispdot : color scheme must be a n x 3 vector of rgb colors")
  
  // some inits
      thePoints = (0 ~ 0) | (1 ~ 0) // this enforces the plot to be at least 1.0 wide!
      theLines = (0 ~ 1)
      thePointStyle = (0 | 0)
      thePointSize = (8 | 8)
      theLineStyle = 0
      thePointColor = 0 // all points have the same color!!! else initialize to (0|0) !!!
      theLineColor = (0 ~ 0 ~ 0)

      theMaxNumberOfColors = rows(inColorScheme)

  if (exist("inFactor") <> 1 && exist("inFactor") <> 2) // not numeric, not text
  // no factor present. Generate *one* dotplot
      
       // round inX to 250 Lines
          // first scale down inX to the interval [0,1]
          theMin = min(inX)
          theMax = max(inX)

          inX = (inX - theMin) ./ (theMax - theMin)
          
          // now truncate 
          inX = round(inX .* 250) ./ 250

          // generate groups
          {theValue, theNumber} = discrete(inX)
          
          // now rescale inX to its original scale and assign
          
          inX = theValue .* (theMax - theMin) + theMin
          
          theMaxFrequency = max(theNumber)
          if (theMaxFrequency < theMaxNumberOfColors)
              theMaxNumberOfColors = theMaxFrequency
          endif

       // generate lines for each observation
       theLength = rows(inX)
       theCurrentIndex = rows(thePoints)
       theAllOne = matrix(theLength)
       thePoints = thePoints | ( 0.25 .* theAllOne ~ inX)
       thePoints = thePoints | ( 0.75 .* theAllOne ~ inX)  
     
       // connect points
       theLines = theLines | ( theCurrentIndex + 1 : theCurrentIndex + theLength ~ theCurrentIndex + 1 + theLength : theCurrentIndex + 2 .* theLength  )
       
       // set styles
       thePointStyle = thePointStyle | 0 .* theAllOne | 0 .* theAllOne
       theLineStyle  = theLineStyle  | 1 .* theAllOne
       thePointSize  = thePointSize  | 8 .* theAllOne | 8 .* theAllOne
           // now here is the tonal highlighting:
                // first assign colors according to the group sizes
                // if group too large, assign maximal color given
       theNumberTooLarge = theNumber > theMaxNumberOfColors
       theColorIndex = ( floor((theNumber-0.5)/theMaxFrequency .* theMaxNumberOfColors) + 1 ) .* (!theNumberTooLarge) + theNumberTooLarge .* theMaxNumberOfColors
                // now assign the corresponding colors
       theLineColor  = theLineColor | (inColorScheme[theColorIndex])
       

  else //  factor present
       // split inX by inFactor and plot by group
  
      error(cols(inFactor).<>1, "dispbox : only one column of discrete data allowed")
      error(rows(inX).<>rows(inFactor), "dispbox : factor must have same length as data")
  
  
      theGroups = discrete(inFactor)

      theNumberOfGroups = rows(theGroups)

      
      theCount = 1

      while (theCount <= theNumberOfGroups)

           // extract one group
           theData = paf(inX, inFactor==theGroups[theCount,1])

               // round theData to 250 Lines
                  // first scale down inX to the interval [0,1]
               theMin = min(theData)
               theMax = max(theData)

               theData = (theData - theMin) ./ (theMax - theMin)
          
               // now truncate 
               theData = round(theData .* 250) ./ 250

               // generate groups
               {theValue, theNumber} = discrete(theData)
          
               // now rescale theData to its original scale and assign
          
               theData = theValue .* (theMax - theMin) + theMin
               
               // if the maximum of frequencies is less than the 
               // number of colors provided, use less colors.
               theMaxFrequency = max(theNumber)
               if (theMaxFrequency < theMaxNumberOfColors)
                   theMaxNumberOfColors = theMaxFrequency
               endif          

            // generate lines for each observation
            theLength = rows(theData)
            theCurrentIndex = rows(thePoints)
            theAllOne = matrix(theLength)
            thePoints = thePoints | ( ((theCount - 1) * 1.5 + 0.25) .* theAllOne ~ theData)
            thePoints = thePoints | ( ((theCount - 1) * 1.5 + 0.75) .* theAllOne ~ theData)  
     
            // connect points
            theLines = theLines | ( theCurrentIndex + 1 : theCurrentIndex + theLength ~ theCurrentIndex + 1 + theLength : theCurrentIndex + 2 .* theLength  )
       
            // set styles
            thePointStyle = thePointStyle | 0 .* theAllOne | 0 .* theAllOne
            theLineStyle  = theLineStyle  | 1 .* theAllOne
            thePointSize  = thePointSize  | 8 .* theAllOne | 8 .* theAllOne
                // now here is the tonal highlighting:
                     // first assign colors according to the group sizes
                     // if group too large, assign maximal color given
            theNumberTooLarge = theNumber > theMaxNumberOfColors
            theColorIndex = ( floor((theNumber-0.5)./theMaxFrequency .* theMaxNumberOfColors) + 1 ) .* (!theNumberTooLarge) + theNumberTooLarge .* theMaxNumberOfColors
                     // now assign the corresponding colors
            theLineColor  = theLineColor | (inColorScheme[theColorIndex])

           
           theCount = theCount + 1
      endo // while more groups left  
   endif // factor present or not

   

   // assign colors in ColorScheme
   freecolor()
   createcolor(inColorScheme)   

   // set the setmask{tpl} values

   setmaskp(thePoints, thePointColor, thePointStyle, thePointSize)
   setmaskl(thePoints, theLines, theLineColor, theLineStyle, 1) 

   // set return value
   dp = thePoints
endp