proc(mux,beta1,beta0,sigmax,sigmau,sigmae,varbeta1,varbeta0)=eivknownratue(w,y,delta)
; -----------------------------------------------------------------
;    Library	eiv
; -----------------------------------------------------------------
;    See_also eivknownatt eivknownvaru
; -----------------------------------------------------------------
;    Macro	eivknownratue
; -----------------------------------------------------------------
;    Description    eivknownratue presents the moment estimates of the 
;                   parameters in the measurement error models, which 
;                   has only one variable x. The ratio of two variances 
;                   of the two measurement errors is known. All of the
;	            	    variables obey normal distributions.
; ----------------------------------------------------------------------------
;   Reference	      Fuller W. A. "Measurement Error Models", Wiley and Sons
;                   1987, section 1.3.
; -----------------------------------------------------------------
;    Link         ../tutorials/eivstart.html Tutorial: EIV in XploRe 
; -----------------------------------------------------------------
;    Usage    {mux,beta1,beta0,sigmax,sigmau,sigmae,varbeta1,varbeta0} = eivknownratue(w,y,delta)
;    Input
;      Parameter  w
;      Definition   n x 1 matrix, the design variables
;      Parameter  y
;      Definition   n x 1 matrix, the response
;      Parameter  delta
;      Definition   scalar, the ratio of the variance of two errors
;    Output
;      Parameter  mux
;      Definition   scalar, the mean value of x
;      Parameter  beta1
;      Definition   scalar, the estimate
;      Parameter beta0
;      Definition  scalar, the estimate
;      Parameter sigmax
;      Definition  scalar, the estimate of the variance of x
;      Parameter sigmau
;      Definition  scalar, the estimate of the variance of u
;      Parameter sigmae
;      Definition  scalar, the estimate of the variance of e
;      Parameter  varbeta1   
;      Definition  scalar, the estimate of the variance of the estimate of beta1
;      Parameter  varbeta0   
;      Definition  scalar, the estimate of the variance of the estimate of beta0
; -----------------------------------------------------------------
;  Example library("eiv")
;	   n = 100
;          randomize(n) 
;          x=9*normal(n)
;          w=x+9*normal(n)
;          y=0.9+0.8*x+0.01*normal(n)
;	   delta =0.0001/81  ; The ratio of var(u)/var(e)
;	   gest=eivknownratue(w,y,delta)
;	   gest.mux
;	   gest.beta1
;	   gest.beta0
;	   gest.sigmax
;	   gest.sigmau
;          gest.sigmae
;          gest.varbeta1
;          gest.varbeta0
; -----------------------------------------------------------------
;   Result Contents of mux
;          [1,] -0.93396 
;          Contents of beta1
;          [1,]  0.92042 
;          Contents of beta0
;          [1,]  0.44547
;          Contents of sigmax
;          [1,]   62.555 
;          Contents of sigmau
;          [1,]   82.616 
;          Contents of sigmae
;          [1,]  0.00010199 
;          Contents of varbeta1
;          [1,]   0.011569
;          Contents of varbeta0
;          [1,]   0.71714
; -----------------------------------------------------------------
;   Author    Hua Liang,  990625
; -----------------------------------------------------------------
     n=rows(w)
     mux=mean(w)
     mxx=var(w)
     mxy=sum((w-mean(w)).*(y-mean(y)))/(rows(w)-1)
     myy=var(y)
     beta1=inv(2*mxy)*(myy-delta*mxx+sqrt((myy-delta*mxx)^2+4*delta*mxy^2))
     beta0=mean(y)-beta1*mean(w)
     sigmax=mxy*inv(beta1)
     sigmau=mxx-sigmax
     sigmae=delta*sigmau
     svv=inv(n-2)*sum((y-mean(y)-beta1*(w-mean(w))).^2)
     sigmauv=-beta1*sigmau 
     varbeta1=inv(n-1)*sigmax^(-2)*(sigmax*svv+sigmau*svv-sigmauv^2)
     varbeta0=inv(n)*svv+(mean(w))^2*varbeta1
endp
