proc(b,bv,m,mg,stat)=eivplmnor(x,t,y,sigma,h,opt)
; ----------------------------------------------------------------------------
; Library	eiv
; ----------------------------------------------------------------------------
;  See_also	gplmest gplmopt
; ----------------------------------------------------------------------------
;   Macro	eivplmnor
; ----------------------------------------------------------------------------
;   Description  eivplmnor fits partially linear EIV model where the 
;		 conditional distribution of y given x and t is normally 
;		 distributed. 
; ----------------------------------------------------------------------------
;   Reference	Severini/Staniswalis, JASA, 1994; Liang, H., Haerdle, W. and
;		Carroll, R. 1997 SFB DP27
; ----------------------------------------------------------------------------
;    Link         ../tutorials/eivstart.html Tutorial: EIV in XploRe 
; -----------------------------------------------------------------
;   Usage	 myfit = eivplmnor(x,t,y,sigma,h{,opt})
;   Input
;     Parameter   x
;     Definition	n x p  matrix, the discrete predictor variables.
;     Parameter   t
;     Definition	n x q  matrix, the continuous predictor variables.
;     Parameter   y
;     Definition	n x 1  vector, the response variables.
;     Parameter   sigma
;     Definition	scalar, the variance of the measurement error.
;     Parameter   h
;     Definition	q x 1  vector, the bandwidth.
;     Parameter   opt
;     Definition	optional, a list with optional input. The macro
;			"gplmopt" can be used to set up this parameter.
;			The order of the list elements is not important.
;			Parameters which are not given are replaced by
;			defaults (see below).
;     Parameter   opt.b0
;     Definition	p x 1  vector, the initial coefficients. If not
;			given, all coefficients are put =0 originally.
;     Parameter   opt.wx
;     Definition	scalar or n x 1 vector, prior weights. If not
;			given, set to 1.
;     Parameter   opt.wt
;     Definition	n x 1  vector, weights for t (trimming factors).
;			If not given, all set to 1.
;     Parameter   opt.tg
;     Definition	ng x 1 vector, a grid for continuous part. If tg is
;			given, the nonparametric function will also be
;			computed on this grid.
;     Parameter   opt.shf
;     Definition	integer, if exists and =1, some output is produced
;			which indicates how the iteration is going on.
;     Parameter   opt.nosort
;     Definition	integer, if exists and =1, the continuous variables
;			t and the grid tg are assumed to be sorted by the
;			1st column. Sorting is required by the algorithm!
;			Hence this option should be given only when data
;			are sorted.
;     Parameter   opt.miter
;     Definition	integer, maximal number of iterations. The default
;			is 10.
;     Parameter   opt.cnv
;     Definition	integer, convergence criterion. The default is 0.0001.
;     Parameter   opt.wtc
;     Definition	n x 1  vector, weights for convergence criterion,
;			w.r.t. m(t) only. If not given, opt.wt is used.
;     Parameter   opt.off
;     Definition	scalar or n x 1 vector, offset in predictor.
;   Output
;     Parameter   myfit.b
;     Definition	k x 1  vector, estimated coefficients.
;     Parameter   myfit.bv
;     Definition	k x k  matrix, estimated covariance matrix for
;			coefficients.
;     Parameter   myfit.m
;     Definition	n x 1  vector, estimated nonparametric part.
;     Parameter   myfit.mg
;     Definition	ng x 1 vector, estimated nonparametric part on grid
;			if tg was given. This component will not exist, if
;			tg was not given.
;     Parameter   myfit.stat
;     Definition	list with the following statistics:
;     Parameter   myfit.stat.deviance
;     Definition	deviance,
;     Parameter   myfit.stat.pearson
;     Definition	generalized pearson's chi^2,
;     Parameter   myfit.stat.r2
;     Definition	pseude R^2,
;     Parameter   myfit.stat.dispersion
;     Definition	dispersion parameter estimate,
;     Parameter   myfit.stat.it
;     Definition	scalar, number of iterations needed.
; ----------------------------------------------------------------------------
;   Example   library("eiv")
;             library("xplore")
;             n=100
;             randomize(n)
;	      sigma=0.81
;             b=1|2
;	      p=rows(b)
;             x=2.*uniform(n,p)-1 
;             t=sort(2.*uniform(n)-1,1)
;             w=x+sqrt(sigma)*uniform(n) 
;             m=0.5*cos(pi.*t)+0.5*t 
;             y=x*b+m+normal(n)./2 
;             h=0.5
;	      sf=eivplmnor(w,t,y,sigma,h)
;	      b~sf.b
;	      dds=createdisplay(1,1)
;	      show(dds,1,1,t~m,t~sf.m)
; ----------------------------------------------------------------------------
;   Result    A partially linear fit for E[y|x,t] is computed.
;	      sf.b contains the coefficients for the linear part.
;	      sf.m contains the estimated nonparametric part
;	      evaluated at observations t. The example gives the
;	      true b together with the EIVPLM estimate sf.b. Also, the
;	      estimated function sf.m is displayed.
; ----------------------------------------------------------------------------
;   Author    Hua Liang, 970828
; ----------------------------------------------------------------------------
; check initial errors
  bad=(rows(dim(x))>2)+(rows(dim(t))>2)
  error(bad>0,"x and t must be vectors or matrices")
  error(rows(x)!=rows(y),"x and y have different number of rows")
  error(rows(x)!=rows(t),"x and t have different number of rows")
  error((rows(dim(y))>1),"y must be a vector")
  n=rows(x)
  p=cols(x)
  q=cols(t)
  bad=(rows(h)!=q)*(rows(h)!=1)+(rows(dim(h))>1)
  error(bad>0,"bandwidth h has wrong dimension")
  if (rows(h)==1)
    h=h.*matrix(q)
  endif
; set the defaults
   b0=0.*matrix(p)
    wx=matrix(n)
    wt=1
    wtc=1
    havegrid=0
    shf=0
    nosort=0
    miter=10
    cnv=0.0001
    off=0
; now check which optional values have been given
  if (exist(opt)>0)
    if (comp(opt,"b0")>0)
      notgood=(rows(opt.b0)!=p)+(rows(dim(opt.b0))!=1)
      warning (notgood>0, "opt.b0 not consistent with x, used zeros instead")
      if (notgood==0)
	b0=opt.b0
      endif
    endif
    if (comp(opt,"wx")>0)
      notgood=(dim(dim(opt.wx))!=1)*(rows(opt.wx)!=n)+(rows(dim(opt.wx))!=1)
      notgood=notgood+(sum(abs(wx))==0)+(min(wx)<0)
      warning(notgood>0, "opt.wx was unusable, used opt.wx=1 instead")
      if (notgood==0)
	wx=opt.wx
      endif
    endif
    if (comp(opt,"off")>0)
      notgood=(dim(dim(opt.off))!=1)*(rows(opt.off)!=n)+(rows(dim(opt.off))!=1)
      warning(notgood>0, "opt.off not consistent with x, used off=1 instead")
      if (notgood==0)
	off=opt.off
      endif
    endif
    if (comp(opt,"wt")>0)
      notgood=(dim(dim(opt.wt))!=1)*(rows(opt.wt)!=n)+(rows(dim(opt.wt))!=1)
      warning(notgood>0, "opt.wt not consistent with t, used wt=1 instead")
      if (notgood==0)
	wt=opt.wt
      endif
    endif
    if (comp(opt,"wtc")>0)
      notgood=(dim(dim(opt.wtc))!=1)*(rows(opt.wtc)!=n)+(rows(dim(opt.wtc))!=1)
      warning(notgood>0, "opt.wtc not consistent with t, used wtc=1 instead")
      if (notgood==0)
	wtc=opt.wtc
      endif
    endif
    if (comp(opt,"tg")>0)
      notgood=(cols(opt.tg)!=q)+(rows(dim(opt.tg))>2)
      warning(notgood>0, "opt.tg not consistent with t, was not used")
      if (notgood==0)
	havegrid=1
	tg=opt.tg
      endif
    endif
    if (comp(opt,"shf")>0)
      shf=(opt.shf==1)*(dim(dim(opt.shf))==1)
    endif
    if (comp(opt,"nosort")>0)
      nosort=(opt.nosort==1)*(dim(dim(opt.nosort))==1)
    endif
    if (comp(opt,"miter")>0)
      miter=floor(opt.miter[1,1])
      notgood=(miter<1)+(dim(dim(opt.miter))!=1)
      warning (notgood>0, "opt.miter was unusable, used default =10 instead")
      miter=notgood*10+(1-notgood)*miter
    endif
    if (comp(opt,"cnv")>0)
      tmp=opt.cnv[1,1]
      notgood=(tmp<=0)+(dim(dim(opt.cnv))!=1)
      warning (notgood>0, "opt.cnv was unusable, used default =0.0001 instead")
      cnv=notgood*cnv+(1-notgood)*tmp
    endif
  endif
  error (countNaN(inv(x'*x))>0, " !! Matrix x is singular !! ")
  if (havegrid==1)
    ng=rows(tg)
  endif
; we need to sort data by t, required by sker!
  if (nosort!=1)
    to=sort(t[,1]~(1:n))[,2]
    tro=sort(to~(1:n))[,2]
    x=x[to]
    y=y[to]
    t=t[to]
    if (rows(wt)==n)
      wt=wt[to]
    endif
    if (rows(wtc)==n)
      wtc=wtc[to]
    endif
    if (rows(wx)==n)
      wx=wx[to]
    endif
    if (rows(off)==n)
      off=off[to]
    endif
    if (havegrid==1)
      tgo=sort(tg[,1]~(1:ng))[,2]
      tgro=sort(tgo~(1:ng))[,2]
      tg=tg[tgo]
    endif
  endif
  m0=0.*matrix(n)
  it=1
  crit=0
  one=matrix(n)
  do
    if (shf==1)
      "'eivplmnor': iteration no."+string("%1.0f", it)
    endif
    xb=x*b0
    tmp=sker(t,h',"qua",wx.*(one~(y-xb-off)~x))
    m=tmp[,2]./tmp[,1]
    xnew=x-tmp[,3:cols(tmp)]./tmp[,1]
    if (havegrid==1)  ; only if grid exists
      tmp=sker(t,h',"qua",wx.*(one~(y-xb-off)),tg)
      mg=tmp[,2]./tmp[,1]
    endif
    B = xnew'*(wt.*xnew)
    bv=inv(B-0.5*n*sigma)
    b=bv*xnew'*(wt.*(y-m))
    if (it>1)
      db=b-b0
      chgb=sqrt( sum(db.*db)./sum(b0.*b0) )
      dm=wx.*(m-m0)
      if (exist(wtc)==1)
	dm=dm.*wtc
	m0=wx.*m0.*wtc
      else
	dm=dm.*wt
	m0=wx.*m0.*wt
      endif
      chgm=sqrt( sum(dm.*dm)./sum(m0.*m0) )
      crit=((chgb<cnv)+(chgm<cnv)>1)
    endif
    it=it+1
    b0=b
    m0=m
  until (crit+(it>miter))
  mu	 =  x*b+m+off
  deviance  = sum((y-mu)^2)
  meany  = mean(y)
  r2	 = sum(wx.*(mu-meany)^2)/sum(wx.*(y-meany)^2)
  pearson = deviance
  it  =it-1
  dispersion = deviance/n
  stat	= list(deviance,pearson,r2,dispersion,it)
  if (nosort!=1)
    m=m[tro]
    if (havegrid==1)
      mg=mg[tgro]
    endif
  endif
endp
