proc()=gamfit(t,y,opt)
; ----------------------------------------------------------------------------
;   Library      gam
; ----------------------------------------------------------------------------
;   See_also     intest, intestpl, gintest, gintestpl, gamopt, gamout
; ----------------------------------------------------------------------------
;   Macro        gamfit
; ----------------------------------------------------------------------------
;   Description  gamfit provides an interactive tool for fitting
;                additive models
; ---------------------------------------------------------------------------
;   Link         ../tutorials/gamstart.html  Generalized Additive Models
; ----------------------------------------------------------------------------
;   Usage        gamfit(t,y{,opt})
;   Input
;     Parameter   t
;     Definition        n x p  matrix, the predictor variables.
;     Parameter   y
;     Definition        n x 1 vector , the observed
;                       response variables
;     Parameter   opt
;     Definition        optional, a list with optional input. The macro
;                       "gamopt" can be used to set up this parameter.
;                       The order of the list elements is not important.
;                       Parameters which are not given are replaced by
;                       deflts (see below).
;     Parameter   opt.x
;     Definition        n x d  matrix, the discrete predictor variables.
;     Parameter   opt.h
;     Definition        p x 1 or 1 x 1 matrix , chosen bandwidth for
;                       the directions of interest
;     Parameter   opt.g
;     Definition        p x 1 or 1 x 1 matrix , chosen bandwidth for
;                       the directions not of interest
;     Parameter   opt.loc
;     Definition        dummy , for loc=0 local constant (Nad. Wats.),
;                       for loc=1 local linear and for loc=2 local
;                       quadratic estimator will be used
;     Parameter   opt.kern
;     Definition        string, specifying the kernel function for backfitting
;                       algorithm
;     Parameter   opt.tg
;     Definition        ng x pg vector, a grid for continuous part. If tg is
;                       given, the nonparametric function will be
;                       computed on this grid.
;     Parameter   opt.shf
;     Definition        integer,  (show-how-far) if exists and =1, an output
;                       is produced which indicates how the iteration
;                       is going on (additive function / point of estimation /
;                       number of iteration).
;     Parameter   opt.code
;     Definition        text string, the short code for the model (e.g.
;                       "bilo" for logit or "noid" for ordinary PLM).
;     Parameter   opt.wx
;     Definition        scalar or n x 1 vector, prior weights. If not given,
;                       all weights are set to 1.
;     Parameter   opt.off
;     Definition        scalar or n x 1 vector, offset in linear predictor.
;                       If not given, set to 0.
;     Parameter   opt.name
;     Definition        string, prefix for the output. If not given, "gam"
;                       is used.
;     Parameter   opt.xvars
;     Definition        p x 1 string vector, variable names for the output.
;                       Note, that only up to 11 characters are used.
;   Output
;     Parameter   gampic  or  opt.name+"pic"
;     Definition        display, containing estimation result
;     Parameter   gamfit
;     Definition        list object, which is made
;                       globally available
;     Parameter   gamfit.m
;     Definition        estimator for the non-linear part
;     Parameter   gamfit.b
;     Definition        p x 1  vector, estimated coefficients of the linear part
;     Parameter   gamfit.bv
;     Definition        p x p  matrix, estimated covariance matrix for b.
;     Parameter   gamfit.const
;     Definition        scalar, estimated constant.
;     Parameter   gamfit.opt
;     Definition        list, internally used option list.
; ----------------------------------------------------------------------------
;   Example   library("gam")
;             randomize(1234)
;             t     = uniform(50,2)*2-1
;             g1    = 2*t[,1]
;             g2    = t[,2]^2
;             g2    = g2 - mean(g2)
;             y     = g1 + g2  + normal(50,1) * sqrt(0.25)
;             gamfit(t,y)
; ----------------------------------------------------------------------------
;   Result    Gamfit allows to choose interactively the model and
;             the estimation procedure and inquires all further needed
;             options and parameters. It runs the non-interactive estimation
;             macro and provides a beautyful output at the end.
; ----------------------------------------------------------------------------
;   Author    Stockmeyer, Sperlich 970819
; ----------------------------------------------------------------------------
;
; check for initial errors
; -------------------------------------------------------------------
  error(rows(dim(t))>2,"t must be vector or matrix")
  error(cols(y)>1,"y must be a vector")
  error(rows(t)!=rows(y),"t and y have different number of rows")
  n = rows(t)
  p = cols(t)
  name = "gam"
  xvars = " "
  backfit = 0
  integration = 1
  havexvars = 0
  generalized = 0
  partlinear = 0
  nocancel = 1
  family = 1
  model = ""
  needcode = 0
  glmmodels = getglobal("glmmodels")
;
; check for option list
; --------------------------------------------------------------------
  if (exist(opt)==0)
    dummy=0
    opt=list(dummy)  ; dummy optional parameter, doesn't matter
  endif
;
; select model
; --------------------------------------------------------------------
  head = "Select additive model"
  item =        " Additive model"
  item = item | " Additive partially linear model"
  item = item | " Generalized additive model"
  item = item | " Generalized additive part. linear model"
  item = item | " Break"
  model = selectitem(head,item,"single")'*(1:rows(item))
  switch
    case ((model==0)||(model==rows(item)))
      nocancel = 0
      model = ""
      break
    case (model==2)
      model = "plam"
      partlinear = 1
      break
    case (model==3)
      model = "gam"
      generalized = 1
      break
    case (model==4)
      model = "gplam"
      generalized = 1
      partlinear = 1
      break
    default
      model = "am"
  endsw
  opt = gamopt("pl",partlinear,opt)
;
; select estimation procedure
; --------------------------------------------------------------------
  switch
  case (nocancel==0)
    break
  case (generalized)
    backfit = 0
    integration = 1
    break
  default
    head = "Select estimation procedure"
    item = " Estimation by integration"
    item = item | " Backfitting algorithm"
    item = item | " Break"
    estim = selectitem(head,item,"single")'*(1:rows(item))
    switch
      case ((estim==0)||(estim==rows(item)))
        nocancel = 0
        break
      case (estim==2)
        backfit = 1
        integration = 0
        break
      default
        backfit = 0
        integration = 1
    endsw
  endsw
;
; check for discrete variables
; --------------------------------------------------------------------
  if ((nocancel)&&(partlinear))
    needx = 1
    if (comp("opt","x")>0)
      x = opt.x
    endif
    if (exist(x)>0)
      if (rows(x)==n)
        needx = 0
      endif
    endif
    while (needx)
      head = " Discrete variables have not been given or correctly specified"
      item = " Run the estimation without discrete variables, on t only ?"
      item = item|" Give the name of the discrete variable x ?"
      item = item|" Break"
      dis = selectitem(head,item,"single")'*(1:3)
      switch
        case ((dis==3)||(dis==0))
          nocancel = 0
          needx = 0
          break
        case (dis==1)
          partlinear = 0
          opt = gamopt("pl",partlinear,opt)
          needx = 0
          if (generalized)
            model = "gam"
          else
            model = "am"
          endif
          break
        case (dis==2)
          xname = readvalue(" Give the name of the discrete x variable","")
          if (existglobal(xname))
            x = getglobal(xname)
          endif
          if (exist(x)>0)
            if (rows(x)==n)
              needx = 0
              opt = gamopt("x",x,opt)
            endif
          endif
          break
      endsw
    endo
  endif
;
; do you want to estimate derivatives ?
; ----------------------------------------------------------------------
  loc = NaN
  needloc = 0
  if ((integration)&&(nocancel))
    needloc = 1
    if (comp("opt","loc")>0)
      loc = opt.loc
    endif
    if ((model!="plam")&&(sum(loc==#(0,1,2))==1))
      needloc = 0
    endif
    if ((model=="plam")&&(sum(loc==#(1,2))==1))
      needloc = 0
    endif
    if (model=="gplam")
      needloc = 0
      loc = 0
      opt = gamopt("loc",loc,opt)
    endif
  endif
  if ((backfit)&&(nocancel))
    needloc = 1
    if (comp("opt","loc")>0)
      loc = opt.loc
    endif
  endif
    if (needloc)
      head = " Up to which derivative the estimation shall be run ?"
      if (model=="plam")
        item = " Function and 1. derivative (local linear)"
      else
        item = " Function (local constant)"
        item = item | " Function and 1. derivative (local linear)"
      endif
      item = item | " Function, 1. and 2. derivative (local quadratic)"
      item = item | " Break"
      loc = selectitem(head,item,"single")'*(1:rows(item))
      switch
        case ((loc==rows(item))||(loc==0))
          nocancel = 0
          break
        case (model=="plam")
          opt = gamopt("loc",loc,opt)
          break
        default
          loc = loc - 1
          opt = gamopt("loc",loc,opt)
      endsw
    endif
;
; check for grid
; ----------------------------------------------------------------------
  havegrid = 0
  if ((nocancel)&&(integration))
    if (comp("opt","tg")>0)
      tg = opt.tg
      ng = rows(tg)
      pg = cols(tg)
      havegrid = 1
    else
      havegrid = 0
    endif
  endif
;
; check for bandwidth h
; ----------------------------------------------------------------------
  if (nocancel)
    needh = 1
    if (comp("opt","h")>0)
      h = opt.h
    endif
    if (exist(h)>0)
      switch
      case ((partlinear==0)&&(havegrid)&&(rows(h)!=1))
        if ((rows(h)==p)||(rows(h)==pg))
          needh = 0
        endif
        break
      case (rows(h)==1)
        needh = 0
        h = matrix(p,1) * h
        opt = gamopt("h",h,opt)
        break
      case (rows(h)==p)
        needh = 0
        break
      endsw
    endif
    if (needh)
      head = "Select Bandwidth h"
      if (p>1)
        head = head|string("",matrix(p-1))
      endif
      h = readvalue(head,matrix(p))
    endif
  endif
;
; check for bandwidth g
; ------------------------------------------------------------------
  if ((nocancel)&&(integration))
    needg = 1
    if (comp("opt","g")>0)
      g = opt.g
    endif
    if (exist(g)>0)
      if (rows(g)==p)
        needg = 0
      endif
      if (rows(g)==1)
        needg = 0
        g = matrix(p,1) * g
        opt = gamopt("g",g,opt)
      endif
    endif
    if (needg)
      head = "Select Bandwidth g"
      if (p>1)
        head = head|string("",matrix(p-1))
      endif
      g = readvalue(head,matrix(p))
    endif
  endif
;
; select kernel
; -------------------------------------------------------------------
  if ((backfit)&&(nocancel))
    if (comp("opt","kern")>0)
      kern = opt.kern
    else
      kernnames = "qua"|"epa"|"gau"
      head = "Select kernel function"
      item = " Quartic kernel"
      item = item | " Epanechnikov kernel"
      item = item | " Gaussian kernel"
      item = item | " Break"
      kern = selectitem(head,item,"single")
      if ((kern[rows(item)])||sum(kern)==0)
        nocancel = 0
      else
        kern = paf(kernnames,kern[1:rows(item)-1])
      endif
    endif
  endif
;
; select distribution of y
; ----------------------------------------------------------------------
  if ((generalized)&&(nocancel))
    code = ""
    needcode = 1
    if (comp("opt","code")>0)
      code = opt.code
    endif
    if (sum(glmmodels.all==code)>0)
      needcode = 0
    endif
    if (needcode)
      familynames = "normal"|"binomial"
      if (model=="gplam")
        familynames = familynames|"poisson"|"negbinomial"
        familynames = familynames|"gamma"|"invgauss"
      endif
      head = "Select exponential family for Y"
      item = " Normal             (Y : any real value)"
      item = item | " Binomial           (Y : 0,1 or 0,..,m)"
      if (model=="gplam")
        item = item|" Poisson            (Y : non-negative integer)"
        item = item|" Negative Binomial  (Y : non-negative integer)"
        item = item|" Gamma              (Y : positive)"
        item = item|" Inverse Gaussian   (Y : positive)"
      endif
      item = item | " Break"
      family = selectitem(head,item,"single")
      if ((family[rows(item)])||sum(family)==0)
        nocancel = 0
      else
        family = paf(familynames,family[1:rows(item)-1])
      endif
    endif
  endif
;
; select the link function
; ---------------------------------------------------------------------
  if ((nocancel)&&(needcode))
    head = "Select a link function"
    switch
    case (family=="normal")
      item = " Identity (canonical)"
      item = item | " Power link"
      codenames = "noid"|"nopow"
      break
    case (family=="binomial")
      item = " Logit (canonical)"
      item = item | " Probit"
      codenames = "bilo"|"bipro"
      break
    case (family=="poisson")
      item = " Logarithmic (canonical,default)"
      item = item | " Power link"
      codenames = "polog"|"popow"
      break
    case (family=="negbinomial")
      item = " Canonical (default)"
      item = item | " Power link"
      codenames = "nbcl"|"nbpow"
      break
    case (family=="gamma")
      item = " Reciprocal (canonical,default)"
      item = item | " Power link"
      codenames = "gacl"|"gapow"
      break
    case (family=="invgauss")
      item = " Squared reciprocal (canonical,default)"
      item = item | " Power link"
      codenames = "igcl"|"igpow"
      break
    endsw
    item = item | " Break"
    cancel = rows(item)
    code = selectitem(head,item,"single")
    if (code[cancel])
      nocancel = 0
    else
      code = paf(codenames,code[1:rows(item)-1])
    endif
  endif
;
; list default values
; ----------------------------------------------------------------------
  if (nocancel)
    askselect = 1
    askdesstat = 1
    asktransfy = 0
    if (sum(y>0)==n)
      asktransfy = 1
    endif
    asktransft = 0
    if (sum(sum(t>0)==n,2)>0)
      asktransft = 1
    endif
    asktransf = 0
    if ((asktransfy)||(asktransft))
      asktransf = 1
    endif
    askgrid = 0
    if ((havegrid==0)&&(integration))
      askgrid = 1
    endif
    askshf = 1
    if (comp("opt","shf")>0)
      if (opt.shf)
        askshf = 0
      endif
    endif
    askwx = 1
    if ((comp("opt","wx")>0)||(model!="gplam"))
      askwx = 0
    endif
    askwt = 1
    if ((comp("opt","wt")>0)||(model!="gplam"))
      askwt = 0
    endif
    askwtc = 1
    if ((comp("opt","wtc")>0)||(model!="gplam"))
      askwtc = 0
    endif
    askweigths = 0
    if ((askwx)||(askwt)||(askwtc))
      askweigths = 1
    endif
    asktvars = 1
    if (comp("opt","tvars")>0)
      asktvars = 0
    endif
    askyvars = 1
    if (comp("opt","yvars")>0)
      askyvars = 0
    endif
    askxvars = 1
    if ((comp("opt","xvars")>0)||(partlinear==0))
      askxvars = 0
    endif
    askvars = 0
    if (asktvars||askxvars||askyvars)
      askvars = 1
    endif
    asknopic = 1
    if (comp("opt","nopic")>0)
      if (opt.nopic)
        asknopic = 0
      endif
    endif
    askname = 1
    if (comp("opt","name")>0)
      askname = 0
    endif
    asktitle = 1
    if (comp("opt","title")>0)
      asktitle = 0
    endif
  endif
;
; change default values
; --------------------------------------------------------------------
  if (nocancel)
    sel = 1
    cancel = ""
    while ((sel!=0)&&(nocancel))
      item = ""
      sellist = list("")
      if (askselect)
        item = item|" Select variables ?"
        append(sellist,askselect)
      endif
      if (asktransf)
        item = item|" Logarithmic transformation of variables ?"
        append(sellist,asktransf)
      endif
      if (askvars)
        item = item|" Name the variables ?"
        append(sellist,askvars)
      endif
      if (askgrid)
        item = item|" Use a grid ?"
        append(sellist,askgrid)
      endif
      if (askshf)
        item = item|" Show how the procedure is going on ?"
        append(sellist,askshf)
      endif
      if (askweigths)
        item = item|" Give weigths ?"
        append(sellist,askweigths)
      endif
      if (asknopic)
        item = item|" Don't show output graph ?"
        append(sellist,asknopic)
      endif
      if (askdesstat)
        item = item|" Show descriptive statistics ?"
        append(sellist,askdesstat)
      endif
      if ((askname)&&(asknopic))
        item = item|" Name of output ?"
        append(sellist,askname)
      endif
      if ((asktitle)&&(asknopic))
        item = item|" Title the output windows ?"
        append(sellist,asktitle)
      endif
      item = item| " Break"
      append(sellist,cancel)
      delete(sellist,1)
      item = item[2:rows(item)]
      if (rows(item)>1)
        sel = selectitem("",item,"single")'*(1:rows(item))
      endif
      switch
        case (sel==0)
          break
        case (comp("sellist","cancel")==sel)
          nocancel = 0
          break
        case (comp("sellist","askgrid")==sel)
          gridstr = " Contruct an equispaced grid with number of gridpoints:"
          gridstr = " Give the name of the grid variable:"|gridstr
          grid = readvalue(gridstr,"gridname"|"0")
          if ((grid[1]!="gridname")&&(grid[2]=="0"))
            if (existglobal(grid[1]))
              tg = getglobal(grid[1])
              ng = rows(tg)
              pg = cols(tg)
              opt = gamopt("tg",tg,opt)
              havegrid = 1
              askgrid = 0
            endif
          endif
          if ((grid[1]=="gridname")&&(grid[2]!="0"))
            nogp = atof(grid[2])
            if (isNumber(nogp))
              if (nogp>0)
                tg = min(t)+(0:nogp-1)*((max(t)-min(t))/(nogp-1))
                tg[nogp] = max(t)
                ng = rows(tg)
                pg = cols(tg)
                opt = gamopt("tg",tg,opt)
                havegrid = 1
                askgrid = 0
              endif
            endif
          endif
        case (comp("sellist","askshf")==sel)
          opt = gamopt("shf",1,opt)
          askshf = 0
        case (comp("sellist","askweigths")==sel)
          str = ""
          if (askwx)
            str = str|" Name the prior weigths variable"
          endif
          if (askwt)
            str = str|" Name the trimming weigths variable"
          endif
          if (askwtc)
            str = str|" Name the conversion criterion weigths variable"
          endif
          str = str[2:rows(str)]
          weigths = readvalue(str,string("",1:rows(str)))
          if (askwx)
            if (existglobal(weigths[1]))
              wx = getglobal(weigths[1])
              opt = gamopt("wx",wx,opt)
            endif
          endif
          if (askwt)
            if (askwx)
              if (existglobal(weigths[2]))
                wt = getglobal(weigths[2])
                opt = gamopt("wt",wt,opt)
              endif
            else
              if (existglobal(weigths[1]))
                wt = getglobal(weigths[1])
                opt = gamopt("wt",wt,opt)
              endif
            endif
          endif
          if (askwtc)
            if (askwx)
              if (askwt)
                if (existglobal(weigths[3]))
                  wtc = getglobal(weigths[3])
                  opt = gamopt("wtc",wtc,opt)
                endif
              else
                if (existglobal(weigths[2]))
                  wtc = getglobal(weigths[2])
                  opt = gamopt("wtc",wtc,opt)
                endif
              endif
            else
              if (askwt)
                if (existglobal(weigths[2]))
                  wtc = getglobal(weigths[2])
                  opt = gamopt("wtc",wtc,opt)
                endif
              else
                if (existglobal(weigths[1]))
                  wtc = getglobal(weigths[1])
                  opt = gamopt("wtc",wtc,opt)
                endif
              endif
            endif
          endif
          askwx = 0
          askwt = 0
          askwtc = 0
          askweigths = 0
        case (comp("sellist","askname")==sel)
          name = readvalue(" Name the output variable",name)
          opt = gamopt("name",name,opt)
          askname = 0
        case (comp("sellist","asknopic")==sel)
          opt = gamopt("nopic",1,opt)
          asknopic = 0
        case (comp("sellist","askdesstat")==sel)
          opt = gamopt("descript",1,opt)
          askdesstat = 0
        case (comp("sellist","asktitle")==sel)
          val = string("%1.0f._Function",1:rows(h))
          str = string("",matrix(rows(h)))
          str[1] = " Title the output windows"
          if (loc)
            val = val|string("%1.0f._First_Derivative",1:rows(h))
            str = str|string("",matrix(rows(h)))
          endif
          if (loc==2)
            val = val|string("%1.0f._Second_Derivative",1:rows(h))
            str = str|string("",matrix(rows(h)))
          endif
          title = readvalue(str,val)
          opt = gamopt("title",title,opt)
          asktitle = 0
        case (comp("sellist","askvars")==sel)
          str = ""
          if (askyvars)
            str = str|"y"
          endif
          if (asktvars)
            str = str|string("t%1.0f",(1:p))
          endif
          if (askxvars)
            str = str|string("x%1.0f",(1:cols(x)))
          endif
          str = str[2:rows(str)]
          vars = readvalue(str,str)
          if (askyvars)
            opt = gamopt("yvars",vars[1],opt)
          endif
          if (asktvars)
            if (askyvars)
              opt = gamopt("tvars",vars[2:p+1],opt)
            else
              opt = gamopt("tvars",vars[1:p],opt)
            endif
          endif
          if (askxvars)
            if (askyvars)
              if (asktvars)
                opt = gamopt("xvars",vars[p+2:rows(vars)],opt)
              else
                opt = gamopt("xvars",vars[2:rows(vars)],opt)
              endif
            else
              if (asktvars)
                opt = gamopt("xvars",vars[p+1:rows(vars)],opt)
              else
                opt = gamopt("xvars",vars[1:rows(vars)],opt)
              endif
            endif
          endif
          askxvars = 0
          asktvars = 0
          askyvars = 0
          askvars = 0
        case (comp("sellist","asktransf")==sel)
          item = ""
          if (asktransfy)
            if (comp("opt","yvars")>0)
              item = item|opt.yvars
            else
              item = item|"y"
            endif
          endif
          if (asktransft)
            s = (sum(t>0)==n)'
            if (comp("opt","tvars")>0)
              item = item|paf(opt.tvars,s)
            else
              item = item|string("t%1.0f",paf(1:p,s))
            endif
          endif
          item = item[2:rows(item)]
          head = " Select logarithmic transformation for:"
          transf = selectitem(head,item)
          yy = y
          tt = t
          if (asktransfy)
            if (transf[1])
              y = log(y)
            endif
            transf = transf[2:rows(transf)]
          endif
          if (asktransft)
            if (sum(transf)>0)
              transf = paf(paf(1:p,s),transf)
              t[,transf] = log(t[,transf])
            endif
          endif
          (sum(log(yy)==y))
          (sum(log(tt)==t))
          asktransf = 0
        case (comp("sellist","askselect")==sel)
          do
            selallt=0
            selallx=0
            if (comp("opt","tvars")>0)
              item = opt.tvars
            else
              item = string("t%1.0f",(1:p))
            endif
            if (partlinear)
              if (comp("opt","xvars")>0)
                item = item|opt.xvars
              else
                item = item|string("x%1.0f",1:cols(x))
              endif
            endif
            exclude = selectitem(" Exclude variables",item)
            excludet = exclude[1:p]
            rowex=rows(exclude)
            if (partlinear && (rowex>p))
              excludex = exclude[p+1:rowex]
            else
              excludex = 0
            endif
            sumext=sum(excludet)
            sumexx=sum(excludex)
            if(sumext>=p)
              selallt=1
              "gamfit: You cannot exclude all variables!"
            endif
            if((rowex>p) &&  (sumexx>=rowex-p))
              selallx=1
              "gamfit: You cannot exclude all variables!"
            endif
          until (selallx+selallt==0)
          if (sumext>0)
            excludet = paf(1:p,1-excludet)
            t = t[,excludet]
            if (comp("opt","tvars")>0)
              opt.tvars = opt.tvars[excludet]
            endif
            if (comp("opt","tg")>0)
              if (cols(opt.tg)==p)
                opt.tg = opt.tg[,excludet]
              endif
            endif
            h = h[excludet]
            if (exist(g)>0)
              g = g[excludet]
            endif
            p = cols(t)
          endif
          if (sumexx>0)
            excludex = paf(1:cols(x),1-excludex)
            x = x[,excludex]
            if (comp("opt","xvars")>0)
              opt.xvars = opt.xvars[excludex]
            endif
          endif
          askselect = 0
      endsw
    endo
  endif
;
; branch to estimation procedures
; ---------------------------------------------------------------------
  switch
    case (nocancel==0)
      break
    case ((model=="am")&&(integration))
      m = intest(t,y,h,g,loc,opt)
      b = 0
      const = mean(y,1)
      gest = list(m,opt)
      break
    case ((model=="plam")&&(integration))
      {m,b,const} = intestpl(x,t,y,h,g,loc,opt)
      gest = list(m,b,const,opt)
      break
    case ((model=="gam")&&(integration))
      m = gintest(code,t,y,h,g,loc,opt)
      b = 0
      const = 0
      gest = list(m,opt)
      break
    case ((model=="gplam")&&(integration))
      {m,b,bv,const} = gintestpl(code,x,t,y,h,g,opt)
      opt = gamopt("bv",bv,opt)
      gest = list(m,b,bv,const,opt)
      break
    case (backfit)
      {m,b,const} = backfit(t,y,h,loc,kern,opt)
      gest = list(m,b,const,opt)
      break
    default
      warning(1,"function not yet installed")
      nocancel = 0
  endsw
;
; produce beautiful output
; -------------------------------------------------------------------
  nopic = 0
  if (comp("opt","nopic")>0)
    nopic = opt.nopic
  endif
  descript = 0
  if (comp("opt","descript")>0)
    descript = opt.descript
  endif
  if ((nocancel)&&((descript)||(nopic!=1)))
    gamout(t,y,m,b,const,opt)
  endif
;
; write results
; --------------------------------------------------------------------
  if (nocancel)
    exec(name+"fit=gest")
    putglobal(name+"fit")
  endif
endp
