proc(b,bv,it,ret)=glmcore(code,x,y,wx,off,ctrl)
; ----------------------------------------------------------------------------
; Library        glm
; ----------------------------------------------------------------------------
;  See_also      glmopt glminit glmest glmstat glmlld glmlink glminvlink
; ----------------------------------------------------------------------------
;   Macro        glmcore
; ----------------------------------------------------------------------------
;   Description  fits a  generalized linear model E[y|x] = G(x*b).
;                This is the core macro for GLM estimation. It assumes
;                that all input variables are given in the right manner.
;                No preparation of data is performed. A more convenient
;                way to estimate a GLM is to call the function glmest.
; ----------------------------------------------------------------------------
;   Keywords     GLM, Generalized Linear Model
; ----------------------------------------------------------------------------
;   Reference    McCullagh/Nelder, Generalized Linear Models, 1989
; ----------------------------------------------------------------------------
;   Link   ../tutorials/glmstart.html Tutorial: GLM in XploRe 
; ----------------------------------------------------------------------------
;   Usage        {b,bv,it,ret} = glmcore(code,x,y,wx,off,ctrl)
;   Input
;     Parameter   code  
;     Definition        text string, the short code for the model (e.g. 
;                       "bilo" for logit or "noid" for ordinary LS).
;     Parameter   x  
;     Definition        n x p  matrix, the predictor variables.
;     Parameter   y  
;     Definition        n x 1  or n x 2  or n x 3 matrix, 
;                       either the response values, 
;                       or sums of response values in 1st column and sums of 
;                       a function of response values in the 2nd column (e.g.
;                       sums of y^2 or log(y)).
;                       (In the case of replicated data, the number of 
;                       replications should be given in wx, y[,1] contains
;                       the sums of all responses for a replication, 
;                       y[,2] contains sums of y^2 or log(y) for a 
;                       replication.)
;     Parameter   wx
;     Definition        n x 1 vector or scalar, weights. Set wx=1 to ignore.
;     Parameter   off
;     Definition        n x 1 vector or scalar, offset. Set off=0 to ignore.
;     Parameter   ctrl  
;     Definition        6 x 1 vector or scalar, contains control parameters
;                         shf   (default=0), 
;                         miter (default=10), 
;                         cnv   (default=0.0001), 
;                         fscor (default=0), 
;                         pow   (default=0, power for power link),
;                         nbk   (default=1, parameter for negative binomial).
;                       Alternatively, one can give here shf only. Set to 0 
;                       to use the defaults.
;                       The parameters correspond to the optional parameters 
;                       which can be given in glminit. 
;                       They are all ignored when not applicable.
;   Output
;     Parameter   b
;     Definition        p x 1  vector, estimated coefficients.
;     Parameter   bv
;     Definition        p x p  matrix, estimated covariance matrix for b.
;                       Not yet corrected for dispersion!
;     Parameter   it
;     Definition        integer, number of iterations needed.
;     Parameter   ret
;     Definition        scalar, return code: 
;                         0  o.k., 
;                         1  maximal number of iterations reached
;                            (if applicable),
;                        -1  missing values have been encountered.
; ----------------------------------------------------------------------------
;   Example   library("glm")
;             ;==========================
;             ;  simulate data 
;             ;==========================
;             n=100
;             b=1|2
;             p=rows(b)
;             x=2.*uniform(n,p)-1
;             y=( 1./(1+exp(-x*b)).>uniform(n) )
;             ;==========================
;             ;  GLM fit 
;             ;==========================
;             lf = glmcore("bilo",x,y,1,0,1)
;             b~lf.b
; ----------------------------------------------------------------------------
;   Result    A logit fit for E[y|x,t] is computed. lf.b contains the 
;             coefficients for the linear part. The example gives the 
;             true b together with the GLM estimate lf.b. 
; ----------------------------------------------------------------------------
;   Author    Marlene Mueller, 2000/05/18
; ----------------------------------------------------------------------------
  n=rows(x)
  p=cols(x)
  one=matrix(n)
;
  glmmodels=getglobal("glmmodels")
;
  binomial  = sum(code==(glmmodels.binomial)) >0
  gaussian  = sum(code==(glmmodels.gaussian)) >0
  poisson   = sum(code==(glmmodels.poisson)) >0
  gamma     = sum(code==(glmmodels.gamma))>0
  igaussian = sum(code==(glmmodels.igaussian))>0
  nbinomial = sum(code==(glmmodels.nbinomial))>0
;
  canonical = sum(code==(glmmodels.canonical)) >0
  direct    = sum(code==(glmmodels.direct)) >0
  power     = sum(code==(glmmodels.power)) >0
;
; set defaults
;
  if (rows(ctrl)==1)
    ctrl=ctrl|10|0.0001|0|0|1
  endif
;
  shf  =(ctrl[1]==1)
  miter=ctrl[2]
  cnv  =ctrl[3]
  fscor=ctrl[4]
  pow  =ctrl[5]*(power)
  nbk  =ctrl[6]*nbinomial
;
  if ((code=="nopow")&&(pow==1))
    code="noid"
    direct=1
    power=0
  endif
;
  if (power*(pow!=1))
;    error(min(min(x),2)<0,"can do only x>=0 with power link")
    error(mean(y[,1])<=0,"mean(y) should be positive for power link")
  endif 
;
  if (direct)
    bv    = inv(x'*(wx.*x))
    b     = bv*(x'*(wx.*(y[,1]-off))) ; need only y[,1] here!
    it    = 0
    ret   = -1
    if (countNaN(b)==0)
      ret = 0
    endif
  else
    switch
      case (binomial)
        if (cols(y)==2) ; we have been through glminit
          mu=y[,2]
          y =y[,1]
        else
          if (max(paf(y,wx.*one))>1)
            mu = (y+0.5)./(wx+1)
            y = y./(wx+(wx==0)) ; divide by wx, but not by 0!
          else
            mu = (y+0.5)./2
          endif
        endif
        break
      case ((power)*(pow!=1))
        mu =0.5*(y[,1]+mean(y[,1]))
        mu =mu.*(mu>0) + mean(y[,1]).*(mu<=0)
        break
      default
        mu =0.5*(y[,1]+mean(y[,1]))
    endsw 
    opts= list(pow,nbk)
    eta = glminvlink(code,mu,opts)
;
    ctrl = 1 
    it = 1
;    y = y./(wx+(wx==0))  ; divide by wx, but do not divide by 0!
    do 
      if (fscor*(1-canonical))
        {ll1,ll2}=glmlld(code+"fs",eta,y[,1],opts)
      else
        {ll1,ll2}=glmlld(code,eta,y[,1],opts)
      endif
      B =  x'*(wx.*ll2.*x)
      bv= -inv(B)
      b =  bv*(x'*(wx.*(ll1-ll2.*(eta-off))))
;
      eta=x*b+off
      mu = glmlink(code,eta,opts)
      chgb=Inf
      chgd=Inf
      if (countNaN(b)==0)
        if (ctrl == 1) 
          deviance= 2*sum(wx.*glmll(code,y,y,opts)-wx.*glmll(code,mu,y,opts))
          crit = 0 
          ctrl = 2  
          bold = b
          bvold = bv
        else 
          chgd = deviance 
          deviance= 2*sum(wx.*glmll(code,y,y,opts)-wx.*glmll(code,mu,y,opts))
          chgd = abs((chgd-deviance)/chgd) 
          db   = b-bold 
          chgb = sqrt((db'*db)./(bold'*bold)) 
          bold = b
          bvold = bv
          crit = ((chgd<cnv)||(chgb<cnv))
          it=ctrl
          ctrl = ctrl +1 
        endif
      else
        crit=1
      endif 
      if (shf)
        if (fscor==1)
          "glmcore/code='"+code+"/fs': iteration no. "+string("%1.0f", it)
        else
          "glmcore/code='"+code+"': iteration no. "+string("%1.0f", it)
        endif
        b
        tmp=    "relative change in b:        "+string("%10.8f", chgb)
        tmp=tmp|"relative change in deviance: "+string("%10.8f", chgd)
        tmp
      endif
      if (1-crit)
      ret=1
      endif
      if (crit)
        ret=0
      endif
    until (crit+(ctrl>miter)) 
    if (countNaN(b)>0)
      if (shf)
        "glmcore: missing values have been encountered!"
      endif
      b=bold
      bv=bvold
      ret=-1
    endif
  endif
endp 









