proc()=glmfit(x,y,opt)
; ----------------------------------------------------------------------------
; Library        glm
; ----------------------------------------------------------------------------
;  See_also      glmopt glmest glmcore glmstat glmout 
; ----------------------------------------------------------------------------
;   Macro        glmfit
; ----------------------------------------------------------------------------
;   Description  helper macro for doglm.
; ----------------------------------------------------------------------------
;   Reference    McCullagh/Nelder, Generalized Linear Models, 1989
; ----------------------------------------------------------------------------
;   Link   ../tutorials/glmstart.html Tutorial: GLM in XploRe 
; ----------------------------------------------------------------------------
;   Usage        glmfit(x,y{,opt})
;   Input
;     Parameter   x  
;     Definition        n x p  matrix, the predictor variables.
;     Parameter   y  
;     Definition        n x 1  vector, the response variables.
;                       (In the case of replicated data, the number of 
;                       replications should be given in opt.wx and y should 
;                       contain the sums of all responses for a replication.)
;     Parameter   opt
;     Definition        optional, a list with optional input. The macro
;                       "glmopt" can be used to set up this parameter.
;                       The order of the list elements is not important.
;                       Parameters which are not given are replaced by 
;                       defaults (see below).
;     Parameter   opt.code
;     Definition        text string, the short code for the model (e.g. 
;                       "bilo" for logit or "noid" for ordinary PLM).
;     Parameter   opt.weights
;     Definition        string, type of weights. Can be "frequency"
;                       for replication counts, or "prior" (default)
;                       for prior weights in weighted regression.
;     Parameter   opt.wx
;     Definition        scalar or n x 1 vector, frequency or prior 
;                       weights. If not given, set to 1.
;     Parameter   opt.off  
;     Definition        scalar or n x 1 vector, offset in linear predictor.
;                       If not given, set to 0.
;     Parameter   opt.name
;     Definition        string, prefix for the output. If not given, "glm"
;                       is used.
;     Parameter   opt.title
;     Definition        string, title for the output. If not given, a default
;                       is set.
;     Parameter   opt.xvars
;     Definition        p x 1 string vector, variable names for the output. 
;                       Note, that only up to 11 characters are used.
;   Output
;     Parameter   glmfitOutput  or  opt.name+"Output"
;     Definition        display, containing estimation result in the left
;                       panel and a plot of the index x*b{+off} vs. y 
;                       (vs. y./wx for binomials) and a plot of th index
;                       vs. the link function.
;     Parameter   glmfitout  or  opt.name+"out"
;     Definition        string vector, global, containing estimation result 
;                       in the left panel as text.
;     Parameter   glmfit  or  opt.name
;     Definition        list, global variable with the components:
;     Parameter   glmfit.b
;     Definition        p x 1  vector, estimated coefficients.
;     Parameter   glmfit.bv
;     Definition        p x p  matrix, estimated covariance matrix for b.
;     Parameter   glmfit.stat
;     Definition        list with components as computed by glmstat:
;                       serror     (standard errors of coefficients), 
;                       tvalue     (t-values for coefficients), 
;                       pvalue     (p-values for coefficients), 
;                       df         (degrees of freedom),
;                       deviance   (deviance),
;                       pearson    (generalized pearson's chi^2),
;                       loglik     (log-likelihood),
;                       dispersion (estimated dispersion =pearson/df),
;                       r2         ((pseudo) coefficient of determination),
;                       adr2       (adjusted (pseudo) coefficient of determination),
;                       aic        (Akaike's AIC criterion),
;                       bic        (Schwarz' BIC criterion), and
;                       it         (number of iterations needed),
;                       ret        (return code, 
;                                     0 if everything went o.k., 
;                                     1 if maximal number of iterations reached, 
;                                    -1 if missing values have been encountered),
;                       nr         (number of replications found in x).
; ----------------------------------------------------------------------------
;   Example   library("glm") 
;             n=100
;             b=1|2
;             p=rows(b)
;             x=2.*uniform(n,p)-1
;             y=x*b+normal(n)./2
;             glmfit(x,y)
; ----------------------------------------------------------------------------
;   Result    You should choose "normal" distribution for Y and identity
;             link function. A ordinary least squares regression
;             is fitted.  'glmfit' contains the estimation results. A
;             graphical display containing these results appears at the end.
; ----------------------------------------------------------------------------
;   Author    Marlene Mueller, 2000/05/18
; ----------------------------------------------------------------------------
;
;    check initial errors
;
  havearray=(rows(dim(x))>2)
  error(havearray,"x must be vector or matrix")
  error(rows(x)!=rows(y),"x and y have different number of rows")
  error((rows(dim(y))>1),"y must be a vector")
;
  n=rows(x)
  p=cols(x)
;
  code=""
  pow=0
  nbk=1
  name="glmfit"
;
  glmmodels=getglobal("glmmodels")
;
; now check which optional values have been given
;
  if (exist(opt)>0)
    if (comp(opt,"code")>0)
      notgood=(sum(opt.code==glmmodels.all)==0)
      warning(notgood>0, "opt.code was unusable, entered interactive mode")
      if (notgood==0)
        code=opt.code
      endif
    endif
    if (comp(opt,"pow")>0)
      notgood=(dim(dim(opt.pow))!=1)
      warning (notgood>0, "opt.pow was unusable, used default =0 instead")
      if (notgood==0)
        pow=opt.pow
      endif
    endif
    if (comp(opt,"nbk")>0)
      notgood=(dim(dim(opt.nbk))!=1)
      warning (notgood>0, "opt.nbk was unusable, used default =1 instead")
      if (notgood==0)
        nbk=opt.nbk
      endif
    endif
    if (comp(opt,"name")>0)
      notgood=(exist(opt.name)!=2) || (dim(dim(opt.name))!=1)
      warning (notgood>0, "opt.name not a single string, used default instead")
      if (notgood==0)
        name=opt.name
      endif
    endif
  else
    dummy=0
    opt=list(dummy)
  endif
;
  if (code=="")
    cancel=1
  else
    cancel=0
  endif
  numfam=0
;
  while (cancel)
    stmp="GLM fit -- Select exponential family for Y"
    atmp="Select ONE exponential family for Y"
    itmp=     " Normal  (Y : any real value)"
    itmp=itmp|" Binomial  (Y : 0,1 or 0,..,m)"
    itmp=itmp|" Poisson  (Y : non-negative integer)"
    itmp=itmp|" Negative Binomial  (Y : non-negative integer)"
    itmp=itmp|" Gamma  (Y : positive)"
    itmp=itmp|" Inverse Gaussian  (Y : positive)"
    itmp=itmp|" Cancel"
    numfam=select1item(stmp,itmp,atmp)
;
    if (numfam!=7)
;
;     %%%%%%%%%%%%%%%%%%%%%%%%%
;     %% normal distribution %% 
;     %%%%%%%%%%%%%%%%%%%%%%%%%
        if (numfam==1)
          stmp="SELECT a link function    "
          atmp="Select ONE link function    "
          itmp=     " Identity (canonical, default)  => 'noid' "
          itmp=itmp|" Power link  => 'nopow'"
          itmp=itmp|" Cancel"
          numlink=select0item(stmp,itmp,atmp)
          switch
            case (numlink==2)
              code="nopow"
              break
            case (numlink==3)
              code=""
              break
            default
            code="noid"
          endsw
        endif
;     %%%%%%%%%%%%%%%%%%%%%%%%%%%
;     %% binomial distribution %% 
;     %%%%%%%%%%%%%%%%%%%%%%%%%%%
        if (numfam==2)
          stmp="SELECT a link function    "
          atmp="Select ONE link function    "
          itmp=     " Logistic (canonical, default)  => 'bilo' "
          itmp=itmp|" Gaussian  => 'bipro'"
          itmp=itmp|" Complementary log-log  => 'bicll'"
          itmp=itmp|" Cancel"
          numlink=select0item(stmp,itmp,atmp)
          switch
            case (numlink==2)
              code="bipro"
              break
            case (numlink==3)
              code="bicll"
              break
            case (numlink==4)
              code=""
              break
            default
              code="bilo"
          endsw
        endif
;     %%%%%%%%%%%%%%%%%%%%%%%%%%
;     %% poisson distribution %% 
;     %%%%%%%%%%%%%%%%%%%%%%%%%%
        if (numfam==3)
          stmp="SELECT a link function    "
          atmp="Select ONE link function    "
          itmp=     " Logarithmic (canonical,default)  => 'nolog'"
          itmp=itmp|" Power link  => 'nopow'"
          itmp=itmp|" Cancel"
          numlink=select0item(stmp,itmp,atmp)
          switch
            case (numlink==2)
              code="popow"
              break
            case (numlink==3)
              code=""
              break
            default
              code="polog"
          endsw
        endif
;     %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
;     %% negative binomial distribution %% 
;     %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        if (numfam==4)
          stmp="SELECT a link function    "
          atmp="Select ONE link function    "
          itmp=     " Canonical (default)  => 'nbcl'"
          itmp=itmp|" Power link  => 'nbpow'"
          itmp=itmp|" Cancel"
          numlink=select0item(stmp,itmp,atmp)
          switch
            case (numlink==2)
              code="nbpow"
              break
            case (numlink==3)
              code=""
              break
            default
              code="nbcl"
          endsw
        endif
;     %%%%%%%%%%%%%%%%%%%%%%%%%%
;     %% gamma distribution   %% 
;     %%%%%%%%%%%%%%%%%%%%%%%%%%
        if (numfam==5)
          stmp="SELECT a link function    "
          atmp="Select ONE link function    "
          itmp=     " Reciprocal (canonical,default)  => 'gacl'"
          itmp=itmp|" Power link  => 'gapow'"
          itmp=itmp|" Cancel"
          numlink=select0item(stmp,itmp,atmp)
          switch
            case (numlink==2)
              code="gapow"
              break
            case (numlink==3)
              code=""
              break
            default
              code="gacl"
          endsw
        endif
;     %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
;     %% inverse gaussian distribution %% 
;     %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        if (numfam==6)
          stmp="SELECT a link function    "
          atmp="Select ONE link function    "
          itmp=     " Squared reciprocal (canonical,default)  => 'igcl'"
          itmp=itmp|" Power link  => 'igpow'"
          itmp=itmp|" Cancel"
          numlink=select0item(stmp,itmp,atmp)
          switch
            case (numlink==2)
              code="igpow"
              break
            case (numlink==3)
              code=""
              break
            default
              code="igcl"
          endsw
        endif
;
      if (code!="")
        cancel=0
      endif
    else
      cancel=0
    endif
  endo
;
  if (numfam!=7) ; otherwise we're canceled ...
;
    nbinomial = sum(code==(glmmodels.nbinomial ))>0
    power     = sum(code==(glmmodels.power)) >0
;
    switch
      case ((power)&&(nbinomial))
        itmopt=       " Power for link function   (0 for logarithm)"
        itmopt=itmopt|" k for negative binomial   (1 for geometric Y)"
        bad=1
        while (bad)
          numopt=readvalue(itmopt,pow|nbk)
          bad=(countNotNumber(numopt)>0)||(numopt[2]<1)
        endo
        pow=numopt[1]
        nbk=numopt[2]
      break
      case ((1-power)&&(nbinomial))
        itmopt=       " k for negative binomial   (1 for geometric Y)"
        bad=1
        while (bad)
          numopt=readvalue(itmopt,nbk)
          bad=(countNotNumber(numopt)>0)||(numopt<1)
        endo
        nbk=numopt
      break
      case ((power)&&(1-nbinomial))
        itmopt=       " Power for link function   (0 for logarithm)"
        bad=1
        while (bad)
          numopt=readvalue(itmopt,pow)
          bad=(countNotNumber(numopt)>0)
        endo
        pow=numopt
      break
      default ; nothing
      break
    endsw
;
    if ((code=="nopow")*(pow==1))  ; avoid iteration if direct is ok
      code="noid"
    endif
;
    opt=glmopt("pow",pow,"nbk",nbk,"name",name,opt)
    {b,bv,stat}=glmest(code,x,y,opt)
;
; produce nice output window :-)
;
    glmout(code,x,y[,1],b,bv,stat,opt)
    exec( name+"=list(b,bv,stat)" )
    putglobal(name)
  endif
endp









