proc()=glmplot(x,y,opt)
; ----------------------------------------------------------------------------
; Library       glm
; ----------------------------------------------------------------------------
;  See_also     glminvlink
; ----------------------------------------------------------------------------
;   Macro       glmplot
; ----------------------------------------------------------------------------
;   Description  glmplot creates a display and plots for a one-dimensional
;                explanatory variable: the distribution, a scatterplot
;                of the marginal influence versus the response and a
;                scatterplot of the variabel versus the response.
; ----------------------------------------------------------------------------
;   Link  ../tutorials/glmstart.html Tutorial: GLM in XploRe 
; ----------------------------------------------------------------------------
;   Usage        glmplot(x,y{,opt})
;   Input
;     Parameter   x  
;     Definition        n x 1  matrix, explanatory variable.
;     Parameter   y
;     Definition        n x 1  response.
;     Parameter   opt
;     Definition        optional, a list with optional input. The macro
;                       "glmopt" can be used to set up this parameter.
;                       The order of the list elements is not important.
;                       Parameters which are not given are replaced by 
;                       defaults (see below).
;     Parameter   opt.code 
;     Definition        text string, the short code for the model (e.g. 
;                       "bilo" for logit or "noid" for ordinary LS).
;     Parameter   opt.weights
;     Definition        string, type of weights. Can be "frequency"
;                       for replication counts, or "prior" (default)
;                       for prior weights in weighted regression.
;     Parameter   opt.wx
;     Definition        scalar or n x 1 vector, frequency or prior 
;                       weights. If not given, set to 1.
;     Parameter   opt.pow
;     Definition        optional, power for power link.
;     Parameter   opt.nbk
;     Definition        scalar, extra parameter k for negative binomial
;                       distribution. If not given, set to 1 (geometric 
;                       distribution).
;     Parameter   opt.xvars
;     Definition        scalar string vector, variable name for the output.
;     Parameter   opt.name
;     Definition        string, prefix for the output. If not given, "glm"
;                       is used.
;   Output
;     Parameter   glmPlot  or  opt.name+"Plot"
;     Definition        display, containing the distribution of x in
;                       the first window (histogram/density), the
;                       marginal influence of x on y in the second
;                       and a scatterplot of x versus y in the third.
; ----------------------------------------------------------------------------
;   Example   library("glm")
;             x=normal(500) 
;             y=(x+normal(500) >0)         ; probit model 
;             opt=glmopt("code","bipro")
;             glmplot(x,y,opt) 
; ----------------------------------------------------------------------------
;   Result    The density of x, the marginal influence of x and a
;             scatter plot of x and y.
; ----------------------------------------------------------------------------
;   Author    Marlene Mueller, 2000/05/18
; ----------------------------------------------------------------------------
;
  glmmodels=getglobal("glmmodels")
;
; now check which optional values have been given
;
  xvars="x"
  name="glm"
  n=rows(x)
  code=""
  pow=0
  nbk=1
  one=matrix(n)
  wx=one
  weights="prior"
;
  if (exist(opt)>0)
    if (comp(opt,"code")>0)
      notgood=(sum(opt.code==glmmodels.all)==0)
      warning(notgood>0, "opt.code was unusable, entered interactive mode")
      if (notgood==0)
        code=opt.code
      endif
    endif
    if (comp(opt,"wx")>0)
      notgood=(dim(dim(opt.wx))!=1)*(rows(opt.wx)!=n)+(rows(dim(opt.wx))!=1)
      notgood=notgood+(sum(abs(wx))==0)+(min(wx)<0)
      error(notgood>0, "weights in opt.wx are bad!")
      if (notgood==0)
        wx=opt.wx
      endif
    endif
    if (comp(opt,"weights")>0)
      notgood=(exist(opt.weights)!=2)
      error(notgood>0, "wrong type of weights in opt.weights!")
      if (notgood==0)
        weights=opt.weights
      endif
    endif
    if (comp(opt,"pow")>0)
      notgood=(dim(dim(opt.pow))!=1)
      warning (notgood>0, "opt.pow was unusable, used default =0 instead")
      if (notgood==0)
        pow=opt.pow
      endif
    endif
    if (comp(opt,"nbk")>0)
      notgood=(dim(dim(opt.nbk))!=1)
      warning (notgood>0, "opt.nbk was unusable, used default =1 instead")
      if (notgood==0)
        nbk=opt.nbk
      endif
    endif
    if (comp(opt,"name")>0)
      notgood=(exist(opt.name)!=2) || (dim(dim(opt.name))!=1)
      warning (notgood>0, "opt.name not a single string, used default instead")
      if (notgood==0)
        name=opt.name
      endif
    endif
    if (comp(opt,"xvars")>0)
      notgood=(exist(opt.xvars)!=2) || (rows(dim(opt.xvars))!=1)
      notgood=notgood || (rows(opt.xvars)!=1)
      warning (notgood>0, "opt.xvars not consistent with x")
      if (notgood==0)
        xvars=opt.xvars
      endif
    endif
  else
    dummy=0
    opt=list(dummy)
  endif
;
  if (code=="")
    cancel=1
  else
    cancel=0
  endif
  numfam=0
;
  while (cancel)
    stmp="GLM fit -- Select exponential family for Y"
    atmp="Select ONE exponential family for Y"
    itmp=     " Normal             (Y : any real value)"
    itmp=itmp|" Binomial           (Y : 0,1 or 0,..,m)"
    itmp=itmp|" Poisson            (Y : non-negative integer)"
    itmp=itmp|" Negative Binomial  (Y : non-negative integer)"
    itmp=itmp|" Gamma              (Y : positive)"
    itmp=itmp|" Inverse Gaussian   (Y : positive)"
    itmp=itmp|" Cancel"
    numfam=select1item(stmp,itmp,atmp)
;
    if (numfam!=7)
;
;     %%%%%%%%%%%%%%%%%%%%%%%%%
;     %% normal distribution %% 
;     %%%%%%%%%%%%%%%%%%%%%%%%%
        if (numfam==1)
          stmp="SELECT a link function    "
          atmp="Select ONE link function    "
          itmp=     " Identity (canonical, default)  => 'noid' "
          itmp=itmp|" Power link                     => 'nopow'"
          itmp=itmp|" Cancel"
          numlink=select0item(stmp,itmp,atmp)
          switch
            case (numlink==2)
              code="nopow"
              break
            case (numlink==3)
              code=""
              break
            default
            code="noid"
          endsw
        endif
;     %%%%%%%%%%%%%%%%%%%%%%%%%%%
;     %% binomial distribution %% 
;     %%%%%%%%%%%%%%%%%%%%%%%%%%%
        if (numfam==2)
          stmp="SELECT a link function    "
          atmp="Select ONE link function    "
          itmp=     " Logistic (canonical, default)  => 'bilo' "
          itmp=itmp|" Gaussian                       => 'bipro'"
          itmp=itmp|" Complementary log-log          => 'bicll'"
          itmp=itmp|" Cancel"
          numlink=select0item(stmp,itmp,atmp)
          switch
            case (numlink==2)
              code="bipro"
              break
            case (numlink==3)
              code="bicll"
              break
            case (numlink==4)
              code=""
              break
            default
              code="bilo"
          endsw
        endif
;     %%%%%%%%%%%%%%%%%%%%%%%%%%
;     %% poisson distribution %% 
;     %%%%%%%%%%%%%%%%%%%%%%%%%%
        if (numfam==3)
          stmp="SELECT a link function    "
          atmp="Select ONE link function    "
          itmp=     " Logarithmic (canonical,default)  => 'nolog'"
          itmp=itmp|" Power link                       => 'nopow'"
          itmp=itmp|" Cancel"
          numlink=select0item(stmp,itmp,atmp)
          switch
            case (numlink==2)
              code="popow"
              break
            case (numlink==3)
              code=""
              break
            default
              code="polog"
          endsw
        endif
;     %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
;     %% negative binomial distribution %% 
;     %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        if (numfam==4)
          stmp="SELECT a link function    "
          atmp="Select ONE link function    "
          itmp=     " Canonical (default)  => 'nbcl'"
          itmp=itmp|" Power link           => 'nbpow'"
          itmp=itmp|" Cancel"
          numlink=select0item(stmp,itmp,atmp)
          switch
            case (numlink==2)
              code="nbpow"
              break
            case (numlink==3)
              code=""
              break
            default
              code="nbcl"
          endsw
        endif
;     %%%%%%%%%%%%%%%%%%%%%%%%%%
;     %% gamma distribution   %% 
;     %%%%%%%%%%%%%%%%%%%%%%%%%%
        if (numfam==5)
          stmp="SELECT a link function    "
          atmp="Select ONE link function    "
          itmp=     " Reciprocal (canonical,default)   => 'gacl'"
          itmp=itmp|" Power link                       => 'gapow'"
          itmp=itmp|" Cancel"
          numlink=select0item(stmp,itmp,atmp)
          switch
            case (numlink==2)
              code="gapow"
              break
            case (numlink==3)
              code=""
              break
            default
              code="gacl"
          endsw
        endif
;     %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
;     %% inverse gaussian distribution %% 
;     %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        if (numfam==6)
          stmp="SELECT a link function    "
          atmp="Select ONE link function    "
          itmp=     " Squared reciprocal (canonical,default)  => 'igcl'"
          itmp=itmp|" Power link                              => 'igpow'"
          itmp=itmp|" Cancel"
          numlink=select0item(stmp,itmp,atmp)
          switch
            case (numlink==2)
              code="igpow"
              break
            case (numlink==3)
              code=""
              break
            default
              code="igcl"
          endsw
        endif
;
      if (code!="")
        cancel=0
      endif
    else
      cancel=0
    endif
  endo
;
  if (numfam!=7) ; otherwise we're canceled ...
;
    nbinomial = sum(code==(glmmodels.nbinomial ))>0
    power     = sum(code==(glmmodels.power)) >0
;
    switch
      case ((power)&&(nbinomial))
        itmopt=       " Power for link function   (0 for logarithm)"
        itmopt=itmopt|" k for negative binomial   (1 for geometric Y)"
        bad=1
        while (bad)
          numopt=readvalue(itmopt,pow|nbk)
          bad=(countNotNumber(numopt)>0)||(numopt[2]<1)
        endo
        pow=numopt[1]
        nbk=numopt[2]
      break
      case ((1-power)&&(nbinomial))
        itmopt=       " k for negative binomial   (1 for geometric Y)"
        bad=1
        while (bad)
          numopt=readvalue(itmopt,nbk)
          bad=(countNotNumber(numopt)>0)||(numopt<1)
        endo
        nbk=numopt
      break
      case ((power)&&(1-nbinomial))
        itmopt=       " Power for link function   (0 for logarithm)"
        bad=1
        while (bad)
          numopt=readvalue(itmopt,pow)
          bad=(countNotNumber(numopt)>0)
        endo
        pow=numopt
      break
      default ; nothing
      break
    endsw
;
    binomial  = sum(code==(glmmodels.binomial)) >0
    poisson   = sum(code==(glmmodels.poisson)) >0
    gamma     = sum(code==(glmmodels.gamma))>0
    igaussian = sum(code==(glmmodels.igaussian))>0
    nbinomial = sum(code==(glmmodels.nbinomial))>0
    power     = sum(code==(glmmodels.power)) >0
;
    if ((code=="nopow")*(pow==1))  ; avoid iteration if direct is ok
      code="noid"
      power=0
    endif
;
    d=(max(x)-min(x))/50
    if (weights=="frequency")
      tmp=bindata(x,d,0,wx.*one)
      tmp.yb=tmp.yb[,2]
    else
      tmp=bindata(x,d,0)
    endif 
    tmp=paf(tmp.xb~tmp.yb,tmp.yb)
    ntmp=rows(tmp)
    if (ntmp>25)      ; density estimate
      h=2.62*1.06*sqrt(var(x))*n^(-0.2)
      tmpx=grid(0,d/h,h/d)
      tmpy=2^sum(tmpx!=0,2)
      wy=(1-tmpx^2)^2*15/16
      wy=wy./sum(tmpy.*wy)
      wx=(1:rows(wy))-1
      {tmpx,tmpy,tmp}=conv(tmp[,1],tmp[,2],wx,wy)
      dens=(tmpx*d)~(tmpy/(n*d))
      setmaskl(dens,(1:rows(dens))', 1, 1, 2)
      setmaskp(dens, 0, 0, 0)
    else      ; histogram
      tmpx=d.*tmp[,1]
      tmpy=tmp[,2]./(d.*n)
      tmp=0.*matrix(ntmp)
      dens=(tmpx|tmpx|(tmpx+d)|(tmpx+d))~(tmpy|tmp|tmpy|tmp)
      tmp=1:ntmp
      tmp=tmp~(tmp+ntmp)~(tmp+3*ntmp)~(tmp+2*ntmp)~tmp
      setmaskl(dens, tmp, 1, 1, 2)
      setmaskp(dens, 0, 0, 0)
    endif
;
    scat=x~y
    setmaskp(scat,4,11,6)
    marg=glmscatter(code,x,y,opt)
;
    exec( name+"Plot=createdisplay(3,1)" )
    exec( "show("+name+"Plot,1,1,dens)" )
    exec( "show("+name+"Plot,2,1,marg)" )
    exec( "show("+name+"Plot,3,1,scat)" )
    titl="title"
    xlm ="xlim"
    xrange=max(x)-min(x)
    xlim=(min(x)-0.1.*xrange)|(max(x)+0.1.*xrange)
;    prn="Print"
;    prin="print("+name+"Plot,"+name+"Plot.ps)"
;
    title="Distribution of "+xvars
    exec( "setgopt("+name+"Plot,1,1,titl,title,xlm,xlim)" )
;    exec( "addbutton("+name+"Plot,1,1,prn,prin)" )
    title="Marginal influence of "+xvars+", Model="+code
    if (power)
      title=title+", pow="+string("%8.4g",pow)
    endif
    if (nbinomial)
      title=title+", nbk="+string("%8.4g",nbk)
    endif
    ylm ="ylim"
    yrange=max(marg[,2])-min(marg[,2])
    ylim=(min(marg[,2])-0.2.*yrange)|(max(marg[,2])+0.2.*yrange)
    exec( "setgopt("+name+"Plot,2,1,titl,title,xlm,xlim,ylm,ylim)" )
;    exec( "addbutton("+name+"Plot,2,1,prn,prin)" )
    title="Scatterplot of "+xvars+" and response" 
    exec( "setgopt("+name+"Plot,3,1,titl,title,xlm,xlim)" )
;    exec( "addbutton("+name+"Plot,3,1,prn,prin)" )
  endif
endp




