proc(r)=gplmest(code,x,t,y,h,opt) 
; ----------------------------------------------------------------------------
; Library       gplm
; ----------------------------------------------------------------------------
;  See_also     gplmopt gplminit gplmcore gplmstat glmest gplmbootstraptest
; ----------------------------------------------------------------------------
;   Macro       gplmest
; ----------------------------------------------------------------------------
;   Description  gplmest fits a  generalized partially linear model 
;                E[y|x,t] = G(x*b + m(t)).
;                This macro offers a convenient interface for GPLM estimation.
;                A preparation of data is performed (inclusive sorting). 
; ----------------------------------------------------------------------------
;   Keywords    GPLM, Generalized Partial Linear Models
; ----------------------------------------------------------------------------
;   Notes       This routine uses in some cases the dynamically loaded 
;               shared library gplm.dll (Windows) or gplm.so (Unix). 
; ----------------------------------------------------------------------------
;   Reference   Speckman, JRSSB, 1988; Hastie/Tibshirani, 1989
; ----------------------------------------------------------------------------
;   Link   ../tutorials/gplmstart.html Tutorial: GPLM in XploRe 
; ----------------------------------------------------------------------------
;   Usage        myfit = gplmest(code,x,t,y,h{,opt})
;   Input
;     Parameter   code  
;     Definition        text string, the short code for the model (e.g. 
;                       "bilo" for logit or "noid" for ordinary PLM).
;     Parameter   x  
;     Definition        n x p  matrix, the discrete predictor variables.
;     Parameter   t  
;     Definition        n x q  matrix, the continuous predictor variables.
;     Parameter   y  
;     Definition        n x 1  vector, the response variables.
;     Parameter   h  
;     Definition        q x 1  vector, the bandwidth vector.
;     Parameter   opt
;     Definition        optional, a list with optional input. The macro
;                       "gplmopt" can be used to set up this parameter.
;                       The order of the list elements is not important.
;                       Parameters which are not given are replaced by 
;                       defaults (see below).
;     Parameter   opt.b0  
;     Definition        p x 1  vector, the initial coefficients. If not 
;                       given, all coefficients are put =0 initially.
;     Parameter   opt.m0  
;     Definition        n x 1  vector, the initial values for the nonparametric
;                       part. If not given, a default is used.
;     Parameter   opt.tg  
;     Definition        ng x 1 vector, a grid for continuous part. If tg is
;                       given, the nonparametric function will also be 
;                       computed on this grid.
;     Parameter   opt.m0g  
;     Definition        ng x 1  vector, the initial values for the 
;                       nonparametric part on the grid. These values are 
;                       ignored if direct update for nonparametric function 
;                       is possible. Otherwise, if not given, it is 
;                       approximated from m0.
;     Parameter   opt.weights
;     Definition        string, type of observation weights. Can be 
;                       "frequency" for replication counts, or "prior" 
;                       (default) for prior weights in weighted regression.
;     Parameter   opt.wx
;     Definition        scalar or n x 1 vector, frequency or prior 
;                       weights. If not given, set to 1.
;     Parameter   opt.wc 
;     Definition        n x 1  vector, weights for convergence criterion,
;                       w.r.t. m(t) only. If not given, opt.wt is used. 
;     Parameter   opt.wt  
;     Definition        n x 1  vector, weights for t (trimming factors). 
;                       If not given, all set to 1.
;     Parameter   opt.off
;     Definition        scalar or n x 1 vector, offset. Can be used for
;                       constrained estimation. If not given, set to 0.
;     Parameter   opt.meth
;     Definition        integer, if -1, a backfitting is performed,
;                       if 1 a profile likelihood method is used, and
;                       0 a simple profile likelihood is used. 
;                       The default is 0.
;     Parameter   opt.fscor
;     Definition        integer, if exists and =1, a Fisher scoring is
;                       performed (instead of the default Newton-Raphson
;                       procedure). This parameter is ignored for
;                       canonical links.
;     Parameter   opt.shf  
;     Definition        integer, if exists and =1, some output is produced 
;                       which indicates how the iteration is going on.
;     Parameter   opt.nosort
;     Definition        integer, if exists and =1, the continuous variables
;                       t and the grid tg are assumed to be sorted by the 
;                       1st column. Sorting is required by the algorithm,
;                       hence you should switch if off only when the data 
;                       are already sorted.
;     Parameter   opt.miter
;     Definition        integer, maximal number of iterations. The default 
;                       is 10.
;     Parameter   opt.cnv
;     Definition        integer, convergence criterion. The default is 0.0001.
;     Parameter   opt.pow
;     Definition        scalar, power for power link. If not given,
;                       set to 0.
;     Parameter   opt.nbk
;     Definition        scalar, extra parameter k for negative binomial
;                       distribution. If not given, set to 1 (geometric 
;                       distribution).
;   Output
;     Parameter   myfit.b
;     Definition        p x 1  vector, estimated coefficients 
;     Parameter   myfit.bv
;     Definition        p x p  matrix, estimated covariance matrix for coeff.
;     Parameter   myfit.m
;     Definition        n x 1  vector, estimated nonparametric part
;     Parameter   myfit.mg
;     Definition        ng x 1  vector, estimated nonparametric part on grid
;     Parameter   myfit.stat
;     Definition        list with the following statistics:
;     Parameter   myfit.stat.deviance
;     Definition        deviance,
;     Parameter   myfit.stat.pearson
;     Definition        generalized pearson's chi^2,
;     Parameter   myfit.stat.loglik
;     Definition        log-likelihood,
;     Parameter   myfit.stat.r2
;     Definition        pseudo R^2,
;     Parameter   myfit.stat.it
;     Definition        scalar, number of iterations needed. 
; ----------------------------------------------------------------------------
;   Example   library("gplm")
;             ;==========================
;             ;  simulate data 
;             ;==========================
;             n=100
;             b=1|2
;             p=rows(b)
;             x=2.*uniform(n,p)-1
;             t=sort(2.*uniform(n)-1,1)
;             m=cos(pi.*t)
;             y=( 1./(1+exp(-x*b-m)).>uniform(n) )
;             ;==========================
;             ;  semiparametric fit 
;             ;==========================
;             h=0.6
;             sf=gplmest("bilo",x,t,y,h)
;             b~sf.b
;             ;==========================
;             ;  plot
;             ;==========================
;             library("plot")
;             true=setmask(sort(t~m),"line","thin")
;             estm=setmask(sort(t~sf.m),"line","blue")
;             plot(true,estm)
; ----------------------------------------------------------------------------
;   Result    A generalized partially linear logit fit for E[y|x,t] is 
;             computed. sf.b contains the coefficients for the linear  
;             part. sf.m contains the estimated nonparametric part 
;             evaluated at observations t. The example gives the 
;             true b together with the GPLM estimate sf.b. Also, the  
;             estimated function sf.m is displayed together with the 
;             true fit m. 
; ----------------------------------------------------------------------------
;   Author    Marlene Mueller, 2000/05/10
; ----------------------------------------------------------------------------
;
  glmmodels=getglobal("glmmodels")
  twoparfam = sum(code==(glmmodels.twoparfam)) >0
;
; Do we have options?
;
  if (exist(opt)==0)
    dummy=0
    opt=list(dummy)  ; dummy optional parameter, doesn't matter
  endif
;
; now we can run gplminit
;
  q=cols(t)
  h=vec(h).*matrix(q)
  {xs,ts,ys,wx,wt,wc,b0,m0,off,ctrl,tg,m0g,tro,tgro}=gplminit(code,x,t,y,h,opt)
  error (countNaN(inv(ts'*(wx.*ts)))>0,"gplmest: Matrix t is singular!")
  ts=ts[,1:q]
;
  pow=ctrl[5]
  nbk=ctrl[6]
;
; everything is prepared for the estimation. run it!
;
  if (countNaN(tg)!=prod(dim(tg)))
    r=gplmcore(code,xs,ts,ys,h,wx,wt,wc,b0,m0,off,ctrl,1,tg,m0g)
  else
    r=gplmcore(code,xs,ts,ys,h,wx,wt,wc,b0,m0,off,ctrl,1)
  endif
;
; now we need to rearrange, if t, tg were sorted
;
  if (countNaN(tro)!=prod(dim(tro)))
    r.m=r.m[tro]
    if (countNaN(tgro)!=prod(dim(tgro)))
      r.mg=r.mg[tgro]
    endif
  endif
;
; produce the statistics of what we have done
;
  stat=gplmstat(code,x,t,y,r.b,r.bv,r.m,r.df,opt)
  if (twoparfam)
    r.bv=stat.dispersion.*r.bv
  endif
;
; remove df, it from r
;
  it=r.it
  ret=r.ret
  tmp=comp(r,"df")
  if (tmp>0)
    delete(r,tmp)
  endif
  tmp=comp(r,"it")
  if (tmp>0)
    delete(r,tmp)
  endif
  tmp=comp(r,"ret")
  if (tmp>0)
    delete(r,tmp)
  endif
;
; append auxiliary stuff to stat
;
  append(stat,it)
  append(stat,ret)
  append(r,stat)
endp 





