proc(stat)=gplmstat(code,x,t,y,b,bv,m,df,opt)
; ----------------------------------------------------------------------------
; Library        gplm
; ----------------------------------------------------------------------------
;  See_also      gplmopt gplminit gplmcore gplmest glmstat
; ----------------------------------------------------------------------------
;   Macro        gplmstat
; ----------------------------------------------------------------------------
;   Description  gplmstat provides some statistics for a fitted GPLM.
; ----------------------------------------------------------------------------
;   Link   ../tutorials/gplmstart.html Tutorial: GPLM in XploRe 
; ----------------------------------------------------------------------------
;   Usage        stat = gplmstat(code,x,t,y,b,m{,wx{,off}})
;   Input
;     Parameter   code  
;     Definition        text string, the short code for the model (e.g. 
;                       "bilo" for logit or "noid" for ordinary PLM).
;     Parameter   x  
;     Definition        n x p  matrix, the discrete predictor variables.
;     Parameter   t  
;     Definition        n x q  matrix, the continuous predictor variables.
;     Parameter   y  
;     Definition        n x 1  vector, the response variables.
;     Parameter   h  
;     Definition        q x 1  vector, the bandwith.
;     Parameter   b  
;     Definition        p x 1  vector, estimated coefficients.
;     Parameter   m
;     Definition        n x 1  vector, estimated nonparametric part
;     Parameter   opt
;     Definition        optional, a list with optional input. The macro
;                       "gplmopt" can be used to set up this parameter.
;                       The order of the list elements is not important.
;                       Parameters which are not given are replaced by 
;                       defaults (see below).
;     Parameter   opt.weights
;     Definition        string, type of observation weights. Can be 
;                       "frequency" for replication counts, or "prior" 
;                       (default) for prior weights in weighted regression.
;     Parameter   opt.wx
;     Definition        scalar or n x 1 vector, frequency or prior 
;                       weights. If not given, set to 1.
;     Parameter   opt.off  
;     Definition        scalar or n x 1 vector, offset in linear predictor.
;                       If not given, set to 0.
;     Parameter   opt.pow
;     Definition        scalar, power for power link. If not given, set 
;                       to 0 (logarithm).
;     Parameter   opt.nbk
;     Definition        scalar, extra parameter k for negative binomial
;                       distribution. If not given, set to 1 (geometric 
;                       distribution).
;   Output
;     Parameter   stat
;     Definition        list with the following statistics:
;     Parameter   stat.serror
;     Definition        standard errors of parameter estimates.
;     Parameter   stat.tvalue
;     Definition        t-values for parameter estimates. 
;     Parameter   stat.pvalue
;     Definition        p-values for significance of parameter estimates. 
;     Parameter   stat.df
;     Definition        degrees of freedom.
;     Parameter   stat.deviance
;     Definition        deviance.
;     Parameter   stat.pearson
;     Definition        generalized pearson's chi^2.
;     Parameter   stat.loglik
;     Definition        log-likelihood.
;     Parameter   stat.dispersion
;     Definition         dispersion parameter estimate =pearson/df.
;     Parameter   stat.r2
;     Definition        (pseudo) R^2.
;     Parameter   stat.adr2
;     Definition        adjusted (pseudo) R^2.
;     Parameter   stat.aic
;     Definition        AIC criterion.
;     Parameter   stat.bic
;     Definition        BIC criterion.
; ----------------------------------------------------------------------------
;   Example   library("gplm")
;             ;==========================
;             ;  simulate data 
;             ;==========================
;             n=100
;             b=6|-6
;             p=rows(b)
;             bv=0.001*unit(p)
;             df=n
;             x=2.*uniform(n,p)-1
;             t=sort(2.*uniform(n)-1,1)
;             m=cos(pi.*t)
;             y=( 1./(1+exp(-x*b-m)).>uniform(n) )
;             ;==========================
;             ;  compute statistics
;             ;==========================
;             stat=gplmstat("bilo",x,t,y,b,bv,m,df)
; ----------------------------------------------------------------------------
;   Result    should be quit good :-)
; ----------------------------------------------------------------------------
;   Author    Marlene Mueller, 2000/05/10
; ----------------------------------------------------------------------------
;
  glmmodels=getglobal("glmmodels")
;
  binomial  = sum(code==(glmmodels.binomial)) >0
  gaussian  = sum(code==(glmmodels.gaussian)) >0
  poisson   = sum(code==(glmmodels.poisson)) >0
  gamma     = sum(code==(glmmodels.gamma))>0
  igaussian = sum(code==(glmmodels.igaussian))>0
  nbinomial = sum(code==(glmmodels.nbinomial))>0
;
  canonical = sum(code==(glmmodels.canonical)) >0
  power     = sum(code==(glmmodels.power)) >0
  twoparfam = sum(code==(glmmodels.twoparfam)) >0
;
  n=rows(x)
  p=cols(x)
  one=matrix(n)
;
  wx=1
  off=0
  pow=0
  nbk=1
  weights="prior"
;
; now check which optional values have been given
;
  if (exist(opt)>0)
    if (comp(opt,"wx")>0)
      notgood=(dim(dim(opt.wx))!=1)*(rows(opt.wx)!=n)+(rows(dim(opt.wx))!=1)
      notgood=notgood+(sum(abs(opt.wx))==0)+(min(opt.wx)<0)
      warning(notgood>0, "opt.wx was unusable, used opt.wx=1 instead")
      if (notgood==0)
        wx=opt.wx
      endif
    endif
    if (comp(opt,"weights")>0)
      notgood=(exist(opt.weights)!=2)
      error(notgood>0, "wrong type of weights in opt.weights!")
      if (notgood==0)
        weights=opt.weights
      endif
    endif
    if (comp(opt,"off")>0)
      notgood=(dim(dim(opt.off))!=1)*(rows(opt.off)!=n)+(rows(dim(opt.off))!=1)
      warning(notgood>0, "opt.off not consistent with x, used off=1 instead")
      if (notgood==0)
        off=opt.off
      endif
    endif
    if (comp(opt,"pow")>0)
      notgood=(dim(dim(opt.pow))!=1)
      warning (notgood>0, "opt.pow was unusable, used default =0 instead")
      if (notgood==0)
        pow=opt.pow
      endif
    endif
    if (comp(opt,"nbk")>0)
      notgood=(dim(dim(opt.nbk))!=1)
      warning (notgood>0, "opt.nbk was unusable, used default =1 instead")
      if (notgood==0)
        nbk=opt.nbk
      endif
    endif
  endif
;
  if (binomial)
    if (weights=="prior")
      if (max(y[,1])>1)
        y = y[,1]./(wx+(wx==0))
      endif
    endif
  endif
;
  opts=list(pow,nbk)
;
  eta=x*b+m+off
  mu = glmlink(code,eta,opts)
;
  if (weights=="frequency")
    nn = sum(wx.*one)
    df = df +nn-n
  else
    nn = n - sum(wx==0)
  endif
;
  deviance = 2*sum(wx.*(glmll(code,y,y,opts)-glmll(code,mu,y,opts)))
  dispersion = deviance/df
  phi=dispersion
;
  p0       = sum(y[,1].*wx)./sum(wx.*matrix(n))  ; need really y[,1] here!
  p0       = p0*matrix(n)
  switch 
    case ((twoparfam)*(1-gaussian))
      append(opts,phi)
      loglik   = sum(wx.*glmll(code,mu,y,opts))
      loglik0  = sum(wx.*glmll(code,p0,y,opts))
      break
    case (gaussian)
      loglikr  = sum(wx.*glmll(code,mu,y,opts))
      loglikr0 = sum(wx.*glmll(code,p0,y,opts))
      append(opts,phi)
      loglik   = sum(wx.*glmll(code,mu,y,opts))
      break
    default
      loglik   = sum(wx.*glmll(code,mu,y,opts))
      loglik0  = sum(wx.*glmll(code,p0,y,opts))
  endsw
;
  switch
    case (binomial)
      pearson= sum( ((y-mu).*(y-mu).*wx) ./ (mu.*(1-mu)) )
      break
    case (poisson)
      pearson= sum( ((y-mu).*(y-mu).*wx) ./ mu )
      break
    case (gamma)
      pearson= sum( ((y-mu).*(y-mu).*wx) ./ (mu^2) )
      break
    case (igaussian)
      pearson= sum( ((y-mu).*(y-mu).*wx) ./ (mu^3) )
      break
    case (nbinomial)
      pearson= sum( ((y-mu).*(y-mu).*wx) ./ (mu^2./nbk +mu) )
      break
    default
      pearson= deviance
  endsw
;
  aic  =-2*loglik +2*(nn-df)
  bic  =-2*loglik +log(nn)*(nn-df)
;
; now adjust bv by dispersion if two-parameter family
;
  if (twoparfam)
    bv=dispersion.*bv
  endif
  serror=xdiag(bv)
  serror=sqrt(serror)
  tvalue=b./serror
  pvalue=2.*cdfn(-abs(tvalue))
;
  stat=list(serror,tvalue,pvalue,df,deviance,pearson,loglik,dispersion,aic,bic)
  if (gaussian)
    r2   = 1-loglikr/loglikr0
    adr2 = 1-(loglikr/df)/(loglikr0/(nn-1))
  else
    r2   = 1-loglik/loglik0
    adr2 = 1-(loglik/df)/(loglik0/(nn-1))
  endif    
  if (r2 >0) ; no r2 without constant!
    append(stat,r2)
  endif
  if (adr2 >0)
    append(stat,adr2)
  endif
endp 


