proc (surv) = hazsurv(data,z)   

; -----------------------------------------------------------------
; Library     hazreg
; -----------------------------------------------------------------
; See_also    hazdat, hazregll, hazbeta, hazcoxb
; -----------------------------------------------------------------
; Macro       hazsurv
; -----------------------------------------------------------------
; Description calculates the conditional survival function, 
;             using the maximum likelihood estimate of 
;             the regression parameter beta obatined through
;             hazbeta. 
; -----------------------------------------------------------------
; Usage       surv = hazsurv(data,z)
; Input       
; Parameter   data 
; Definition  n x (p+4) matrix, 
;             column 1:  the sorted observed survival time t, 
;             column 2:  the cosorted censoring indicator delta,
;             column 3:  labels l,  
;             column 4:  number of ties at time t[i], cosorted,
;             columns 5 to p+4: the cosorted covariate matrix z.
; Parameter   z 
; Definition  p x 1 matrix, the covariate values for which the 
;             conditional survival curve is estimated.
; Output
; Parameter   surv  
; Definition  n x 2 matrix, the first column is the sorted 
;             t, followed by the estimated survival function 
;             at the points of t, conditional on z.
; -----------------------------------------------------------------
; Example  library("hazreg") 
;            n = 20
;            p = 2
;            beta = 1|2                      ; regression parameter
;            z = uniform(n,p) - 0.5          ; covariates
;            y = -log(1-uniform(n))          ; exponential survival
;            y = y./exp(z*beta)              ; covariate effects
;            c = 4*uniform(n)                ; uniform censoring
;            t = min(y~c,2)                  ; censored time             
;            delta = (y<=c)                  ; censoring indicator            
;            
;         {data,ties} = hazdat(t,delta, z)   ; preparing data
;         z1 = 0.1|0.3
;         surv = hazsurv(data, z1)  
;                                            ; estimation of the
;                                            ; conditional survival
;                                            ; function                              
; -----------------------------------------------------------------
; Result     The conditional survival function is estimated. 
; -----------------------------------------------------------------
; Keywords   conditional survival function
; -----------------------------------------------------------------
; Author     Lijian Yang 000308
; -----------------------------------------------------------------
 
  {betahat,betak,ck} = hazbeta(data)
  {bhaz,bsurv} = hazcoxb(data)
  p = rows(betahat)
  t = data[,1]
  bsurv = bsurv[,2]

  zbeta = z'*betahat
  ezbeta = exp(zbeta)

  surv = (bsurv).^(ezbeta)
  
surv = t~surv
  
endp


