proc (mh) = lpderest(x,h,q,p,K,d) 
; -----------------------------------------------------------------
; Library        smoother
; -----------------------------------------------------------------
;  See_also      lpregest lpderrot
; -----------------------------------------------------------------
;   Macro        lpderest
; -----------------------------------------------------------------
;   Description  estimates the q-th derivative of a regression 
;                function using local polynomial kernel regression.
;                The computation uses WARPing.
; -----------------------------------------------------------------
;   Reference    Fan and Gijbels (1995): Local Polynomial Fitting
; 
;                Fan and Marron (1994): Binning for local polynomials
; 
;                Haerdle (1991): Smoothing Techniques
; -----------------------------------------------------------------
;   Usage        mh = lpderest (x, h {,q {,p {,K} {,d}}}) 
;   Input
;     Parameter  x  
;     Definition   n x 2, the data. In the first column the
;                  independent, in the second column the
;                  dependent variable.
;     Parameter  h 
;     Definition   scalar, bandwidth. If not given, the rule of thumb
;                  bandwidth computed by lpderrot is used.
;     Parameter  q
;     Definition   integer <=4, order of derivative. If not 
;                  given, q=1 (first derivative) is used.
;     Parameter  p
;     Definition   integer, order of polynomial. If not given, 
;                  p=q+1 is used.
;     Parameter  K  
;     Definition   string, kernel function on [-1,1] or Gaussian
;                  kernel "gau". If not given, the Quartic kernel
;                  "qua" is used.
;     Parameter  d  
;     Definition   scalar, discretization binwidth. d must be smaller
;                  than h. If not given, the minimum of h/3 and 
;                  (max(x[,1])-min(x[,1]))/100 is used. 
;   Output
;     Parameter  mh  
;     Definition   m x 2 matrix, the first column is a grid and the 
;                  second column contains the derivative estimate on 
;                  that grid.
; -----------------------------------------------------------------
;   Example   library("smoother") 
;             library("plot")
;             ;
;             x = read("motcyc.dat")                  
;             mh = lpderest(x,5)      ; estimate function
;             ;
;             mh = setmask(mh, "line","blue")
;             plot(x,mh)                         
; -----------------------------------------------------------------
;   Result    The derivative regession estimate (blue) using   
;             Quartic kernel and bandwidth h=5 is pictured. 
; -----------------------------------------------------------------
;   Author    Marlene Mueller, 2000/03/28 - 14:04
; -----------------------------------------------------------------
  n=rows(x)
  error(cols(x)<>2,"lpderest: cols(x) <> 2")
; 
  if (exist(h)==0)
    h=lpderrot(x)
  endif
;   
  if (exist(q)==0)    
    q=1                                   ; default is 1st derivative    
  endif    
  if (exist(p)==0)    
    p=q+1                                 ; default is q+1    
  endif    
  if (exist(K)==0)
    K="qua"
  endif
  if (exist(K)==1) ; no K but d
    d=K
    K="qua"
  endif
  if (exist(d)==0)
    d=(max(x[,1])-min(x[,1]))/100
    d=min(d~(h./3),2)
  endif
;
  error(h .<=d,"denest: h smaller than d -- Please modify h or d!")
;
  {xb,yb}=bindata(x[,1],d,0,x[,2])        ; bin data    
  g=aseq(0,h/d,d/h)    
  wy=_K(g)                                ; kernel weights    
  wx=aseq(0,rows(wy),1)    
  {xc,c,or}=conv(xb,yb,wx,wy)    
  s=c[,1]    
  t=c[,2]    
  nc=rows(c) 
;   
  i=1    
  while (i.<= p)    
    wy=wy.*g    
    sym=i-2.*floor(i./2) 
    {xc,c,or}=conv(xb,yb,wx,wy,sym) 
    s=s~c[,1] 
    t=t~c[,2] 
    i=i+1 
  endo 
  while (i.<= 2*p) 
    wy=wy.*g 
    sym=i-2.*floor(i./2)  
    {xc,c,or}=conv(xb,yb[,1],wx,wy,sym)
    s=s~c 
    i=i+1 
  endo 
  s1=s[,1:(p+1)] 
  i=2 
  while (i.<= p+1) 
    s1=s1|s[,i:(p+i)] 
    i=i+1 
  endo
  is=#(aseq(0,p+1,nc)*matrix(1,nc))
  is=is+sort(#(aseq(1,nc,1)*matrix(1,p+1)))
; 
  s1=index(s1,is)
  s1= s1'
  reshape("s1",#(p+1,p+1,nc))
  s1 = inv(s1)  
  reshape("s1",#(p+1,(p+1)*nc,1))
  s = s1'
  is=#(aseq(1,p+1,1)*matrix(1,nc))
;  
  s=paf(s,is.=q+1) 
  mh = ((-1)^q).*prod(aseq(1,q,1)).*sum(s.*t,2) 
  mh=(d.*xc)~(mh./h^q)
endp 

