proc (mh) = lpderxest(x,h,q,p,v) 
; -----------------------------------------------------------------
; Library        smoother
; -----------------------------------------------------------------
;  See_also      lpregest lpderrot
; -----------------------------------------------------------------
;   Macro        lpderxest
; -----------------------------------------------------------------
;   Description  estimates the q-th derivative of a regression 
;                function using local polynomial kernel regression
;                with Quartic kernel.
; -----------------------------------------------------------------
;   Reference    Fan and Gijbels (1995): Local Polynomial Fitting
; 
;                Fan and Marron (1994): Binning for local polynomials
; 
;                Haerdle (1991): Smoothing Techniques
; -----------------------------------------------------------------
;   Usage        mh = lpderxest (x, h {,q {,p {,K} {,v}}}) 
;   Input
;     Parameter  x  
;     Definition   n x 2, the data. In the first column the
;                  independent, in the second column the
;                  dependent variable.
;     Parameter  h 
;     Definition   scalar, bandwidth. If not given, the rule of thumb
;                  bandwidth computed by lpderrot is used.
;     Parameter  q
;     Definition   integer <=2, order of derivative. If not 
;                  given, q=1 (first derivative) is used.
;     Parameter  p
;     Definition   integer, order of polynomial. If not given, 
;                  p=q+1 is used for q<2, p=q is used for q=2.
;     Parameter  v  
;     Definition   m x 1, values of the independent variable on 
;                  which to compute the regression. If not given, 
;                  x is used.
;   Output
;     Parameter  mh  
;     Definition   n x 2 or m x 2 matrix, the first column is the 
;                  sorted first column of x or the sorted v, the 
;                  second column contains the derivative estimate  
;                  on the values of the first column.
; -----------------------------------------------------------------
;   Example   library("smoother") 
;             library("plot")
;             ;
;             x = read("motcyc.dat")                  
;             mh = lpderxest(x,5)      ; estimate function
;             ;
;             mh = setmask(mh, "line","blue")
;             plot(x,mh)                         
; -----------------------------------------------------------------
;   Result    The derivative regession estimate (blue) using   
;             Quartic kernel and bandwidth h=5 is pictured. 
; -----------------------------------------------------------------
;   Author    Marlene Mueller, 980413  
; -----------------------------------------------------------------
  n=rows(x)
  error(cols(x)<>2,"lpderxest: cols(x) <> 2")
; 
  if (exist(h)==0)
    h=lpderrot(x)
  endif
;   
  if (exist(q)==0)    
    q=1                                   ; default is 1st derivative    
  endif    
  if (exist(p)==0) 
    if (q<2)   
      p=q+1                                 ; default is q+1    
    else
      p=q
    endif
  endif  
  error(q>2,"lpregxest: q must be <= 2")
  error(p>2,"lpregxest: p must be <= 2")
;  
  tmp=sort(x)
  x=tmp[,1]
  y=tmp[,2]
  if (exist(v)==0)
    v=x
  else
    v=sort(v)
  endif
;
  if (q==0 && p==0)
    mh=sker(x,h,"qua",matrix(n)~y,v)
    mh=v~ (mh[,2]./mh[,1])
  else
    mh=locpol(x, v, y, h, p-1) 
    switch
      case (q==0)
        mh=v~(mh[,1])
        break
      case (q==1)
        mh=v~(mh[,2])
        break
      default; (q==2)
        mh=v~(mh[,3].*2)
        break
    endsw
  endif
endp 

