proc(Vtest) = lvtest (y, w, z, hw, hz)
; ----------------------------------------------------------------------
; Library      smoother
; ----------------------------------------------------------------------
; See_also     epa  
; ----------------------------------------------------------------------
; Macro        lvtest
; ----------------------------------------------------------------------
; Description  This quantlet tests for significance of a subset or of the whole set 
;              of continuous regresssors in a nonparametric regression. 
; -----------------------------------------------------------------
; Notes        The test is asymptotic and one-sided standard normal. Two estimators 
;              of the asymptotic variance are considered, one biased in small 
;              samples but quickly evaluated, and a less biased one but requiring
;              more computational time. 
;
;	             The first argument y is the vector of observations of the dependent
;	             variable. If we test for significance of a subset of regressors, the 
;              second argument w is the matrix of observations of the explanatory 
;              variables that are not under test, and the third argument z is the 
;              matrix of observations of the regressors under test. The fourth 
;              argument is the vector of bandwidths for w, the fifth argument is 
;              the vector of bandwidths for z. 
;              If we test the significance of the whole set of regressors, the 
;              second argument is the matrix of observations on this set, and the 
;              third argument is the vector of bandwidths.
;
;              In both cases, there is no default argument for the bandwidths, thus, 
;              these vectors must be supplied by the user. 
;
;              The test statistic is evaluated by using the multidimensional 
;              Epanechnikov kernel.
;              The function displays the value of the test statistic and its P-Value.
; ----------------------------------------------------------------------        
; Usage        test = lvtest(y,w,z,hw,hz)       
; Input
;   Parameter  y
;   Definition n dimensional vector
;   Parameter  w  
;   Definition n x p matrix
;   Parameter  z  
;   Definition n x k matrix
;   Parameter  hw  
;   Definition p1 dimensional vector
;   Parameter  hz  
;   Definition k dimensional vector
; Output
;   Parameter  test  
;   Definition test statistic and its P-value
; ----------------------------------------------------------------------
; Example    ; We test here for the significance of the subset of regressors
;            ; z. Since by construction, this subset is significant, the 
;            ; evaluated P-value is equal to zero.
;            ;
;	           library("smoother")
;            setenv("outputstringformat", "%s")
;            func("diagrv")
;            randomize(0)
;            u = normal(100,1)
;            w = normal (100,1)
;            z = normal (100,1)
;            y = w^3 - w + 2*z + u
;            test = lvtest (y,w,z,0.4,0.4)
;        
; ----------------------------------------------------------------------
;  Result
;    Contents of result
;
;    [1,] Test Statistic   P-Value  
;    [2,] ________________________  
;    [3,]  
;    [4,]      5.8210      0.0000 
;
; ----------------------------------------------------------------------  
; Example
;        ; Here, the subset of regressors z is not significant. Thus, the 
;        ; P-value of the test is high.
;        library("smoother")
;        setenv("outputstringformat", "%s")
;        func("diagrv")
;        randomize(0)
;        u = normal(100,1)
;        w = normal (100,1)
;        z = normal (100,1)
;        y = w^3 - w +  u
;        test = lvtest (y,w,z,0.4,0.4)  
; ----------------------------------------------------------------------
;  Result
;    Contents of result
;
;    [1,] Test Statistic   P-Value  
;    [2,] ________________________  
;    [3,]  
;    [4,]      0.0735      0.4707 
;
; ----------------------------------------------------------------------
; Reference
;   Lavergne, P. and Q. Vuong (1998): Nonparametric Significance Testing.
;    Mimeo INRA and University of Southern California.  
;
; ----------------------------------------------------------------------
;   Keywords
;      test, nested models, kernel
; ----------------------------------------------------------------------
;   Author      Pascal Lavergne, Gilles Teyssiere, 980914
; ---------------------------------------------------------------------- 
;; Check the arguments
;; 
  error(rows(y)!= rows(w),"The vector y and w do not have the same number of observations")
  targ5 = exist(hz)
  if (targ5 == 0)
    JTest = 1	
    hz    =  z
    z     =  w
  else
      error (rows(y)!= rows(z),"The vector y and z do not have the same number of observations")
      JTest = 0
  endif
;; end of checking of arguments
  n = rows(y)
  en = matrix(n,1) 
  on = zeros(n,1)
  uest = zeros(n,1)
  ker = zeros(n,n)
  if (JTest == 0); Significance test for a subset of regressors
    ler = zeros(n,n)
    x = w~z
    p1 = cols(w)
    hw = hw'
    hz = hz'
    h = hw~hz
    hp1 = prod(hw,2) 
    hp = prod(h,2) 
    xh = x./h
    xg = xh[,1:p1]
	
    i = 1
    while (i <= n)
      xih = xh[i,]
      dh = xih-xh 
      kerk = epa(dh) 
      kerk = kerk/hp
      ker[i,] = kerk'
      xig = xg[i,] 
      dg = xig-xg 
      kerl = epa(dg) 
      kerl[i,] = 0
      kerl = kerl/hp1
      ler[i,] = kerl'
      fnn = sum(kerl,1)
      if (fnn != 0)
        uest[i,]=y[i]*fnn - sum(y.*kerl,1)
      endif
      i = i+1
    endo
    ker = diagrv(ker,on)
    in = uest'*ker*uest
 
    aux11 = y'*((ler*(ler')).*ker)*y
    aux12 = en'*((ler*(y.*(ler'))).*ker)*y
    aux13 = en'*((ler*((y^2).*(ler'))).*ker)*en
    kl = diagrv(ler.*ker,on)
    aux21 = (uest)'*kl*y
    aux22 = (y.*uest)'*kl*en
    kll = diagrv((ler^2).*ker,on)
    v3n =  (en'*kll*(y^2))- (y'*kll*y)
    v1n =(aux11-(2*aux12)+aux13)
    v2n = aux21 - aux22
    vn =(in-v1n-(2*v2n)-(2*v3n))/(n*(n-1)*(n-2)*(n-3))
    ;; Calcul de la variance 
    omega2 = (uest^2)'*(ker^2)*(uest^2)
    omega2 = 2*hp*omega2/(n*((n-1)^5))
    sdo2  = sqrt(omega2)
  else ; Case of joint significance test
    x = z
    hz = hz'
    h = hz
    hp = prod(h,2) 
    xh = x./h
	
    i = 1
    while (i <= n)
      xih = xh[i,]
      dh = xih-xh 
      kerk = epa(dh) 
      kerk = kerk/hp
      ker[i,] = kerk'
      i = i+1
    endo
    uest = n*y - sum(y)
    ker = diagrv(ker,on)
    in = uest'*ker*uest
    
    aux11 = y'*ker*y
    aux12 = sum(ker*y)*sum(y)
    aux13 = sum(ker*en)*sum(y^2)
    aux22 = sum(ker*(y^2))
    v1n =(aux11-(2*aux12)+aux13)
    v2n = aux22 - aux11
    vn =(in-v1n-(2*v2n))/(n*(n-1)*(n-2)*(n-3))
    ;; Calcul de la variance 
    omega2 = (uest^2)'*(ker^2)*(uest^2)
    omega2 = 2*hp*omega2/(n*((n-1)^5))
    sdo2  = sqrt(omega2)
  endif
  Vtest = n*sqrt(hp)*vn/sdo2; statistique de test
  pval = 1-cdfn(Vtest); calcul des P-Values
  ;
  ; Output display
  line1 ="Test Statistic   P-Value "
  line2 ="________________________ "
  st1 = string("%11.4f",Vtest)
  st2 = string("%12.4f",pval)
  st3 = st1+st2
  result = line1|line2|""|st3
  result
endp
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; End of macro lvtest
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;




















