proc (mu, sigma) = mleev0 (x)
; -----------------------------------------------------------------
;   Library       finance
; -----------------------------------------------------------------
;   See_also      lrseev
; -----------------------------------------------------------------
;   Macro         mleev0
; -----------------------------------------------------------------
;   Description   Maximum likelihood estimator for Gumbel (EV0) model
; -----------------------------------------------------------------
;   Usage         {mu, sigma} = mleev0 (x)
;   Input
;     Parameter   x
;     Definition      vector
;   Output
;     Parameter   mu
;     Definition      scalar, estimated location parameter
;     Parameter   sigma
;     Definition      scalar, estimated scale parameter	    
; -----------------------------------------------------------------
;   Example   library("finance")
;             randomize(0)
;	            x=randx("ev0",100)
;             m=mleev0(x)
;             m
; -----------------------------------------------------------------
;   Result    Contents of m.mu
;	            [1,] -0.18003
;             Contents of m.sigma
;             [1,]   1.1118 
; -----------------------------------------------------------------
;   Author    Michael Thomas  990503
; -----------------------------------------------------------------
  n = rows (x)
  xm = mean (x)
  alpha0 = sqrt (var (x) * 6 / pi)
  step = 0
  do
    alpha1 = alpha0
    y = exp (-x / alpha0)
    h0 = sum (y)
    h1 = sum (x .* y)
    h2 = sum (x .* x .* y)
    alpha0 = alpha0 - (alpha0 - xm + h1 / h0) / (1 + (h0 * h2 - h1 * h1) / (alpha0 * alpha0 * h0 * h0))
    step = step + 1
  until (step > 50 || abs (alpha0 - alpha1) < 1e-10)
  sigma = alpha0
  mu = -alpha0 * log (mean (exp (-x / alpha0)))
endp
