proc() = plothistwtd(inX, inWeights, inTicks)
; ---------------------------------------------------------------------
; Library     plot
; ---------------------------------------------------------------------
; See_also    plothist plotbox plotdot
; ---------------------------------------------------------------------
; Macro       plothistwtd
; ---------------------------------------------------------------------
; Description Plots a weighted histogram of x
; ---------------------------------------------------------------------
; Usage       plothistwtd(inX, inWeights {,inTicks})
; Input       
; Parameter   inX
; Definition  n x 1 Vector of continuous data.
; Parameter   inWeights
; Definition  n x 1 Vector of weights. 
; Parameter   inTicks
; Definition  m x 1 Vector. Specifying the (m-1) intervalls for the 
;             histogram.
; Output      
; ---------------------------------------------------------------------
; Notes       The intervalls are taken to be (inTicks[1], inTicks[2]),
;             (inTicks[2], inTicks[3]), ..., (inTicks[m-1], inTicks[m]),
;             where each left margin is element of the intervall and each
;             right margin is not.
;
;             inTicks[i] < inTicks[i+1] must hold for all i=1, ..., m-1.
;
;             The weights provided are automatically rescaled to sum up
;             to one. If the sum of weights is sufficiently close to zero,
;             the weights are applied "as is".
; ---------------------------------------------------------------------
; Example     library("plot")
;             Mean = #(1, 1.1, 1.6, 1.2, 2.5, 2.2, 2.9, 3.2, 4.1)
;             Fraction = #(0.5, 0.7, 3, 1.1, 1.2, 4.0, 1.3, 2.8, 0.9) 
;             plothist(Mean, 1:5)
;             plothistwtd(Mean, Fraction, 1:5)
; ---------------------------------------------------------------------
; Result      You see two histograms - one standard histogram and one
;             weighted histogram. 
;             In this example means for different subgroups of a dataset
;             have been computed. In the standard histogram all means 
;             have the same weight, disregarding the size of the underlying 
;             sample.
;             In the weighted histogram the sample size (here as fraction
;             of the whole dataset) is provided.
;             Note the different shape (and hence interpretation!) of the
;             two histograms. 
; ---------------------------------------------------------------------
; Keywords    histogram, high level graphics, weighted plots 
; ---------------------------------------------------------------------
; Author      Stephan R. W. Lauer, 990305
; ---------------------------------------------------------------------

   if (exist("inX") <> 1)
      error(1, "plothistwtd: The data must be numerical")
   endif

   if (exist("inWeights") <> 1)
      error(1, "plothistwtd: The weights must be numerical")
   endif

   if (rows(inX) <> rows(inWeights))
      error(1, "plothistwtd: data and weights must be of the same length")
   endif

   // rescale weights
   if (abs(sum(inWeights)) > 0.05) // weights sufficiently different from sum zero
      inWeights = inWeights ./ sum(inWeights)
   endif
   // else leave weights unchanged...

   if (exist("inTicks") <> 1)
      // no or no valid tickmarks specified, calculate our own ones
      // use by default 12 equidistant groups, anchored at integers
      inTicks = 0:12 // init inTicks
      inTicks = inTicks .* (ceil(max(inX))-floor(min(inX)))/12
      inTicks = inTicks + floor(min(inX))
   endif

   theNumberOfBins = rows(inTicks) - 1
   theNumberOfCases = rows(inX) 

   // now calculate the vector of heights and widths of the bins
   // the widths are given via inTicks, the heights have to be 
   // calculated from the frequencies of inX weighted with inWeights

   theWidths = matrix(theNumberOfBins)
   theHeights = matrix(theNumberOfBins)   
 
   count = 1
   while (count <= theNumberOfBins)
         theWidths[count] = inTicks[count+1] - inTicks[count]
         if ( sum((inX >= inTicks[count]) && (inX < inTicks[count + 1])) > 0) // if cases present
              // extract weights for this group and sum up
              theHeights[count] = sum(paf(inWeights,  (inX >= inTicks[count]) && (inX < inTicks[count + 1])  ))
         else theHeights[count] = 0.0
         endif       
   count = count + 1
   endo

   // now set up the boxes

   //  UpperLeft (UL) +--+ UpperRight (UR)
   //                 |  |
   //                 |  |
   //  LowerLeft (LL) +--+ LowerRight (LR)

   theLLCorners = inTicks[1:theNumberOfBins] ~ (0 * matrix(theNumberOfBins))
   theLRCorners = inTicks[2:(theNumberOfBins + 1)] ~ (0 * matrix(theNumberOfBins))
   theULCorners = inTicks[1:theNumberOfBins] ~ theHeights[1:theNumberOfBins]
   theURCorners = inTicks[2:(theNumberOfBins + 1)] ~ theHeights[1:theNumberOfBins]

   thePoints = theLLCorners | theLRCorners | theURCorners | theULCorners

   // connect LL--LR
   theLines = (1:theNumberOfBins) ~ ((theNumberOfBins + 1):(2*theNumberOfBins))
   
   // connect LR--UR
   theLines = theLines | (((theNumberOfBins + 1):(2*theNumberOfBins)) ~ ((2*theNumberOfBins + 1):(3*theNumberOfBins)))

   // connect UR--UL
   theLines = theLines | (((2*theNumberOfBins + 1):(3*theNumberOfBins)) ~ ((3*theNumberOfBins + 1):(4*theNumberOfBins)))

   // connect UL--LL
   theLines = theLines | (((3*theNumberOfBins + 1):(4*theNumberOfBins)) ~ (1:theNumberOfBins))


   // set points and lines
   setmaskp(thePoints, 0, 0, 8) // invisible
   setmaskl(thePoints, theLines, 0, 1, 1) // black, solid

   // show results
   wtdhistogram = createdisplay(1, 1)
   show(wtdhistogram, 1, 1, thePoints)
  
endp