proc(l)=readmatrix(filename, separator)
; -----------------------------------------------------------------------
; Library      ista
; -----------------------------------------------------------------------
; See_also     readlist, writelist
; -----------------------------------------------------------------------
; Macro        readmatrix
; -----------------------------------------------------------------------
; Description  Reads a matrix with mixed text and number columns as ASCII 
;              data from a file.
; -----------------------------------------------------------------------
; Usage        l =  readmatrix (filename{, separator})
; Input
;   Parameter  filename
;   Definition string
;   Parameter  separator
;   Definition string (optional)
; Output
;   Parameter  l.ctype
;   Definition type of column vectors in the file (0 = numeric, 10 = text)
;   Parameter  l.number
;   Definition number matrix (optional)
;   Parameter  l.text
;   Definition text matrix (optional)
; -----------------------------------------------------------------------
; Note         The input separator is an optional argument. The default is
;              set to " \t\n" which means that the blank, TAB and NEWLINE 
;              character is used to separate between the data fields in
;              the file. 
;  
;              The output parameters l.number and l.text are optional.
;              They are not part of the l if no numbers or no text
;              is found in the data file.
;      
;              In case that your data are not in matrix format you get
;              beside the error message the following information in the
;              output window:
;
;              [ 1,] readmatrix : /user/haerdle/XploRe/data/nue2.dat 
;              [ 2,]  
;              [ 3,] Found  381 line(s) with  12 column(s) 
;              [ 4,] Found    3 line(s) with  13 column(s) 
;              [ 5,] Found    2 line(s) with  14 column(s) 
;              [ 6,]  
;              [ 7,] Line   40 :  13 column(s) 
;              [ 8,] Line   58 :  14 column(s) 
;              [ 9,] Line   62 :  14 column(s) 
;              [10,] Line   98 :  13 column(s) 
;              [11,] Line  111 :  13 column(s) 

; -----------------------------------------------------------------------
; Example      ; loads the library util
;              library ("util")
;              x = readmatrix("bank2")
;              x
; -----------------------------------------------------------------------
; Result       Content of object x.ctype
;              [1,] 0.000000 
;              [2,] 0.000000 
;              [3,] 0.000000 
;              [4,] 0.000000 
;              [5,] 0.000000 
;              [6,] 0.000000 
;              Content of object x.number
;              [  1,] 214.800000 131.000000 131.100000 9.000000 9.700000 141.000000 
;              [  2,] 214.600000 129.700000 129.700000 8.100000 9.500000 141.700000 
;              [  3,] 214.800000 129.700000 129.700000 8.700000 9.600000 142.200000 
;              ...
;              [198,] 214.800000 130.300000 130.400000 10.600000 11.100000 140.000000 
;              [199,] 214.700000 130.700000 130.800000 11.200000 11.200000 139.400000 
;              [200,] 214.300000 129.900000 129.900000 10.200000 11.500000 139.600000 
; -----------------------------------------------------------------------
; Example      x = read("car")
;              x
; -----------------------------------------------------------------------
; Result       
;              Content of object x.ctype
;              [ 1,] 10.000000 
;              [ 2,] 0.000000 
;              [ 3,] 0.000000 
;              [ 4,] 0.000000 
;              [ 5,] 0.000000 
;              [ 6,] 0.000000 
;              [ 7,] 0.000000 
;              [ 8,] 0.000000 
;              [ 9,] 0.000000 
;              [10,] 0.000000 
;              [11,] 0.000000 
;              [12,] 0.000000 
;              [13,] 0.000000 
;              Content of object x.number
;              [ 1,] 4099.000000 22.000000 3.000000 2.000000 2.500000 27.500000 11.000000 2930.000000 186.000000 40.000000 121.000000 3.580000 
;              [ 2,] 4749.000000 17.000000 3.000000 1.000000 3.000000 25.500000 11.000000 3350.000000 173.000000 40.000000 258.000000 2.530000 
;              [ 3,] 3799.000000 22.000000 NaN NaN 3.000000 18.500000 12.000000 2640.000000 168.000000 35.000000 121.000000 3.080000 
;              ...
;              [72,] 6850.000000 25.000000 4.000000 3.000000 2.000000 23.500000 16.000000 1990.000000 156.000000 36.000000 97.000000 3.780000 
;              [73,] 7140.000000 23.000000 4.000000 3.000000 2.500000 37.500000 12.000000 2160.000000 172.000000 36.000000 97.000000 3.740000 
;              [74,] 11995.000000 17.000000 5.000000 3.000000 2.500000 29.500000 14.000000 3170.000000 193.000000 37.000000 163.000000 2.980000 
;              Content of object x.text
;              [ 1,] AMC_Concord 
;              [ 2,] AMC_Pacer 
;              [ 3,] AMC_Spirit
;              ... 
;              [72,] VW_Scirocco 
;              [73,] VW_Dasher 
;              [74,] Volvo_260 
; -----------------------------------------------------------------------
; Author       Sigbert Klinke, 971123
; -----------------------------------------------------------------------
  if (exist("separator")<>2)
    separator = " \t\n"
  endif
  x   = readascii(filename, separator)
  col = 0|paf(1:rows(x.type), x.type==20)
  col = col[2:rows(col)]-col[1:(rows(col)-1)]-1
  outstr = "readmatrix : " + filename | "" 
  if (sum(col[1].<>paf(col, col)))
    {xb , yb} = bindata (col, 1, -0.5)
    colno = sort(paf (xb~yb, yb))
    if (colno[1,1]==0)
      colno = colno[2:rows(colno)]
    endif
    outstr = outstr|string ("Found %4.0f line(s) with %3.0f column(s)", colno[,2], colno[,1])
    colno = colno[2:rows(colno)]
    lines = paf(1:rows(col)~col, sum(col.=colno[,1]', 2))
    outstr = outstr|""|string ("Line %4.0f : %3.0f column(s)", lines[,1], lines[,2])
    outstr
    error (1, "readmatrix: Data are not in matrix format")
  endif
  col = col[1]
  x.data = paf(x.data, x.type<>20)
  x.type = paf(x.type, x.type<>20)
  i = 0
  while (i<col)
    i    = i+1
    ind  = i+(0:(rows(x.type)/col-1))*col
    data = x.data[ind]
    type = x.type[ind]
    if (sum(type==10))
      if (exist("text")<>2)
        text = data
      else
        text = text~data
      endif  
      if (exist("ctype")<>1)
        ctype = 10
      else
        ctype = ctype|10
      endif
    else
      if (exist("number")<>1)
        number = atof(data)
      else
        number = number~atof(data)
      endif  
      if (exist("ctype")<>1)
        ctype = 0
      else
        ctype = ctype|0
      endif
    endif
  endo
  l = list(ctype)
  if (exist("number")==1)
    append (l, number)
  endif
  if (exist("text")==2)
    append (l, text)
  endif
endp