proc(hopt,ch)=regxbwcrit(crit, x, h, K)
; -----------------------------------------------------------------
; Library        smoother
; -----------------------------------------------------------------
;  See_also      regxbwsel regbwsel regxest
; -----------------------------------------------------------------
;   Macro        regxbwcrit
; -----------------------------------------------------------------
;   Keywords     kernel smoothing, kernel regression,
;                bandwidth selection
; -----------------------------------------------------------------
;   Description  determines the optimal from a range of bandwidths 
;                by one using the resubstitution estimator with one
;                of the following penalty functions:
;                Shibata's penalty function (shi), 
;                Generalized Cross Validation (gcv),
;                Akaike's Information Criterion (aic),
;                Finite Prediction Error (fpe),
;                Rice's T function (rice).
; -----------------------------------------------------------------
;   Reference    Haerdle (1990): Smoothing Techniques
; 
;                Haerdle, Mueller, Sperlich, Werwatz (1999): Non- and
;                Semiparametric Modelling
; -----------------------------------------------------------------
;   Usage        {hopt, ch} = regxbwcrit(crit, x {,h {,K} {,d} })
;   Input
;     Parameter  crit  
;     Definition     string, criterion for bandwidth selection:
;                    "shi", "gcv", "aic", "fpe", "rice".
;     Parameter  x  
;     Definition     n x 2 vector, the data.
;     Parameter  h  
;     Definition     m x 1 vector, vector of bandwidths.
;     Parameter  K  
;     Definition     string, kernel function on [-1,1] or Gaussian
;                    kernel "gau". If not given, "qua" is used.
;   Output
;     Parameter  hopt  
;     Definition     scalar, optimal bandwidth. 
;     Parameter  ch  
;     Definition     m x 2 vector, the criterion function for h values.
; -----------------------------------------------------------------
;   Example   library("smoother")
;             x=read("nicfoo")
;             h=grid(0.05,0.1,10)
;             {hopt,ch}=regxbwcrit("gcv",x,h)
;             hopt
;             library("plot")
;             ch=setmask(ch,"line","blue")
;             plot(ch)
;             setgopt(plotdisplay,1,1,"title",string("hopt=%1.6g",hopt))
; -----------------------------------------------------------------
;   Result    hopt is the LSCV optimal bandwidth for these data.
;             The resulting curve for the LSCV criterion is plotted.
; -----------------------------------------------------------------
;   Author    Marlene Mueller, 990413
; ----------------------------------------------------------------- 
  error(cols(x)<>2, "regxbwcrit: cols(x) <> 2")
;
  if (exist(K)==0)
    K="qua"
  endif
  if (exist(K)==1) ; no K but v
    v=K
    K="qua"
  endif
;
  tmp=sort(x)
  x=tmp[,1]
  y=tmp[,2]
  if (exist(v)==0)
    v=x
  endif
;
  n=rows(x)
  m=rows(h)
  ch=matrix(m)-1
  K0=_K(0)
;
  i=0
  while (i<m)
    i=i+1
    if (K!="gau")
      tmp=sker(x,h[i],K,matrix(n)~y,v)
    else
      tmp=sker(x,5.*h[i],"ngau",matrix(n)~y,v)
    endif
    fh=tmp[,1]
    mh=tmp[,2]./(fh+(tmp[,2].==0).*(fh.==0))
    xi=K0./(n.*h[i].*fh)
    xi=xi+ (max(xi)-min(xi)).*(1e-10)
;
    switch
      case (crit=="shi")
        xi= 1+2.*xi
        break;
      case (crit=="gcv")
        xi= 1./((1- xi)^2)
        break;
      case (crit=="aic")
        xi= exp(2.*xi)
        break;
      case (crit=="fpe")
        xi= (1+xi)./(1-xi)
        break;
      case (crit=="rice")
        xi= 1./(1-2.*xi)
        break;
      default;
        error(1,"denbwcrit: Criterion "+crit+" does not exist!")
        break;
    endsw
;
;mh.==NaN
    miss=isNumber(mh)
    xi=paf(xi,miss)
    yh=paf(y,miss)
    mh=paf(mh,miss)
    res=(yh-mh)
    ch[i]=((res.*xi)'*res)/rows(mh)
  endo
  ch=h~ch
  hopt = ch[minind(ch[,2]),1]
endp



