proc(filtX,KG,PreviousPs,clipInd) = rlsfil(y,mu,Sig,H,F,Q,R,bs)

; -----------------------------------------------------------------
; Library        kalman
; -----------------------------------------------------------------
;  See_also      ksmoother, kem, kfilter, kfilter2, 
;                        calibrLS, rICfil, kemitor, kemitor2
; -----------------------------------------------------------------
;   Macro        rlsfil
; -----------------------------------------------------------------
;   Description  Calculates a filtered time serie (uni- or 
;                multivariate) using a robust, recursive Filter 
;                based on LS-optimality, the rLS-filter.
;                additionally to the Kalman-Filter one needs to
;                specify the degree of robustness one wants to
;                achieve; this is done either by specifying a clipping
;                height or by specifying a relative loss w.r.t. the 
;                classical Kalman Filter in the ideal model in terms
;                of MSE.
;                The state-space model is assumed to be in the
;                following form:
;
;                y_t = H x_t + v_t
;
;                x_t = F x_t-1 + w_t
;
;                x_0 ~ (mu,Sig), v_t ~ (0,Q), w_t ~ (0,R)
;
;                All parameters are assumed known.
;
; -----------------------------------------------------------------
;   Keywords     robust Kalman filter, Kalman filter, Kalman smoother, filter
; -----------------------------------------------------------------
;   Usage         {filtX,KG,PreviousPs,clipInd} = rlsfil(y,mu,Sig,H,F,Q,R,bs)    
;   Input
;     Parameter   y  
;     Definition      T x m matrix of observed time series, 
;                     T is the number of observations,
;                     m is the dimension of time series
;     Parameter   mu    
;     Definition      n x 1 vector, the mean of the initial state
;     Parameter   Sig    
;     Definition      n x n covariance matrix of the initial state
;     Parameter   H    
;     Definition      m x n matrix
;     Parameter   F    
;     Definition      n x n matrix
;     Parameter   Q   
;     Definition      m x m variance-covariance matrix
;     Parameter   R
;     Definition      n x n variance-covariance matrix 
;     Parameter   bs
;     Definition        T vector, the clipping heights ; are to be produced by calibrLS
;                       or numerical (uniform) clipping height
;   Output                                                           
;     Parameter   filtX
;     Definition      T x n matrix of filtered time series
;     Parameter   KG
;     Definition      T x n x m "vector" of Kalman-Gain-matrices for different times
;     Parameter   PreviousPs
;     Definition      T x n x n "vector" of P_t|t's
;     Parameter   clippInd
;     Definition      T  vector of  1 (if rlsfil clipped at instance t) and 0 (else)
; -----------------------------------------------------------------
;   Example   library("kalman")
;             library("plot")
;
;             serie = read("kalmanao.dat")
;;             produced from kalman1.dat by manipulating observations 50, 60, 90
;;
;            y = serie[,2]
;             mu = 10
;             Sig = 0
;             H = 1
;             F = 1
;             Q = 9
;             R = 9
;             T=dim(y)
;
;             e=0.05
;             N=100
;             eps=0.01
;             itmax=15
;             aus=4   ;lots of output, may be set to 0
;
;             ergLS=calibrLS(T,Sig,H,F,Q,R,e,N,eps,itmax,aus)
;                     ; gets the clipping heights to efficiency loss e
;             
;             b=ergLS.b
;
;             res = kfilter2(y,mu,Sig, H,F,Q,R)
;             fx = res.filtX
;             res= rlsfil(y,mu,Sig, H,F,Q,R,b)
;             frx = res.filtX
;             
;             origy= serie[,1]~serie[,2]
;             origy= setmask(origy, "line", "blue", "thin")
;             fx = serie[,1]~fx
;             fx = setmask(fx, "line", "red", "thin")
;             frx = serie[,1]~frx
;             frx = setmask(frx, "line", "green", "thin")
;             clip=serie[,1]~(res.clipInd)
;             clip=paf(clip,clip[,2]==1)
;             clip[,2]=0
;             setmaskp(clip,4, 3, 4)  
;             
;             disp = createdisplay(1,1)
;             show(disp,1,1, origy,fx,frx,clip)
;             setgopt(disp,1,1, "title", "KalmanData1 + AO's in t=50,60,90")
;             setgopt(disp,1,1, "xlabel", "t") 
;             setgopt(disp,1,1, "ylabel", "y, rLS, Kalman") 
; -----------------------------------------------------------------
;   Result    Serie of observations is displayed with blue colour,
;             Kalman-filtered serie is displayed with red colour,
;             rls-filtered serie is displayed with green colour.
;             The stages rlsfil has clipped at are marked by a flag.
; -----------------------------------------------------------------
;   Example   library("kalman")
;             library("plot")
; 
;             T=50
;             e=0.1
;
;            
;             mu = #(20,0)
;             Sig = #(0,0)~#(0,0)
;             H  = #(0.3,-0.3)~#(1,1)
;             F  = #(1,0)~#(1,0)
;             R  = #(0,0)~#(0,9)
;             mid=#(0,0)
;             Qid= #(9,0)~#(0,9)
;             mcont=#(25,30)
;             Qcont=0.1.*Qid
;             AOr=0.1
;
;             randomize(0)
;             ErrX = epscontnorm(T,0,mid,R,mcont,Qcont,0) ;;  no contamination in ErrX
;             ErrY = epscontnorm(T,AOr,mid,Qid,mcont,Qcont,0) ;;  10% contamination in ErrY
;             sim = kemitor2(T,mu,H,F,(ErrY.X),(ErrX.X));;  simulates a length 100 path of this state space model under AO's
;             y=sim.Y         ;;  observation          
;             Xact=sim.X     ;;  actual state
;
;             N=10000
;             eps=0.01
;             itmax=15
;             aus=4  ;lots of output, may be set to 0
;
;             ergLS=calibrLS(T,Sig,H,F,Qid,R,e,N,eps,itmax,aus)
;                    ; gets the clipping heights to efficiency loss e
;             
;             b=ergLS.b
;
;             res = kfilter2(y,mu,Sig, H,F,Qid,R)
;             fx = res.filtX
;             res= rlsfil(y,mu,Sig, H,F,Qid,R,b)
;             frx = res.filtX
;
;             i=(1:T)
;             
;             Xact1 = i~(Xact[,1])
;             Xact1 = setmask(Xact1, "line", "blue", "thin")
;             fx1 = i~(fx[,1])
;             fx1 = setmask(fx1, "line", "red", "thin")
;             frx1= i~(frx[,1])
;             frx1 = setmask(frx1, "line", "green", "thin")
;             ym1=max(vec( (Xact[,1])~(fx[,1])~(frx[,1]) ) ) ;Ind-flags on top of graphics
;             ym2=min(vec( (Xact[,1])~(fx[,1])~(frx[,1]) ) )  ;clip-flags on bottom of graphics
;             flagInd=i~(ErrY.Ind)
;             flagInd=paf(flagInd,flagInd[,2]==1)
;             flagInd[,2]=ym1*((ym1>0)*1.1+(ym1<0)*0.9)
;             flagclip=i~(res.clipInd)
;             flagclip=paf(flagclip,flagclip[,2]==1)
;             flagclip[,2]= ym2*((ym2<0)*1.1+(ym2>0)*0.9)
;             setmaskp(flagInd,4, 3, 4)  
;             setmaskp(flagclip,2, 4, 4)  
;             disp = createdisplay(1,2)
;             show(disp,1,1,Xact1,fx1,frx1,flagInd,flagclip)
;             setgopt(disp,1,1, "title", "simulated Model under AO -- 1st coord.")
;             setgopt(disp,1,1, "xlabel", "t") 
;             setgopt(disp,1,1, "ylabel", "x, rLS, Kalman") 
;             
;             Xact2 = i~(Xact[,2])
;             Xact2 = setmask(Xact2, "line", "blue", "thin")
;             fx2 = i~(fx[,2])
;             fx2 = setmask(fx2, "line", "red", "thin")
;             frx2= i~(frx[,2])
;             frx2 = setmask(frx2, "line", "green", "thin")
;             ym1=max(vec( (Xact[,2])~(fx[,2])~(frx[,2]) ) )
;             ym2=min(vec( (Xact[,2])~(fx[,2])~(frx[,2]) ) )
;             flagInd[,2]=ym1*((ym1>0)*1.1+(ym1<0)*0.9)
;             flagclip[,2]= ym2*((ym2<0)*1.1+(ym2>0)*0.9)
;             setmaskp(flagInd,4, 3, 4)  
;             setmaskp(flagclip,2, 4, 4)  
;             show(disp,1,2,Xact2,fx2,frx2,flagInd,flagclip)
;             setgopt(disp,1,2, "title", "simulated Model under AO -- 2nd coord.")
;             setgopt(disp,1,2, "xlabel", "t") 
;             setgopt(disp,1,2, "ylabel", "x, rLS, Kalman") ;              
; -----------------------------------------------------------------
;   Result    Original serie of states is displayed with blue colour,
;                conventionally filtered serie is displayed with red colour.
;                robustly filtered serie is displayed with green colour.
;                Red Flags are displayed where actual AO's where simulated
;                and green ones where the robustly filtered serie was clipped.
;             (y is a lagged bivariate MA process with errors.)
; -----------------------------------------------------------------
;   Example   library("kalman")
;             library("plot")
;
;             serie = read("kalman2.dat")
;             y = serie[,2]
;             mu = #(0,0)
;             Sig = #(0,0)~#(0,0)
;             H = #(1,0)'
;             F = #(0.5,1)~#(-0.3,0)
;             R = #(1,0)~#(0,0)
;             Q = 4
;             T = dim(y)
;
;             e=0.05
;             N=10000
;             eps=0.01
;             itmax=15
;             aus=4;;   lots of output;; may be set to 0
;
;             ergLS=calibrLS(T,Sig,H,F,Q,R,e,N,eps,itmax,aus)
;                    ; gets the clipping heights to efficiency loss e
;             
;             b=ergLS.b
;
;             res= kfilter2(y,mu,Sig, H,F,Q,R)
;             fx = res.filtX
;             fy=(H*fx')'
;             res= rlsfil(y,mu,Sig, H,F,Q,R,b)
;             frx = res.filtX
;             fry=(H*frx')'
;
;             origy = serie[,1]~serie[,2]
;             origy = setmask(origy, "line", "blue", "thin")
;
;             fy = serie[,1]~fy
;             fy = setmask(fy, "line", "red", "thin")
;             fry = serie[,1]~fry
;             fry = setmask(fry, "line", "green", "thin")
;             flags = serie[,1]~(res.clipInd)
;             flags=paf(flags,flags[,2]==1)
;             flags[,2]=0
;             setmaskp(flags,4, 3, 4)
;  
;             disp = createdisplay(1,1)
;             show(disp,1,1,origy,fy,fry,flags)
;             setgopt(disp,1,1, "title", "KalmanData2 in Observation Space")
;             setgopt(disp,1,1, "xlabel", "t") 
;             setgopt(disp,1,1, "ylabel", "y, y-rLS, y-Kalman") 
;              
; -----------------------------------------------------------------
;   Result    Original serie of observations displayed with blue colour,
;             kalman-filtered serie y_t|t is displayed with red colour,
;             rLS-filtered serie y_t|t is displayed with green colour.
;                Red Flags are displayed where actual AO's where simulated
;                and where the robustly filtered serie was clipped.
;             (y is an AR(2) process with errors.)
; -----------------------------------------------------------------
;   Author    P.Ruckdeschel 991010 
; -----------------------------------------------------------------
  
  
  dimX = rows(F)                                ;get dimension of state vector
  dimY = rows(Q)                                ;get dimension of observation vector
  T = rows(y)                                   ;get number of observations (T) 

  if (dim(bs)==1)
     b=0.*y+bs
  else
     b=bs
  endif

;;;;;;;;;;;;;;;;;;;;;;;;error checks;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  Rs=svd(R)
  Qs=svd(Q)
  Sigs=svd(Sig)
  ;
  error((sum (Rs.l<0) > 0 )||(R<>R')||(cols(R) <> rows(R)), "Matrix R is not a Covariance.")
  error((sum (Sigs.l<0) > 0)||(Sig<>Sig')||(cols(Sig) <> rows(Sig)) , "Matrix Sig is not a Covariance.")
  error((sum (Qs.l<0) > 0)||(Q<>Q')||(cols(Q) <> rows(Q)) , "Matrix Q is not a Covariance..")
  error(cols(F) <> rows(F), "Matrix F is not square.")
  error(cols(R) <> rows(F), "Matrices F and R mismatch.")
  error(cols(R) <> cols(H), "Matrices R and H mismatch.")
  error(rows(Q) <> rows(H), "Matrices Q and H mismatch.")
  error(dim(mu) <> cols(F), "Vector mu and matrix F mismatch.")
  error(((dim(bs) <> T) &&(dim(bs)<>1)), "wrong dimension for clipping b's")
  j=0
 while (j<T)
      j=j+1
      error(b[j]<=0 , "uncorrect clipping height b[_j_]")
 endo

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


  PreviousPs = matrix(T+1,dimX,dimX)
  KG=matrix(T,dimX,dimY) 

  PreviousX = mu                                ;set starting state x_0|0 = mu
  PreviousP = Sig                               ;set starting covariance matrix P_0|0 = Sigma

  PreviousPs[1,,] = reshape(PreviousP,#(1,dimX,dimX))
  KG=matrix(T,dimX,dimY)*0 
 
  filtX = matrix(T,dimX) .* 0                                ;prepare the result matrix
  clipInd= matrix(T)*0


  
  i = 1                                         ;main loop
  while (i <= T)
    
; Step of the Kalman recursion - the notation follows Brockwell&Davis
    pom = (F * PreviousP * F' + R)              ;temporary variable, P_(t|t-1) in fact
    Delta = H * pom * H' + Q
    DeltaInv = ginv(Delta)
    KG0=pom* H' * DeltaInv 
    
    Innovation = (y[i,]' - H * F * PreviousX)
    pInnov=KG0* Innovation   ;temporary variable, correction of the predicted value after obseration in fact    

    norm=sqrt(sum(pInnov^2))
    norm1=(norm<b[i])*b[i]+(norm>=b[i])*norm ; necessary to avoid division by 0
    clipInd[i]=(norm>=b[i])
    norm=(norm>=b[i])*b[i]/norm1+(norm<b[i]) ; norm is a weight function, =1 if |pInnovc|<b[i],  =b[i]/|pInnovc| else   

    CurrentX = F * PreviousX +pInnov.*norm     
    CurrentP = pom - pom * H' * DeltaInv * H * pom  
    
; Step of the loop 
    
    filtX [i,] = (CurrentX)'             ;note the transposition 
    PreviousX = CurrentX
    PreviousP = CurrentP 
   
    
; storing Kalman Gain and Prediction Covariance
    PreviousPs[i+1,,] = reshape(PreviousP,#(1,dimX,dimX))
    KG[i,,]=reshape(KG0,#(1,dimX,dimY))


    i = i + 1 
  endo
endp
