proc(datt) = sure2d(dat,l,pro)
; -----------------------------------------------------------------------
; Library      wavelet
; -----------------------------------------------------------------------
; See_also     dwt invdwt stein hardthres softthres sure 
; -----------------------------------------------------------------------
; Link         ../tutorials/wavestart.html Wavelet tutorial
; -----------------------------------------------------------------------
; Macro        sure2d
; -----------------------------------------------------------------------
; Description  Sure denoises wavelet coefficients.
;              If the stein procedure is chosen, the mean
;              squared error is minimized. MSE is estimated by Stein's
;              unbiased risk estimator based on the variance of the 
;              coefficients. Sure computes then the optimal threshold for       
;              the father wavelets and each level of mother wavelets.
;              If soft or hard thresholding is chosen, only the mother
;              wavelet coefficients will be denoised.
; -----------------------------------------------------------------------
; Usage        datt = sure2d (dat,l,pro)
; Input
;   Parameter  dat 
;   Definition n x n array, wavelet coefficients
;   Parameter  l 
;   Definition integer, l^2 is the number of the father wavelet coefficients
;   Parameter  pro 
;   Definition string, "stein","soft" or "hard" indicating the chosen 
;                      wished method
; Output
;   Parameter  datt
;   Definition n x n array, thresholded wavelet coefficients
; -----------------------------------------------------------------------
; Example      ; loads the library wavelet
;              library ("wavelet")
;              n = 32
;              x = grid(0|0, (1./n)|(1./n), n|n ) 
;              ; computes a noisy step function 
;              y = 0.1*(x[,1].<=0.4) + 2*(abs(x[,1]-0.5).<0.1)
;              y = y + 0.5*(abs(x[,1]- 0.7)<0.1) 
;              y = y + 2* x[,2]^2
;              y = y + normal(n^2) *0.5
;              y = reshape(y,n|n)
;              ; computes the wavelet coefficients 
;              coeff  = fwt2 (y, 2, daubechies2,10) 
;              ; thresholds the coefficients 
;              coefft = sure2d (coeff, 2,"stein") 
;              ; computes the inverse wavelet transform 
;              ys = invfwt2 (coefft, 2, daubechies2,10) 
;              d = createdisplay(1,1)
;              tdat = x~vec(ys)
;              setmaskp(tdat, 4, 4, 8)
;              show(d,1,1, tdat, x~vec(y))
; -----------------------------------------------------------------------
; Result       ys denoised estimate of y
; -----------------------------------------------------------------------
; Author       Yurii Golubev, Stefan Sperlich 980203
; -----------------------------------------------------------------------
  n  = rows(dat) 
  n2 = n/2 
  d = vec(dat[(1+n2):n,]~dat[1:n2,(1+n2):n]) 
  vari = mean(d^2) 
vari
  switch 
  case (pro=="stein") 
    dat[(1+n2):n,] = matrix(n2,n)*0
    dat[1:n2,(1+n2):n] = matrix(n2,n2)*0
    do
      nn = n2
      n2 = n2 / 2 
      d  = vec(dat[(1+n2):nn,1:nn]~dat[1:n2,(1+n2):nn])
      {dn, risk} = stein (d, vari) 
      dat[(1+n2):nn,1:nn] = reshape(dn[1:(n2*nn)],n2|nn)
      dat[1:n2,(1+n2):nn] = reshape(dn[(1+n2*nn):rows(dn)],n2|n2)
    until (n2 <= l)
    d  = vec(dat[1:n2,1:n2])
    {dn, risk} = stein (d, vari) 
    dat[1:n2,1:n2] = reshape(dn,n2|n2)
  break 
  case (pro=="hard")
   ; fdat= dat[1:l,1:l]
    ht  = sqrt(2*vari*log(n^2))             
    dat = dat.*(abs(dat).>abs(ht))    
   ; dat[1:l,1:l] = fdat           
  break
  case (pro=="soft")
   ; fdat= dat[1:l,1:l]
    ht  = sqrt(2*vari*log(n^2))            
    dat = (dat-ht).*(dat.>ht)+(dat+ht).*(dat.<-ht)    
   ; dat[1:l,1:l] = fdat           
  break
  endsw 
  ;
  datt = dat
endp
