; -----------------------------------------------------------------
; Library        cafpe
; -----------------------------------------------------------------
;  See_also     tp/cafpe/fgrfsv
; -----------------------------------------------------------------
;   Macro       fpefsv
; -----------------------------------------------------------------
;   Keywords     nonparametric time series analysis, lag selection,
;                Final Prediction Error, CAFPE, local linear regression
; -----------------------------------------------------------------
;   Description Quantlet to compute (C)AFPE given estimates for B and C and the
;               asymptotically optimal bandwidth. If a scalar bandwidth is given
;		    it computes AFPE, CAFPE using this bandwidth. If a vector bandwidth
;		    is given, it only computes Ahat and the residuals. 
;
; -----------------------------------------------------------------
;   Usage   { afpe,cafpe,wnsig2hatc,Ahat,resid } = fpefsv(xsj,yorig,hA,xtj,kernel,estimator,Bhat,Chat,perA,Ksqint,ntotal);
;   Input
;	 Parameter          ATTENTION:
;      Definition this quantlet requires to open locling.dll, density.dll (NT)
;               or locling.so, denc.so (UNIX). This can be done with the quantlet cafpeload or 
;               directly with
;               garb = dlopen ("<path>\locling.dll") on NT,
;               garb = dlopen ("<path>/locling.so") on UNIX.

;       Parameter   xsj
;       Definition  (ny x d) matrix with explanatory variables for all dependenet observations
;       Parameter   yorig
;       Definition  (ny x 1) vector with dependent variable
;       Parameter   hA
;       Definition  scalar estimate of asymptotically optimal bandwidth or (d x 1) vector of bandwidth.
;									  For the vector bandwidth only the residuals are computed while zeros are returned
;									  for afpe,cafpe,wnsig2hatc
;       Parameter   xtj
;       Definition  (nx x d) matrix with values at which function has  to be estimated
;       Parameter   kernel
;       Definition  character variable to select kind of kernel: "gaussian": Gaussian kernel
;       Parameter   estimator
;       Definition  character variable for selection nonparametric estimator
;                   "loclin": local linear estimator; other estimators not
;                   implemented
;       Parameter   Bhat
;       Definition  scalar, estimate of B as in TY, Section 3, YT, Section 3.4
;       Parameter   Chat
;       Definition  scalar, estimate of C as in TY, Section 2, YT, Section 3.4
;       Parameter   perA
;       Definition  scalar, parameter used for screening off 0 <= per_A <= 1 percent of the observations
;                   with the lowest density for estimating A, see  eq. (3.1) and Section 5 in TY or TA

;       Parameter   Ksqint
;       Definition  scalar, constant of kernel ||K||2^2 = integral of K(u)^2 du
;       Parameter   ntotal
;       Definition  scalar, number of observations for estimating B
;   Output
;       Parameter   afpe
;       Definition  scalar, local linear estimate of AFPE in TY, eq. (3.4),
;									  for vector bandwidth a zero is returned 
;       Parameter   cafpe
;       Definition  scalar, local linear estimate of CAFPE in TY, eq. (4.1)
;									  for vector bandwidth a zero is returned 
;       Parameter   wnsig2hatc
;       Definition  scalar, local linear estimate of A using eq. (3.3) in TY
;									  for vector bandwidth a zero is returned 
;       Parameter   Ahat
;       Definition  scalar, local linear estimate of A in TY, eq. (3.1)
;       Parameter   resid
;       Definition (ny x 1) vector of estimated residuals

; -----------------------------------------------------------------
;   Reference Tjostheim, D. and Auestad, B. (TA) (1994) Nonparametric
;             identification of nonlinear time series: selecting significant lags.
;             Journal of the American Statistical Association 89, 1410-1419.
;
;             Tschernig, R. and Yang, L. (TY) (2000) Nonparametric
;             Lag Selection for Time Series,
;             Journal of Time Series Analysis,
;             forthcoming  (http://www.blackwellpublishers.co.uk)
;
;             Yang, L. & Tschernig, R. (YT) (1999) Multivariate bandwidth
;               selection for local linear regression,  Journal of the Royal Statistical
;               Society, Series B, 61, 793 - 815.
; -----------------------------------------------------------------
;   Author    Tschernig  000420
; -----------------------------------------------------------------

proc (afpe,cafpe,wnsig2hatc,Ahat,resid) = fpefsv(xsj,yorig,hA,xtj,kernel,estimator,Bhat,Chat,perA,Ksqint,ntotal);

; ------------------ initialize -------------------------------

    sigK            = 1;
    dw              = cols(xtj);


; ----------------- estimate mean functions  --------------------

    loo             = 0             ; no leave-one out
    fandg           = 0             ; only cond. mean

    switch
    case (estimator == "loclin");
        lorq    = 1;    /* local linear     */
        break
    endsw

    { mA,gsqA,denA,err } = fvllc(xsj,yorig,hA,xtj,kernel,lorq,fandg,loo);

    ghat        = mA[,1];
    betahat     = 0;

    if (err == 1);
        warning(err,"quantlet fpefsv: function estimation has missing values at given hA, 1e20 returned for AFPE, CAFPE")
        hA;
        afpe        = 1e20;
        cafpe       = 1e20;
        wnsig2hatc  = 1e20;
        Ahat        = 1e20;
        ; ?? here endp not possible
    endif

; -------------------------- compute Ahat ----------------------------

    wA          = wei(denA,perA,1);
    resid       = yorig - ghat;
    res2        = resid^2 .* wA;
    n           = rows(wA);
    Ahat        = sum(res2)/n;

; ---------------------- compute CAFPE, AFPE --------------------------

		if (rows(hA) == 1)
    	afpe      = Ahat + 2*(pdfn(0))^dw / (n*hA^dw) * Bhat;
    	cafpe     = afpe * ( 1 + dw*n^(-4/(dw+4)) );
    	wnsig2hatc= Ahat - (Ksqint^dw - 2*(pdfn(0))^dw) / (n*hA^dw) *Bhat - sigK^4*hA^4/4 * Chat;
		else
			afpe			= 0
			cafpe			= 0
			wnsig2hatc= 0
		endif
endp;
