; -----------------------------------------------------------------
; Library        cafpe
; -----------------------------------------------------------------
;  See_also     tp/cafpe/fpenpsl, tp/cafpe/cafpefull, tp/cafpe/cafpe
; -----------------------------------------------------------------
;   Macro       fpenps
; -----------------------------------------------------------------
;   Keywords     nonparametric time series analysis, lag selection,
;                Final Prediction Error, CAFPE, local linear regression
; -----------------------------------------------------------------
;   Description Quantlet to conduct lag selection for nonlinear autoregressive
;               models. It can be based on either the
;               local linear estimation of the Asymptotic Final Prediction
;               (AFPE) or a corrected version (CAFPE)
;
; -----------------------------------------------------------------
;   Usage  (crmin,crpro,crstore,crstoreadd,hstore,hstoretest) = fpenps(xraw,xresid,lagmax,volat,startval,robden,estimator,kernel,selcrit,perA,perB,searchmethod,dmax);
;   Input
;	 Parameter          ATTENTION:
;      Definition this quantlet requires to open locling.dll, density.dll (NT)
;               or locling.so, denc.so (UNIX). This can be done with the quantlet cafpeload or 
;               directly with
;               garb = dlopen ("<path>\locling.dll") on NT,
;               garb = dlopen ("<path>\density.dll") on NT,
;               garb = dlopen ("<path>/locling.so") on UNIX,
;               garb = dlopen ("<path>/denc.so") on UNIX.

;       Parameter   xraw
;       Definition  (n x 1) vector of univariate time series
;       Parameter   xresid
;       Definition  (nr x 1) vector of residuals for selecting lags
;                   of conditional volatility function; if not needed
;                   set xresid = 0
;       Parameter   lagmax
;       Definition  scalar, largest lag to be considered
;       Parameter   volat
;       Definition  "no": lag selection for conditional mean function;
;                   "resid": lag selection for conditional volatility function,
;                   the residuals of fitting a conditional mean
;                   function have to be contained in xresid
;       Parameter   startval
;       Definition  character variable, to control treatment of starting values
;                   "different": for each lag vector as few starting
;                                values are used as necessary;
;                   "same": for each lag vector the same starting value is used which
;                           is determined by the largest lag used in the lag selection
;                           quantlet xorigxe
;       Parameter   robden
;       Definition  character variable for switching on/off robustification
;                               of density estimation a la Tjostheim & Auestad (1994),
;                               see also Section 5 in TY
;                               "yes": on; "no": off
;       Parameter   estimator
;       Definition  character variable for selection nonparametric estimator
;                   "loclin": local linear estimator; other estimators not
;                   implemented
;	  Parameter   kernel
;	  Definition  character variable for selecting kernel for density estimation,
;                   must currently be "gaussian"
;       Parameter      selcrit
;       Definition  character variable to select lag selection critierion:
;                   "lqafpe":  estimating the asymptotic Final Prediction Error (AFPE) using
;                              local linear estimation and a  plug-in bandwidth based on partial
;                              local quadratic estimator
;                   "lqcafpe": estimating the corrected asymptotic Final Prediction Error (CAFPE) using
;                              local linear estimation and a  plug-in bandwidth based on partial
;                              local quadratic estimator
;       Parameter   perA
;       Definition  scalar, parameter used for screening off 0 <= per_A <= 1 percent of the observations
;                   with the lowest density for estimating A, see  eq. (3.1) and Section 5 in TY
;       Parameter   perB
;       Definition  scalar, parameter like per_A but for screening of observations
;                   per_B lowest density for estimating B
;       Parameter   searchmethod
;       Definition  character variable for determining search method
;                   "full": full search over all possible input variable combinations;
;                   "directed":  directed search a la Tjostheim & Auestad (1994)
;       Parameter   dmax
;       Definition  scalar, maximal number of lags
;   Output
;       Parameter   crmin
;       Definition     vector that stores for all considered lag combinations
;                      in the first dmax rows the selected lag vector
;                      in the dmax+1 row the estimated criterion
;                      in the dmax+2 row the computed Ahat if (C)AFPE is
;                      used
;                      in the dmax+3 row the estimated variance of the white
;                      noise process
;       Parameter   crpro
;       Definition     matrix that stores for each number of lags
;                      in the first dmax rows the selected lag vector
;                      in the dmax+1 row the estimated asymptotically
;                      optimal bandwidth for estimating A and (C)AFPE
;                      in the dmax+2 row the used bandwidth for estimating B
;                      in the dmax+3 row the used bandwidth for estimating C
;                      in the dmax+4 row the estimated criterion
;                      in the dmax+5 row the computed Ahat if (C)AFPE is
;                      used
;                      in the dmax+6 row the estimated variance of the white
;                      noise process
;       Parameter   crstore
;       Definition  matrix that stores lag vector and criterion value for
;                      all lag combinations and bandwidth values considered
;                      in the first dmax rows all considered lag vector are
;                      stored
;                      in the dmax+1 to dmax+number of bandwidths in grid
;                      the estimated criterion for each lag vector and
;                      bandwidth combination is stored
;       Parameter    crstoreadd
;       Definition   matrix that stores those criteria that
;                      are evaluated in passing for all
;                      lag combinations where all values for one lag
;                      combination are stored in one column, see program
;                      for details
;       Parameter    hstore
;       Definition    row vector that stores the bandwidths used in
;                      computing (C)AFPE  for each lag vector
;       Parameter   hstoretest
;       Definition    matrix that stores for each lag vector in one column
;                      the estimated asymptotically optimal
;                      bandwidth, hS(m+2) and hC (see section 5 in TY)

; -----------------------------------------------------------------
;   Reference Tjostheim, D. and Auestad, B. (1994) Nonparametric
;                           identification of nonlinear time series: selecting significant lags.
;                           Journal of the American Statistical Association 89, 1410-1419.
;
;                           Tschernig, R. and Yang, L. (TY) (2000), Nonparametric
;             Lag Selection for Time Series,
;             Journal of Time Series Analysis,
;             forthcoming  (http://www.blackwellpublishers.co.uk)

; -----------------------------------------------------------------
;   Author    Tschernig  000420
; -----------------------------------------------------------------

proc(crmin,crpro,crstore,crstoreadd,hstore,hstoretest) = fpenps(xraw,xresid,lagmax,volat,startval,robden,estimator,kernel,selcrit,perA,perB,searchmethod,dmax);

hAgiven		= 0					; activates computation of scalar plug-in bandwidth
; --------------organize exogenous variables -------------------------

; ========================== Initialize ===============================


if (kernel == "gaussian");
    Ksqint = 1 / (2*sqrt(pi))       ; kernel constant ||K||2^2  in TY
endif;

; ------------------- Set of lags and output matrices -----------------

if (searchmethod == "full");

    lagnor= lagmax                  ; total number of variables to be considered
    error(lagnor < dmax, "quantlet fpenps: ERROR: check dmax; the maximal number of input variables is larger than the total number of input variables");

    lagcomb    = zeros(dmax,1)      ; vector that contains number of all possible combinations of lags for a given number of lags
    dc         = 1;
    do
        lagcomb[dc] = binom(lagnor,dc);
        dc          = dc + 1;
        until (dc > dmax)

    lagcombtot= sum(lagcomb);

    crpro      = zeros(dmax,dmax+6) ; matrix that stores optimal model for
                                    ; each dc, that is all lags, bandwidths
                                    ; and criterion, bias corrected
                                    ; residual variance and residual variance
else

   lagcombtot = sum(#(lagmax:lagmax-dmax));
   crpro       = zeros(dmax,7)      ; matrix that stores optimal model for
                                    ; each dc, that is additional lag,
                                    ; bandwidths and criterion, bias corrected
                                    ; residual variance and residual variance

endif;

selmet = substr(selcrit,1,3);

crstore     = zeros(dmax+1,lagcombtot)
                                    ; matrix that stores criterion computed during all loops
crstoreadd  = zeros(4,lagcombtot)
                                    ; matrix that stores other criteria that are computed for free
crstoretest = zeros(dmax+3,lagcombtot)
                                    ; matrix that stores criterion, bias corrected residual variance
                                    ; and residual variance  computed during dc, lagsc loops
hstore      = zeros(1,lagcombtot)
                                    ; like crstore for all bandwiths hA but without lags
hstoretest  = zeros(3,lagcombtot)   ; like crstoretest for hA, hB, hC but without lags
crmin       = zeros(1,dmax+3)       ; vector with best model

lagv        = #(1:lagmax)     ; vector that includes values or all lags considered


; ================== start loop over set of lags ========================

; ---------------------- full search --------------------------------

if (searchmethod == "full");

    lagcomball    = lagfull(lagnor,dmax);

    i = 1
    do

        lags                = lagcomball[,i];
        lags                = paf(lags,lags > 0)    ;due to init. possibly more rows in lagno than sel.
        d                   = rows(lags);
        crstore[1:d,i]      = lags[1:d];
        crstoretest[1:d,i]  = lags[1:d];

       "quantlet fpenps: lags" lags
        error(sum(lags)==0,"quantlet fpenps: lags are all zeros")
        { crito,Ahat,sig2wnhat,hA,hB,hC,critgrid,garb,garb,resid } = fpenpsl(xraw,xresid,lags,lagmax,hAgiven,volat,startval,robden,estimator,kernel,selcrit,perA,perB,Ksqint);
        hAgrid                              = hA
        crstore[dmax+1:dmax+rows(hAgrid),i] = critgrid[,1];
        if (cols(critgrid) > 1)     ; store results on criteria that come for free
            crstoreadd[1:rows(hAgrid)*(cols(critgrid)-1),i] = vec(critgrid[,2:cols(critgrid)]);
        endif;
        crstoretest[dmax+1:dmax+3,i]        = crito|Ahat|sig2wnhat;
        hstore[1:rows(hAgrid),i]            = hAgrid;
        hstoretest[1:3,i]                   = hA|hB[1]|hC[1];

        i                                   = i + 1;

        until (i > lagcombtot)

; ----------create crpro matrix for full search ----------------------

    dc     = 1;
    do

        if (dc == 1);
            garb =  0;
        else;
            garb = sum(lagcomb[1:dc-1]);
        endif;

        crminih     = minind( (crstoretest[dmax+1,garb+1:garb+lagcomb[dc]])' );
        crminihv    = min(   (crstoretest[dmax+1,garb+1:garb+lagcomb[dc]])' );
        crminihv    = crminihv~crminih;

        crpro[dc,]  = ((crstoretest[1:dmax,garb+crminih])')~(hstoretest[1:3,garb+crminih]')~(crstoretest[dmax+1:dmax+3,garb+crminih]');
        dc          = dc + 1;

    until (dc > dmax)


; ---------------------- directed search --------------------------------

else

    lagdc1  = 0                     ; vector that includes all lags already selected for smaller dc;
    i       = 1                     ; counter that runs over dc and lagc loop in order
                                    ; to store all fpe's estimated
; ------------------ Start loop over number of lags ---------------

    dc         = 1;                /* counter for loop of number of lags */
    do

        "quantlet fpenps: additional lag considered now" lagdc1
             { lagno,lagnor } = lagdir1(lagmax,dc,lagdc1);

; -------- do actual estimation for all lags selected in lagno -------

        lagc      = 1               ; counter for testing loop
        do

             "quantlet fpenps:   number of lags considered, lagc" dc~lagc
            switch
            case (dc == 1)          ; in first loop start with lag 1
                lags            = lagc;
                break
            default                 ; determine vector of lags that determines total specification for following testing
                lags            = lagdc1|lagno[lagc];
                break
            endsw

            d                   = rows(lags);

            crstore[1:d,i]      = lags[1:d];
            crstoretest[1:d,i]  = lags[1:d];
            "quantlet fpenps: lags" lags

            { crito,Ahat,sig2wnhat,hA,hB,hC,critgrid,garb,garb,resid } = fpenpsl(xraw,xresid,lags,lagmax,hAgiven,volat,startval,robden,estimator,kernel,selcrit,perA,perB,Ksqint)
	    hAgrid 				= hA    

            crstore[dmax+1:dmax+rows(hAgrid),i] = critgrid[,1];
            if (cols(critgrid) > 1) ; store results on criteria that come for free
                crstoreadd[1:rows(hAgrid)*(cols(critgrid)-1),i] = vec(critgrid[,2:cols(critgrid)]);
            endif;
            crstoretest[dmax+1:dmax+3,i]        = crito|Ahat|sig2wnhat;
            hstore[1:rows(hAgrid),i]            = hAgrid;
            hstoretest[1:3,i]                   = hA|hB|hC;

            i           = i + 1;
            lagc        = lagc + 1;

        until (lagc > min((lagmax)|lagnor))

        crminih         = minind( (crstoretest[dmax+1,i-lagc+1:i-1])' );
        crminihv        = min(    (crstoretest[dmax+1,i-lagc+1:i-1])' );
        crminihv        = crminihv~crminih;
        crpro[dc,]      = crstoretest[dc,i-lagc+crminih]~(hstoretest[1:3,i-lagc+crminih]')~(crstoretest[dmax+1:dmax+3,i-lagc+crminih]');

        lagdc1      = crstoretest[1:dc,i-lagc+crminihv[,2]] ; store optimal lags
        dc   = dc + 1;
    until (dc > dmax);

      "";
    " ------ done with dc - loop ------ ";

endif;

; ========================== find overall minimum ======================


switch
case (searchmethod == "full")       ; index for crpro
    crprocs   = dmax;
    break
case (searchmethod == "directed");
    crprocs   = 1;
    break
default
    "quantlet fpenps: searchmethod must be full or directed";
    break
endsw


; ----------------------- add white noise model ---------------------------

switch
case (volat == "no");
    sig2hat    = (var(xraw));
    break
case (volat == "resid")
    sig2hat    = (var(xresid));
    break
default
    break
endsw;

switch
case (substr(selcrit,1,3) == "lq")
    crpro      = (zeros(1,crprocs)~zeros(1,3)~sig2hat*matrix(1,3))|crpro;
    break
default
        "quantlet fpenps: no valid selection criterion"
        break
endsw;

; ---------------------- get index of overall minimum and get crmin -------

crmini = minind( crpro )
crmini = crmini[,crprocs+4]


if (searchmethod == "full");
    crmin[1,1:dmax]             = crpro[crmini,1:dmax];
    crmin[1,dmax+1:dmax+3]      = crpro[crmini,crprocs+4:crprocs+6];
else
    if (crmini == 1)                ; no lags
        crmin[1,1]              = (crpro[1,1])';
    else;                           ; no zero lag
        crmin[1,1:crmini-1]     = (crpro[2:crmini,1])';
    endif;
    crmin[1,dmax+1:dmax+3]      = crpro[crmini,5:7];
endif;

                                ; delete all unused rows
crstoreadd    = paf(crstoreadd,sum(crstoreadd')' == 0); Xplsave

endp
