; -----------------------------------------------------------------
; Library        cafpe
; -----------------------------------------------------------------
;  See_also     tp/cafpe/dencp
; -----------------------------------------------------------------
;   Macro       fvllc
; -----------------------------------------------------------------
;   Keywords    multivariate local linear estimation, multivariate 
;               partial local quadratic estimation
; -----------------------------------------------------------------
;   Description Quantlet for multivariate local linear or partial local quadratic
;               estimation using C++ - routines via dlls.
;               It can estimate conditional means, conditional volatilities, 
;		first derivatives, second direct derivatives, 
;		conditional densities with full or
;               leave-one-out and density estimation with all possible
;               data or only lagged data.
;               Only the gaussian kernel can be used.
;
; -----------------------------------------------------------------
;   Usage   { mA,gsqA,denA,err } = fvllc(Xsj,Yorig,h,Xtj,kernreg,lorq,fandg,loo);
;   Input
;	 Parameter ATTENTION:

;      Definition         this quantlet requires to open locling.dll, density.dll (NT)
;               or locling.so, denc.so (UNIX). This can be done with the quantlet cafpeload or 
;               directly with
;               garb = dlopen ("<path>\locling.dll") on NT,
;               garb = dlopen ("<path>/locling.so") on UNIX.

;       Parameter   Xsj
;       Definition  (n x d) matrix of regressors
;       Parameter   Yorig
;       Definition  (ny x 1) vector of dependent data
;       Parameter    h
;       Definition  scalar or (d x 1) vector, bandwidth
;       Parameter   Xtj
;       Definition (nxest x d) matrix of data at which
;                  estimation is conducted
;       Parameter  kernreg
;       Definition must currently be "gaussian"
;       Parameter  lorq
;       Definition 1=local linear, 2=local quadratic with direct second derivatives
;       Parameter  fandg
;       Definition 0=only mean function, 1=also volatility function
;                 (be careful: volatility function may not be positive)
;       Parameter  loo
;       Definition 0=take all, 1 = leave-one-out (only works if n=ny)
;   Output
;       Parameter  mA
;       Definition (nxest x (lorq*d+1)) matrix of
;                   cond. mean, cond. first der., cond. direct
;                   sec. der. (lorq=2)
;       Parameter  gsqA
;       Definition (nxest x (lorq*d+1)) matrix of
;                  cond. volatility, cond. first der.,
;                  cond. direct sec. der. (lorq=2)
;       Parameter  denA
;       Definition (nxest x 1) vector of cond. density
;       Parameter  err
;       Definition 0 if matrix inversion in dll ok, 1 if matrix inversion failed

; -----------------------------------------------------------------
;   Reference Fan, J. and Gijbels, I. (1996) Local Polynomial Modelling
;             and Its Applications. Chapman and Hall
; -----------------------------------------------------------------
;   Author    Tschernig  000420
; -----------------------------------------------------------------


proc (mA,gsqA,denA,err) = fvllc(Xsj,Yorig,h,Xtj,kernreg,lorq,fandg,loo);


;  prepare input matrices and check inputs 

    error( !((lorq==1) || (lorq==2)), "quantlet fvllc: choose lorq = 1 or 2, see header of fvllc")
    error( !((loo==0) || (loo==1)), "quantlet fvllc: choose loo = 0 or 1, see header of fvllc")
    error( !((fandg==0) || (fandg==1)), "quantlet fvllc: choose fandg = 0 or 1, see header of fvllc")
    error( prod(h .<= 0)==1, "quantlet fvllc: choose positive bandwidth vector h");
    error( !(kernreg == "gaussian"), "quantlet fvllc: choose kerntype=gaussian")

    dc                     = cols(Xtj);

    error( (rows(Xtj) != rows(Xsj)) && (loo==1), "quantlet fvllc: be careful: leave-one-out only works if Xsj[1:rows(yorig),] and Xtj are the same");

    xorigc = Xsj;               /* do not transpose for C++-routines */
    yorigc = Yorig;
    xestc  = Xtj;


    nc             = rows(xorigc);        /* get dimension of C++ mat. */
    nyc            = rows(Yorig);
    nxestc         = rows(xestc);


   error(nyc != nc, "quantlet fvllc: check rows of Yorig and Xtj") ;

   switch
   case( rows(h) == 1) ;
                hc     = h*matrix(dc,1);        /* bandwidth vector    */
                break
   case( rows(h) == dc);
                hc     = h;
                break
   default;
                error(!((rows(h)==1) || (rows(h) == dc)), "quantlet fvllc: choose h scalar or of correct dimension");
                break
   endsw


   error(cols(xorigc) != cols(xestc),"quantlet  fvllc: xsj and xtj do not have the same number of regressors");

;     initialize output variables 

   mA                      = zeros(nxestc,lorq*dc+1);    /* cond. mean */
   gsqA           = zeros(nxestc,lorq*dc+1);    /* cond. vol */
   denA    = zeros(nxestc,1);
   seasc  = 1|1;


;   call C++ routines 

    tempp=  dlcall ("locling",xorigc,nc,dc,yorigc,nyc,xestc,nxestc,hc, seasc, mA, gsqA, denA, lorq, fandg, loo);

;   check output 

    if (isNaN(mA))
           err             = 1;
                    warning (err==1,"quantlet fvllc: inversion in local linear estimator did not work because probably the bandwidth is too small");
    else
                            err          = 0
    endif;

endp
