proc (archt) = annarchtest(y,arg1,arg2,arg3)
; ----------------------------------------------------------------------
; Library 	  times
; ----------------------------------------------------------------------
;  See_also	  archtest
;----------------------------------------------------------------------
;   Macro	  annarchtest
; ----------------------------------------------------------------------
;   Description  This macro calculates either the Lagrange Multiplier (LM) form
;                or the T-Rsquare (TR2) form of a test for conditional 
;                heteroskedasticity based on Artificial Neural Networks.
;                The first argument of the function is the vector of residuals,
;		 the second optional argument is the order of the test, the third
;                optional argument is the number of hidden units of the Neural 
;		 Network. The second optional argument is either a vector or a
;		 scalar. If no second argument is provided, the default orders
;		 are 2, 3, 4, and 5. The third argument may be either a vector,
;                or a scalar. If both second and third arguments are vectors,
;                the test will be calculated for all combinations of orders and
;		 hidden units. If no third argument is provided, the number of
;                hidden units by default is set to 3. The fourth argument is
;                the form of the test. This argument is a string of characters,
;                which can be either "LM" or "TR2". The default fourth argument
;                is "LM", i.e., the Lagrange Multiplier form.
;                The macro returns in the first column the order of the test,  
; 		 in the second column the number of hidden units, in the third
;                column the value of the test, in the fourth column the 
;                95% critical value of the null hypothesis for that order,
;                and in the fifth column the P-value of the test. 
; ----------------------------------------------------------------------
;   Usage         h = annarchtest(y {,nlags{,nodes{,testform}}})
;   Input 
;     Parameter   y 
;     Definition  vector
;     Parameter   lags
;     Definition  vector
;     Parameter   nodes
;     Definition  vector
;     Parameter   testform
;     Definition  string
;     Output
;     Parameter   h
;     Definition  vector of strings
; ----------------------------------------------------------------------
;  Example    ;Calculation of the ANN test for ARCH effects. As no optional
;             ;argument is provided, the test is evaluated for the default  
;             ;lag orders 2,3,4, and 5, for an Artificial Neural Network with
;             ;3 hidden units. The Lagrange Multiplier form is evaluated.
;             library("times") 
;             randomize(0)
;             y = genarch(0.2|0.1,0.5,400)
;             h = annarchtest(y)
;             ;h = annarchtest(y,"LM"); equivalent instruction as "LM" is the default argument
;             h 
; ----------------------------------------------------------------------
;  Result
;    Contents of h
    
;
;    [1,] Lag order  Nb of hidden Statistic  95% Critical Value  P-Value  
;    [2,]             units  
;    [3,] _______________________________________________________________ 
;    [4,]  
;    [5,]      2         3         14.38089       5.97378        0.00075 
;    [6,]      3         3          8.61509       7.80251        0.03487 
;    [7,]      4         3         13.96671       9.47844        0.00740 
;    [8,]      5         3         18.23967      11.06309        0.00266 
;
; ----------------------------------------------------------------------
;  Example    ;In this example, we calculate the TR2 form of the test for ARCH 
;             ;effects. As no other optional argument is provided, the test is
;             ;computed for the four lag orders 2,3,4, and 5, with 3 hidden 
;             ;units
;             library("times") 
;             randomize(0)
;             y = genarch(0.2|0.1,0.5,400)
;             h = annarchtest(y,"TR2")
;             h 
; ----------------------------------------------------------------------
;   Result
;     Contents of h
;
;     [1,] Lag order  Nb of hidden Statistic  95% Critical Value  P-Value  
;     [2,]             units  
;     [3,] ______________________________________________________________  
;     [4,]  
;     [5,]      2         3         10.94810       5.97378        0.00419 
;     [6,]      3         3          6.50613       7.80251        0.08942 
;     [7,]      4         3         10.54149       9.47844        0.03223 
;     [8,]      5         3         13.79811      11.06309        0.01694 
; ----------------------------------------------------------------------
;  Example    ;Calculation of the TR2 form of the ANN test for ARCH effects.
;             ;As the first numerical optional argument is equal to 3, the 
;             ;lag order is equal to three. The number of hidden units is the
;             ;default one, i.e, 3 units.
;             library("times") 
;             randomize(0)
;             y = genarch(0.2|0.1,0.5,400)
;             h = annarchtest(y,3,"TR2")
;             h 
; ----------------------------------------------------------------------
;   Result
;     Contents of h
;
;     [1,] Lag order  Nb of hidden Statistic  95% Critical Value  P-Value  
;     [2,]             units  
;     [3,] ______________________________________________________________  
;     [4,]  
;     [5,]      3         3         15.71618       7.80251        0.00130 
;
; ----------------------------------------------------------------------
;  Example    ; We calculate here the default LM form of the test for ARCH 
;             ;effects. The lag orders are given by the vector k, i.e., 
;             ;2,3, and 4, the number of hidden units is given by the second
;             ;optional argument, i.e., the scalar 4.
;             library("times") 
;             randomize(0)
;             y = genarch(0.2|0.1,0.5,400)
;             k = #(2,3,4)
;             h = annarchtest(y,k,4)
;             h 
; ----------------------------------------------------------------------
;   Result
;     Contents of h
;
;     [1,] Lag order  Nb of hidden Statistic  95% Critical Value  P-Value  
;     [2,]             units  
;     [3,] ______________________________________________________________  
;     [4,]  
;     [5,]      2         4         21.86099       5.97378        0.00002 
;     [6,]      3         4         19.27466       7.80251        0.00024 
;     [7,]      4         4         22.65275       9.47844        0.00015 

;
; ----------------------------------------------------------------------
;  Example    ;Calculation of the LM form of the test for ARCH effects.
;             ;We investigate here all the configurations with the number of
;             ;lags being equal to 2, 3, and 4, and the number of hidden units
;             ; being equal to 2,3 and 4 as well. 
;             library("times") 
;             randomize(0)
;             y = genarch(0.2|0.1,0.5,400)
;             k = #(2,3,4)
;             h = annarchtest(y,k,k)
;             h 
; ----------------------------------------------------------------------
;   Result
;     Contents of h
;
;     [ 1,] Lag order  Nb of hidden Statistic  95% Critical Value  P-Value  
;     [ 2,]             units  
;     [ 3,] ______________________________________________________________  
;     [ 4,]  
;     [ 5,]      2         2         14.35325       5.97378        0.00076 
;     [ 6,]      3         2         14.01169       7.80251        0.00289 
;     [ 7,]      4         2          2.78118       9.47844        0.59507 
;     [ 8,]      2         3         23.17970       5.97378        0.00001 
;     [ 9,]      3         3         18.55083       7.80251        0.00034 
;     [10,]      4         3         28.83286       9.47844        0.00001 
;     [11,]      2         4         20.51651       5.97378        0.00004 
;     [12,]      3         4         16.01652       7.80251        0.00113 
;     [13,]      4         4         18.16912       9.47844        0.00114 

;
; ----------------------------------------------------------------------
;  Example    ; Calculation of the TR2 form of the ANN test for ARCH effects.
;             ; The lag order is given by the first argument, i.e., 3, the
;             ;number of units are the components of the second argument k,
;             ; i.e., 2,3 and 4.
;             library("times") 
;             randomize(0)
;             y = genarch(0.2|0.1,0.5,400)
;             k = #(2,3,4)
;             h = annarchtest(y,3,k,"TR2")
;             h 
; ----------------------------------------------------------------------
;   Result
;     Contents of h
;
;     [1,] Lag order  Nb of hidden Statistic  95% Critical Value  P-Value  
;     [2,]             units  
;     [3,] ______________________________________________________________  
;     [4,]  
;     [5,]      3         2         15.11203       7.80251        0.00172 
;     [6,]      3         3         12.17561       7.80251        0.00681 
;     [7,]      3         4         16.90235       7.80251        0.00074 
;
;-----------------------------------------------------------------------
;   Reference
;     Caulet, R. & A. Peguin-Feissolle (1997): A Test for Conditional 
;           Heteroskedasticity Based on Artificial Neural Networks.
;           GREQAM DT 97A09.  
; ----------------------------------------------------------------------
;   Keywords
;      ARCH tests, neural networks, tests
; ----------------------------------------------------------------------
;   Author        Gilles Teyssiere, 980602
; ----------------------------------------------------------------------
  ;;  default parameters
  lags = #(2,3,4,5);  ARCH test lag orders
  nodes = 3        ;  number of hidden nodes
  ttest = 0        ;  the LM form of the test
  targ1 = exist(arg1)
  targ2 = exist(arg2)
  targ3 = exist(arg3)
  if (targ1 != 0)
    if (targ1 == 1)     
      lags = arg1
      if (targ2 == 1)
        nodes = arg2
        targ3 = exist(arg3)
        if (targ3 == 2)         
          if (arg3 == "TR2")
            ttest = 1
          endif
        endif
      else
        if (targ2 == 2)
          if (arg2 == "TR2")
            ttest = 1
          endif      
        endif
      endif
    else
      if (targ1 == 2)
        if (arg1 == "TR2")
          ttest = 1
        endif
      endif
    endif
  endif
  ;;
  T = rows(y)
  ssy = y^2 
  iq = 1
  iA = 1
  nbn = rows(lags)
  nbq = rows(nodes)
  archt = matrix(nbn*nbq,3)
  while (iq <= nbq) 
    in = 1
    while (in <= nbn)
      ; random generation of ANN weights
      n = lags[in]
      q = nodes[iq]    
      ; lambdas are between -2 and 2
      lambda = 2*(2*uniform(n+1,q) - matrix(n+1,q)) 
      ; regressand of the auxiliary regression
      ty = ssy[n+1:T]
      ty = ty/mean(ty) -1
      ; regressors of the auxiliary regression
      MA = matrix(T-n)~ssy[1:T-n,] 
      j = 2
      while (j <= n)
        idx = T-n+j-1
        MA = MA~ssy[j:idx,]
        j = j+1
      endo     
      tx = matrix(T-n)~(1/(1+exp(-MA* lambda[,1])))
      j = 2
      while (j <= q)
        tx = tx~(1/(1+exp(-MA*lambda[,j])))
        j = j+1
      endo
      txy = tx'*ty
      if (ttest == 0)
        test = (txy'*inv(tx'*tx)*txy)/2 ; LM FORM
      else
        test = (T-n)*txy'*inv(tx'*tx)*txy/(ty'*ty); TR2 FORM
      endif
      archt[iA,1] = n
      archt[iA,2] = q
      archt[iA,3] = test
      iA = iA+1     
      in = in+1
    endo
    iq = iq+1
  endo
  ;;
  CritOrder = matrix(nbn*nbq)*0.95
  ;;archt = archt~qfc(CritOrder,archt[,1])~(1-cdfc(archt[,3],archt[,1]))
  ;
  ; Output display
  line1 ="Lag order  Nb of hidden Statistic  95% Critical Value  P-Value "
  line2 ="            units "                          
  line3 ="______________________________________________________________ "
  st1 = string("%6.0f",archt[,1])
  st2 = string("%10.0f",archt[,2])
  st3 = string("%17.5f",archt[,3])
  st4 = string("%14.5f",qfc(CritOrder,archt[,1]))
  st5 = string("%15.5f",1-cdfc(archt[,3],archt[,1]))
  st6 = st1+st2+st3+st4+st5
  archt = line1|line2|line3|""|st6  
endp  
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; end of macro annarchtest
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
 
 
