proc (gamma, mu, sigma) = dpgp (x, k)
; -----------------------------------------------------------------
;   Library       finance
; -----------------------------------------------------------------
;   See_also      pickandsgp mlegp mlegp0 hillgp1 momentgp dpgpdiag
; -----------------------------------------------------------------
;   Macro         dpgp
; -----------------------------------------------------------------
;   Description   Drees-Pickands estimator for GP model.
; -----------------------------------------------------------------
;   Usage         {gamma,mu,sigma} = dpgp(x,k)
;   Input
;     Parameter   x
;     Definition      vector
;     Parameter   k
;     Definition      integer, number of upper extremes
;   Output
;     Parameter   gamma
;     Definition      scalar, estimated shape parameter
;     Parameter   mu
;     Definition      scalar, estimated location parameter
;     Parameter   sigma
;     Definition      scalar, estimated scale parameter	    
; -----------------------------------------------------------------
;   Example   library("finance")
;             randomize(0)
;     	      x=randx("gp",100,1)
;             m=dpgp(x,50)
;             m
; -----------------------------------------------------------------
;   Result    Contents of m.gamma
;             [1,]   1.5921 
;	            Contents of m.mu
;	            [1,]  0.31505 
;             Contents of m.sigma
;             [1,]  0.35449 
; -----------------------------------------------------------------
;   Author    Michael Thomas  990503
; -----------------------------------------------------------------
  n = rows (x)
  error (n < 4, "Date set must have at least 4 elements")
  if (k < 4)
    warning (k < 4, "Drees-Pickands (GP) needs at least 4 exceedances")
    k = 4
  endif
  if (k > n)
    warning (k > n, "Number of exceedances is too large")
    k = n
  endif

  x = sort (x)
  gamma = dpgpgamma (x, k)
  mu = x [n - k + 1]
  sigma = gpsigmaest (gamma, x [n - k + 1:n])
  r = fittail (gamma, mu, sigma, k, n)
  mu = r.mu
  sigma = r.sigma
endp
