proc(mux,beta1,beta0,sigmax,sigmae,varbeta1,varbeta0)=eivknownvarumod(alpha,w,y,sigmau)
; -----------------------------------------------------------------
;    Library	eiv
; -----------------------------------------------------------------
;    See_also eivknownvaru
; -----------------------------------------------------------------
;    Macro	eivknownvarumod
; -----------------------------------------------------------------
;    Description    eivknownvarumod presents modified moment estimates 
;                   of parameters for the measurement error models, 
;                   which has only one variable x. The variance of 
;                   measurement error sigma_u is known. All of the 
;                   variables obey normal distributions.
; ----------------------------------------------------------------------------
;   Reference	     The theoretical discussion see section 2.5 of Fuller(1987):
;                  Measurement Error Models, page 163-172.
; -----------------------------------------------------------------
;    Link         ../tutorials/eivstart.html Tutorial: EIV in XploRe 
; -----------------------------------------------------------------
;    Usage    {mux,beta1,beta0,sigmax, sigmae,varbeta1,varbeta0} =eivknownvarumod(alpha,w,y,sigmau)
;    Input
;      Parameter alpha
;      Definition  scalar
;      Parameter  w
;      Definition   n x 1 matrix, the design variables
;      Parameter  y
;      Definition   n x 1 matrix, the response
;      Parameter  sigmau
;      Definition   scalar, the variance of measurement error sigma_u
;    Output
;      Parameter  mux
;      Definition   scalar, the mean value of x
;      Parameter  beta1
;      Definition   scalar, the estimate
;      Parameter beta0
;      Definition  scalar, the estimate
;      Parameter sigmax
;      Definition  scalar, the estimate of the variance of x
;      Parameter sigmae
;      Definition  scalar, the estimate of the variance of error e
;      Parameter  varbeta1   
;      Definition  scalar, the estimate of the variance of the estimate of beta1
;      Parameter  varbeta0   
;      Definition  scalar, the estimate of the variance of the estimate of beta0
; -----------------------------------------------------------------
;  Example library("eiv")
;	   n = 100
;          randomize(n)
;	   x=9*normal(n)
;          w=x+9*normal(n)
;	   y=0.9+0.8*x+0.01*normal(n)
;          alpha=5
;	   sigmau=81
;	   gest=eivknownvarumod(alpha,w,y,sigmau)
;	   gest.mux
;	   gest.beta1
;	   gest.beta0
;	   gest.sigmax
;	   gest.sigmae
;          gest.varbeta1   ; the estimate of the variance of the estimate of beta1
;          gest.varbeta0   ; the estimate of the variance of the estimate of beta0
; -----------------------------------------------------------------
;   Result Contents of mux
;          [1,] -0.93396 
;          Contents of beta1
;          [1,]  0.84348 
;          Contents of beta0
;          [1,]  0.3736
;          Contents of sigmax
;          [1,]    64.17 
;          Contents of sigmae
;          [1,]  7.3409
;          Contents of varbeta1
;          [1,]   0.032728
;          Contents of varbeta0
;          [1,]   0.62606
; -----------------------------------------------------------------
;   Author    Hua Liang,  961215
; -----------------------------------------------------------------
     n=rows(w)
     mux=mean(w)
     mxx=var(w)
     mxy=sum((w-mean(w)).*(y-mean(y)))/(rows(w)-1)
     myy=var(y)
     lambda=inv(sigmau)*(mxx-inv(myy)*mxy^2)
     Hxx=0
      if(lambda>=1+inv(n-1))
        Hxx=mxx-sigmau
       else
        Hxx=mxx-(lambda-inv(n-1))*sigmau
       endif
     beta1=inv(Hxx+alpha*inv(n-1)*sigmau)*mxy
     beta0=mean(y)-beta1*mean(w)
     sigmax=mxx-sigmau
     sigmae=var(y)-beta1^2*sigmax
     sigmavv=inv(n-2)*(n-1)*(myy-2*beta1*mxy+beta1^2*mxx)
     varbeta1=inv(n-1)*(inv(Hxx)*sigmavv+Hxx^(-2)*(sigmau*sigmavv+beta1^2*sigmau^2))
     svv=inv(n-2)*sum((y-mean(y)-beta1*(w-mean(w))).^2)
     varbeta0=inv(n)*svv+(mean(w))^2*varbeta1
endp
