proc(beta1,beta0,varbeta1,varbeta0)=eivlinearinstr(x,w,y)
; -----------------------------------------------------------------
;    Library	eiv
; -----------------------------------------------------------------
;    See_also   eivlinearinstrvec
; -----------------------------------------------------------------
;    Macro	eivlinearinstr
; -----------------------------------------------------------------
;    Description    eivlinearinstr presents the moment estimates of the
;		    regression coefficients in the measurement error
;		    model with single predictor, which has an instrumental
;		    variable W. All of the variables obey normal
;		    distributions. All parameters are estimated by
;		    moment method in measurement error models.
; ----------------------------------------------------------------------------
;   Reference	 Fuller W. A. "Measurement Error Models", Wiley and Sons
;              1987, page 50-53.
; -----------------------------------------------------------------
;    Link         ../tutorials/eivstart.html Tutorial: EIV in XploRe 
; -----------------------------------------------------------------
;    Usage    {beta1,beta0,varbeta1,varbeta0} = eivlinearinstr(x,w,y)
;    Input
;      Parameter  x
;      Definition   n x 1 matrix, the design variables
;      Parameter  w
;      Definition   n x 1 matrix, the instrumental variables
;      Parameter  y
;      Definition   n x 1 matrix, the response
;    Output
;      Parameter  beta1
;      Definition   scalar, the estimate
;      Parameter beta0
;      Definition  scalar, the estimate
;      Parameter  varbeta1   
;      Definition  scalar, the estimate of the variance of the estimate of beta1
;      Parameter  varbeta0   
;      Definition  scalar, the estimate of the variance of the estimate of beta0
; -----------------------------------------------------------------
;  Example library("eiv")
;          n=100
;          randomize(n)
;          V=uniform(n,3).*#(4,3,4)'+#(3,3,3)'
;          y=V[,1]
;          x=V[,2]
;          w=V[,3]
;          gest=eivlinearinstr(x,w,y)
;          gest.beta1
;          gest.beta0
;          gest.varbeta1   
;          gest.varbeta0	   
; -----------------------------------------------------------------
;  Result   Contents of beta1
;           [1,]  -1.1709 
;           Contents of beta0
;           [1,]   10.135 
;           Contents of varbeta1
;           [1,]   7.3944 
;           Contents of varbeta0
;           [1,]   152.02  
; -----------------------------------------------------------------
;   Author    Hua Liang,  970730
; -----------------------------------------------------------------
     n=rows(x)
     myxw=sum((y-mean(y)~x-mean(x)).*(w-mean(w)))/(n-1)
     beta1=inv(myxw[,2])*myxw[,1]
     beta0=mean(y)-beta1*mean(x)
     svv=inv(n-2)*sum((y-mean(y)-beta1*(x-mean(x))).^2)
     varbeta1=inv(n-1)*(myxw[,2])^(-2)*var(w)*svv
     varbeta0=inv(n)*svv+mean(x)^2*varbeta1
endp
